# SPDX-FileCopyrightText: Bosch Rexroth AG
#
# SPDX-License-Identifier: MIT

import os
import json
import threading

import ctrlxdatalayer
from comm.datalayer import NodeClass
from ctrlxdatalayer.provider import Provider
from ctrlxdatalayer.provider_node import (
    ProviderNode,
    ProviderNodeCallbacks,
    NodeCallback,
)
from ctrlxdatalayer.variant import Result, Variant, VariantType
from ctrlxdatalayer.metadata_utils import (
    MetadataBuilder,
    AllowedOperation,
    ReferenceType,
)
from ctrlxdatalayer.client import Client
from sample.schema.InertialValue2 import InertialValue2
from sample.schema.InertialValue3 import InertialValue3
from helper.datalayer import convert_data_to_json
from datetime import datetime
import time
from collections import deque


class MyProviderNode:
    """

    Methode failed - Error: 408
    DL_TIMEOUT
    original error: DL_TIMEOUT

    Methode failed - Error: 400
    DL_TYPE_MISMATCH
    Can't read type information
    """

    def __init__(self, provider: Provider, nodeAddress: str, typeAddress: str,
                 initialValue: Variant, client: Client):
        """__init__"""
        self._cbs = ProviderNodeCallbacks(
            self.__on_create,
            self.__on_remove,
            self.__on_browse,
            self.__on_read,
            self.__on_write,
            self.__on_metadata,
        )
        self._providerNode = ProviderNode(self._cbs)
        self._provider = provider
        self._nodeAddress = nodeAddress
        self._typeAddress = typeAddress
        self._data = initialValue
        self._metadata = self.create_metadata()
        self._client = client
     
    def create_metadata(self) -> Variant:
        builder = MetadataBuilder(allowed=AllowedOperation.READ | AllowedOperation.WRITE | AllowedOperation.BROWSE | AllowedOperation.CREATE)
        builder = builder.set_display_name(self._nodeAddress)
        builder = builder.set_node_class(NodeClass.NodeClass.Method)
        builder.add_reference(ReferenceType.read(), self._typeAddress)
        builder.add_reference(ReferenceType.write(), self._typeAddress)
        builder.add_reference(ReferenceType.create(), self._typeAddress)
        return builder.build()

    def read_data_from_node(self, node, retries=3, delay=2):
        """Liest Daten von einem Data Layer Knoten mit mehreren Versuchen und Rückoff-Strategie"""
        attempt = 0
        while attempt < retries:
            try:
                result, data = self._client.read_sync(node)
                if result == Result.OK:
                    json_data = convert_data_to_json(data)
                    print(f"Data read from {node}: {json_data}", flush=True)
                    return json_data
                elif result == Result.TIMEOUT:
                    error_message = f"Timeout beim Lesen von {node}, Versuch {attempt + 1} von {retries}"
                    print(error_message, flush=True)
                else:
                    error_message = f"FEHLER beim Lesen von {node}: {result}"
                    print(error_message, flush=True)
                    return error_message
            except Exception as e:
                error_message = f"Fehler beim Lesen der Daten vom Knoten: {e}"
                print(error_message, flush=True)
                return error_message

            attempt += 1
            time.sleep(delay)  # Wartezeit zwischen den Versuchen

        return f"Fehler beim Lesen der Daten vom Knoten nach {retries} Versuchen: DL_TIMEOUT"

    def save_node_with_timestamp(self, node, mount_point, period, interval, retries=3, delay=2):
        """Speichert Daten von einem Data Layer Node auf einem USB-Stick mit Zeitstempeln"""

        def create_directory(path):
            if not os.path.exists(path):
                try:
                    os.makedirs(path)
                except OSError as e:
                    print(f"Fehler beim Erstellen des Verzeichnisses {path}: {e}", flush=True)
                    return False
            return True

        # Sicherstellen, dass node und mount_point Strings sind
        if isinstance(node, bytes):
            node = node.decode('utf-8')
        if isinstance(mount_point, bytes):
            mount_point = mount_point.decode('utf-8')

        # Erstellen des übergeordneten Ordners "history"
        history_folder = os.path.join(mount_point, "history")
        if not create_directory(history_folder):
            return f"Fehler beim Erstellen des Verzeichnisses {history_folder}"

        print(f"History folder created at: {history_folder}", flush=True)

        # Zielverzeichnis und Dateiname aus dem Node-Pfad erstellen
        node_path_parts = node.split('/')
        node_filename = node_path_parts[-1] + ".json"
        target_directory = os.path.join(history_folder, *node_path_parts[:-1])

        if not create_directory(target_directory):
            return f"Fehler beim Erstellen des Verzeichnisses {target_directory}"

        print(f"Target directory created at: {target_directory}", flush=True)

        dest_path = os.path.join(target_directory, node_filename)
        print(f"Destination path: {dest_path}", flush=True)

        data_collector = DataCollector(self._client, node, buffer_size=period // interval, interval=interval)
        collector_thread = threading.Thread(target= data_collector.start)
        print("Thread gestartet")
        collector_thread.start()

        # Wartezeit, um den Puffer zu füllen
        time.sleep(period)
        data_collector.stop()
        collector_thread.join()

        data_collection = data_collector.buffer.get_all()

        # Datenstruktur erstellen
        final_data_structure = {
            "Name": node_path_parts[-1],
            "Data": data_collection
        }

        try:
            # Daten in die Datei schreiben
            with open(dest_path, 'w') as file:
                json.dump(final_data_structure, file, indent=4)
            print(f"Data saved to {dest_path}", flush=True)
            return f"Daten gespeichert auf {dest_path}"
        except IOError as e:
            print(f"IOError: {e}", flush=True)
            return f"Fehler beim Speichern der Daten: {e}"
        except Exception as e:
            print(f"Unexpected error: {e}", flush=True)
            return f"Ein unerwarteter Fehler ist aufgetreten: {e}"

    def save_node_to_usb(self, node, mount_point):
        """Speichert Daten von einem Data Layer Knoten auf einem USB-Stick"""
    
        # Sicherstellen, dass node und mount_point Strings sind
        if isinstance(node, bytes):
            node = node.decode('utf-8')
        if isinstance(mount_point, bytes):
            mount_point = mount_point.decode('utf-8')

        # Zielverzeichnis und Dateiname aus dem Node-Pfad erstellen
        node_path_parts = node.split('/')
        node_filename = node_path_parts[-1] + ".json"
        target_directory = os.path.join(mount_point, *node_path_parts[:-1])

        # Sicherstellen, dass das Zielverzeichnis existiert
        if not os.path.exists(target_directory):
            try:
                os.makedirs(target_directory)
            except OSError as e:
                return f"Fehler beim Erstellen des Verzeichnisses {target_directory}: {e}"

        dest_path = os.path.join(target_directory, node_filename)

        # Daten vom Knoten lesen
        data_to_save = self.read_data_from_node(node)
        if data_to_save.startswith("Fehler"):
            return data_to_save  # Fehler beim Lesen des Knotens zurückgeben

        # Nur der letzte Teil des Node-Pfads als Name
        node_name = node_path_parts[-1]

        # Datenstruktur erstellen
        data_structure = {
            "Name": node_name,
            "Value": json.loads(data_to_save)
        }

        try:
            # Daten in die Datei schreiben
            with open(dest_path, 'w') as file:
                json.dump(data_structure, file, indent=4)
            return f"Daten gespeichert auf {dest_path}"
        except IOError as e:
            return f"Fehler beim Speichern der Daten: {e}"
        except Exception as e:
            return f"Ein unerwarteter Fehler ist aufgetreten: {e}"

    def register_node(self):
        """register_node"""
        return self._provider.register_node(self._nodeAddress, self._providerNode)

    def unregister_node(self):
        """unregister_node"""
        self._provider.unregister_node(self._nodeAddress)
        self._metadata.close()
        self._data.close()

    def set_value(self, value: Variant):
        """set_value"""
        self._data = value

    def __on_create(self, userdata: ctrlxdatalayer.clib.userData_c_void_p, address: str, data: Variant, cb: NodeCallback):
        """__on_create"""
        print("__on_create() address:", address, "userdata:", userdata, flush=True)
        cb(Result.OK, data)

    def __on_remove(self, userdata: ctrlxdatalayer.clib.userData_c_void_p, address: str, cb: NodeCallback):
        """__on_remove"""
        print("__on_remove() address:", address, "userdata:", userdata, flush=True)
        cb(Result.UNSUPPORTED, None)

    def __on_browse(self, userdata: ctrlxdatalayer.clib.userData_c_void_p, address: str, cb: NodeCallback):
        """__on_browse"""
        print("__on_browse() address:", address, "userdata:", userdata, flush=True)
        with Variant() as new_data:
            new_data.set_array_string([])
            cb(Result.OK, new_data)

    def __on_read(self, userdata: ctrlxdatalayer.clib.userData_c_void_p, address: str, data: Variant, cb: NodeCallback):
        """__on_read"""
        print("__on_read() address:", address, "data:", self._data, "userdata:", userdata, flush=True)
        
        # Überprüfen, ob _data ein Array von Strings ist
        if self._data.get_type() == VariantType.ARRAY_STRING:
            # Konvertieren des Arrays in einen einzelnen String
            combined_data = " ".join(self._data.get_array_string())
            new_data = Variant()
            new_data.set_string(combined_data)
        else:
            new_data = self._data
        
        cb(Result.OK, new_data)

    def __on_write(self, userdata: ctrlxdatalayer.clib.userData_c_void_p, address: str, data: Variant, cb: NodeCallback):
        print("__on_write() address:", address, "data:", data, "userdata:", userdata, flush=True)

        # Check if the data type is Flatbuffers
        if data.get_type() != VariantType.FLATBUFFERS:
            print("Flatbuffer Error")
            cb(Result.TYPE_MISMATCH, None)
            return

        # Extract the flatbuffer data
        buffer = data.get_flatbuffers()
        
        

        if "doc-safe" in address:
            root = InertialValue2.GetRootAs(buffer, 0)
            try:
                node = root.Node()
                mount = root.Mount()
                if isinstance(node, bytes):
                    node = node.decode('utf-8')
                if isinstance(mount, bytes):
                    mount = mount.decode('utf-8') 
            except ValueError:
                print("doc-safe Value Error")
                cb(Result.TYPE_MISMATCH, None)
                return
            
            result_list = self.save_node_to_usb(node, mount)
            print("Saved node result: ", result_list)
            cb(Result.OK, None)
          

        if "history" in address:
            root = InertialValue3.GetRootAs(buffer, 0)
            try:
                node = root.Node()
                mount = root.Mount()
                period = root.Period()
                interval = root.Interval()

                if isinstance(node, bytes):
                    node = node.decode('utf-8')
                if isinstance(mount, bytes):
                    mount = mount.decode('utf-8')

                # Sicherstellen, dass period und interval Ganzzahlen sind
                if not isinstance(period, int):
                    period = int(period)
                if not isinstance(interval, int):
                    interval = int(interval)

                # Timeout-Wert berechnen
                timeout_value = 1000 * period * interval

                # Setzen des Timeouts für die Methode
                result = self._provider.set_timeout_node(self._providerNode, timeout_value)

                if result == Result.OK:
                    print(f"Timeout-Wert für Methode {self._nodeAddress} erfolgreich auf {timeout_value} ms gesetzt")
                else:
                    print(f"Fehler beim Setzen des Timeout-Werts für Methode {self._nodeAddress}: {result}")
                    cb(Result.FAILED, None)
                    return


            except (ValueError, TypeError) as e:
                print(f"Error while decoding values: {e}", flush=True)
                cb(Result.TYPE_MISMATCH, None)
                return
        
            # Mehrere Versuche mit Rückoff-Strategie
            retries = 3
            delay = 2
            for attempt in range(retries):
                result_list = self.save_node_with_timestamp(node, mount, period, interval)
                if "Fehler" not in result_list:
                    print("Saved Node result: ", result_list)
                    cb(Result.OK, None)
                    break
                else:
                    print(f"Attempt {attempt + 1} failed: {result_list}", flush=True)
                    time.sleep(delay)
            else:
                # Alle Versuche fehlgeschlagen
                cb(Result.FAILED, None)


        # Create a new Variant with the result
        result_variant = Variant()
        result_variant.set_array_string(result_list)
        self._data = result_variant
        cb(Result.OK, self._data)

    def __on_metadata(self, userdata: ctrlxdatalayer.clib.userData_c_void_p, address: str, cb: NodeCallback):
        """__on_metadata"""
        print("__on_metadata() address:", address, flush=True)
        cb(Result.OK, self._metadata)


class RingBuffer:
    def __init__(self, size):
        self.size = size
        self.buffer = deque(maxlen= size)

    def add(self, item):
        self.buffer.append(item)

    def get_all(self):
        return list(self.buffer)


class DataCollector:
    def __init__(self, client: Client, node, buffer_size=5, interval=1):
        self.client = client
        self.node = node
        self.interval= interval
        self.buffer= RingBuffer(buffer_size)
        self.running = False

    def start(self):
        """Startet den DataCollector, um Daten in regelmäßigen Abständen zu lesen und in den Puffer 
        zu speichern"""
        self.running = True
        while self.running:
            # Daten vom Knoten lesen und einen  Zeitstempel hinzufügen
            data = self.fetch_data_from_node(self.node)
            timestamp = datetime.now().strftime("%d.%m.%Y, %H:%M:%S")
            self.buffer.add({"Time": timestamp, "Value": data})
            # Wartet das angegebene Intervall ab
            time.sleep(self.interval)

    def stop(self):
        self.running = False

    def fetch_data_from_node(self, node):
        """Liest Daten von einem Data Layer Knoten mit mehreren Versuchen und Rückoff-Strategie"""
        retries = 3 #Versuche
        delay = 2 #Wartezeit zwischen den Versuchen
        for attempt in range(retries):
            try:
                result, data = self.client.read_sync(node)
                if result == Result.OK:
                    #Konvertiere die Daten in Json und gebe sie zurück 
                    json_data = convert_data_to_json(data)
                    return json_data
                elif result == Result.TIMEOUT:
                    #Wartezeit, bevor ein weiterer Versuch unternommen wird
                    time.sleep(delay)
            except Exception as e:
                return f"Fehler beim Lesen der Daten vom Knoten: {e}"

        return f"Fehler beim Lesen der Daten vom Knoten nach {retries} Versuchen: DL_TIMEOUT"