#!/usr/bin/env python3
# SPDX-FileCopyrightText: Bosch Rexroth AG
#
# SPDX-LicenseCopyrightText: MIT

import os
import time



import faulthandler 
import signal

import ctrlxdatalayer
from ctrlxdatalayer.variant import Result

from helper.ctrlx_datalayer_helper import get_client, get_provider
from ctrlxdatalayer.variant import Variant


from app.provider_node_alldata import MyProviderNode



__close_app = False
addr_root="sdk/usb"
bfbs_file = "sampleSchema.bfbs"
bfbs_file2 = "sampleSchema2.bfbs"
bfbs_file3 = "sampleSchema3.bfbs"



def handler(signum, frame):
    global __close_app
    __close_app = True



def usb_connection(
        provider: ctrlxdatalayer.provider.Provider,
        client: ctrlxdatalayer.client.Client):
    
    initial_value = Variant()
    initial_value.set_array_string(["/media/sdb1"])  # Set initial value as empty array
    print("Jetzt krachts")
    
    # Setze Pfade basierend auf der Umgebung
    snap_path = os.getenv("SNAP")
    bfbs_paths = {
        "types/sample/inertialvalue": os.path.join(snap_path if snap_path else "./bfbs/", bfbs_file),
        "types/sample/inertialvalue2": os.path.join(snap_path if snap_path else "./bfbs/", bfbs_file2),
        "types/sample/inertialvalue3": os.path.join(snap_path if snap_path else "./bfbs/", bfbs_file3)
    }

    # Registriere Typen
    for type_name, bfbs_path in bfbs_paths.items():
        result = provider.register_type(type_name, bfbs_path)
        if result != Result.OK:
            print(f"WARNING Registering {type_name} failed with: {result}", flush=True)

    # Erstelle und registriere Nodes
    nodes_info = [
        ("sdk/usb/doc-safe", "types/sample/inertialvalue2"),
        ("sdk/usb/history", "types/sample/inertialvalue3")
    ]

    for node_address, type_name in nodes_info:
        node = MyProviderNode(provider, node_address, type_name, initial_value, client)
        result = node.register_node()
        if result != Result.OK:
            print(f"WARNING Registering node {node_address} failed with: {result}", flush=True)


    print("INFO Running endless loop...", flush=True)
    while provider.is_connected() and not __close_app:
        time.sleep(1.0)

    print("ERROR ctrlX Data Layer Provider is disconnected", flush=True)


def main():
    signal.signal(signal.SIGINT,handler)
    signal.signal(signal.SIGTERM, handler)
    signal.signal(signal.SIGABRT, handler)

    print()
    print("===========================================================================")
    print("ctrlX Application in Python:")
    print("- Reads data from USB ")
    print("- Provides USB data as Data Layer nodes")
    print()
    print("Will be restarted by the snap system on error.")
    print("===========================================================================")
    print()

    faulthandler.enable()

    system = ctrlxdatalayer.system.System("")
    system.start(False)

    ip = "192.168.1.1"  # "10.0.2.2" for ctrlX COREvirtual with Port Forwarding
    user = "boschrexroth"
    password = "boschrexroth"
    ssl_port = 443  # 8443 for ctrlX COREvirtual with Port Forwarding

    client, client_connection_string = get_client(system, ip, user, password, ssl_port)
    if client is None:
        print(f"ERROR Could get ctrlX Datalayer client connection: {client_connection_string}", flush=True)
        system.stop(False)
        return

    print(f"INFO ctrlX Datalayer client connection succeeded: {client_connection_string}", flush=True)

    provider, provider_connection_string = get_provider(system, ip, user, password, ssl_port)
    if provider is None:
        print(f"ERROR Could get ctrlX Datalayer provider connection: {provider_connection_string}", flush=True)
        provider.close()
        client.close()
        system.stop(False)
        return

    print(f"INFO ctrlX Datalayer provider connection succeeded: {provider_connection_string}", flush=True)
    usb_connection(provider,client)

    #Endless loop
    while client.is_connected():
        time.sleep(5.0)
    print("Error Data Layer client is disconnected -exiting application. Will be restarted automatically. ")
    


if __name__ == '__main__':
    main()


    

    