/**
 * MIT License
 *
 * Copyright (c) 2020-2022 Bosch Rexroth AG
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

// The example app datalayer.register.node creates a new provider
// with node 'sample-cpp-registermethod' and different type elements to the ctrlX Data Layer.

#include <iostream>
#include <filesystem>
#include <signal.h>
#include <thread>

#include "comm/datalayer/datalayer.h"
#include "comm/datalayer/datalayer_system.h"

#include "ctrlx_datalayer_helper.h"


#include "comm/datalayer/scope_generated.h"



// Add some signal Handling so we are able to abort the program with sending sigint
bool endProcess = false;

static void sigHandler(int sig, siginfo_t *siginfo, void *context)
{
  endProcess = true;
}

using comm::datalayer::IProviderNode;

// Basic class Provider node interface for providing data to the system
class MyProviderNode : public IProviderNode
{
private:
  comm::datalayer::Variant _data;



public:
  MyProviderNode(comm::datalayer::Variant data)
  {
    _data = data;
    //createMetadata();
  };

  virtual ~MyProviderNode() override{};

  // Create function of an object. Function will be called whenever a object should be created.
  virtual void onCreate(const std::string &address, const comm::datalayer::Variant *data, const comm::datalayer::IProviderNode::ResponseCallback &callback) override
  {
    callback(comm::datalayer::DlResult::DL_FAILED, nullptr);
  }

  // Read function of a node. Function will be called whenever a node should be read.
  virtual void onRead(const std::string &address, const comm::datalayer::Variant *data, const comm::datalayer::IProviderNode::ResponseCallback &callback) override
  {
    callback(comm::datalayer::DlResult::DL_FAILED, nullptr);
  }

  // Write function of a node. Function will be called whenever a node should be written.
  virtual void onWrite(const std::string &address, const comm::datalayer::Variant *data, const comm::datalayer::IProviderNode::ResponseCallback &callback) override
  {
      //method sqrt was called. now calculating the sqrt

      comm::datalayer::Variant resultVariant;
      if(data->getType() != comm::datalayer::VariantType::FLOAT32){
        callback(comm::datalayer::DlResult::DL_TYPE_MISMATCH, nullptr);
      }
      if(double(*data) <= 0){
        callback(comm::datalayer::DlResult::DL_TYPE_MISMATCH, nullptr);
      }
      double sqrtResult = sqrt(double(*data));
      resultVariant.setValue(sqrtResult);
      callback(comm::datalayer::DlResult::DL_OK, &resultVariant);
  }

  // Remove function for an object. Function will be called whenever a object should be removed.
  virtual void onRemove(const std::string &address, const comm::datalayer::IProviderNode::ResponseCallback &callback) override
  {
    callback(comm::datalayer::DlResult::DL_FAILED, nullptr);
  }

  // Browse function of a node. Function will be called to determine children of a node.
  virtual void onBrowse(const std::string &address, const comm::datalayer::IProviderNode::ResponseCallback &callback) override
  {
    callback(comm::datalayer::DlResult::DL_FAILED, nullptr);
  }

  // Read function of metadata of an object. Function will be called whenever a node should be written.
  virtual void onMetadata(const std::string &address, const comm::datalayer::IProviderNode::ResponseCallback &callback) override
  {
    // Take metadata from metadata.mddb
    callback(comm::datalayer::DlResult::DL_FAILED, nullptr);
  }
};

int main()
{
  comm::datalayer::DatalayerSystem datalayerSystem;
  comm::datalayer::DlResult result;
  // Starts the ctrlX Data Layer system without a new broker because one broker is already running on ctrlX device
  datalayerSystem.start(false);

  std::cout << "INFO Register 'sample-cpp-registermethod' as root element with 1 node 'sqrt'" << std::endl;

  //comm::datalayer::IProvider *provider = getProvider(datalayerSystem); // ctrlX CORE (virtual) with ip 192.168.1.1
  comm::datalayer::IProvider *provider = getProvider(datalayerSystem, "10.0.2.2", "boschrexroth", "boschrexroth", 8443); // ctrlX CORE virtual with port forwarding
  if (provider == nullptr)
  {
    std::cout << "ERROR Getting provider connection failed." << std::endl;
    datalayerSystem.stop(false);
    return 1;
  }

  //create a client, which can create the data layer scope
  //comm::datalayer::IClient *client = getClient(datalayerSystem); //ctrrlX CORE (virtual) with ip 192.168.1.1
  comm::datalayer::IClient *client = getClient(datalayerSystem, "10.0.2.2", "boschrexroth", "boschrexroth", 8443); // ctrlX CORE virtual with port forwarding
  if (client == nullptr)
  {
    std::cout << "ERROR Getting client connection failed." << std::endl;
    datalayerSystem.stop(false);
    return 1;
  }

  comm::datalayer::Variant scopeVariant;
  flatbuffers::FlatBufferBuilder builder;
  auto str = builder.CreateString("sample-cpp-registermethod/**");
  std::vector<flatbuffers::Offset<flatbuffers::String>> permissionR = {str};
  auto scope = comm::datalayer::CreateScopeDirect(builder, 
                                            "datalayer.sqrt",
                                            "Use sqrt method", 
                                            "Allows you to use the sqrt method provided by register.datalayer.method",
                                            nullptr,
                                            nullptr,
                                            nullptr,
                                            &permissionR);
  builder.Finish(scope);
  scopeVariant.shareFlatbuffers(builder);
  result = client->createSync("datalayer/security/scopes", &scopeVariant);



  // Register a node as float value
  comm::datalayer::Variant sqrt;
  sqrt.setValue(4.0f);
  std::cout << "INFO Register node 'sample-cpp-registermethod/sqrt'  " << std::endl;
  result = provider->registerNode("sample-cpp-registermethod/sqrt", new MyProviderNode(sqrt));
  if (STATUS_FAILED(result))
  {
    std::cout << "WARN Register node 'sample-cpp-registermethod/sqrt' failed with: " << result.toString() << std::endl;
  }

  // Register Mddb file for the Metadata database on the ctrlX Data Layer
  auto snapDir = snapPath();
  std::filesystem::path dir;
  if (snapDir == nullptr)
    dir = "compiled"; // Build environment: Compiled files are stored into that sub directory
  else
    dir = snapDir; // Snap environment: Compiled files are stored into the $SNAP directory

  std::filesystem::path fileMddb = dir / "metadata.mddb";
  result = provider->registerType("datalayer", fileMddb);
  if (STATUS_FAILED(result))
  {
    std::cout << "WARN Register " << fileMddb << " failed with: " << result.toString() << std::endl;
  }


  // Prepare signal structure to interrupt the endless loop with ctrl + c
  struct sigaction act;
  memset(&act, '\0', sizeof(act));
  act.sa_sigaction = &sigHandler;
  act.sa_flags = SA_SIGINFO;
  sigaction(SIGINT, &act, NULL);

  std::cout << "INFO Running endless loop - end with Ctrl+C" << std::endl;
  while (endProcess == false)
  {
    if (provider->isConnected() == false)
    {
      std::cout << "ERROR Datalayer connection broken!" << std::endl;
      break;
    }

    std::this_thread::sleep_for(std::chrono::milliseconds(1000));
  }

  std::cout << "INFO Exiting application" << std::endl;
  if (isSnap())
  {
    std::cout << "INFO Restarting automatically" << std::endl;
  }
 
  provider->unregisterNode("sample-cpp-registermethod/sqrt");

  // Clean up datalayer instances so that process ends properly
  provider->stop();
  delete provider;

  datalayerSystem.stop(false); // Attention: Doesn't return if any provider or client instance is still runnning

  return endProcess ? 0 : 1;
}