export interface IStringable {
    toString: () => string;
}
export interface IBaseExpression<T> {
    /**
     * Inserts a tag name in the expression.
     */
    tag: (name: string) => T;
    /**
     * Inserts a field name in the expression.
     */
    field: (name: string) => T;
    /**
     * Chains on a value to the expression. An error will be thrown if the
     * value is a type we can't represent in InfluxQL, primarily `null` or
     * `undefined.`
     */
    value: (value: any) => T;
}
export interface IExpressionHead extends IBaseExpression<IBinaryOp> {
}
export interface IExpressionTail extends IBaseExpression<IExpressionHead> {
}
export interface IBinaryOp {
    /**
     * Adds an 'AND' operator
     */
    and: IExpressionTail;
    /**
     * Adds an 'OR' operator
     */
    or: IExpressionTail;
    /**
     * Adds a '+' addition symbol
     */
    plus: IExpressionTail;
    /**
     * Adds a '*' multiplication symbol
     */
    times: IExpressionTail;
    /**
     * Adds a '-' subtraction symbol
     */
    minus: IExpressionTail;
    /**
     * Adds a '/' division symbol
     */
    div: IExpressionTail;
    /**
     * Adds a '=' symbol
     */
    equals: IExpressionTail;
    /**
     * Adds a '=~' comparator to select entries matching a regex.
     */
    matches: IExpressionTail;
    /**
     * Adds a '!~' comparator to select entries not matching a regex.
     */
    doesntMatch: IExpressionTail;
    /**
     * Adds a '!=' comparator to select entries not equaling a certain value.
     */
    notEqual: IExpressionTail;
    /**
     * Adds a '>' symbol
     */
    gt: IExpressionTail;
    /**
     * Adds a '>=' symbol
     */
    gte: IExpressionTail;
    /**
     * Adds a '<' symbol
     */
    lt: IExpressionTail;
    /**
     * Adds a '<=' symbol
     */
    lte: IExpressionTail;
}
/**
 * Expression is used to build filtering expressions, like those used in WHERE
 * clauses. It can be used for fluent and safe building of queries using
 * untrusted input.
 *
 * @example
 * e => e
 *   .field('host').equals.value('ares.peet.io')
 *   .or
 *   .field('host').matches(/example\.com$/)
 *   .or
 *   .expr(e => e
 *     .field('country').equals.value('US')
 *     .and
 *     .field('state').equals.value('WA'));
 *
 * // Generates:
 * // "host" = 'ares.peet.io' OR "host" ~= /example\.com$/ OR \
 * //   ("county" = 'US' AND "state" = 'WA')
 */
export declare class Expression implements IExpressionHead, IExpressionTail, IBinaryOp {
    private readonly _query;
    /**
     * Inserts a tag reference into the expression; the name will be
     * automatically escaped.
     * @param name
     * @return
     */
    tag(name: string): this;
    /**
     * Inserts a field reference into the expression; the name will be
     * automatically escaped.
     * @param name
     * @return
     */
    field(name: string): this;
    /**
     * Inserts a subexpression; invokes the function with a new expression
     * that can be chained on.
     * @param fn
     * @return
     * @example
     * e.field('a').equals.value('b')
     *   .or.expr(e =>
     *     e.field('b').equals.value('b')
     *     .and.field('a').equals.value('c'))
     *   .toString()
     * // "a" = 'b' OR ("b" = 'b' AND "a" = 'c')
     */
    exp(fn: (e: Expression) => Expression): this;
    /**
     * Value chains on a value to the expression.
     *
     *  - Numbers will be inserted verbatim
     *  - Strings will be escaped and inserted
     *  - Booleans will be inserted correctly
     *  - Dates will be formatted and inserted correctly, including INanoDates.
     *  - Regular expressions will be inserted correctly, however an error will
     *    be thrown if they contain flags, as regex flags do not work in Influx
     *  - Otherwise we'll try to call `.toString()` on the value, throwing
     *    if we cannot do so.
     *
     * @param value
     * @return
     */
    value(value: any): this;
    /**
     * Chains on an AND clause to the expression.
     */
    get and(): this;
    /**
     * Chains on an OR clause to the expression.
     */
    get or(): this;
    /**
     * Chains on a `+` operator to the expression.
     */
    get plus(): this;
    /**
     * Chains on a `*` operator to the expression.
     */
    get times(): this;
    /**
     * Chains on a `-` operator to the expression.
     */
    get minus(): this;
    /**
     * Chains on a `/` operator to the expression.
     */
    get div(): this;
    /**
     * Chains on a `=` conditional to the expression.
     */
    get equals(): this;
    /**
     * Chains on a `=~` conditional to the expression to match regexes.
     */
    get matches(): this;
    /**
     * Chains on a `!`` conditional to the expression to match regexes.
     */
    get doesntMatch(): this;
    /**
     * Chains on a `!=` conditional to the expression.
     */
    get notEqual(): this;
    /**
     * Chains on a `>` conditional to the expression.
     */
    get gt(): this;
    /**
     * Chains on a `>=` conditional to the expression.
     */
    get gte(): this;
    /**
     * Chains on a `<` conditional to the expression.
     */
    get lt(): this;
    /**
     * Chains on a `<=` conditional to the expression.
     */
    get lte(): this;
    /**
     * Converts the expression into its InfluxQL representation.
     * @return
     */
    toString(): string;
}
/**
 * Measurement creates a reference to a particular measurement. You can
 * reference it solely by its name, but you can also specify the retention
 * policy and database it lives under.
 *
 * @example
 * m.name('my_measurement') // "my_measurement"
 * m.name('my_measurement').policy('one_day') // "one_day"."my_measurement"
 * m.name('my_measurement').policy('one_day').db('mydb') // "mydb"."one_day"."my_measurement"
 */
export declare class Measurement {
    private _parts;
    /**
     * Sets the measurement name.
     * @param name
     * @return
     */
    name(name: string): this;
    /**
     * Sets the retention policy name.
     * @param retentionPolicy
     * @return
     */
    policy(retentionPolicy: string): this;
    /**
     * Sets the database name.
     * @param db
     * @return
     */
    db(db: string): this;
    /**
     * Converts the measurement into its InfluxQL representation.
     * @return
     * @throws {Error} if a measurement name is not provided
     */
    toString(): string;
}
export declare type measurement = {
    measurement: string | ((m: Measurement) => IStringable);
};
export declare type where = {
    where: string | ((e: IExpressionHead) => IStringable);
};
export declare function parseMeasurement(q: measurement): string;
export declare function parseWhere(q: where): string;
