#include <QtEndian>
#include "my_bsec.h"
#include <QDebug>
#include <QDataStream>

My_BSEC::My_BSEC(QObject *parent) : QObject(parent)
{
    this->init();
    qDebug() << "BSEC instance  constructed!";
}

void My_BSEC::init(){
    const uint8_t bsec_config_iaq[454] = // 300s 28d 3v3
         {0,8,4,1,61,0,0,0,0,0,0,0,174,1,0,0,48,0,1,0,0,168,19,73,64,49,119,76,0,0,225,68,137,65,0,191,205,204,204,190,0,0,64,191,225,122,148,190,0,0,0,0,216,85,0,100,0,0,0,0,0,0,0,0,28,0,2,0,0,244,1,225,0,25,0,0,128,64,0,0,32,65,144,1,0,0,112,65,0,0,0,63,16,0,3,0,10,215,163,60,10,215,35,59,10,215,35,59,9,0,5,0,0,0,0,0,1,88,0,9,0,229,208,34,62,0,0,0,0,0,0,0,0,218,27,156,62,225,11,67,64,0,0,160,64,0,0,0,0,0,0,0,0,94,75,72,189,93,254,159,64,66,62,160,191,0,0,0,0,0,0,0,0,33,31,180,190,138,176,97,64,65,241,99,190,0,0,0,0,0,0,0,0,167,121,71,61,165,189,41,192,184,30,189,64,12,0,10,0,0,0,0,0,0,0,0,0,229,0,254,0,2,1,5,48,117,100,0,44,1,112,23,151,7,132,3,197,0,92,4,144,1,64,1,64,1,144,1,48,117,48,117,48,117,48,117,100,0,100,0,100,0,48,117,48,117,48,117,100,0,100,0,48,117,48,117,100,0,100,0,100,0,100,0,48,117,48,117,48,117,100,0,100,0,100,0,48,117,48,117,100,0,100,0,44,1,44,1,44,1,44,1,44,1,44,1,44,1,44,1,44,1,44,1,44,1,44,1,44,1,44,1,8,7,8,7,8,7,8,7,8,7,8,7,8,7,8,7,8,7,8,7,8,7,8,7,8,7,8,7,112,23,112,23,112,23,112,23,112,23,112,23,112,23,112,23,112,23,112,23,112,23,112,23,112,23,112,23,255,255,255,255,255,255,255,255,220,5,220,5,220,5,255,255,255,255,255,255,220,5,220,5,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,48,117,0,0,0,0,92,90,0,0};

    uint8_t work_buffer[BSEC_MAX_PROPERTY_BLOB_SIZE] = {0};
    bsec_init();
    bsec_set_configuration(bsec_config_iaq,484,work_buffer,sizeof(work_buffer));
    sensorConfiguration();
    qDebug() << "BSEC instance init!";
}

void My_BSEC::sensorConfiguration(){
    uint8_t n_requested_virtual_sensors = 6;
    bsec_sensor_configuration_t requested_virtual_sensors[6];


    requested_virtual_sensors[0].sensor_id = BSEC_OUTPUT_IAQ;
    requested_virtual_sensors[0].sample_rate = BSEC_SAMPLE_RATE_ULP;
    requested_virtual_sensors[1].sensor_id = BSEC_OUTPUT_RAW_TEMPERATURE;
    requested_virtual_sensors[1].sample_rate = BSEC_SAMPLE_RATE_ULP;
    requested_virtual_sensors[2].sensor_id = BSEC_OUTPUT_RAW_HUMIDITY;
    requested_virtual_sensors[2].sample_rate = BSEC_SAMPLE_RATE_ULP;
    requested_virtual_sensors[3].sensor_id = BSEC_OUTPUT_STATIC_IAQ;
    requested_virtual_sensors[3].sample_rate = BSEC_SAMPLE_RATE_ULP;
    requested_virtual_sensors[4].sensor_id = BSEC_OUTPUT_CO2_EQUIVALENT;
    requested_virtual_sensors[4].sample_rate = BSEC_SAMPLE_RATE_ULP;
    requested_virtual_sensors[5].sensor_id = BSEC_OUTPUT_BREATH_VOC_EQUIVALENT;
    requested_virtual_sensors[5].sample_rate = BSEC_SAMPLE_RATE_ULP;

    // Allocate a struct for the returned physical sensor settings
    bsec_sensor_configuration_t required_sensor_settings[BSEC_MAX_PHYSICAL_SENSOR];
    uint8_t  n_required_sensor_settings = BSEC_MAX_PHYSICAL_SENSOR;

    // Call bsec_update_subscription() to enable/disable the requested virtual sensors
    bsec_update_subscription(requested_virtual_sensors, n_requested_virtual_sensors, required_sensor_settings, &n_required_sensor_settings);
    this->sensorControl();
    qDebug() << "BSEC instance sensor config!";
}

void My_BSEC::sensorControl()
{
    bsec_bme_settings_t lastBSECBMESettings;
    memcpy(&lastBSECBMESettings,&BSECBMESettings,sizeof (BSECBMESettings));
    bsec_sensor_control(0,&BSECBMESettings);
    //lastBSECBMESettings.next_call = BSECBMESettings.next_call;
    if (memcmp(&lastBSECBMESettings,&BSECBMESettings,sizeof (BSECBMESettings))){
        qDebug() << "BSEC new BME settings calculated:";
        if (BSECBMESettings.next_call/1000000000 == 3){
            qDebug() << "LP Mode cylce time 3s";
        }
        else{
            qDebug() << "ULP Mode cylce time 300s";
        }
        qDebug() << "Heater temp:" << BSECBMESettings.heater_temperature;
        qDebug() << "Heater dur:" << BSECBMESettings.heating_duration;
        qDebug() << "Heater temp overs:" << BSECBMESettings.temperature_oversampling;
        qDebug() << "Heater hum over:" << BSECBMESettings.humidity_oversampling;
        qDebug() << "Heater pres over:" << BSECBMESettings.pressure_oversampling;
        qDebug() << "Heater prcess data:" << BSECBMESettings.process_data;
        qDebug() << "Heater run_gas:" << BSECBMESettings.run_gas;
    }

}

bool My_BSEC::calculateData(QByteArray inputArray){
    bsec_library_return_t bsecStatus;

    if(inputArray.length() != 46){
        qWarning() << "My_BSEC Data Array Error. Length = " << inputArray.length();
        return false;
    }

    int i = 0;
    quint64 timestamp = qFromBigEndian(inputArray.mid(i,8).toUInt(Q_NULLPTR,16));
    timestamp *= 1000000;
    i += 8;

    quint8 status = inputArray.mid(i,2).toUInt(Q_NULLPTR,16);
    i += sizeof (status) * 2;

    quint8 gas_index = inputArray.mid(i,2).toUInt(Q_NULLPTR,16);
    i += sizeof (gas_index) * 2;

    quint8 meas_index = inputArray.mid(i,2).toUInt(Q_NULLPTR,16);
    i += sizeof (meas_index) * 2;

    quint32 temperatureInt = qFromBigEndian(inputArray.mid(i,8).toUInt(Q_NULLPTR,16));
    float *temperature = reinterpret_cast<float*>(&temperatureInt);
    i += sizeof (temperatureInt) * 2;

    quint32 pressureInt = qFromBigEndian(inputArray.mid(i,8).toUInt(Q_NULLPTR,16));
    float *pressure = reinterpret_cast<float*>(&pressureInt);
    i += sizeof (pressureInt) * 2;

    quint32 humidityInt = qFromBigEndian(inputArray.mid(i,8).toUInt(Q_NULLPTR,16));
    float *humidity = reinterpret_cast<float*>(&humidityInt);
    i += sizeof (pressureInt) * 2;

    quint32 gas_resistanceInt = qFromBigEndian(inputArray.mid(i,8).toUInt(Q_NULLPTR,16));
    float *resistance = reinterpret_cast<float*>(&gas_resistanceInt);

    bsecInputs[0].sensor_id = BSEC_INPUT_HUMIDITY;
    bsecInputs[0].signal = *humidity;
    bsecInputs[0].time_stamp = timestamp;

    bsecInputs[1].sensor_id = BSEC_INPUT_TEMPERATURE;
    bsecInputs[1].signal = *temperature;
    bsecInputs[1].time_stamp = timestamp;

    bsecInputs[2].sensor_id = BSEC_INPUT_GASRESISTOR;
    bsecInputs[2].signal = *resistance;
    bsecInputs[2].time_stamp = timestamp;

    nrBSECOutputs = BSEC_NUMBER_OUTPUTS;
    bsecStatus = bsec_do_steps(bsecInputs, NR_BSEC_INPUTS, bsecOutputs, &nrBSECOutputs);
    sensorControl();
    if(bsecStatus < 0){
        qDebug() << "BSEC error:" << bsecStatus;
    }
    if(bsecStatus > 0){
        qDebug() << "BSEC warning:" << bsecStatus;
    }
    if(nrBSECOutputs == 0){
        qDebug() << "BSEC did no steps! Timestamp: " << (float)timestamp / 1000000000;
        return false;
    }
    qDebug() << "BSEC did steps. Timestamp: " << (float)timestamp / 1000000000;
    if(saveCycles > 0){
        curCycle ++;
        if (curCycle >= saveCycles){
            qDebug() << "Will save state. Cycle: " << curCycle;
            curCycle = 0;
            writeState();
        }
    }
    return true;
}



float My_BSEC::getSensorSingal(uint8_t outputSensorID){
    for (int i = 0 ; i < nrBSECOutputs ; i++){
        if(bsecOutputs[i].sensor_id == outputSensorID)
            return bsecOutputs[i].signal;
    }
    return 0;
}

qint8 My_BSEC::getSensorAcc(uint8_t outputSensorID){
    for (int i = 0 ; i < nrBSECOutputs ; i++){
        if(bsecOutputs[i].sensor_id == outputSensorID)
            return bsecOutputs[i].accuracy;
    }
    return 0;
}

float My_BSEC::getCO2Equivalent(){
    return getSensorSingal(BSEC_OUTPUT_CO2_EQUIVALENT);
}

qint8 My_BSEC::getCO2Acc(){
    return getSensorAcc(BSEC_OUTPUT_CO2_EQUIVALENT);
}


float My_BSEC::getTemperature(){
    return getSensorSingal(BSEC_OUTPUT_RAW_TEMPERATURE);
}

float My_BSEC::getHumidity(){
    return getSensorSingal(BSEC_OUTPUT_RAW_HUMIDITY);
}

float My_BSEC::getIAQ(){
    return getSensorSingal(BSEC_OUTPUT_IAQ);
}

float My_BSEC::getStaticIAQ(){
    return getSensorSingal(BSEC_OUTPUT_STATIC_IAQ);
}

qint8 My_BSEC::getIAQAcc(){
    return getSensorAcc(BSEC_OUTPUT_IAQ);
}

bool My_BSEC::readState(){
    unsigned char inputData[BSEC_MAX_STATE_BLOB_SIZE];
    unsigned char workBuffer[BSEC_MAX_PROPERTY_BLOB_SIZE];
    int status = 0;
    QDataStream inputStateStream;

    if ( stateDataFile->open(QIODevice::ReadOnly ) ){
        inputStateStream.setDevice(stateDataFile);
        qDebug() <<"State file opened!";
    }
    else{
        qDebug() << "Cannot open state file!";
        return false;
    }

    quint32 stateSize = inputStateStream.readRawData(reinterpret_cast<char*>(inputData),BSEC_MAX_STATE_BLOB_SIZE);
    if(stateSize){
        status = bsec_set_state(inputData,stateSize,workBuffer,BSEC_MAX_PROPERTY_BLOB_SIZE);
        if (status == 0){
            qDebug() << "State loaded." << stateSize << "bytes";
        }else{
            qDebug() << "State not loaded. Error code: " << status;
            stateDataFile->close();
            return false;
        }
    }
    else{
        qDebug() << "State not loaded. No data.";
    }
    stateDataFile->close();
    return true;
}
void My_BSEC::setStateDataFile(QFile *stateDataFile){
    this->stateDataFile = stateDataFile;
    qDebug() << "Set state file: " << this->stateDataFile->fileName();
}
void My_BSEC::setSaveCycles(int cycles){
    this->saveCycles = cycles;
    qDebug() << "Set save cycles: " << saveCycles;
}
bool My_BSEC::writeState(){
    QDataStream outputStateStream;
    if ( stateDataFile->open(QIODevice::WriteOnly ) ){
        outputStateStream.setDevice(stateDataFile);
        qDebug() <<"State file opened!";
    }
    else{
        qDebug() << "Cannot open state file!";
        return false;
    }
    unsigned char saveData[BSEC_MAX_STATE_BLOB_SIZE];
    unsigned char workBuffer[BSEC_MAX_PROPERTY_BLOB_SIZE];
    quint32 stateSize = 0;
    int status = 0;
    status = bsec_get_state(0,saveData,BSEC_MAX_STATE_BLOB_SIZE,workBuffer,BSEC_MAX_PROPERTY_BLOB_SIZE,&stateSize);
    if (status){
        qDebug() << "State not saved. Error code: " << status;
        stateDataFile->close();
        return false;
    }
    outputStateStream.writeRawData(reinterpret_cast<char*>(saveData),stateSize);
    qDebug() << "State saved. " << stateSize << "bytes";
    stateDataFile->close();
    return true;
}

