import sys
import os
import json
import time
from datetime import datetime
import uuid
import re
import threading
from enum import IntEnum
import math as Math
import random
import queue


if 'ptvsd' in sys.modules:
    import ptvsd
    ptvsd.break_into_debugger()
#if 'debugpy' in sys.modules:
#    debugpy.breakpoint()


def print_value(name, value):
    print((str(name) + ': ' + str(value)) if name is not None else str(value))


def randint(min, max):
    return random.randint(min, max)


__registry = {}


def namespace(c):
    cname = c.__name__
    if cname not in __registry:
        __registry[cname] = c
    for name in dir(c):
        fn = getattr(c, name)
        if not name.startswith("__"):  # and callable(fn)
            setattr(__registry[cname], name, fn)
    return __registry[cname]


_exception_bucket = queue.Queue()

threading.current_thread().name = 'main-0'
_active_threads = [threading.current_thread()]
_active_threads_changed = None


def rename_current_thread(name):
    if _active_threads_changed:
        _active_threads_changed('renamed', threading.current_thread(), name)
    threading.current_thread().name = name


class ThreadExit(BaseException): ...


class ExcThread(threading.Thread):
    name_seq = 0
    trace_line = 0

    def __init__(self, name, target=None, args=()):
        super().__init__(target=target, daemon=True, args=args)
        self.offset = None
        if hasattr(threading.current_thread(), 'offset'):
            self.offset = threading.current_thread().offset
        ExcThread.name_seq += 1
        curr_name = threading.current_thread().name.split(':')
        if len(curr_name) == 2 and len(name.split(':')) == 1:
            name = curr_name[0] + ':' + name
        self.name = name + '-' + str(ExcThread.name_seq)
        self.trace_line = 0
        self.killed = False
        self.exception = None

    def run(self):
        _active_threads.append(self)
        if _active_threads_changed:
            _active_threads_changed('added', self)
        try:
            self._target(*self._args, **self._kwargs)
        except ThreadExit:
            pass
        except (Exception, SystemExit) as e:
            _exception_bucket.put(sys.exc_info())
            self.exception = e
        _active_threads.remove(self)
        if _active_threads_changed:
            _active_threads_changed('removed', self)

    def exit(self, code):
        if self._tstate_lock is not None:
            self._tstate_lock.release()
        self._stop()
        if sys.platform != 'win32':
            import ctypes
            libpthread = ctypes.cdll.LoadLibrary('libpthread.so.0')
            libpthread.pthread_cancel(ctypes.c_ulong(self.ident))
            _active_threads.remove(self)
            if _active_threads_changed:
                _active_threads_changed('removed', self)
        if code != 0:
            self.exception = SystemExit(code)
        self.killed = True


def watch_thread_exceptions():
    common.send_trace('{ "line": 0, "origin": "' + threading.current_thread().name + '" }')
    rename_current_thread('watch-0')
    while True:
        try:
            type, value, traceback = _exception_bucket.get(timeout=1)
            raise value
        except queue.Empty:
            pass


class objdict(dict):
    def __getattr__(self, name):
        if name in self:
            return self[name]
        else:
            raise AttributeError("No such attribute: " + name)

    def __setattr__(self, name, value):
        self[name] = value

    def __delattr__(self, name):
        if name in self:
            del self[name]
        else:
            raise AttributeError("No such attribute: " + name)


_before_main_hooks = []


@namespace
class common:
    def on_before_main(callback):
        _before_main_hooks.append(callback)

    def before_main():
        [hook() for hook in _before_main_hooks]

    def get(obj, prop, arr=None):
        # pylint: disable=unsubscriptable-object
        got = obj[prop] if prop else obj
        if arr != None:
            for item in got:
                arr.append(item)
        return got

    def set(obj, prop, value):
        # pylint: disable=unsupported-assignment-operation
        obj[int(prop) if isinstance(obj, list) else prop] = value
        return obj

    def json_parse(s):
        return objdict(json.loads(s)) if s else None
    
    def json_stringify(s):
        return json.dumps(s) if s else None

    def sleep(seconds):
        time.sleep(seconds)

    def contains(arr, item):
        try:
            arr.index(item)
        except:
            return False
        return True
    
    def timestamp():
        return int(datetime.now().timestamp() * 1000000)

    def keys(obj, arr):
        # pylint: disable=no-value-for-parameter
        for key in obj.keys():
            arr.append(key)
    
    def guid():
        return str(uuid.uuid4())
    
    def header_id():
        tn = threading.current_thread().name.split(':')
        return os.path.splitext(os.path.basename(__file__))[0] if len(tn) == 1 else tn[0]
    
    def to_string(obj):
        return str(obj)

    def to_number(obj):
        if isinstance(obj, str) and len(obj) == 1 and ord(obj) > 57:
            return ord(obj)
        f = float(obj)
        return f if f % 1 != 0 else int(f)

    def to_boolean(obj):
        return (True if obj == 'true' or obj == 'True' or obj == 'TRUE' else False) if isinstance(obj, str) else False if obj == 0 else True

    def is_array(obj):
        return isinstance(obj, list)

    def set_bit(value, bit):
        return value | 1 << bit

    def clear_bit(value, bit):
        return value & ~(1 << bit)

    def test_bit(value, bit):
        return (value >> bit) % 2 != 0

    def toggle_bit(value, bit):
        return common.clear_bit(value, bit) if common.test_bit(value, bit) else common.set_bit(value, bit)

    def start_project(id, service=False):
        hex = open(os.path.join(os.path.dirname(__file__), id + '.py'), encoding='utf-8').read()
        offset = hex.count("\n", 0, hex.rfind('#' * 80)) + 1
        hex = hex[hex.rfind('#' * 80) + 81:].rsplit("\n", 3)[0]
        t = ExcThread(name=id + ':main', target=exec, args=(compile(hex, id + '.py', 'exec'), globals()))
        t.offset = offset
        # if service:
        #     def wrapper():
        #         time.sleep(5)
        #         t.start()
        #         print("service started: " + id)
        #     ExcThread(name=id + ':service', target=wrapper).start()
        # else:
        t.start()
        print("project started: " + id)
        # t.join()
        while True:
            time.sleep(.1)
            try:
                next(tt for tt in _active_threads if tt.name.startswith(id))
            except:
                break
        if not service:
            common.shutdown_project(id)
        return t.exception.code if t.exception is not None else 0 

    def shutdown_project(id):
        # only works when tracing enabled
        for h in filter(lambda h: h.origin.startswith(id), events.handlers):
            events.handlers.remove(h)
            common.send_trace('{ "listener": "' + h.origin + '", "event": "removed", "type": "' + h.ident + '" }')
        current = threading.current_thread()
        for t in filter(lambda t: t.name.startswith(id), _active_threads):
            if isinstance(t, ExcThread) and t != current:
                try:
                    t.exit(0)
                except:
                    print(str(sys.exc_info()))
        print("project stopped: " + id)

    def send_trace(content):
        # only works when tracing enabled
        try:
            publish_trace(content + '\n')
        except NameError:
            pass


@namespace
class loops:
    def forever(callback):
        def repeat():
            while (True):
                # pylint: disable=not-callable
                callback()
        ExcThread(name='forever', target=repeat).start()

    def pause(ms):
        time.sleep(ms/1000)

    def main(callback):
        def inner():
            callback()
            ret = threading.current_thread().name.split(':')
            if len(ret) == 1:
                control.exit(0)
            else:
                common.shutdown_project(ret[0])

        ExcThread(name='main', target=inner).start()


@namespace
class control:
    def run_in_background(callback):
        ExcThread(name='background', target=callback).start()
    
    def exit(code=0):
        current = threading.current_thread()
        for t in _active_threads:
            if isinstance(t, ExcThread) and t != current:
                t.exit(code)
        sys.exit(code)

    def __log(priority, text):
        print(text)


class Registration:
    def __init__(self, ident, evid, handler):
        self.ident = ident
        self.evid = evid
        self.handler = handler
        self.origin = threading.current_thread().name


class Event:
    def __init__(self, ident, evid, value):
        self.ident = ident
        self.evid = evid
        self.value = value


@namespace
class events:
    handlers = []
    queue = []

    def emit(id, evid, value=None):
        events.queue.append(Event(id, evid, value))

    def listen(id, evid, handler):
        registration = Registration(id, evid, handler)
        events.handlers.append(registration)
        common.send_trace('{ "listener": "' + registration.origin + '", "event": "added", "type": "' + registration.ident + '" }')
    
    def call(id, evid, value=None, wait=True, handler=None):
        for h in events.handlers if handler is None else [handler]:
            if h.ident == id and h.evid == evid:
                t = ExcThread(name='handler', target=h.handler, args=[value] if value is not None else ())
                t.start()
                if wait:
                    t.join()

    def loop():
        while True:
            while len(events.queue) > 0:
                e = events.queue.pop(0)
                for h in events.handlers:
                    if h.ident == e.ident and h.evid == e.evid:
                        events.call(e.ident, e.evid, e.value, False, h)
            time.sleep(0.001)


ExcThread(name='eventbus', target=events.loop).start()


class RestClient:
    def __init__(self, url, token=None, headers={}):
        self.url = url
        self.token = token
        self.headers = headers
        self.__session = None
    
    @property
    def session(self):
        if self.__session is None:
            import requests
            requests.packages.urllib3.disable_warnings()
            self.__session = requests.Session()
        return self.__session

    def clean_parameters(self, obj):
        # pylint: disable=not-an-iterable
        if obj == None:
            return obj
        if isinstance(obj, list):
            ret = []
            for o in obj:
                ret.append(self.clean_parameters(
                    o) if isinstance(o, dict) else o)
            return ret
        elif isinstance(obj, (str, bool, float, int)):
            return obj
        else:
            ret = {}
            for key, value in obj.items():
                if isinstance(value, dict):
                    r = self.clean_parameters(value)
                    if r != None and len(r) != 0:
                        ret[key] = r
                elif value != None:
                    ret[key] = value
            return ret
    
    def getHeaders(self):
        ret = {}
        ret.update(self.headers)
        if self.token:
            ret.update({ "Authorization": self.token })
        return ret

    def get(self, path, extract_value=True):
        response = self.session.get(self.url + path, headers=self.getHeaders(), verify=False)
        if response.status_code >= 400:
            raise Exception('Request error: ' + response.text + ' (' + str(response.status_code) + ')')
        else:
            data = response.json()
            if extract_value:
                return data['value'] if 'value' in data else [] if 'type' in data and data['type'] == 'arstring' else data
            else:
                return data

    def post(self, path, obj=None):
        response = self.session.post(self.url + path, headers=self.getHeaders(),
                                verify=False,
                                data=obj if isinstance(obj, (str, bool, float, int)) else None, 
                                json=self.clean_parameters(obj) if not obj is None and not isinstance(obj, (str, bool, float, int)) else None)
        if response.status_code >= 400:
            raise Exception('Request error: ' + response.text + ' (' + str(response.status_code) + ')')
        else:
            return response

    def put(self, path, obj=None):
        response = self.session.put(self.url + path, headers=self.getHeaders(),
                                verify=False,
                                data=obj if isinstance(obj, (str, bool, float, int)) else None, 
                                json=self.clean_parameters(obj) if not obj is None and not isinstance(obj, (str, bool, float, int)) else None)
        if response.status_code >= 400:
            raise Exception('Request error: ' + response.text + ' (' + str(response.status_code) + ')')
        else:
            return response

    def remove(self, path):
        response = self.session.delete(self.url + path, headers=self.getHeaders(), verify=False)
        if response.status_code >= 400:
            raise Exception('Request error: ' + response.text + ' (' + str(response.status_code) + ')')
        else:
            return response


@namespace
class rest:
    def create_client(api_url, auth_token=None, headers={}):
        return RestClient(api_url, auth_token, headers)


class ConsolePriority(IntEnum):
    Debug = 0
    Log = 1
    Warning = 2
    Error = 3
    Silent = 4
class EventIdPrexix(IntEnum):
    PIN = 0
    AXIS = 1

import os
import sys
import socket
import threading
import re

if 'ptvsd' not in sys.modules and 'debugpy' not in sys.modules:
    _tracesock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    __traceconns = []
    __mainFrame = sys._getframe()
    __current_process_message = None
    __source = open(__file__, encoding='utf-8').read()
    __mainStart = __source.count("\n", 0, __source.rfind("################################################################################")) + 1


    def publish_trace(msg):
        for con in __traceconns:
            try:
                con.sendall(
                    msg.encode(), socket.MSG_NOSIGNAL)  # pylint: disable=no-member
            except Exception:
                try:
                    con.close()
                    __traceconns.remove(con)
                except Exception:
                    pass


    def traceit(frame, event, arg):
        try:
            __file__
        except NameError:
            pass
        else:
            current_thread = threading.current_thread()
            if (frame.f_code.co_filename == __file__ and frame.f_lineno > __mainStart) or (frame.f_code.co_filename == current_thread.name[0:current_thread.name.find(':')] + '.py'):
                if event == "line":
                    lineno = frame.f_lineno
                    if hasattr(current_thread, 'offset') and current_thread.offset is not None:
                        if current_thread.killed:
                            raise ThreadExit()
                        lineno = frame.f_lineno + current_thread.offset
                    publish_trace('{ "line": ' + str(lineno) + ', "origin": "' + current_thread.name + '" }\n')
                    current_thread.trace_line = lineno
                elif event == "return":
                    publish_trace('{ "line": 0, "origin": "' + current_thread.name + '" }\n')
                    current_thread.trace_line = 0
        return traceit


    def enable_trace():
        __mainFrame.f_trace = traceit
        sys.settrace(traceit)
        threading.settrace(traceit)
    

    # enable_trace()
    common.on_before_main(enable_trace)


    _active_threads_changed = lambda event, thread, param='': publish_trace('{ "thread": "' + thread.name + '", "event": "' + event + '", "param": "' + param + '" }\n')


    def accept_traceconns():
        _tracesock.bind(('0.0.0.0', (os.getpid() % 10000) + 55000))
        _tracesock.listen()
        while True:
            connection, client_address = _tracesock.accept()
            # enable_trace()
            __traceconns.append(connection)
            start_receive(connection)
            connection.sendall(('{ "threads": [' + ', '.join(map(lambda at: '{ "thread": "' + at.name + '", "line": ' + str(at.trace_line if hasattr(at, 'trace_line') else 0) + ' }', _active_threads)) + '] }\n').encode())
            connection.sendall(('{ "listeners": [' + ', '.join(map(lambda h: '{ "listener": "' + h.origin + '", "type": "' + h.ident + '" }', events.handlers)) + '] }\n').encode())
            if __current_process_message is not None:
                connection.sendall(__current_process_message.encode())


    def start_receive(connection):
        receiver = ExcThread(name='receiver')
        def receive_from():
            while True:
                try:
                    data = connection.recv(4096).decode()
                    try:
                        received = common.json_parse(data)
                        if received['type'] == 'command':
                            id = re.sub(r'(?<!^)(?=[A-Z])', '_', received['id']).lower() # to snake case
                            ExcThread(name='execute', target=eval(id), args=((objdict(item) if isinstance(item, dict) else item for item in received['params']))).start()
                    except:
                        print('Error on processing received data: ' + str(data))
                        print(str(sys.exc_info()))
                except Exception:
                    receiver.exit(1)
                    break
        receiver._target = receive_from
        receiver.start()


    class TraceThread(ExcThread):
        def exit(self, code):
            _tracesock.close()
            super().exit(code)


    TraceThread(name='trace', target=accept_traceconns).start()


    __oprint = print


    def nprint(*objects, sep=' ', end='\n', file=sys.stdout, flush=False):
        #__builtins__.print(*objects, sep=sep, end=end, file=file, flush=flush)
        __oprint(str(*objects).encode('utf-8'), sep=sep, end=end, file=file, flush=flush)
        trace_message = '{ "console": "' + sep.join(objects).replace('"', '\\"') + '" }\n'
        publish_trace(trace_message)
        if sep.join(objects).startswith('#processId: '):
            global __current_process_message
            __current_process_message = trace_message


    print = nprint

common.POLLING_INTERVAL = 100
@namespace
class tools:
    def extract(obj, prop, arr = None):
        return common.get(obj, prop, arr)
    def create_obj(str):
        return common.json_parse(str)
    def to_string(obj):
        return common.to_string(obj)
    def to_number(obj):
        return common.to_number(obj)
    def to_boolean(obj):
        return common.to_boolean(obj)
    def get_time_stamp():
        return common.timestamp() / 1000
    def measure_execution_time(body):
        start_time = common.timestamp()
        body()
        end_time = common.timestamp()
        diff = (end_time - start_time) / 1000
        events.emit('tools', 'values', diff)
    def on_execution_time_measured(handler):
        any_handler = handler
        events.listen('tools', 'values', any_handler)
    def start_project(id):
        return common.start_project(id)
    def shutdown_project(id):
        common.shutdown_project(id)

import importlib


MotionLib = importlib.import_module('motion')
DatalayerLib = importlib.import_module('datalayer')

@namespace
class server:
    def get_path_param(request, name):
        comps = request.path.split('/')
        for i in range(len(comps)):
            if comps[i] == name:
                return comps[i + 1]
        return None

    def strip_brackets(s):
        return s[1:-1]
    
    def extract(value, ruri: str, parent=False):
        if value is not None:
            tokens = ruri.split('/')
            for i in range(3, (len(tokens) - 1) if parent else len(tokens)):
                value = value[tokens[i]] if common.to_number(tokens[i][0]) > 57 else value[common.to_number(tokens[i])]
        return value

    def last_part(ruri: str):
        tokens = ruri.split('/')
        return tokens[len(tokens) - 1]


class ResourceEvent(IntEnum):
    READ = 0
    CREATE = 1
    UPDATE = 2
    DELETE = 3
class ResourceType(IntEnum):
    BOOLEAN = 0
    INTEGER = 1
    DECIMAL = 2
    STRING = 3
    STRUCTURE = 4

@namespace
class server:
    RESOURCE_EVENT_PREFIX = 'resource'
    class ResourceRequest:
        def __init__(self, params, path, data, response, origin):
            self.params = params
            self.path = path
            self.data = data
            self.response = response
            self.origin = origin
    def get_param(request, name):
        arr = []
        common.keys(request.params, arr)
        return common.get(request.params, name) if common.contains(arr, name) else None
    def get_content(request):
        return request.data
    def respond(request, value):
        request.origin.respond(request.response, value)
    class Resource:
        def __init__(self, name, type):
            self.name = name
            self.type = type
            self.value = None
        def get_value(self):
            return self.value
        def set_value(self, value):
            self.value = value
        def on_request(self, event, handler):
            wrapper = lambda params: handler(params)
            events.listen(server.RESOURCE_EVENT_PREFIX + self.name, event, wrapper)
    def create_sub_resource_provider(factory, resource, schema, model_name, parent_path):
        keys = []
        obj = schema['decomposition'][model_name]
        common.keys(obj, keys)
        for i in range(len(keys)):
            prop = keys[i]
            model = common.get(obj, keys[i])
            path = parent_path + '/' + prop
            rpt = factory.create_resource_provider(resource, path, model, schema['schema'])
            if model[0] == '[' and common.to_number(model[1]) < 91:
                path2 = path + '/{' + prop + 'idx}'
                model2 = server.strip_brackets(model)
                rpt2 = factory.create_resource_provider(resource, path2, model2, schema['schema'])
                server.create_sub_resource_provider(factory, resource, schema, model2, path2)
            elif common.to_number(model[0]) < 91:
                server.create_sub_resource_provider(factory, resource, schema, model, path)
    def create_resource_provider(factory, resource, schema):
        model_name = resource.name
        rp = factory.create_resource_provider(resource,
    model_name,
    model_name,
    schema['schema'] if schema else None)
        if schema:
            server.create_sub_resource_provider(factory, resource, schema, model_name, model_name)
        return rp

class DatalayerResult(IntEnum):
    OK = 0

# TODO remove fix
import ctypes
class CDLL(ctypes.CDLL):
    def __init__(self, name, mode=ctypes.DEFAULT_MODE, handle=None, use_errno=False, use_last_error=False, winmode=None):
        if name == "libcomm_datalayer.so":
            name = os.environ["SNAP_COMMON"] + "/solutions/activeConfiguration/scripts/libraries/bosch/ctrlxdatalayer/libcomm_datalayer.so"
        super().__init__(name, mode, handle, use_errno, use_last_error, winmode)
ctypes.CDLL = CDLL

import os
import ctrlxdatalayer

from comm.datalayer import NodeClass

_datalayer_system = ctrlxdatalayer.system.System("")
_datalayer_system.start(False)

_datalayer_constr = "ipc://" if os.environ.get(
    'SNAP_DATA') is not None else "tcp://boschrexroth:boschrexroth@host.docker.internal?sslport=8443"


@namespace
class ctrlxsdk:
    client = _datalayer_system.factory().create_client(_datalayer_constr)
    subscription = client.create_subscription_sync(
        ctrlxdatalayer.subscription.create_properties(
            "tiger-client-sub", publish_interval=common.POLLING_INTERVAL),
        lambda res, items, user: [events.emit("DLSUB@" + ctrlxsdk.strip(item.get_address()), 0, getattr(
            item.get_data(), "get_" + item.get_data().get_type().name.lower())()) for item in filter(lambda item: item.get_data().get_type().name != "UNKNON", items)]
    )[1]
    provider = None
    nodes = dict()

    def strip(address: str):
        return (address[1:] if address.startswith('/') else address).strip()

    def get_provider():
        if ctrlxsdk.provider is None:
            ctrlxsdk.provider = _datalayer_system.factory().create_provider(_datalayer_constr)
            ctrlxsdk.provider.start()
        return ctrlxsdk.provider

    def fix_type(type: str):
        return "float32" if type == "float" else "float64" if type == "double" else type

    def read(address: str, type: str):
        res, data = ctrlxsdk.client.read_sync(ctrlxsdk.strip(address))
        try:
            return getattr(data, "get_" + ctrlxsdk.fix_type(type))()
        except:
            return None

    def write(address: str, type: str, value):
        with ctrlxdatalayer.variant.Variant() as data:
            getattr(data, "set_" + ctrlxsdk.fix_type(type))(value)
            ctrlxsdk.client.write_sync(ctrlxsdk.strip(address), data)

    def listen(address: str, callback):
        address = ctrlxsdk.strip(address)
        ctrlxsdk.subscription.subscribe(address)
        events.listen("DLSUB@" + address, 0, callback)

    def replace_pathholders(path: str):
        return re.sub(r"\{[^\}]*\}", "*", path)

    def provide(address: str, rp):
        address = ctrlxsdk.strip(address)
        type_address = ""
        if rp.schema is not None and rp.model != 'string' and rp.model != 'number' and rp.model != 'boolean' and not rp.model.startswith('enum:'):
            type_address = "types/ide/" + common.header_id() + "/" + address
            import tempfile
            tmp = tempfile.NamedTemporaryFile(suffix=".fbs", delete=False)
            try:
                tmp.write(rp.schema.encode("utf-8"))
                tmp.flush()
                import subprocess
                subprocess.check_output(os.environ["SNAP_COMMON"] + "/solutions/activeConfiguration/scripts/libraries/bosch/ctrlxdatalayer/flatc -b --schema -o " + tempfile.gettempdir() + " " + tmp.name, shell=True)
                bfbs_path = os.path.join(tempfile.gettempdir(), tmp.name[:-4] + ".bfbs")
                bfbs = open(bfbs_path, 'rb').read()
                os.unlink(bfbs_path)
            finally:
                tmp.close()
                os.unlink(tmp.name)
            rp.schema = ctrlxdatalayer.variant.Variant()
            rp.schema.set_flatbuffers(bytearray(bfbs))
            ctrlxsdk.get_provider().register_type_variant(type_address, rp.schema)
        meta = ctrlxdatalayer.metadata_utils.MetadataBuilder.create_metadata(
            "", "tiger dynamic resource", "", "", NodeClass.NodeClass.Collection if rp.model[0] == '[' else NodeClass.NodeClass.Variable,
            read_allowed=True, write_allowed=True, create_allowed=True, delete_allowed=True, browse_allowed=True,
            type_path=type_address
        )
        rp.node = ctrlxdatalayer.provider_node.ProviderNode(
            ctrlxdatalayer.provider_node.ProviderNodeCallbacks(
                rp.on_create,
                rp.on_delete,
                rp.on_browse,
                rp.on_read,
                rp.on_update,
                lambda userdata, address, cb: cb(DatalayerResult.OK, meta)
            )
        )
        full_address = "ide/" + common.header_id() + "/" + address
        ctrlxsdk.nodes[full_address] = rp.node
        ctrlxsdk.get_provider().register_node(full_address, rp.node)

    def to_variant(value, type, schema=None, model=None):
        is_array = isinstance(value, list)
        data = ctrlxdatalayer.variant.Variant()
        if type == ResourceType.BOOLEAN or model == 'boolean':
            data.set_array_bool8(value) if is_array else data.set_bool8(value)
        elif type == ResourceType.INTEGER or (model is not None and model.startswith('enum')):
            data.set_array_uint64(
                value) if is_array else data.set_uint64(value)
        elif type == ResourceType.DECIMAL or model == 'number':
            data.set_array_float64(
                value) if is_array else data.set_float64(value)
        elif type == ResourceType.STRING or model == 'string':
            data.set_array_string(
                value) if is_array else data.set_string(value)
        else:
            result, data2, err = _datalayer_system.json_converter().parse_json_complex(common.json_stringify(value), schema)
            if result == ctrlxdatalayer.variant.Result.OK:
                data.set_flatbuffers(data2.get_flatbuffers())
            else:
                return err
        return data

    def from_variant(data, schema=None):
        if data.get_type().name.lower() == 'flatbuffers':
            result, data2 = _datalayer_system.json_converter().converter_generate_json_complex(data, schema, 2)
            if result == ctrlxdatalayer.variant.Result.OK:
                return common.json_parse(data2.get_string())
        elif not hasattr(data, "get_" + data.get_type().name.lower()):
            return None
        else:
            return getattr(data, "get_" + data.get_type().name.lower())()

    def shutdown():
        ctrlxsdk.subscription.unsubscribe_all()
        if ctrlxsdk.provider is not None:
            for address in ctrlxsdk.nodes:
                ctrlxsdk.provider.unregister_node(address)
            ctrlxsdk.provider.stop()


@namespace
class ctrlxsdk:
    class ResourceProviderFactory:
        def create_resource_provider(self, resource, path, model = None, schema = None):
            path = ctrlxsdk.replace_pathholders(path)
            rp = ctrlxsdk.ResourceProvider(resource, path, model, schema)
            ctrlxsdk.provide(path, rp)
            return rp
    class ResourceProvider:
        def __init__(self, resource, path, model = None, schema = None):
            self.resource = resource
            self.path = path
            self.model = model
            self.schema = schema + '\nroot_type ' + (server.strip_brackets(self.model) if self.model[0] == '[' else self.model) + ';' if schema else schema
        def respond(self, res, value, address = None):
            res(DatalayerResult.OK,
                ctrlxsdk.to_variant(server.extract(value, address) if address and value else value,
                    self.resource.type,
                    self.schema,
                    self.model))
        def on_read(self, userdata, address, data, cb):
            events.call(server.RESOURCE_EVENT_PREFIX + self.resource.name,
                ResourceEvent.READ,
                server.ResourceRequest(userdata, address, None, cb, self))
            if self.resource.value:
                self.respond(cb, self.resource.value, address)
        def on_create(self, userdata, address, data, cb):
            events.call(server.RESOURCE_EVENT_PREFIX + self.resource.name,
                ResourceEvent.CREATE,
                server.ResourceRequest(userdata,
                    address,
                    ctrlxsdk.from_variant(data, self.schema),
                    cb,
                    self))
        def on_update(self, userdata, address, data, cb):
            events.call(server.RESOURCE_EVENT_PREFIX + self.resource.name,
                ResourceEvent.UPDATE,
                server.ResourceRequest(userdata,
                    address,
                    ctrlxsdk.from_variant(data, self.schema),
                    cb,
                    self))
            if self.resource.value:
                common.set(server.extract(self.resource.value, address, True),
                    server.last_part(address),
                    ctrlxsdk.from_variant(data, self.schema))
                cb(DatalayerResult.OK,
                    ctrlxsdk.to_variant('OK', ResourceType.STRING))
        def on_delete(self, userdata, address, cb):
            events.call(server.RESOURCE_EVENT_PREFIX + self.resource.name,
                ResourceEvent.DELETE,
                server.ResourceRequest(userdata, address, None, cb, self))
        def on_browse(self, userdata, address, cb):
            if self.resource.value:
                value = server.extract(self.resource.value, address)
                arr = []
                if common.is_array(value):
                    for i in range(len(value)):
                        arr.append(common.to_string(i))
                else:
                    common.keys(value, arr)
                cb(DatalayerResult.OK,
                    ctrlxsdk.to_variant(arr, ResourceType.STRING))
            else:
                cb(DatalayerResult.OK,
                    ctrlxsdk.to_variant([], ResourceType.STRING))
@namespace
class ctrlxsdk:
    factory = ctrlxsdk.ResourceProviderFactory()

class MoveType(IntEnum):
    ABSOLUTE = 0
    RELATIVE = 1
class AxisDir(IntEnum):
    POSITIVE = 0
    NEGATIVE = 1
class AxisMeaning(IntEnum):
    MAIN_AXIS_X = 0
    MAIN_AXIS_Y = 1
    MAIN_AXIS_Z = 2
    MAIN_AXIS_A = 3
    MAIN_AXIS_B = 4
    MAIN_AXIS_C = 5
    FREE_AXIS_7 = 6
    FREE_AXIS_8 = 7
    FREE_AXIS_9 = 8
    FREE_AXIS_10 = 9
    FREE_AXIS_11 = 10
    FREE_AXIS_12 = 11
    FREE_AXIS_13 = 12
    FREE_AXIS_14 = 13
    FREE_AXIS_15 = 14
    FREE_AXIS_16 = 15
class AxisType(IntEnum):
    LINEAR = 0
    ROTATIONAL = 1

@namespace
class control:
    def pause_until_motion_ready(obj):
        while not obj.ready():
            loops.pause(common.POLLING_INTERVAL)

    def pause_until_axis_position_passed(axis, pos, dir, tolerance=0):
        while True:
            if (dir.value == 0 and axis.value >= pos - tolerance) or (dir.value == 1 and axis.value <= pos + tolerance):
                break
            loops.pause(common.POLLING_INTERVAL)

@namespace
class motion:
    class Point:
        def __init__(self, config):
            self.config = config
        def get_value(self):
            if len(self.config['positions']) == 1:
                value = self.config['positions'][0]['value']
                return value
            else:
                value2 = []
                for index in range(len(self.config['positions'])):
                    value2.append(self.config['positions'][index])
                ret = value2
                return ret
    def create_kin_position(value, meaning):
        return { "value": value, "meaning": meaning }

class PinType(IntEnum):
    DIGITAL_IN = 0
    DIGITAL_IO = 1
    ANALOG_IN = 2
    ANALOG_IO = 3
class PinEvent(IntEnum):
    RISE = 0
    FALL = 1

@namespace
class pins:
    class DigitalInPin:
        def __init__(self, address):
            self.address = address
            self.type = PinType.DIGITAL_IN
        def register_interrupt(self, callbacks):
            pins.on_event(self, PinEvent.RISE, callbacks['interruptCb'])
            pins.on_event(self, PinEvent.FALL, callbacks['continueCb'])
    class DigitalInOutPin:
        def __init__(self, address):
            self.address = address
            self.type = PinType.DIGITAL_IO
        def register_interrupt(self, callbacks):
            pins.on_event(self, PinEvent.RISE, callbacks['interruptCb'])
            pins.on_event(self, PinEvent.FALL, callbacks['continueCb'])
    class AnalogInPin:
        def __init__(self, address):
            self.address = address
            self.type = PinType.ANALOG_IN
    class AnalogInOutPin:
        def __init__(self, address):
            self.address = address
            self.type = PinType.ANALOG_IO
    def create_pin(address, type, service = None):
        pin = pins.DigitalInPin(address) if type == PinType.DIGITAL_IN else pins.DigitalInOutPin(address) if type == PinType.DIGITAL_IO else pins.AnalogInPin(address) if type == PinType.ANALOG_IN else pins.AnalogInOutPin(address) if type == PinType.ANALOG_IO else None
        pin.service = service if service else pins.pins_service
        pin.service.initialize(pin)
        return pin
    def digital_read(pin):
        return pins.pins_service.read_digital(pin)
    def digital_write(pin, value):
        pins.pins_service.write_digital(pin, value)
    def on_event(pin, event, body):
        address = pin.address
        events.listen(str(EventIdPrexix.PIN) + address,
            1 if event == PinEvent.RISE else 2,
            body)
    def analog_read(pin):
        return pins.pins_service.read_analog(pin)
    def analog_write(pin, value):
        pins.pins_service.write_analog(pin, value)
@namespace
class control:
    def pause_until_pin_value(pin, value):
        while pins.pins_service.read_digital(pin) != value:
            loops.pause(common.POLLING_INTERVAL)

@namespace
class ctrlx:
    rest_client_instance = RestClient('http://localhost:8069')

    def rest_client():
        return ctrlx.rest_client_instance
    
    def get_simple_type(obj) -> str:
        return "string" if isinstance(obj, str) else "bool8" if isinstance(obj, bool) else "uint64" if isinstance(obj, int) else "double"


@namespace
class control:
    def exit(code=0):
        ctrlxsdk.shutdown()
        current = threading.current_thread()
        for t in _active_threads:
            if isinstance(t, ExcThread) and t != current:
                t.exit(code)
        if code == 0:
            for i in range(len(motion.config["kins"])):
                MotionLib.detach_obj(motion.config["kins"][i]["address"])
            for i in range(len(motion.config["axes"])):
                MotionLib.detach_obj(motion.config["axes"][i]["address"])
        # DatalayerLib.create("/script/instances/" + common.header_id() + "/cmd/abort", 0)
        sys.exit(code)
        # raise Exception("Process killed programmatically. Code: " + str(code))


@namespace
class ctrlx:
    def set_opstate(opstate):
        target_state = 'Running' if opstate == 'Booting' else opstate
        DatalayerLib.create('/motion/cmd/opstate', opstate)
        while ctrlx.get_opstate() != target_state:
            loops.pause(common.POLLING_INTERVAL)
    def get_opstate():
        return DatalayerLib.read('/motion/state/opstate')
    def motion_exists(type, key):
        return common.contains(DatalayerLib.read('/motion/' + type + '?type=browse'), key)
    def wait_motion_cmd(motion_obj, cmd_id):
        result = MotionLib.get_cmd_state(motion_obj, cmd_id)
        while result == 'ACTIVE' or result == 'PREPARED' or result == 'INBUFFER' or result == 'CREATED':
            loops.pause(common.POLLING_INTERVAL)
            result = MotionLib.get_cmd_state(motion_obj, cmd_id)
        return result == 'DONE'
    def clear_axis_error(address):
        if MotionLib.get_axs_plc_open_state(address) == 'ERRORSTOP':
            ctrlx.wait_motion_cmd(address, MotionLib.axs_cmd_reset(address))
    def clear_kin_error(address):
        if MotionLib.get_kin_plc_open_state(address) == 'ERRORSTOP':
            ctrlx.wait_motion_cmd(address, MotionLib.kin_cmd_reset(address))
    class Axis:
        def __init__(self, name, axis_params, parent):
            self.name = name
            self.axis_params = axis_params
            self.parent = parent
            self.update_value()
        def low_motion_interrupt(self):
            ctrlx.wait_motion_cmd(self.name, MotionLib.axs_cmd_interrupt(self.name))
            self.update_value()
        def low_motion_continue(self):
            MotionLib.axs_cmd_continue(self.name)
        def get_value(self):
            self.update_value()
            return self.value
        def move(self, type, value, velocity, acceleration):
            if type == MoveType.ABSOLUTE:
                ctrlx.wait_motion_cmd(self.name,
                    MotionLib.axs_cmd_pos_abs(self.name, value, velocity, acceleration, acceleration))
            else:
                ctrlx.wait_motion_cmd(self.name,
                    MotionLib.axs_cmd_pos_rel(self.name, value, velocity, acceleration, acceleration))
            self.update_value()
        def chained_move(self, type, value, velocity, acceleration, body):
            if type == MoveType.ABSOLUTE:
                MotionLib.axs_cmd_pos_abs(self.name, value, velocity, acceleration, acceleration)
            else:
                MotionLib.axs_cmd_pos_rel(self.name, value, velocity, acceleration, acceleration)
            body()
        def moving(self):
            state = MotionLib.get_axs_plc_open_state(self.name)
            return state == "DISCRETE_MOTION" or state == "COORDINATED_MOTION"
        def stop(self):
            ctrlx.wait_motion_cmd(self.name, MotionLib.axs_cmd_abort(self.name))
        def ready(self):
            return MotionLib.get_axs_plc_open_state(self.name) == "STANDSTILL"
        def update_value(self):
            self.value = DatalayerLib.read('/motion/axs/' + self.name + '/state/values/actual/pos')
    class ContSlope:
        def prepare(self, name, dist1, dist2):
            DatalayerLib.create_json("/motion/kin/" + name + "/cmd/wait-for-signal",
                common.json_stringify({ "signalId": 99, "autoReset": True }))
            DatalayerLib.create_json("/motion/kin/" + name + "/cmd/wait-prepare", '{}')
            self.change(name, dist1, dist2)
            MotionLib.kin_cmd_opt_cont_motion_p(name)
        def change(self, name, dist1, dist2):
            MotionLib.kin_cmd_opt_poly_trans_p(name, dist1, dist2)
        def dispose(self, name):
            DatalayerLib.write('motion/state/functions/somo/signals/99', True)
            MotionLib.kin_cmd_opt_poly_trans_p(name)
    class BlockSlope:
        def prepare(self, name, dist1, dist2):
            self.change(name, dist1, dist2)
        def change(self, name, dist1, dist2):
            MotionLib.kin_cmd_opt_blend_p(name, dist1, dist2)
        def dispose(self, name):
            MotionLib.kin_cmd_opt_blend_p(name)
    class Kinematic:
        def __init__(self, name, axes):
            self.name = name
            self.axes = axes
            self.slope = None
            self.values = []
            self.update_values()
        def get_positions(self):
            self.update_values()
            positions = []
            i = 0
            while i < 16 and i < len(self.values):
                positions.append({ "value": self.values[i], "meaning": i })
                i += 1
            return positions
        def low_motion_interrupt(self):
            ctrlx.wait_motion_cmd(self.name, MotionLib.kin_cmd_interrupt(self.name))
            self.update_values()
        def low_motion_continue(self):
            MotionLib.kin_cmd_continue(self.name)
        def move_internal(self, type, values, velocity, acceleration):
            j = len(values)
            while j < 16:
                values.append(0)
                j += 1
            if type == MoveType.ABSOLUTE:
                return MotionLib.kin_cmd_move_lin_abs(self.name,
                    values,
                    'PCS',
                    velocity,
                    acceleration,
                    acceleration)
            else:
                return MotionLib.kin_cmd_move_lin_rel(self.name,
                    values,
                    'PCS',
                    velocity,
                    acceleration,
                    acceleration)
        def move_positions(self, type, positions, velocity, acceleration):
            current_positions = self.get_positions()
            values = []
            for k in range(len(current_positions)):
                cur_pos = current_positions[k]["value"]
                for l in range(len(positions)):
                    if positions[l]["meaning"] == k:
                        cur_pos = positions[l]["value"]
                        break
                values.append(cur_pos)
            self.move(type, values, velocity, acceleration)
        def move(self, type, values, velocity, acceleration):
            if not self.slope:
                ctrlx.wait_motion_cmd(self.name,
                    self.move_internal(type, values, velocity, acceleration))
                self.update_values()
            else:
                self.move_internal(type, values, velocity, acceleration)
        def chained_move(self, type, values, dist1, dist2, velocity, acceleration, body):
            MotionLib.kin_cmd_opt_blend(self.name, dist1, dist2)
            self.move_internal(type, values, velocity, acceleration)
            body()
        def blending_scope(self, slope, dist1, dist2, body):
            self.slope = slope
            self.slope.prepare(self.name, dist1, dist2)
            body()
            self.slope.dispose(self.name)
            count = 1
            while count < 100 and not self.moving():
                loops.pause(common.POLLING_INTERVAL)
                count += 1
            control.pause_until_motion_ready(self)
            self.update_values()
        def change_slope(self, dist1, dist2):
            if self.slope:
                self.slope.change(self.name, dist1, dist2)
            else:
                pass
        def moving(self):
            state2 = MotionLib.get_kin_plc_open_state(self.name)
            return state2 == "MOVING" or state2 == "STOPPING"
        def stop(self):
            ctrlx.wait_motion_cmd(self.name, MotionLib.kin_cmd_abort(self.name))
        def ready(self):
            return MotionLib.get_kin_plc_open_state(self.name) == "STANDBY"
        def update_values(self):
            for m in range(len(self.axes)):
                self.axes[m].update_value()
            self.values = MotionLib.get_kin_ipo_pos(self.name, "PCS")
    class Factory:
        def enable_axis(self, address):
            if MotionLib.get_axs_plc_open_state(address) == 'DISABLED':
                MotionLib.axs_cmd_power(address, True)
        def enable_kin(self, address):
            if MotionLib.get_kin_plc_open_state(address) == 'DISABLED':
                MotionLib.kin_cmd_enable(address)
        def add_axes_to_kin(self, address, axes):
            for n in range(len(axes)):
                if MotionLib.get_axs_plc_open_state(axes[n].name) == "COORDINATED_MOTION":
                    ctrlx.wait_motion_cmd(axes[n].name,
                        MotionLib.axs_cmd_remove_from_kin(axes[n].name))
            for o in range(len(axes)):
                ctrlx.wait_motion_cmd(axes[o].name,
                    MotionLib.axs_cmd_add_to_kin(axes[o].name, address, False))
        def create_axis(self, name, params, parent):
            MotionLib.attach_obj(name)
            ctrlx.clear_axis_error(name)
            self.enable_axis(name)
            return ctrlx.Axis(name, params, parent)
        def create_kinematic(self, name, axes):
            MotionLib.attach_obj(name)
            ctrlx.clear_kin_error(name)
            self.enable_kin(name)
            self.add_axes_to_kin(name, axes)
            return ctrlx.Kinematic(name, axes)
    class PinsService:
        pins = []
        def __init__(self):
            pass
        def initialize(self, pin):
            untyped = pin
            pin.current = self.read_digital(untyped) if pin.type == PinType.DIGITAL_IN or pin.type == PinType.DIGITAL_IO else self.read_analog(pin)
            self.pins.append(pin)
            
            def my_function(current):
                previous = pin.current
                if pin.type == PinType.DIGITAL_IN or pin.type == PinType.DIGITAL_IO:
                    if current and not previous:
                        events.emit(str(EventIdPrexix.PIN) + pin.address, 1)
                    elif not current and previous:
                        events.emit(str(EventIdPrexix.PIN) + pin.address, 2)
                else:
                    pass
                pin.current = current
            ctrlxsdk.listen(str(common.get(pins.config, 'addressPrefix')) + pin.address,
                my_function)
            
        def read_digital(self, pin):
            return DatalayerLib.read(str(common.get(pins.config, 'addressPrefix')) + pin.address)
        def write_digital(self, pin, value):
            DatalayerLib.write(str(common.get(pins.config, 'addressPrefix')) + pin.address,
                value)
        def read_analog(self, pin):
            return DatalayerLib.read(str(common.get(pins.config, 'addressPrefix')) + pin.address)
        def write_analog(self, pin, value):
            DatalayerLib.write_json(str(common.get(pins.config, 'addressPrefix')) + pin.address,
                common.json_stringify({ "type": 'uint16', "value": value }))
    class Variable:
        def __init__(self, address, vtype):
            self.address = address
            self.vtype = vtype
        def get_value(self):
            return ctrlxsdk.read(self.address, self.vtype)
        def set_value(self, value):
            ctrlxsdk.write(self.address, self.vtype, value)
        def on_change(self, handler):
            initial_value_received = False
            def my_function2(params):
                nonlocal initial_value_received
                if initial_value_received:
                    handler(params)
                else:
                    initial_value_received = True
            wrapper = my_function2
            ctrlxsdk.listen(self.address, wrapper)
@namespace
class motion:
    factory = ctrlx.Factory()
    CONT_SLOPE = ctrlx.ContSlope()
    BLOCK_SLOPE = ctrlx.BlockSlope()
    def interrupt_motion(obj):
        temp = obj
        typed = temp
        typed.low_motion_interrupt()
    def continue_motion(obj):
        temp2 = obj
        typed2 = temp2
        typed2.low_motion_continue()
    def on_interrupt_handler(event_source, obj, body):
        def interrupt_fn():
            motion.interrupt_motion(obj)
            body(True)
        def continue_fn():
            motion.continue_motion(obj)
            body(False)
        event_source.register_interrupt({ "interruptCb": interrupt_fn, "continueCb": continue_fn })
    def move_jump_absolute(kin, pos, velocity, acceleration, start_height, end_height, max_height):
        current = kin.values
        first = [current[0], current[1], max_height]
        second = [pos[0], pos[1], max_height]
        p = len(pos)
        while p < 16:
            pos.append(0)
            first.append(0)
            second.append(0)
            p += 1
        MotionLib.kin_cmd_opt_blend_p(kin.name,
            max_height - current[2] - start_height,
            Math.sqrt(Math.pow((pos[0] - current[0]), 2) + Math.pow((pos[1] - current[1]), 2)) / 2)
        MotionLib.kin_cmd_move_lin_abs(kin.name,
            first,
            'PCS',
            velocity,
            acceleration,
            acceleration,
            0,
            0)
        MotionLib.kin_cmd_opt_blend(kin.name,
            Math.sqrt(Math.pow((pos[0] - current[0]), 2) + Math.pow((pos[1] - current[1]), 2)) / 2,
            max_height - pos[2] - end_height)
        MotionLib.kin_cmd_move_lin_abs(kin.name,
            second,
            'PCS',
            velocity,
            acceleration,
            acceleration,
            0,
            0)
        kin.move(MoveType.ABSOLUTE, pos, velocity, acceleration)
@namespace
class pins:
    pins_service = ctrlx.PinsService()

################################################################################

@namespace
class motion:
    config = { "axes": [{ "type": 0, "limits": { "position": { "min": 0, "max": 1000, "unit": "mm" }, "velocity": { "min": -1000, "max": 1000, "unit": "mm/s" }, "acceleration": { "min": -100, "max": 100, "unit": "m/s^2" }, "torque": { "min": -100, "max": 100, "unit": "Nm" } }, "axisParams": { "x": 0, "y": 0, "z": 0, "xr": 0, "yr": 0, "zr": 0 }, "id": "X", "address": "X" },    { "type": 0, "limits": { "position": { "min": 0, "max": 1000, "unit": "mm" }, "velocity": { "min": -1000, "max": 1000, "unit": "mm/s" }, "acceleration": { "min": -100, "max": 100, "unit": "m/s^2" }, "torque": { "min": -100, "max": 100, "unit": "Nm" } }, "axisParams": { "x": 0, "y": 0, "z": 0, "xr": 0, "yr": 0, "zr": 0 }, "id": "Y", "address": "Y" },    { "type": 0, "limits": { "position": { "min": 0, "max": 1000, "unit": "mm" }, "velocity": { "min": -1000, "max": 1000, "unit": "mm/s" }, "acceleration": { "min": -100, "max": 100, "unit": "m/s^2" }, "torque": { "min": -100, "max": 100, "unit": "Nm" } }, "axisParams": { "x": 0, "y": 0, "z": 0, "xr": 0, "yr": 0, "zr": 0 }, "id": "Z", "address": "Z" }], "kins": [{ "limits": { "velocity": { "min": 0, "max": 6000, "unit": "mm/s" }, "acceleration": { "min": 0, "max": 2, "unit": "m/s^2" } }, "units": { "position": "mm", "torque": "Nm" }, "axes": [{ "direction": 0, "axis": "X", "meaning": 0 },    { "direction": 0, "axis": "Y", "meaning": 1 },    { "direction": 0, "axis": "Z", "meaning": 2 }], "id": "Robot", "address": "Robot" }], "points": [{ "positions": [{ "meaning": 0, "value": 0 },    { "meaning": 1, "value": 0 },    { "meaning": 2, "value": 0 }], "reference": "Robot", "id": "pointStart" },    { "positions": [{ "meaning": 0, "value": 100 },    { "meaning": 1, "value": 0 },    { "meaning": 2, "value": 0 }], "reference": "Robot", "id": "point1" },    { "positions": [{ "meaning": 0, "value": 100 },    { "meaning": 1, "value": 100 },    { "meaning": 2, "value": 0 }], "reference": "Robot", "id": "point2" },    { "positions": [{ "meaning": 0, "value": 100 },    { "meaning": 1, "value": 100 },    { "meaning": 2, "value": 100 }], "reference": "Robot", "id": "point3" }] }
@namespace
class motion:
    X = motion.factory.create_axis("""X""",
    { "x": 0, "y": 0, "z": 0, "xr": 0, "yr": 0, "zr": 0 },
    None)
    Y = motion.factory.create_axis("""Y""",
    { "x": 0, "y": 0, "z": 0, "xr": 0, "yr": 0, "zr": 0 },
    None)
    Z = motion.factory.create_axis("""Z""",
    { "x": 0, "y": 0, "z": 0, "xr": 0, "yr": 0, "zr": 0 },
    None)
    robot = motion.factory.create_kinematic("""Robot""", [X, Y, Z])
    point_start = motion.Point(motion.config['points'][0])
    point1 = motion.Point(motion.config['points'][1])
    point2 = motion.Point(motion.config['points'][2])
    point3 = motion.Point(motion.config['points'][3])
    def points_pointStart():
        return motion.point_start.get_value()
    def points_point1():
        return motion.point1.get_value()
    def points_point2():
        return motion.point2.get_value()
    def points_point3():
        return motion.point3.get_value()

@namespace
class ctrlx:
    config = { "variables": [{ "id": "bPyFlag", "address": "plc/app/Application/sym/GVL_BASE/bPyFlag", "vtype": "bool8" },    { "id": "strPyInfo", "address": "plc/app/Application/sym/GVL_BASE/strPyInfo", "vtype": "string" },    { "id": "iPyStop", "address": "plc/app/Application/sym/GVL_BASE/iPyStop", "vtype": "int16" }] }
@namespace
class ctrlx:
    b_py_flag = ctrlx.Variable("""plc/app/Application/sym/GVL_BASE/bPyFlag""", """bool8""")
    str_py_info = ctrlx.Variable("""plc/app/Application/sym/GVL_BASE/strPyInfo""",
    """string""")
    i_py_stop = ctrlx.Variable("""plc/app/Application/sym/GVL_BASE/iPyStop""", """int16""")

@namespace
class pins:
    config = { "addressPrefix": "/fieldbuses/ethercat/master/instances/ethercatmaster/realtime_data/", "pins": [] }

@namespace
class server:
    config = { "resources": [] }


common.before_main()
################################################################################

ctrlx.str_py_info.set_value("VISUAL Start DemoVisual")
i_count = 0

def on_main():
    global i_count
    for index in range(4):
        loops.pause(1000)
        i_count = i_count + 1
        ctrlx.str_py_info.set_value("Move Cycle  " + str(i_count))
        motion.robot.move_positions(MoveType.ABSOLUTE, motion.point_start.get_value(), 100, 1)
        motion.robot.move_positions(MoveType.ABSOLUTE, motion.point1.get_value(), 100, 1)
        motion.robot.move_positions(MoveType.ABSOLUTE, motion.point2.get_value(), 100, 1)
        motion.robot.move_positions(MoveType.ABSOLUTE, motion.point3.get_value(), 100, 1)
    ctrlx.str_py_info.set_value("VISUAL Program finshed")
    control.exit(0)
loops.main(on_main)


watch_thread_exceptions()
