/**
 * WebIQ Visuals widget template.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "cx-select", // {%= ui_type %}
 *     "name": null,
 *     "template": "custom/controls/cx-select" // {%= template_path %}
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default CSS class applied on widget root element
 * name {string}: Name of widget set to data-name attribute
 * template {string}: Path to template file
 *
 * @version 1.2 changed by boschrexroth
 */
(function () {
    'use strict';

    // variables for reference in widget definition
    const templatePath = "custom/controls/cx-select" // "{%= template_path %}"
    const uiType = "cx-select"    // widget keyword (data-ui) {%= ui_type %}
    const className = uiType           // widget name in camel case {%= constructor_name %}
    const isContainer = false

    // example - default configuration
    const defaultConfig = {
        "class-name": uiType,
        "name": null,
        "template": templatePath,
        "label": uiType,
        "item": null
    };

    // setup module-logger
    const ENABLE_LOGGING = true //enable logging with 'log' function
    const RECORD_LOG = false //store logged messages in logger memory
    const logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG)
    const fLog = logger.fLog //force log - logs message even if logger is disabled
    const log = logger.log;  //log message if logger is enabled

    // declare private functions - START
    // (CUSTOM ADDITION++)
    /**
     * enable/disable all listeners stored in listenrs[]
     * @param {Dom} self reference to widget element
     */
    function enableListeners(self, bEnable = true) {
        if (bEnable)
            self.vars.listeners.forEach((l) => { l.enable(); });
        else
            self.vars.listeners.forEach((l) => { l.disable(); });
    }
    /**
     * convert text list in csv format to json object 
     * @param {*} self 
     * @param {*} sCsv 
     */
    function textList2Json(self, sCsv) {
        const vr = self.vars
        sCsv = sCsv.trimLeft()
        if (sCsv.startsWith("{")) {
            vr.txtList = JSON.parse(sCsv)
        } else {
            // create text list from string
            const arW = sCsv.split(";")
            const iLen = arW.length
            vr.txtList = []
            for (let i = 0; i < iLen; i++) {
                const arC = arW[i].split("=")
                if (arC[1] !== "") {
                    vr.txtList[arC[0]] = arC[1];
                }
            }
        }
    }
    /**
     * Retrieves json text list, which is empty when not defined
     * @param {string} sName name of text list
     * @returns {object} text list object
     */
    function getTextList(self, sName) {
        let myList = cxt.textList[sName]
        if (typeof cxt.textList !== 'undefined') {
            myList = cxt.textList[sName]
        }
        if (cxt.isNullOrUndef(myList)) {
            try {
                let myList = self.vars.textLists[sName]
                return myList
            } catch {
                return {}
            }
        } else {
            return myList
        }
    }
    /**
     * Returns value for key from JSON list (e.g. 0=off 1=on)
     * @param {string} key name of list
     * @returns json list
     */
    function parseTextList(self, key) {
        const vr = self.vars
        try {
            let res = shmi.localize(vr.txtList["" + key])
            return res ? res : `?TxtLst?=${key}`
        } catch (e) {
            return `?TxtLst?=${key}`
        }
    }
    function updateValue(self, value) {
        const el = self.vars.elements
        if (el.value) {
            el.value.textContent = parseTextList(self, value)
        }
    }
    /**
     * subscribe item if defined & stores it in an array
     * @param {object} self  reference to WebIQ widget
     * @param {string} sItem name of item
     */
    function subscribeIfExists(self, sItem) {
        const { im } = self.imports;
        if (sItem)
            self.vars.subscribes.push(im.subscribeItem(sItem, self))
    }
    // (/CUSTOM ADDITION--)
    // declare private functions - END

    // definition of new widget extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        // default configuration settings - all available options have to be initialized!
        config: defaultConfig,
        // instance variables
        vars: {
            // (CUSTOM ADDITION++)
            elements: {
                //references for DOM elements accessed with JS code
                label: null, //element to display label text
                value: null, //element to display item value
                unit: null //element to display unit text
            },
            item: null,      //reference for PLC item subscription token - will be set when widget is enabled
            itemVal: null,   //value of item
            listeners: [],   //event listeners
            subscribes: []   //subscribed items
            // (CUSTOM ADDITION--)
        },
        // imports added at runtime
        imports: {
            // example - add import via shmi.requires(...)
            im: "visuals.session.ItemManager",
            // (CUSTOM ADDITION++)
            io: "visuals.io"
            // (CUSTOM ADDITION--)
        },

        // array of custom event types fired by this widget
        events: [],

        // functions to extend or override the BaseControl prototype
        prototypeExtensions: {
            // called when config-file (optional) is loaded and template (optional) is inserted into base element
            onInit: function () {
                // (CUSTOM ADDITION++)
                log("onInit")
                const self = this
                const el = self.vars.elements
                const vr = self.vars
                const cf = self.config
                // disable context menu
                self.element.addEventListener('contextmenu', (e) => e.preventDefault(), false)

                self.element.title = cf.title
                let myWidget = self.element.firstChild
                // create label
                if (cf.label) {
                    el.label = cxt.addChild(myWidget, "button", "divLabel")
                    el.label.textContent = shmi.localize(cf.label)
                }
                // create value with lock icon
                el.myDiv = cxt.addChild(myWidget, "div", "divFrame")
                let imgLock = cxt.addChild(el.myDiv, "img", "cxLock")
                imgLock.src = cxt.cfgMsgBox.imgLock

                el.value = cxt.addChild(el.myDiv, "button", "divValue")
                cxt.addChild(el.myDiv, "button", "divMenu", "☰")

                // add onClick handler to execute an function
                if (cf.item) {
                    //get reference to Mouse- & TouchListener constructors
                    const arListeners = [self.imports.io.MouseListener, self.imports.io.TouchListener]

                    // define the onClick handler
                    const handler = {
                        onClick: async function (x, y, event) {
                            let sOpt = ""
                            let oVal = await cxt.getItem(cf.item)
                            for (const key in vr.txtList) {
                                sOpt += `${vr.txtList[key]}\t${key}\n`
                            }
                            oVal = await cxt.msgBox("select", "select option", "ok", sOpt, "\n", "\t", oVal)
                            if (oVal !== null) cxt.setItem(cf.item, oVal)
                        }
                    };

                    //create listeners for root element (=self.element) of the widget
                    arListeners.forEach(listen => {
                        self.vars.listeners.push(
                            new listen(self.element, handler)
                        )
                    });
                }
                // (CUSTOM ADDITION--)
            },
            // called when widget is enabled
            onEnable: async function () {
                // (CUSTOM ADDITION++)
                log("onEnable")
                const self = this
                const cf = self.config
                const vr = self.vars
                const im = self.imports.im

                // Subscribe to item updates
                if (cf.selectType === "tlName") {
                    if (cf.tlName) vr.txtList = getTextList(self, cf.tlName)
                } else {
                    subscribeIfExists(self, cf.tlItem)
                }

                subscribeIfExists(self, self.config.item)
                enableListeners(self, true)

                self.vars.toolTip = cxt.isTooltipEnabled() ? im.subscribeItem(TOOLTIP_ITEM, self) : null
                // (CUSTOM ADDITION--)
            },
            // called when widget is disabled
            onDisable: function () {
                // (CUSTOM ADDITION++)
                log("onDisable")
                const self = this
                const vr = self.vars
                enableListeners(self, false)
                // Stop listening to all items in subscribes[]
                vr.subscribes.forEach((s) => {
                    s.unlisten();
                })
                vr.subscribes = []
                if (vr.toolTip) vr.toolTip.unlisten(); vr.toolTip = null
                // (CUSTOM ADDITION--)
            },
            // called when widget is locked - disable mouse- and touch-listeners
            onLock: function () {
                // (CUSTOM ADDITION++)
                log("onLock")
                const self = this
                enableListeners(self, false)
                self.element.classList.add("locked")
                // (/CUSTOM ADDITION--)
            },
            // called when widget is unlocked - enable mouse- and touch-listeners
            onUnlock: function () {
                // (CUSTOM ADDITION++)
                log("onUnlock")
                const self = this
                enableListeners(self, true)
                self.element.classList.remove("locked")
                // (CUSTOM ADDITION--)
            },
            // called by ItemManager when value of subscribed item changes and once on initial subscription
            onSetValue: function (value, type, name) {
                // (CUSTOM ADDITION++)
                const self = this;
                const cf = self.config
                const vr = self.vars

                log("onSetValue:" + name)
                if (type === null)  // when type===null do nothing, because params are invalid
                    return
                if (name === cf.item) {
                    updateValue(self, value)
                    vr.itemVal = value
                } else if (name === cf.tlItem) {
                    // create text list from string
                    textList2Json(self, value)
                    updateValue(this, vr.itemVal)
                }
                else if (name === TOOLTIP_ITEM)
                    cxt.tooltip(self, value)
                // (CUSTOM ADDITION--)
            },
            // called by ItemManager to provide properties (min & max values etc.) of subscribed item
            onSetProperties: function (min, max, step, name, type, warnMin, warnMax, prewarnMin, prewarnMax, digits) {
                // (CUSTOM ADDITION++)

                // (CUSTOM ADDITION--)
            },
            // called when widget is deleted - used for instance clean-up
            onDelete: function () {
                log("onDelete")
            }
        }
    };

    // definition of new widget extending BaseControl - END

    // generate widget constructor & prototype using the control-generator tool
    shmi.requires("visuals.tools.control-generator").generate(definition);
})();
