/**
 * WebIQ Visuals widget template.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "cx-button", // {%= ui_type %}
 *     "name": null,
 *     "template": "custom/controls/cx-button" // {%= template_path %}
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default CSS class applied on widget root element
 * name {string}: Name of widget set to data-name attribute
 * template {string}: Path to template file
 *
 * @version 1.2 changed by boschrexroth
 */
(function () {
    'use strict';

    // variables for reference in widget definition
    const templatePath = "custom/controls/cx-button" // "{%= template_path %}"
    const uiType = "cx-button"                       // widget keyword (data-ui) {%= ui_type %}
    const className = uiType                         // widget name in camel case {%= constructor_name %}
    const isContainer = false

    // example - default configuration
    const defaultConfig = {
        "class-name": uiType,
        "name": null,
        "template": templatePath,
        "label": uiType,
        "item": null
    };

    // setup module-logger
    const ENABLE_LOGGING = false //enable logging with 'log' function
    const RECORD_LOG = false //store logged messages in logger memory
    const logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG)
    const fLog = logger.fLog //force log - logs message even if logger is disabled
    const log = logger.log; //log message if logger is enabled

    // declare private functions - START
    // (CUSTOM ADDITION++)
    function enableListeners(self) {
        self.vars.listeners.forEach((l) => {
            l.enable();
        });
    }

    function disableListeners(self) {
        self.vars.listeners.forEach((l) => {
            l.disable();
        });
    }
    /**
     * executes a ui-action of function
     * @param {*} jFunc function of json Ui-action
     */
    function execFunc(jFunc) {
        if (typeof jFunc === "function")
            jFunc()
        else if (jFunc != null)
            new shmi.visuals.core.UiAction(jFunc).execute()
    }

    function setLabel(sLabel, self) {
        const el = self.vars.elements
        if (el.txtLabel && sLabel) el.txtLabel.textContent = shmi.localize(sLabel)
    }
    function setImg(sImgUrl, self) {
        const el = self.vars.elements
        if (el.imgBtn && sImgUrl) el.imgBtn.src = sImgUrl
    }
    // (/CUSTOM ADDITION--)
    // declare private functions - END

    // definition of new widget extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        // default configuration settings - all available options have to be initialized!
        config: defaultConfig,
        // instance variables
        vars: {
            // (CUSTOM ADDITION++)
            elements: {
                //references for DOM elements accessed with JS code
                label: null, //element to display label text
                unit: null,  //element to display unit text
                bar: null,   //element of colored bar
                value: null  //element to display item value
            },
            whileAction: null,
            min: 0,          //minimum value for item, will be replaced by actual item minimum if defined
            max: 100,        //maximum value for item, will be replaced by actual item maximum if defined
            item: null,      //reference for PLC item subscription token - will be set when widget is enabled
            listeners: [],   //event listeners
            subscribes: []   //subscribed items
            // (/CUSTOM ADDITION--)
        },
        // imports added at runtime
        imports: {
            // example - add import via shmi.requires(...)
            im: "visuals.session.ItemManager",
            // (CUSTOM ADDITION++)
            io: "visuals.io"
            // (/CUSTOM ADDITION--)
        },

        // array of custom event types fired by this widget
        events: [],

        // functions to extend or override the BaseControl prototype
        prototypeExtensions: {
            // called when config-file (optional) is loaded and template (optional) is inserted into base element
            onInit: function () {
                // (CUSTOM ADDITION++)
                const self = this
                log("onInit " + self.config.name)
                const el = self.vars.elements
                const cf = self.config
                // disable context menu
                self.element.addEventListener('contextmenu', (e) => e.preventDefault(), false)

                // Store references to the DOM elements for performance reasons
                el.ledStatus = shmi.getUiElement("cxBtnLed", self.element);

                el.widget = el.ledStatus.parentNode
                el.imgLock = shmi.getUiElement("cxLock", self.element);
                el.imgLock.src = cxt.cfgMsgBox.imgLock
                el.boxInfo = shmi.getUiElement("cxBtnInfo", self.element);
                el.txtLabel = shmi.getUiElement("cxBtnLabel", self.element);
                el.imgBtn = shmi.getUiElement("cxBtnImg", self.element)
                el.imgBox = el.imgBtn.parentNode

                // setup position of image
                let elBoxInfo = el.txtLabel.parentNode.parentNode.classList
                if (cf.imgPosition === "left") {
                    elBoxInfo.add("rowRev")
                } else if (cf.imgPosition === "right") {
                    elBoxInfo.add("row")
                } else if (cf.imgPosition === "top") {
                    elBoxInfo.add("colRev")
                } else if (cf.imgPosition === "bottom") {
                    elBoxInfo.add("col")
                } else {
                    cf.imgUrlOn = null
                    elBoxInfo.add("centerField")
                    el.imgBox.style.display = "none"
                }

                if (!cf.labelOn) cf.labelOff = null
                if (!cf.imgUrlOn) {
                    cf.imgUrlOff = null
                    el.imgBtn.style.display = "none"
                }
                setLabel(cf.labelOn, self)
                setImg(cf.imgUrlOn, self)

                // when no status item defined: use ctrl item as status
                cf.itemStatus = (!cf.itemStatus) ? cf.itemCtrl : cf.itemStatus
                if (cf.itemCtrl === cf.itemStatus) el.ledStatus.style.visibility = "hidden"

                // add onClick handler to execute an function
                if (el.widget) {
                    //get reference to Mouse- & TouchListeners
                    const arListeners = [self.imports.io.MouseListener, self.imports.io.TouchListener]
                    let myListeners = null
                    if (cf.btnType === "on-off") {
                        myListeners = {
                            onPress: function (x, y, event) {
                                log("on-off press")
                                if (cf.itemCtrl && !cxt.isNullOrUndef(cf.valueOn)) cxt.setItem(cf.itemCtrl, cf.valueOn)
                                // When no item is define, but ui-actions
                                if (!cf.itemCtrl) {
                                    el.widget.classList.add("cxBtnBoxOn")
                                    setLabel(cf.labelOn, self)
                                    setImg(cf.imgUrlOn, self)
                                }
                                execFunc(cf.actionDn)
                                if (cf.actionWhile && cf.intervalWhilePressed)
                                    self.vars.whileAction = setInterval(() => {
                                        new shmi.visuals.core.UiAction(cf.actionWhile).execute()
                                    }, cf.intervalWhilePressed)
                            },
                            onRelease: function (x, y, event) {
                                log("on-off release")
                                if (self.vars.whileAction) {
                                    clearInterval(self.vars.whileAction)
                                    self.vars.whileAction = null
                                }
                                if (cf.itemCtrl && !cxt.isNullOrUndef(cf.valueOff)) cxt.setItem(cf.itemCtrl, cf.valueOff)
                                // When no item is define, but ui-actions
                                if (!cf.itemCtrl) {
                                    el.widget.classList.remove("cxBtnBoxOn")
                                    setLabel(cf.labelOff, self)
                                    setImg(cf.imgUrlOff, self)
                                }

                                execFunc(cf.actionUp)
                            }
                        }
                    } else if (cf.btnType === "set") {
                        myListeners = {
                            onPress: function (x, y, event) {
                                log("set press")
                                if (cf.itemCtrl && !cxt.isNullOrUndef(cf.valueOn)) cxt.setItem(cf.itemCtrl, cf.valueOn)
                                execFunc(cf.actionDn)
                            },
                            onRelease: function (x, y, event) {
                                log("set release")
                                execFunc(cf.actionUp)
                            }
                        }
                    } else /* toggle */ {
                        myListeners = {
                            onPress: async function (x, y, event) {
                                log("toggle press")
                                if (cf.itemCtrl) {
                                    let myStatus = await cxt.getItem(cf.itemStatus)
                                    cxt.setItem(cf.itemCtrl, (myStatus == cf.valueOn) ? cf.valueOff : cf.valueOn)
                                }
                                execFunc(cf.actionDn)
                            },
                            onRelease: function (x, y, event) {
                                log("toggle release")
                                execFunc(cf.actionUp)
                            }
                        }
                    }

                    //create listeners for root element (=self.element) of the widget
                    arListeners.forEach(listen => {
                        self.vars.listeners.push(
                            new listen(self.element, myListeners)
                        )
                    });
                }
                // (CUSTOM ADDITION--)
            },
            // called when widget is enabled
            onEnable: async function () {
                // (CUSTOM ADDITION++)
                const self = this
                log("onEnable " + self.config.name)
                const cf = self.config
                const im = self.imports.im
                enableListeners(self)

                // Subscribe items
                if ((cf.itemCtrl) && (cf.itemCtrl !== cf.itemStatus)) {
                    self.vars.subscribes.push(im.subscribeItem(cf.itemCtrl, self))
                }
                if (cf.itemStatus) {
                    self.vars.subscribes.push(im.subscribeItem(cf.itemStatus, self))
                }
                // If TOOLTIP_ITEM not defined => widget not correct enabled
                self.vars.toolTip = cxt.isTooltipEnabled() ? im.subscribeItem(TOOLTIP_ITEM, self) : null

                // (/CUSTOM ADDITION--)
            },
            // called when widget is disabled
            onDisable: function () {
                // (CUSTOM ADDITION++)
                const self = this
                const vr = self.vars
                log("onDisable " + self.config.name)
                disableListeners(self);
                // Stop listening to item, i.e. unsubscribe from item updates
                if (vr.toolTip) vr.toolTip.unlisten(); vr.toolTip = null

                self.vars.subscribes.forEach((s) => {
                    s.unlisten();
                })
                self.vars.subscribes = []
                // (/CUSTOM ADDITION--)
            },
            // called when widget is locked - disable mouse- and touch-listeners
            onLock: function () {
                // (CUSTOM ADDITION++)
                log("onLock")
                const self = this
                disableListeners(self)
                self.element.classList.add("locked")
                // (/CUSTOM ADDITION--)
            },
            // called when widget is unlocked - enable mouse- and touch-listeners
            onUnlock: function () {
                // (CUSTOM ADDITION++)
                log("onUnlock")
                const self = this;
                enableListeners(self)
                self.element.classList.remove("locked")
                // (/CUSTOM ADDITION--)
            },
            // called by ItemManager when value of subscribed item changes and once on initial subscription
            onSetValue: function (value, type, name) {
                // (CUSTOM ADDITION++)
                const self = this;
                // console.log("onSetValue: " + self.config.name + ` ${name}=${value}`)
                if (type === null)  // when type===null do nothing, because params are invalid
                    return
                const cf = self.config
                const el = self.vars.elements
                if (name === cf.itemStatus) {
                    if (value == cf.valueOn) {
                        el.widget.classList.add("cxBtnBoxOn")
                        setLabel(cf.labelOn, self)
                        setImg(cf.imgUrlOn, self)
                    } else {
                        el.widget.classList.remove("cxBtnBoxOn")
                        setLabel(cf.labelOff, self)
                        setImg(cf.imgUrlOff, self)
                    }
                } else if (name === cf.itemCtrl) {
                    el.ledStatus.style.visibility = (value == cf.valueOn) ? "visible" : "hidden"
                } else if (name === TOOLTIP_ITEM) {
                    cxt.tooltip(self, value)
                }
                // (/CUSTOM ADDITION--)
            },
            // called by ItemManager to provide properties (min & max values etc.) of subscribed item
            onSetProperties: function (min, max, step, name, type, warnMin, warnMax, prewarnMin, prewarnMax, digits) {
                // (CUSTOM ADDITION++)
                const self = this
                log("onSetProperties  " + self.config.name + ` name:${name} type:${type} min:${min} max:${max}`)
                if (type === null)  // when type===null do nothing, because params are invalid
                    return

                const cf = self.config
                if (type === TYP_STRING) {
                    // set empty string when '' is set
                    cf.valueOn = (cf.valueOn === "''") ? "" : cf.valueOn
                    cf.valueOff = (cf.valueOff === "''") ? "" : cf.valueOff
                } else {
                    cf.valueOn = cf.valueOn ? parseFloat(cf.valueOn) : cf.valueOn
                    cf.valueOff = cf.valueOff ? parseFloat(cf.valueOff) : cf.valueOff
                }
                // (/CUSTOM ADDITION--)
            },
            // called when widget is deleted - used for instance clean-up
            onDelete: function () {
                log("onDelete")
            }
        }
    };

    // definition of new widget extending BaseControl - END

    // generate widget constructor & prototype using the control-generator tool
    shmi.requires("visuals.tools.control-generator").generate(definition);
})();
