/**
 * Visuals - Industrial Visualization Framework for JavaScript
 *
 * Copyright © 2012-2024 Smart HMI GmbH
 *
 * All rights reserved
 *
 * No part of this website or any of its contents may be reproduced, copied, modified or
 * adapted, without the prior written permission of Smart HMI.
 *
 * Commercial use and distribution of the contents of the website is not allowed without
 * express and prior written permission of Smart HMI.
 *
 *
 * Web: http://www.smart-hmi.de
 *
 * @version 2.15.7 c927faeb.44385 25-01-2024 14:05:49
 */

/**
 * Alarm History
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "alarm-history",
 *     "name": null,
 *     "template": "default/alarm-history"
 * }
 *
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 *
 */
(function() {
    'use strict';

    //variables for reference in control definition
    var className = "AlarmHistory", //control name in camel-case
        uiType = "alarm-history", //control keyword (data-ui)
        isContainer = true;

    //example - default configuration
    var defConfig = {
        "class-name": uiType,
        "name": null,
        "template": "default/" + uiType,
        "label": uiType,
        "dateformat": "${alarmlist_dateformat}"
    };

    //setup module-logger
    var ENABLE_LOGGING = false,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    //declare private variables - START
    var alarmGridConfig = {
            "name": "alarms-historic",
            "fields": ["id", "index", "level", "come", "gone", "group", "needAck", "isAck", "alarmAck", "ackUser", "json"]
        },
        alarmTableConfig = {
            "label": "${alarmlist_history_title}",
            "table": alarmGridConfig.name,
            "name": "alarm-table",
            "class-name": "complex-table2 alarms",
            "field-datagrid-col-map": {
                "level": 2,
                "description": 1,
                "alarm-start": 3,
                "alarm-end": 4,
                "alarm-group": 5,
                "detail": 10
            },
            "select-mode": "SINGLE",
            "default-field-control-map": {
                "level": {
                    "ui-type": "toggle-display",
                    "config": {
                        "template": "default/alarm-history/toggle-display-alarm-levels",
                        "conditions": [
                            "%VALUE%==0",
                            "%VALUE%==1",
                            "%VALUE%==2"
                        ],
                        "class-name": "toggle-display icon-only no-background"
                    }
                },
                "description": {
                    "ui-type": "text2",
                    "config": {
                        "class-name": "text2 multiline",
                        "options": [],
                        "pattern": "${alarm_title_<%= VALUE %>}"
                    }
                },
                "alarm-start": {
                    "ui-type": "text2-date",
                    "config": {
                        "class-name": "text2 text2-date multiline",
                        "dateformat": "$DD.$MM.$YYYY, $HH:$mm:$ss"
                    }
                },
                "alarm-end": {
                    "ui-type": "text2-date",
                    "config": {
                        "class-name": "text2 text2-date multiline",
                        "dateformat": "$DD.$MM.$YYYY, $HH:$mm:$ss"
                    }
                },
                "alarm-group": {
                    "ui-type": "text2",
                    "config": {
                        "class-name": "text2",
                        "options": [],
                        "pattern": "${alarm_group_<%= VALUE %>}"
                    }
                },
                "detail": {
                    "ui-type": "local-script",
                    "config": {
                        "module": "visuals.tools.alarms.ls.alarmDetails"
                    }
                }
            },
            "default-field-headers": {
                "level": "${alarmlist_level_header}",
                "description": "${alarmlist_title_header}",
                "alarm-start": "${alarmlist_come_header}",
                "alarm-end": "${alarmlist_gone_header}",
                "alarm-group": "${alarmlist_group_header}",
                "detail": "${alarmlist_detail_header}"
            },
            "_comment": "expr is passed to expr parameter array of shmi.visuals.core.DataGridManager.setFilter",
            "filters": [{
                "label": "${alarmlist_level_info}",
                "template": "default/alarm-history/level-service",
                "class-name": "toggle-button icon-and-text",
                "field": "level",
                "expr": 0
            }, {
                "label": "${alarmlist_level_warning}",
                "template": "default/alarm-history/level-warning",
                "class-name": "toggle-button icon-and-text",
                "field": "level",
                "expr": 1
            }, {
                "label": "${alarmlist_level_alarm}",
                "template": "default/alarm-history/level-alarm",
                "class-name": "toggle-button icon-and-text",
                "field": "level",
                "expr": 2
            }],
            "default-layout": {
                "class-name": "layout-std",
                "_comment": "default == no additional css layout class",
                "column-org": {
                    "col1": {
                        "fields": ["level"],
                        "width": "5%"
                    },
                    "col2": {
                        "fields": ["description"],
                        "width": "20%"
                    },
                    "col3": {
                        "fields": ["alarm-start"],
                        "width": "20%"
                    },
                    "col4": {
                        "fields": ["alarm-end"],
                        "width": "20%"
                    },
                    "col5": {
                        "fields": ["alarm-group"],
                        "width": "20%"
                    },
                    "col6": {
                        "fields": ["detail"],
                        "width": "15%"
                    }
                },
                "line-height": "39px"
            },
            "sortable-fields": [
                "level",
                "description",
                "alarm-start",
                "alarm-end",
                "alarm-group"
            ],
            "delete-selected-rows": false,
            "show-nof-rows": true,
            "show-buttons-table-min-width-px": 400,
            "text-mode": "SINGLELINE",
            "responsive-layouts": [
                {
                    "class-name": "layout-compact",
                    "table-max-width-px": 700,
                    "v-scroll-options": ["V_SWIPE"],
                    "select-mode": "SINGLE",
                    "column-org": {
                        "col1": {
                            "fields": ["level"],
                            "width": "15%"
                        },
                        "col2": {
                            "fields": ["description"],
                            "width": "30%"
                        },
                        "col3": {
                            "fields": ["alarm-start", "alarm-end"],
                            "width": "25%"
                        },
                        "col4": {
                            "fields": ["detail"],
                            "width": "25%"
                        }
                    },
                    "line-height": "79px"
                }
            ],
            "default-nof-buffered-rows": 150,
            "buffer-size": 500
        };
    //declare private variables - END

    //declare private functions - START
    function getAnchorElements(self) {
        self.vars.anchors.from = shmi.getUiElement("from-anchor", self.element);
        self.vars.anchors.to = shmi.getUiElement("to-anchor", self.element);
        self.vars.anchors.table = shmi.getUiElement("table-anchor", self.element);
        self.vars.anchors.filterButton = shmi.getUiElement("filter-button", self.element);
        self.vars.anchors.filterOverlay = shmi.getUiElement("filter-overlay", self.element);
    }

    function createControls(self) {
        var anchors = self.vars.anchors,
            controls = self.vars.controls;

        controls.table = shmi.createControl("complex-table2", anchors.table, alarmTableConfig, "DIV");
        controls.from.date = shmi.createControl("select-date", anchors.from, { "label": "start" }, "DIV");
        controls.from.time = shmi.createControl("select-time", anchors.from, { "label": "start", "isUTC": true }, "DIV");
        controls.to.date = shmi.createControl("select-date", anchors.to, { "label": "end" }, "DIV");
        controls.to.time = shmi.createControl("select-time", anchors.to, { "label": "end", "isUTC": true }, "DIV");

        controls.from.time.setValue(0);
        controls.to.time.setValue(0);
    }

    function computeTimestamp(dateTimestamp, hours, minutes, seconds) {
        var date = new Date(dateTimestamp * 1000);
        date.setHours(hours, minutes, seconds, 0);

        return date.getTime() / 1000;
    }

    function computeTimestampFromControls(ctrlGroup) {
        return computeTimestamp(
            ctrlGroup.date.getValue(),
            ctrlGroup.time.getHours(),
            ctrlGroup.time.getMinutes(),
            ctrlGroup.time.getSeconds()
        );
    }

    function setFilter(self) {
        var from = computeTimestampFromControls(self.vars.controls.from),
            to = computeTimestampFromControls(self.vars.controls.to),
            dt = shmi.requires("visuals.tools.date");
        shmi.addClass(self.element, "filter-active");
        self.vars.activeFilterLabel.textContent = shmi.localize("${alarmlist_timefilter}");
        self.vars.activeFilterFrom.textContent = dt.formatDateTime(from, { datestring: shmi.localize(self.config.dateformat) });
        self.vars.activeFilterTo.textContent = dt.formatDateTime(to, { datestring: shmi.localize(self.config.dateformat) });
        self.vars.grid.setFilter(3, [from, to]);
    }

    function clearFilter(self) {
        self.vars.grid.clearFilter(3);
        shmi.removeClass(self.element, "filter-active");
    }

    function attachChangeListeners(self) {
        var toks = self.vars.tokens,
            from = self.vars.controls.from,
            to = self.vars.controls.to;

        toks.push(from.date.listen("change", function(evt) {
            log(uiType, "from date:", evt.detail.value);
            log(uiType, "total from:", new Date(computeTimestampFromControls(from) * 1000));
            to.date.setValue(from.date.getValue() + 86400);
            to.time.setValue(from.time.getValue());
        }));
        toks.push(from.time.listen("change", function(evt) {
            log(uiType, "from time:", evt.detail.value);
            log(uiType, "total from:", new Date(computeTimestampFromControls(from) * 1000));
        }));

        toks.push(to.date.listen("change", function(evt) {
            log(uiType, "to date:", evt.detail.value);
        }));
        toks.push(to.time.listen("change", function(evt) {
            log(uiType, "to time:", evt.detail.value);
        }));
    }

    //declare private functions - END

    //definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            grid: null,
            table: null,
            startDate: null,
            endDate: null,
            todayButton: null,
            resetButton: null,
            timeFilterLabel: null,
            activeFilterLabel: null,
            activeFilterFrom: null,
            activeFilterTo: null,
            anchors: {
                from: null,
                to: null,
                table: null,
                filterButton: null,
                filterOverlay: null
            },
            controls: {
                table: null,
                from: {
                    date: null,
                    time: null
                },
                to: {
                    date: null,
                    time: null
                }
            },
            tokens: [],
            listeners: []
        },
        /* imports added at runtime */
        imports: {
            /* example - add import via shmi.requires(...) */
            im: "visuals.session.ItemManager",
            /* example - add import via function call */
            qm: function() {
                return shmi.visuals.session.QueryManager;
            },
            dgm: "visuals.session.DataGridManager",
            iter: "visuals.tools.iterate.iterateObject"
        },
        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this,
                    grid = self.imports.dgm.getGrid(alarmGridConfig.name),
                    iter = self.imports.iter,
                    missingElements = false,
                    io = shmi.requires("visuals.io");

                if (!grid) {
                    grid = self.imports.dgm.grids[alarmGridConfig.name] = new shmi.visuals.core.DataGridDB(alarmGridConfig.name, shmi.c(alarmGridConfig.db), alarmGridConfig.table, alarmGridConfig.fields, alarmGridConfig.conditions);
                    log("grid created:", alarmGridConfig.name);
                } else {
                    console.debug(uiType, "grid exists:", alarmGridConfig.name);
                }
                self.vars.grid = grid;
                self.vars.resetButton = shmi.getUiElement("reset-button", self.element);
                var resetButtonHandler = {
                    onClick: function() {
                        clearFilter(self);
                    }
                };
                self.vars.timeFilterLabel = shmi.getUiElement("filter-label", self.element);
                self.vars.timeFilterLabel.textContent = shmi.localize("${alarmlist_set_timefilter}");
                self.vars.activeFilterLabel = shmi.getUiElement("time-frame-label", self.element);
                self.vars.activeFilterFrom = shmi.getUiElement("start-date", self.element);
                self.vars.activeFilterTo = shmi.getUiElement("end-date", self.element);

                var resetMl = new io.MouseListener(self.vars.resetButton, resetButtonHandler),
                    resetTl = new io.TouchListener(self.vars.resetButton, resetButtonHandler);
                self.vars.listeners.push(resetMl, resetTl);

                var dateformat = shmi.localize(self.config.dateformat);
                alarmTableConfig["default-field-control-map"]["alarm-start"].config.dateformat = dateformat;
                alarmTableConfig["default-field-control-map"]["alarm-end"].config.dateformat = dateformat;

                getAnchorElements(self);
                iter(self.vars.anchors, function(elem, name) {
                    if (!elem) {
                        console.error(className, "element not found:", name + "-anchor");
                        missingElements = true;
                    }
                });

                if (!missingElements) {
                    createControls(self);
                    var filterHandler = {
                            onClick: function() {
                                var filterClass = "filter-open";
                                if (!self.locked) {
                                    if (!shmi.hasClass(self.element, filterClass)) {
                                        shmi.addClass(self.element, filterClass);
                                        if (!self.vars.firstOpened) {
                                            self.vars.controls.to.date.setValue(self.vars.controls.from.date.getValue() + 86400);
                                            self.vars.controls.to.time.setValue(self.vars.controls.from.time.getValue());
                                            self.vars.firstOpened = true;
                                        }
                                    } else {
                                        shmi.removeClass(self.element, filterClass);
                                        setFilter(self);
                                    }
                                }
                            }
                        },
                        ml = new io.MouseListener(self.vars.anchors.filterButton, filterHandler),
                        tl = new io.TouchListener(self.vars.anchors.filterButton, filterHandler);
                    self.vars.listeners.push(ml, tl);

                    var overlayHandler = {
                            onClick: function() {
                                var filterClass = "filter-open";
                                shmi.removeClass(self.element, filterClass);
                                setFilter(self);
                            }
                        },
                        mlo = new io.MouseListener(self.vars.anchors.filterOverlay, overlayHandler),
                        tlo = new io.TouchListener(self.vars.anchors.filterOverlay, overlayHandler);
                    self.vars.listeners.push(mlo, tlo);
                }
            },
            /* called when control is enabled */
            onEnable: function() {
                var self = this,
                    from = self.vars.controls.from,
                    nowTime = Math.floor((new Date(shmi.getServerTime() * 1000)).getTime() / 1000),
                    nowTimeDate = Math.floor((nowTime) / 86400) * 86400;

                self.controls.forEach(function(c) {
                    c.enable();
                });

                from.date.setValue(nowTimeDate - 86400);
                from.time.setValue(nowTime);

                attachChangeListeners(self);
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });
            },
            /* called when control is disabled */
            onDisable: function() {
                var self = this;
                self.controls.forEach(function(c) {
                    c.disable();
                });
                self.vars.tokens.forEach(function(t) {
                    t.unlisten();
                });
                self.vars.firstOpened = false;
                self.vars.tokens = [];
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
            },
            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
            },
            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });
            },
            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {

            },
            /** Sets min & max values and stepping of subscribed variable **/
            onSetProperties: function(min, max, step) {

            }
        }
    };

    //definition of new control extending BaseControl - END

    //generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

(function() {
    /**
     * module to display alarm-detail messages in alarm-table.
     *
     */

    var MODULE_NAME = "visuals.tools.alarms.ls.alarmDetails",
        ENABLE_LOGGING = true,
        RECORD_LOG = false,
        logger = shmi.requires("visuals.tools.logging").createLogger(MODULE_NAME, ENABLE_LOGGING, RECORD_LOG),
        fLog = logger.fLog,
        log = logger.log,
        module = shmi.pkg(MODULE_NAME);

    // MODULE CODE - START

    /* private variables */

    /* private functions */

    /**
     * Implements local-script run function.
     *
     * This function will be called each time a local-script will be enabled.
     *
     * @author Felix Walter <walter@smart-hmi.de>
     * @param {LocalScript} self instance reference of local-script control
     */
    module.run = function(self) {
        var tokens = [],
            detailsButton = null,
            im = shmi.requires("visuals.session.ItemManager"),
            nv = shmi.requires("visuals.tools.numericValues"),
            uc = shmi.visuals.tools.unitClasses,
            itemHandler = im.getItemHandler(),
            alarmInfo = null;
        self.vars = self.vars || {};
        detailsButton = shmi.createControl("button", self.element.parentNode, { label: "${alarmlist_show_detail}" }, "DIV");
        itemHandler.setValue = function(value) {
            alarmInfo = JSON.parse(value);
            if (alarmInfo && Array.isArray(alarmInfo.items)) {
                alarmInfo.items.forEach(function(info, idx) {
                    info.formattedValue = nv.formatNumber(value, {
                        unit: info.unit,
                        precision: info.digits
                    });

                    if (typeof info.unit === "number") {
                        var adapter = uc.getSelectedAdapter(info.unit);
                        if (adapter) {
                            info.unit = adapter.unitText;
                            info.value = adapter.outFunction(info.value);
                        }
                    }

                    if (typeof info.digits === "number" && info.digits >= 0) {
                        info.value = info.value.toFixed(info.digits);
                    }
                });
            }
        };
        tokens.push(detailsButton.listen("click", function() {
            var localeVar = shmi.evalString("${alarm_msg_<%= VALUE %>}", { VALUE: alarmInfo.index }),
                translated = shmi.localize(localeVar);
            shmi.notify(shmi.evalString(translated, alarmInfo));
        }));
        if (self.config.item) {
            tokens.push(im.subscribeItem(self.config.item, itemHandler));
        }
        /* called when this local-script is disabled */
        self.onDisable = function() {
            self.run = false; // from original .onDisable function of LocalScript control
            tokens.forEach(function(t) {
                t.unlisten();
            });
            tokens = [];
            shmi.deleteControl(detailsButton, true);
        };
    };
    // MODULE CODE - END

    fLog("module loaded");
})();

/**
 * Control "alarm-info""
 *
 * Configuration options (default):
 *
 * {
        "class-name": uiType,
        "name": null,
        "template": "default/alarm-info",
        "label": uiType,
        "noAlarm": null,
        "action": null,
        "lastAlarmsNum": null,
        "lastAlarmMsg": null,
        "lastAlarmClass": null,
        "userGroups": "",
        "enableCycle": true,
        "cycleInterval": 1000,
        "showAlarm": true,
        "showWarn": true,
        "showInfo": true
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 *
 */
(function() {
    'use strict';

    //variables for reference in control definition
    const className = "AlarmInfo", //control name in camel-case
        uiType = "alarm-info", //control keyword (data-ui)
        isContainer = false;

    //example - default configuration
    const defConfig = {
        "class-name": uiType,
        "name": null,
        "template": "default/alarm-info",
        "label": uiType,
        "noAlarm": null,
        "action": null,
        "userGroups": null,
        "enableCycle": true,
        "cycleInterval": 1000,
        "showAlarm": true,
        "showWarn": true,
        "showInfo": true,
        "groupFilter": null
    };

    //setup module-logger
    const ENABLE_LOGGING = true,
        RECORD_LOG = false,
        logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG),
        log = logger.log;

    // Constants and private fields
    const ALARM_STATE_NAMES = ["service", "preWarn", "warn"];

    //declare private functions - START

    /**
     * hide - hide element (apply CSS class "hidden")
     *
     * @param {HTMLElement} el element to hide
     */
    function hide(el) {
        if (el) {
            shmi.addClass(el, "hidden");
        }
    }

    /**
     * show - show element (remove CSS class "hidden")
     *
     * @param {HTMLElement} el element to hide
     */
    function show(el) {
        if (el) {
            shmi.removeClass(el, "hidden");
        }
    }

    /**
     * parseAlarmGroupRange - parse range of alarm groups
     *
     * @param {string} rangeText input text (e.g. `3-7`)
     * @returns {number[2]|null} range start & end tuple or `null` if none could be parsed
     */
    function parseAlarmGroupRange(rangeText) {
        const tuple = rangeText.split("-");

        if (tuple.length === 2) {
            const start = parseInt(tuple[0]),
                end = parseInt(tuple[1]);

            if (!(isNaN(start) || isNaN(end)) && start > 0 && end > 1 && end > start) {
                return [start, end];
            }
        }

        return null;
    }

    /**
     * parseAlarmGroupFilter - parse group ID filter from config string
     *
     * @param {string} groupFilter comma separated group ID filter
     * @returns {number[]} array of group IDs
     */
    function parseAlarmGroupFilter(groupFilter) {
        if (typeof groupFilter !== "string") {
            return null;
        }

        let entries = groupFilter.split(",");
        entries = entries.map((entry) => {
            if (entry.includes("-")) {
                return parseAlarmGroupRange(entry);
            }

            const groupId = parseInt(entry);
            if (!isNaN(groupId) && groupId >= 0) {
                return groupId;
            }

            return null;
        });
        entries = entries.filter((entry) => entry !== null);
        const rangeEntries = [];
        entries.forEach((entry) => {
            if (Array.isArray(entry)) {
                let idx = entry[0];
                while (idx <= entry[1]) {
                    if (!(rangeEntries.includes(idx) && entries.includes(idx))) {
                        rangeEntries.push(idx);
                    }
                    idx += 1;
                }
            }
        });
        entries.push(...rangeEntries);

        return entries.length ? entries : null;
    }

    /**
     * testAlarmState - test & update state of displayed alarm
     *
     * @param {object} self instance reference
     */
    function testAlarmState(self) {
        const alarmArray = Object.values(self.imports.am.alarms).map((value) => value.properties);
        if (Array.isArray(alarmArray)) {
            if (alarmArray.length < 1) {
                clearAlarmMsg(self);
            } else {
                clearInt(self);
                if (!self.config.enableCycle) {
                    // standard mode: lookup for last Alarm of highest class
                    self.vars.activeAlarmList = getLastHighest(self, alarmArray);
                } else {
                    // cycleMode: cycle display of all active alarms
                    self.vars.activeAlarmList = getEnabledAlarms(self, alarmArray);
                    if (self.vars.activeAlarmList.length > 1) {
                        self.vars.interval = setInterval(displayAlarms.bind(null, self), self.config.cycleInterval);
                    }
                }
                displayAlarms(self);
            }
        }
    }

    // helper: displays alarms in activeAlarmList
    function displayAlarms(self) {
        if (self.vars.activeAlarmList.length === 0) {
            clearAlarmMsg(self);
        }
        if (self.vars.currentAlarm >= self.vars.activeAlarmList.length) {
            self.vars.currentAlarm = 0;
        }
        clearAlarmState(self);
        const alarm = self.vars.activeAlarmList[self.vars.currentAlarm++];
        let msg = null;
        if (alarm) { // evaluate context items etc.
            msg = shmi.evalString("${alarm_title_<%= value %>}", { value: alarm.index });
            msg = shmi.localize(msg);
            msg = shmi.evalString(msg, alarm);
            if (!isNaN(alarm.severity)) {
                setAlarmState(alarm.severity, self);
            }
        } else {
            msg = shmi.localize(self.config.noAlarm);
        }
        self.vars.lastAlarmMsg.textContent = msg;

        if (self.vars.lastAlarmsNum) {
            self.vars.lastAlarmsNum.textContent = self.vars.lastAlarmsNumValue;
        }
    }

    // helper: get alarm object of the last Alarm and highest class (=severity)
    function getLastHighest(self, alarmList) {
        let lastAlarm = null;
        getEnabledAlarms(self, alarmList).forEach(function(alarmInfo) {
            // Find latest alarm with the highest occurring severity
            if (lastAlarm === null) {
                lastAlarm = alarmInfo;
            } else if (alarmInfo.severity >= lastAlarm.severity && alarmInfo.timestamp_in > lastAlarm.timestamp_in) {
                lastAlarm = alarmInfo;
            }
        });

        return lastAlarm === null ? [] : [lastAlarm];
    }
    // helper: select alarms to be displayed dependig on the config.notShowWarn and config.notShowInfo parameters
    function getEnabledAlarms(self, alarmList) {
        const enabledAlarms = alarmList.filter(function(alarmInfo) {
            if (Array.isArray(self.vars.groupFilter)) {
                if (!self.vars.groupFilter.includes(alarmInfo.group)) {
                    return false;
                }
            }

            //filter inactive alarms
            if (!alarmInfo.active) {
                if (!alarmInfo.acknowledgeable) {
                    return false;
                } else if (alarmInfo.acknowledged) {
                    return false;
                }
            }

            // Filter alarms based on control configuration
            return (alarmInfo.severity === 2 && self.config.showAlarm) ||
                (alarmInfo.severity === 1 && self.config.showWarn) ||
                (alarmInfo.severity === 0 && self.config.showInfo);
        });
        self.vars.lastAlarmsNumValue = enabledAlarms.length;
        return enabledAlarms;
    }
    // helper: clears the alarm message
    function clearAlarmMsg(self) {
        clearInt(self);
        self.vars.currentAlarm = 0;
        clearAlarmState(self);
        self.vars.lastAlarmMsg.textContent = shmi.localize(self.config.noAlarm);
    }
    // helper: clears the interval
    function clearInt(self) {
        if (self.vars.interval) {
            clearInterval(self.vars.interval);
            self.vars.interval = null;
        }
    }

    // helper: clears the alarm class (state)
    function clearAlarmState(self) {
        self.vars.stateIcons.forEach(hide);
        ALARM_STATE_NAMES.forEach(function(lvl) {
            shmi.removeClass(self.element, lvl);
        });
    }
    // helper: sets alarm class (state)
    function setAlarmState(state, self) {
        if (state in self.vars.stateIcons) {
            show(self.vars.stateIcons[state]);
        }
        shmi.addClass(self.element, ALARM_STATE_NAMES[state]);
    }
    //declare private functions - END

    //definition of new control extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            hidden: false,
            clickElement: null,
            lastAlarmsNum: null,
            lastAlarmsNumValue: 0,
            lastAlarmMsg: null,
            lastAlarmMsgValue: 0,
            lastAlarmClass: null,
            lastAlarmClassValue: 0,
            listeners: [],
            tokens: [],
            stateIcons: [],
            activeAlarmList: [],
            currentAlarm: 0,
            interval: 0,
            alarmSubscriber: null,
            groupFilter: null
        },
        /* imports added at runtime */
        imports: {
            /* example - add import via shmi.requires(...) */
            im: "visuals.session.ItemManager",
            /* example - add import via function call */
            am: "visuals.session.AlarmManager"
        },
        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                const self = this,
                    io = shmi.requires("visuals.io");

                let ml = null, // MouseListener
                    tl = null, // TouchListener
                    handler = null,
                    userGroups = null,
                    um = null,
                    user = null;

                if (typeof self.config.userGroups === "string" && self.config.userGroups.length > 0) {
                    userGroups = self.config.userGroups.split(",");
                    um = shmi.requires("visuals.session.UserManager");
                    user = um.currentUser;
                    if (user.groupList.filter(function(n) {
                        return userGroups.indexOf(n) !== -1;
                    }).length < 1) {
                        // disable and hide
                        hide(shmi.getUiElement('alarm-info-wrapper', self.element));
                        self.vars.hidden = true;
                        return;
                    }
                }

                self.vars.groupFilter = parseAlarmGroupFilter(self.config.groupFilter);

                self.vars.lastAlarmsNum = shmi.getUiElement('last-alarms-num', self.element);
                if (!self.vars.lastAlarmsNum) {
                    log(uiType, "last-alarms-num not found in template");
                }
                self.vars.lastAlarmMsg = shmi.getUiElement('last-alarm-msg', self.element);
                if (!self.vars.lastAlarmMsg) {
                    log(uiType, "last-alarm-msg not found in template");
                } else {
                    self.vars.lastAlarmMsg.textContent = shmi.localize(self.config.noAlarm);
                }
                self.vars.clickElement = shmi.getUiElement('alert-list-button', self.element);
                if (self.vars.clickElement && Array.isArray(self.config.action)) {
                    handler = {
                        onClick: function(x, y, e) {
                            const core = shmi.requires("visuals.core"),
                                action = new core.UiAction(self.config.action);
                            action.execute();
                        }
                    };
                    ml = new io.MouseListener(self.vars.clickElement, handler);
                    tl = new io.TouchListener(self.vars.clickElement, handler);
                    self.vars.listeners.push(ml, tl);
                } else if (!self.vars.clickElement) {
                    log(uiType, "click element not found in template");
                }
            },
            /* called when control is enabled */
            onEnable: function() {
                const self = this;
                if (self.vars.hidden) {
                    log("Alarm Info is disabled");
                    return;
                }
                self.vars.stateIcons = shmi.getUiElements('state', self.element) || [];
                self.vars.alarmSubscriber = self.imports.am.subscribeAlarms(self, function(alarm, isLast) {
                    if (isLast) {
                        testAlarmState(self);
                    }
                });
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });
            },
            /* called when control is disabled */
            onDisable: function() {
                const self = this;

                self.vars.tokens.forEach(function(t) {
                    t.unlisten();
                });
                self.vars.tokens = [];

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
                self.imports.am.unsubscribeAlarms(self.vars.alarmSubscriber);
                clearInt(self);
            },
            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                const self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
            },
            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                const self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });
            },
            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {

            },
            /** Sets min & max values and stepping of subscribed variable **/
            onSetProperties: function(min, max, step) {

            }
        }
    };

    //definition of new control extending BaseControl - END

    //generate control constructor & prototype using the control-generator tool
    shmi.requires("visuals.tools.control-generator").generate(definition);
})();

/**
 * Alpha Num Keyboard
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * label {string}: label for the overlay container
 * key-preview {boolean}, shows button name preview on touch pressed only
 * auto-key-focus {boolean}: false, // sets tabindex="0" on each key
 * show-enter {boolean]: false, // show the enter button, also disable enter to apply input
 * password-input {boolean}: false, display input as password --> input via real keyboard disabled
 * select-box-enabled {boolean}: true, show keyboard select box
 * value {}: null, // existing value of e.g. input-field
 * callback {function}: null, used to pass value to other origin-control
 *
 * Additional information about the configuration is located in the tutorial: keyboard-layout-configuration
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "AlphaNumKeyBoard", // control name in camel-case
        uiType = "alpha-num-keyboard", // control keyword (data-ui)
        isContainer = true;

    // example - default configuration
    var defConfig = {
        "class-name": uiType,
        "name": null,
        "template": "default/alpha-num-keyboard",
        "label": "",
        "key-preview": true, // shown on touch only
        "show-enter": false,
        "auto-key-focus": false, // set tabindex="0" to key
        "password-input": false,
        "select-box-enabled": true,
        "keyboards": {}
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false,
        logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG),
        fLog = logger.fLog,
        log = logger.log;

    // definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            listeners: [],
            controls: [],
            tokens: [],
            keys: [],
            keyboardMap: {},
            activeModifiers: {},
            activeKeyboard: null,
            activePalette: null,
            activeKeys: null,
            currentText: "",
            keyTypeListeners: {},
            previewContainer: null,
            shiftKeyActive: false,
            showPassword: false
        },
        /* imports added at runtime */
        imports: {
        },
        events: [],
        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                // listeners add by shmi mouse and touch listeners
                self.vars.keyListeners = {
                    onPress: onKeyPressListener.bind(null, self),
                    onClick: onKeyClickListener.bind(null, self),
                    onRelease: onKeyReleaseListener.bind(null, self)
                };

                // custom listeners added with browser method 'addEventListener'
                self.vars.eventListeners = {
                    mouseOver: onKeyMouseInListener.bind(null, self),
                    mouseOut: onKeyMouseOutListener.bind(null, self),
                    mouseUp: onKeyMouseUpListener.bind(null, self),
                    mouseDown: onKeyMouseDownListener.bind(null, self),
                    touchStart: onKeyTouchStartListener.bind(null, self),
                    touchEnd: onKeyTouchEndListener.bind(null, self),
                    touchMove: onKeyTouchMoveListener.bind(null, self),
                    keyDown: onKeyDownListener.bind(null, self),
                    keyUp: onKeyUpListener.bind(null, self)
                };

                // html element to insert keyboard
                self.vars.keyboardContainer = shmi.getUiElement('keyboard-container', self.element);

                // get configurations
                var keyboardConfig = shmi.requires("shmi.visuals.session.keyboards"),
                    localeInfo = shmi.requires("visuals.session.localeInfo"),
                    um = shmi.requires("visuals.session.UserManager");

                if (!keyboardConfig) {
                    return;
                }
                self.config.keyboards = keyboardConfig;

                // get and set current locale
                if (um.currentUser && um.currentUser.loggedIn && keyboardConfig && localeInfo.locales[um.currentUser.locale] &&
                    localeInfo.locales[um.currentUser.locale].keyboard && keyboardConfig[localeInfo.locales[um.currentUser.locale].keyboard]) {
                    self.vars.currentLocale = localeInfo.locales[um.currentUser.locale].keyboard;
                } else if (localeInfo.locales[localeInfo.default] && localeInfo.locales[localeInfo.default].keyboard &&
                    keyboardConfig[localeInfo.locales[localeInfo.default].keyboard]) {
                    self.vars.currentLocale = localeInfo.locales[localeInfo.default].keyboard;
                } // if currentLocale coun't be set the first keyboard configuration is used

                // init keyboard
                var languages = Object.keys(keyboardConfig); // keys/names of all configured keyboard languages

                // initialize seperate keyboard control like keyboard language selection
                initKeyboardControls(self);

                languages.forEach(function(langName) {
                    // create keyboards for each language registered
                    var keyboard = createKeyboard(self, langName, keyboardConfig[langName]);
                    self.vars.keyboardContainer.appendChild(keyboard.element);
                    // save the keyboard in a map with name as key (keyboard is not a control!)
                    self.vars.keyboardMap[langName] = keyboard;
                });

                // has been temporary preview container, can be used as final preview container if needed
                self.vars.previewContainer = shmi.getUiElement('text-preview', self.element);
                self.vars.currentText = self.config.value || self.vars.currentText;
                updatePreviewTextContent(self);
                self.vars.previewContainer.focus();

                var activePaletteElement = null;
                // set initially active keyboard and palette name
                if (self.vars.currentLocale) {
                    self.vars.activeKeyboard = self.vars.keyboardMap[self.vars.currentLocale];
                } else {
                    self.vars.activeKeyboard = self.vars.keyboardMap[Object.keys(self.vars.keyboardMap)[0]];
                }
                activePaletteElement = Object.keys(self.vars.activeKeyboard.paletteMap)[0];

                // set initial activePalette and activeKeyMap to be referenced later
                self.vars.activePalette = self.vars.activeKeyboard.paletteMap[activePaletteElement];
                self.vars.activeKeyMap = self.vars.activePalette.keyMap;

                // initially show the first keyboard while all others are hidden by default with display: none!
                self.vars.activeKeyboard.element.style.display = 'block';

                if (self.config['key-preview']) {
                    // add css class "key-preview-enabled" to enable the popup of the key name preview
                    shmi.addClass(self.element, 'key-preview-enabled');
                }

                if (self.config['password-input']) {
                    shmi.addClass(self.element, 'password-input');
                }

                if (!self.config['show-enter']) {
                    shmi.addClass(self.element, 'hide-enter');
                }

                // set/enable listeners for each key in active key map
                enableActiveKeyMapListeners(self);
            },
            /**
             * Enables the Alpha Num Keyboard
             */
            onEnable: function() {
                var self = this;
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });
                self.vars.controls.forEach(function(c) {
                    c.enable();
                });

                document.addEventListener('keydown', self.vars.eventListeners.keyDown);
                document.addEventListener('keyup', self.vars.eventListeners.keyUp);
                shmi.log("[Alpha Num Keyboard] enabled", 1);
            },
            /**
             * Disables the Alpha Num Keyboard
             *
             */
            onDisable: function() {
                var self = this;
                self.hide();

                document.removeEventListener('keydown', self.vars.eventListeners.keyDown);
                document.removeEventListener('keyup', self.vars.eventListeners.keyUp);

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
                self.vars.controls.forEach(function(c) {
                    c.disable();
                });
                self.vars.tokens.forEach(function(t) {
                    t.unlisten();
                });
                shmi.log("[Alpha Num Keyboard] disabled", 1);
            },
            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
                self.vars.controls.forEach(function(c) {
                    c.lock();
                });
            },
            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });
                self.vars.controls.forEach(function(c) {
                    c.unlock();
                });
            },
            /**
             * Displays the Alpha Num Keyboard
             */
            show: function() {
                shmi.removeClass(this.element, 'closed');
                shmi.addClass(this.element, 'open');
            },
            /**
             * Hides the Alpha Num Keyboard from DOM
             */
            hide: function() {
                shmi.removeClass(this.element, 'open');
                shmi.addClass(this.element, 'closed');
            },
            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(val, type, name) {

            },
            /** Sets min & max values and stepping of subscribed variable **/
            onSetProperties: function(min, max, step) {

            }
        }
    };

    // #        Building the Keyboard       # //

    /**
     * initiate creation of keyboard
     *
     * @param {object} self
     * @param {string} keyboardName name of keyboard - de,us,...
     * @param {object} keyboardConfig configuration of keyboard
     * @returns {{paletteNames: string[], name: *, element: HTMLElement}} Object that contains the palette names, the keyboard name and element
     */
    function createKeyboard(self, keyboardName, keyboardConfig) {
        var paletteNames = Object.keys(keyboardConfig),
            keyboardElement = document.createElement('DIV'),
            keyboardPaletteMap = {};

        shmi.addClass(keyboardElement, 'keyboard');
        shmi.addClass(keyboardElement, 'keyboard-' + keyboardName.toLowerCase());
        keyboardElement.setAttribute('data-name', 'keyboard-' + keyboardName);

        paletteNames.forEach(function(paletteName, idx) {
            var paletteConfig = keyboardConfig[paletteName],
                palette = createKeyboardPalette(self, paletteName, paletteConfig),
                id = getUnusedId(keyboardPaletteMap);
            palette.id = id;
            palette.element.setAttribute('data-key-id', id);

            // sets display on first palette of the keyboard to flex and others to none to show the first one initially
            palette.element.style.display = (idx === 0 ? 'flex' : 'none');
            keyboardElement.appendChild(palette.element);
            keyboardPaletteMap[id] = palette;
        });

        return {
            paletteMap: keyboardPaletteMap,
            name: keyboardName,
            element: keyboardElement
        };
    }

    /**
     * creates a keyboard palette, wich represents one face
     * (e.g. letter, numbers, special characters, ...)
     *
     * @param {object} self
     * @param {string} paletteName name of palette
     * @param {object} paletteRows configuration of key-rows from palette
     * @returns {object} palette with name, keymap and element
     */
    function createKeyboardPalette(self, paletteName, paletteRows) {
        var paletteElement = document.createElement('DIV'),
            createdPaletteRows = [],
            keyboardPaletteKeyMap = {};

        shmi.addClass(paletteElement, 'keyboard-palette');
        shmi.addClass(paletteElement, 'keyboard-palette-' + paletteName.toLowerCase());
        paletteElement.setAttribute('data-name', 'palette-' + paletteName);

        // creating rows of keys for the palette
        paletteRows.forEach(function(rowConfig) {
            var paletteRow = createKeyboardRow(self, rowConfig);
            paletteElement.appendChild(paletteRow.element);
            createdPaletteRows.push(paletteRow);
            // creating keys for each row
            paletteRow.keys.forEach(function(key) {
                var id = getUnusedId(keyboardPaletteKeyMap);
                key["data-key-id"] = id;
                key.element.setAttribute('data-key-id', id);
                keyboardPaletteKeyMap[key["data-key-id"]] = key;
            });
        });

        return {
            name: paletteName,
            element: paletteElement,
            keyMap: keyboardPaletteKeyMap
        };
    }

    /**
     * create a keyboard row, which contains an variable amount of keys
     *
     * @param {object} self
     * @param {object} rowConfig configuration of the keys contained in this row
     * @returns {object} row with element and created keys
     */
    function createKeyboardRow(self, rowConfig) {
        var keyRow = document.createElement('DIV'),
            rowKeys = [];
        shmi.addClass(keyRow, 'key-row');
        rowConfig.forEach(function(keyConfig, idx) {
            var key = createKeyboardKey(self, keyConfig.name, keyConfig, (idx < rowConfig.length / 2));
            keyRow.appendChild(key.element);
            rowKeys.push(key);
            if (key.alternatives) {
                key.alternatives.forEach(function(alt) {
                    rowKeys.push(alt);
                });
            }
        });

        return {
            keys: rowKeys,
            element: keyRow
        };
    }

    /**
     * create a keyboard key
     *
     * @param {object} self
     * @param {string} keyName  id to identify key later
     * @param {object} keyConfig configuration of key
     * @returns {object} key with name, id, configuration and element
     */
    function createKeyboardKey(self, keyName, keyConfig, leftSide) {
        var keyBox = document.createElement('DIV'),
            key = document.createElement('DIV'),
            keyPrev = null,
            keyPrevContent = null,
            alternativeKeys = null;
        if (self.config['key-preview']) {
            keyPrev = document.createElement('DIV');
            keyPrevContent = document.createElement('SPAN');
        }

        // register all used key types to the listeners array
        self.vars.keyTypeListeners[keyConfig.type] = self.vars.keyTypeListeners[keyConfig.type] || {};

        var escapedKeyName = String(keyConfig.name || keyName).replace(/"/g, '\\"');
        shmi.addClass(keyBox, 'key-box');
        shmi.addClass(key, 'key');
        shmi.addClass(key, 'key-' + escapedKeyName);
        key.setAttribute('data-key-name', escapedKeyName);

        if (self.config["auto-key-focus"]) {
            key.setAttribute('tabindex', 0);
        }
        if (keyConfig['class-name']) {
            shmi.addClass(key, keyConfig['class-name']);
        }
        if (Number(keyConfig.size) !== 0) {
            shmi.addClass(keyBox, 'key-box-flex-' + keyConfig.size);
        }
        shmi.addClass(key, 'key-' + keyConfig.type);
        key.setAttribute('data-key-type', keyConfig.type);
        key.innerText = keyConfig.name || '';

        keyBox.appendChild(key);

        // create preview container
        if (keyPrev) {
            shmi.addClass(keyPrev, 'key-preview');
            keyPrevContent.innerText = keyConfig.name || '';
            keyBox.appendChild(keyPrev);
            keyPrev.appendChild(keyPrevContent);
        }

        // create alternative-key container and keys
        if (keyConfig.alternatives) {
            alternativeKeys = document.createElement('DIV');
            shmi.addClass(alternativeKeys, 'alt-keys');
            if (leftSide) {
                shmi.addClass(alternativeKeys, 'left-side');
            } else {
                shmi.addClass(alternativeKeys, 'right-side');
            }
            keyConfig.alternatives.forEach(function(config) {
                var altKey = createKeyboardKey(self, config.name, config);
                alternativeKeys.appendChild(altKey.element);
            });
            shmi.addClass(alternativeKeys, 'numofKeys-' + keyConfig.alternatives.length);
            keyBox.appendChild(alternativeKeys);
        }
        keyConfig.element = keyBox;
        return keyConfig;
    }

    /**
     * update key config when modifier (shift) is pressed
     *
     * @param {object} self
     * @param {object} keyConfig configuration of key
     * @param {object} values modifier values
     */
    function setKeyConfig(self, keyConfig, values) {
        // set html key value
        var newName = values.name || keyConfig.name;
        if (newName) {
            keyConfig.element.firstChild.textContent = newName;
            if (self.config['key-preview']) {
                var prevElem = keyConfig.element.getElementsByTagName('span')[0];
                if (prevElem) {
                    prevElem.innerText = newName;
                }
            }
        }

        // set class for key element
        var newClassName = values['class-name'];
        if (newClassName) {
            if (newClassName.charAt(0) === '!') {
                shmi.removeClass(keyConfig.element, newClassName.substring(1));
            } else {
                shmi.addClass(keyConfig.element, newClassName);
            }
        }

        // set html key size by class name
        var css = keyConfig.element.className,
            size = values.size || keyConfig.size,
            sizeClassName = size ? 'key-box-flex-' + size : null;
        if (keyConfig.element.className.indexOf(sizeClassName) === -1 && sizeClassName) {
            keyConfig.element.className = (" " + css + " ").replace(/\bkey-box-flex-\S*\b/g, ' ');
            shmi.addClass(keyConfig.element, sizeClassName);
        }

        // set html key type
        var type = values.type || keyConfig.type;
        keyConfig.element.firstChild.setAttribute('data-key-type', type);
    }

    /**
     * enable listener of currently displayed keys
     *
     * @param {object} self
     */
    function enableActiveKeyMapListeners(self) {
        Object.keys(self.vars.activeKeyMap).forEach(function(key) {
            var ml = new shmi.visuals.io.MouseListener(self.vars.activeKeyMap[key].element, self.vars.keyListeners),
                tl = new shmi.visuals.io.TouchListener(self.vars.activeKeyMap[key].element, self.vars.keyListeners),
                keyElem = self.vars.activeKeyMap[key].element.firstChild;
            ml.enable();
            tl.enable();
            self.vars.listeners.push(ml, tl);

            // mouse
            keyElem.addEventListener('mouseover', self.vars.eventListeners.mouseOver);
            keyElem.addEventListener('mouseout', self.vars.eventListeners.mouseOut);
            keyElem.addEventListener('mousedown', self.vars.eventListeners.mouseDown);
            keyElem.addEventListener('mouseup', self.vars.eventListeners.mouseUp);
        });

        // touch
        self.vars.activePalette.element.addEventListener('touchstart', self.vars.eventListeners.touchStart);
    }

    /**
     * disable listener of currently displayed keys
     *
     * @param {object} self
     */
    function disableActiveKeyMapListeners(self) {
        self.vars.listeners.forEach(function(listener) {
            listener.disable();
        });
        self.vars.listeners = [];

        Object.keys(self.vars.activeKeyMap).forEach(function(key) {
            var keyElem = self.vars.activeKeyMap[key].element.firstChild;
            keyElem.removeEventListener('mouseover', self.vars.eventListeners.mouseOver);
            keyElem.removeEventListener('mousedown', self.vars.eventListeners.mouseDown);
            keyElem.removeEventListener('mouseout', self.vars.eventListeners.mouseOut);
            keyElem.removeEventListener('mouseup', self.vars.eventListeners.mouseUp);

            self.vars.activePalette.element.removeEventListener('touchstart', self.vars.eventListeners.touchStart);
        });
    }

    /**
     * init Keyboard Controls
     *  - selectBox for keyboard selection
     *  - submit button
     *  - close/abort button
     *
     * @param {object} self
     */
    function initKeyboardControls(self) {
        if (self.config["select-box-enabled"] && Object.keys(self.config.keyboards).length > 1) {
            // selectbox if multiple keyboards are available
            // language selection
            var localeInfo = shmi.requires("visuals.session.localeInfo"),
                languageCtrl = shmi.getUiElement('language-control', self.element),
                selectBoxConfig = {
                    "label": "",
                    "options": []
                },
                languages = Object.keys(self.config.keyboards);

            if (languages && languages.length > 0) {
                languages.forEach(function(langName, idx) {
                    selectBoxConfig.options.push({
                        "label": localeInfo.keyboards[langName].label,
                        "value": langName
                    });
                });
                var selectBox = shmi.createControl("select-box", languageCtrl, selectBoxConfig, "DIV");
                shmi.waitOnInit(selectBox, function() {
                    // default selection
                    if (self.vars.currentLocale) {
                        selectBox.setValue(self.vars.currentLocale);
                    } else {
                        self.config['default-keyboard'] = self.config['default-keyboard'] || languages[0];
                        selectBox.setValue(self.config['default-keyboard']);
                    }

                    self.vars.tokens.push(selectBox.listen("change", function onChange(evt) {
                        // disable and hide current keyboard
                        disableActiveKeyMapListeners(self);
                        self.vars.activePalette.element.style.display = 'none';
                        self.vars.activeKeyboard.element.style.display = 'none';

                        // reset active keyboard and palette
                        self.vars.activeKeyboard = self.vars.keyboardMap[evt.detail.value];
                        var activePaletteElement = null;
                        activePaletteElement = Object.keys(self.vars.activeKeyboard.paletteMap)[0];
                        self.vars.activePalette = self.vars.activeKeyboard.paletteMap[activePaletteElement];
                        self.vars.activeKeyMap = self.vars.activePalette.keyMap;

                        // make keyboard visible
                        self.vars.activeKeyboard.element.style.display = 'block';
                        self.vars.activePalette.element.style.display = 'flex';

                        enableActiveKeyMapListeners(self);
                    }));
                });
                shmi.addClass(languageCtrl, 'enabled');
                self.vars.controls.push(selectBox);
            }
        }

        // close button
        var closeButtonContainer = shmi.getUiElement('close-button', self.element),
            closeButtonConfig = {
                "class-name": "button icon-only",
                "label": "Cancel",
                "template": "default/button_with_bg_pic"
            },
            closeButton = shmi.createControl("button", closeButtonContainer, closeButtonConfig, "DIV");

        self.vars.tokens.push(
            closeButton.listen("click", function onclick(evt) {
                log("Fire: Close Keyboard");
                shmi.fire('keyboard-cancel', {
                    "success": false,
                    "input": self.vars.currentText
                });
            })
        );
        self.vars.controls.push(closeButton);

        // submit button
        var submitButtonContainer = shmi.getUiElement('submit-button', self.element),
            submitButtonConfig = {
                "class-name": "button icon-only",
                "template": "default/button_with_bg_pic"
            },
            submitButton = shmi.createControl("button", submitButtonContainer, submitButtonConfig, "DIV");

        self.vars.tokens.push(
            submitButton.listen("click", function onclick(evt) {
                log("Fire: Submit Keyboard");
                shmi.fire('keyboard-submit', {
                    "success": true,
                    "input": self.vars.currentText
                });
            })
        );
        self.vars.controls.push(submitButton);

        // pw button
        if (self.config["password-input"]) {
            const pwButtonContainer = shmi.getUiElement('pw-button', self.element),
                pwButtonConfig = {
                    "class-name": "button icon-only",
                    "template": "default/button_with_bg_pic"
                },
                pwButton = shmi.createControl("button", pwButtonContainer, pwButtonConfig, "DIV");

            self.vars.tokens.push(
                pwButton.listen("click", function onclick(evt) {
                    self.vars.showPassword = !self.vars.showPassword;
                    updatePreviewTextContent(self);
                })
            );
            self.vars.controls.push(pwButton);
        }
    }

    var idAlphabet = [
        "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M",
        "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z"
    ];

    /**
     * getRandomId - generates random ID string of variable length
     *
     * @param  {number} length length of generated ID string
     * @return {string}        random ID
     */
    function getRandomId(length) {
        var rid = "";
        if (length === undefined) {
            length = 3;
        }
        for (var i = 0; i < length; i++) {
            rid += idAlphabet[Math.floor(Math.random() * (idAlphabet.length - 1))];
        }
        return rid;
    }

    /**
     * Get a unused random id
     *
     * @param obj unused in this context
     * @return    unused id
     */
    function getUnusedId(obj) {
        var id = null;
        do {
            id = getRandomId(8);
        } while (shmi.objectHasOwnProperty(obj, id));
        return id;
    }

    // #        Keyboard handler        # //

    /**
     * fireKeyboardClick - fires action of a pressed/clicked keyboard key
     *
     * @param {object} self
     * @param {object} key key where the event was triggert
     * @param {object} event event from mouse, key or touch event
     */
    function fireKeyboardClick(self, key, event) {
        // execute action depending on the type of the pressed/clicked key
        switch (key.type) {
        case 'letter':
        case 'number':
        case 'character':
            // simply add the value of the pressed key to the preview text content
            //addLetterToCurrentText(self, key);
            addLetterIntoCurrentText(self, key);
            break;
        case 'palette':
            // a key of type palette has to have a palette attribute with the name of the palette to be activated
            if (key.palette) {
                setKeyboardPalette(self, key);
            } else {
                console.error('[alpha-num-keyboard] Missing palette attribute value on key of type palette');
            }
            break;
        case 'modifier':
            // a key of type modifier has to have a property value of modify
            var modConfig = key.modify ? key.modify.split(':') : null,
                type, mode, name;
            // modify value has to contain type:mode:name
            if (modConfig && modConfig.length === 3) {
                type = modConfig[0];
                mode = modConfig[1];
                name = modConfig[2];
                if (!isModifierActive(self, name)) {
                    setModifier(self, name, mode, type, event);
                } else {
                    // Check if this is unset twice in case of the clip or clipOnce callback fires too!
                    // The unsetModifier method only executes if the modifier is still active
                    shmi.removeClass(event.target, 'clipped');
                    unsetModifier(self, name, mode, type, event);
                }
            }
            break;
        case 'resolver':
            keyResolver(self, key);
            break;
        default:
            log('uncatched keyboard event: ' + key.type);
        }
    }

    /**
     * is fired when a key with type: resolver is pressed
     * (e.g. backspace, submit, ..)
     *
     * @param {object} self
     * @param {object} key key where the event was triggert
     */
    function keyResolver(self, key) {
        var resolve = key.resolve;
        if (resolve) {
            // define addition actions on resolve which is meant to modify the current text
            // add any resolve case with further methods
            switch (resolve) {
            case 'backspace':
                deleteLetterFromCurrentText(self);
                break;
            case 'submit':
                log("Fire: Submit Keyboard");
                shmi.fire('keyboard-submit', {
                    "success": true,
                    "input": self.vars.currentText
                });
                break;
            default:
                log("Missing Default Case");
            }
        }
    }

    /**
     * set Modifier
     * a modifier is somthing like `shift` or `alt`
     *
     * @param {object} self
     * @param {string} name  name of modifiers, like `shift`
     * @param {string} mode modifier-mode for the switch-case, like `clip` or `clipOnce`
     * @param {string} type type of keys that can be modified by this modifier (e.g. `letter`)
     * @param {object} setEvent event from mouse, key or touch event
     */
    function setModifier(self, name, mode, type, setEvent) {
        if (!isModifierActive(self, name)) {
            var id = getActivePaletteId(self);
            self.vars.activeModifiers[id] = self.vars.activeModifiers[id] || [];
            self.vars.activeModifiers[id].push(name);
            // the unsetModifier is registered with the modification mode and type listener
            createModificationTypeListeners(self, name, mode, type, setEvent);
            Object.keys(self.vars.activeKeyMap).forEach(function(key) {
                if (self.vars.activeKeyMap[key].type === type) {
                    var values = evaluateKeyModifiers(self, self.vars.activeKeyMap[key]);
                    setKeyConfig(self, self.vars.activeKeyMap[key], values);
                }
            });
        }
    }

    /**
     * unset Modifiers
     * a modifier is somthing like `shift` or `alt`
     *
     * @param {object} self
     * @param {string} name  name of modifiers, like `shift`
     * @param {string} mode modifier-mode for the switch-case, like `clip` or `clipOnce`
     * @param {string} type type of keys that can be modified by this modifier (e.g. `letter`)
     * @param {object} event event from mouse, key or touch event
     */
    function unsetModifier(self, name, mode, type, event) {
        if (isModifierActive(self, name)) {
            var id = getActivePaletteId(self),
                idx = self.vars.activeModifiers[id].indexOf(name);
            // remove the of the active modifiers array
            self.vars.activeModifiers[id].splice(idx, 1);
            Object.keys(self.vars.activeKeyMap).forEach(function(key) {
                var values = evaluateKeyModifiers(self, self.vars.activeKeyMap[key]);
                setKeyConfig(self, self.vars.activeKeyMap[key], values);
            });
        }
    }

    /**
     * create function for when modifier-mode is `clip`
     * (deutsch: dauerhaftes Feststell dieser Taste)
     *
     * @param {object} self
     * @param {string} name name of modifiers, like `shift`
     * @param {string} mode modifier-mode, `clip`
     * @param {string} type type of keys that can be modified by this modifier (e.g. `letter`)
     * @param {string} id   identifier
     * @param {object} event event from mouse, key or touch event
     */
    function createClipModifierFunction(self, name, mode, type, id, event) {
        return function modClip(unsetEvent, key) {
            // listener callback gets triggered on each click on a button of the specified type
            if (unsetEvent !== event && key.modify === type + ':' + mode + ':' + name) {
                // only unset the clip modifier if key has the same modify type
                delete self.vars.keyTypeListeners[type][mode][id]; // remove registered listener callback
                unsetModifier(self, name, mode, type, event); // unset the modifer
                return false; // prevent any further actions: boolean
            }
            return true;
        };
    }

    /**
     * create function for when modifier-mode is `clipOnce`
     * (deutsch: einmaliges Feststell dieser Taste)
     *
     * @param {object} self
     * @param {string} name name of modifiers, like `shift`
     * @param {string} mode modifier-mode, `clipOnce`
     * @param {string} type type of keys that can be modified by this modifier (e.g. `letter`)
     * @param {string} id   identifier
     * @param {object} event event from mouse, key or touch event
     */
    function createClipOnceModifierFunction(self, name, mode, type, id, event) {
        return function modClipOnce(unsetEvent, key) {
            // listener callback gets triggered on each click on a button of the specified type
            if (unsetEvent !== event) {
                // only unset the modifier if it did not get set by the current event
                if (key.type === type || key.modify === type + ':' + mode + ':' + name) {
                    // only unset modifier if the pressed key is of type being modified or has same modify type
                    shmi.removeClass(event.target, 'clipped');
                    delete self.vars.keyTypeListeners[type][mode][id];
                    unsetModifier(self, name, mode, type, event);
                    return false; // prevent any further actions: boolean
                }
            }
            return true;
        };
    }

    /**
     * creates listener and function that is called, when button is pressed and a modifier is active
     *
     * @param {object} self
     * @param {string} name name of modifiers, like `shift`
     * @param {string} mode modifier-mode for the switch-case, like `clip` or `clipOnce`
     * @param {string} type type of keys that can be modified by this modifier (e.g. `letter`)
     * @param {object} event event from mouse, key or touch event
     */
    function createModificationTypeListeners(self, name, mode, type, event) {
        var id = getRandomId(8);
        if (self && name && mode && type && event) {
            // these are different types of modifier modes
            switch (mode) {
            // the clip modifier mode keeps the modifier enabled until the button gets pressed again
            case 'clip':
                // ensure the mode is an object
                self.vars.keyTypeListeners[type][mode] = self.vars.keyTypeListeners[type][mode] || {};
                // define callback in listener for a pressed button of specific type and mode identified by a random id
                self.vars.keyTypeListeners[type][mode][id] = createClipModifierFunction(self, name, mode, type, id, event);
                break;
            // the clipOnce modifier mode keeps the modifier enable until one other button is pressed
            case 'clipOnce':
                // add class 'clipped', contains the same style as 'pressed', doesn't get removed on release
                shmi.addClass(event.target, 'clipped');
                // ensure key type listener for mode is an object
                self.vars.keyTypeListeners[type][mode] = self.vars.keyTypeListeners[type][mode] || {};
                // define callback in listener for a pressed button of specific type and mode identified by a random id
                self.vars.keyTypeListeners[type][mode][id] = createClipOnceModifierFunction(self, name, mode, type, id, event);
                break;
            default:
                log("Undefined modifier mode", mode);
            }
        }
    }

    /**
     * get all modifier of given key
     *
     * @param {object} self
     * @param {object} key key context
     * @returns {object} modifier of key
     */
    function evaluateKeyModifiers(self, key) {
        const values = {},
            modifiers = shmi.cloneObject(key.modifiers),
            iterObj = shmi.requires("visuals.tools.iterate.iterateObject");

        iterObj(modifiers || {}, (mapped, modifier) => {
            const modVars = modifier.split(':');
            if (modVars.length !== 2) {
                console.error("[Keyboard] - modifier not configured right", key);
            } else if (isModifierActive(self, modVars[0])) {
                // add class if modifier is active
                values[modVars[1]] = mapped;
            } else if (modVars[1] === 'class-name') {
                // and remove it if the modifier is inactive
                values[modVars[1]] = '!' + mapped;
            }
        });

        return values;
    }

    /**
     * set keyboard palette by given key config
     *
     * @param {object} self
     * @param {object} key key, that initiates palatte change
     */
    function setKeyboardPalette(self, key) {
        var paletteName = key.palette,
            newPalette = null;

        // search for palette with paletteName within available palettes of active keyboard
        newPalette = Object.values(self.vars.activeKeyboard.paletteMap).filter(function(palette) {
            return palette && (palette.name === (paletteName));
        })[0];

        if (newPalette) {
            // hide current palette
            self.vars.activePalette.element.style.display = 'none';

            // disable all listeners of the current active palette
            disableActiveKeyMapListeners(self);

            self.vars.activePalette = newPalette;
            self.vars.activeKeyMap = self.vars.activePalette.keyMap;

            // show new palette
            // important: shown palette element has to be display flex
            self.vars.activePalette.element.style.display = 'flex';

            // enable the new active palette key listeners
            enableActiveKeyMapListeners(self);
        } else {
            console.error('[Keyboard] - can not find palette with name: "' + paletteName + '"');
        }
    }

    /**
     * check if given modifier is active
     *
     * @param {object} self
     * @param {string} modifierName name of modifier
     * @returns {boolean} if modifier by modifier name is currently active or not
     */
    function isModifierActive(self, modifierName) {
        var id = getActivePaletteId(self);
        return self.vars.activeModifiers[id] && self.vars.activeModifiers[id].indexOf(modifierName) !== -1;
    }

    /**
     * get id of active palette
     *
     * @returns {string}
     */
    function getActivePaletteId(self) {
        return self.vars.activeKeyboard.name.toLowerCase() + ':' + self.vars.activePalette.name.toLowerCase();
    }

    /**
     * removeAllPressedHighlights - removes all `pressed` classes.
     * (this is why we have a clipped class with the same styles too ;) )
     *
     * @param {object} self
     */
    function removeAllPressedHighlights(self) {
        self.vars.activePalette.element.querySelectorAll('.key.pressed').forEach(function(pressedElement) {
            shmi.removeClass(pressedElement, 'pressed');
        });
    }

    /**
     * removeAllAltKeys - removes all `show-alt-keys` classes.
     *
     * @param {object} self
     */
    function removeAllAltKeys(self) {
        clearTimeout(self.vars.altKeysTimeout); // removes timeout to show alternative keys
        self.vars.activePalette.element.querySelectorAll('.key.show-alt-keys').forEach(function(pressedElement) {
            shmi.removeClass(pressedElement, 'show-alt-keys');
        });
    }

    /**
     * fire type listener
     *
     * @param {object} self
     * @param {object} key key configuration
     * @param {object} event event from mouse, key or touch event
     * @returns {boolean} if further actions should be prevented or be allowed
     */
    function fireTypeListeners(self, key, event) {
        var iter = shmi.requires("visuals.tools.iterate").iterateObject;
        var type = key.type,
            prevented = false;
        if (type) {
            var typeListeners = self.vars.keyTypeListeners[type];
            iter(typeListeners, function(listenerCallbacks, mode) {
                iter(listenerCallbacks, function(callback, id) {
                    if (callback(event, key) === false) {
                        prevented = true;
                    }
                });
            });
        }
        return !prevented;
    }

    /**
     * get key by value
     *
     * @param {object} self
     * @param {object} event event from mouse, key or touch event
     * @returns {object} key with the given value
     */
    function getKeyByValue(self, event) {
        var keyMap = null;
        keyMap = Object.values(self.vars.activeKeyMap).filter(function(key) {
            var eventKey = event.key.toLowerCase(),
                keyName = key.name;
            return keyName === eventKey;
        })[0];
        return keyMap ? self.vars.activeKeyMap[keyMap["data-key-id"]] : null;
    }

    /**
     * delete letter from current text
     *
     * @param {object} self
     */
    function deleteLetterFromCurrentText(self, forwardMode) {
        var oldText = String(self.vars.currentText),
            newText = oldText,
            newIndex = 0;

        if (self.vars.previewContainer.selectionStart === self.vars.previewContainer.selectionEnd) {
            if (self.vars.previewContainer.selectionStart !== 0) {
                if (forwardMode) {
                    if (getSymbols(oldText.substr(self.vars.previewContainer.selectionStart, self.vars.previewContainer.selectionStart + 2))[1] === "") {
                        newText = oldText.substr(0, self.vars.previewContainer.selectionStart) + oldText.substr(self.vars.previewContainer.selectionEnd + 2);
                        newIndex = self.vars.previewContainer.selectionStart;
                    } else {
                        newText = oldText.substr(0, self.vars.previewContainer.selectionStart) + oldText.substr(self.vars.previewContainer.selectionEnd + 1);
                        newIndex = self.vars.previewContainer.selectionStart;
                    }
                } else if (getSymbols(oldText.substr(self.vars.previewContainer.selectionStart - 2, self.vars.previewContainer.selectionStart))[1] === "") {
                    newText = oldText.substr(0, self.vars.previewContainer.selectionStart - 2) + oldText.substr(self.vars.previewContainer.selectionEnd);
                    newIndex = self.vars.previewContainer.selectionStart - 2;
                } else {
                    newText = oldText.substr(0, self.vars.previewContainer.selectionStart - 1) + oldText.substr(self.vars.previewContainer.selectionEnd);
                    newIndex = self.vars.previewContainer.selectionStart - 1;
                }
            }
        } else {
            newText = oldText.substr(0, self.vars.previewContainer.selectionStart) + oldText.substr(self.vars.previewContainer.selectionEnd);
            newIndex = self.vars.previewContainer.selectionStart;
        }

        self.vars.currentText = newText;
        updatePreviewTextContent(self);
        self.vars.previewContainer.selectionStart = self.vars.previewContainer.selectionEnd = newIndex;
    }

    /**
     * function to seperate all symbols in a string
     * Source: https://mathiasbynens.be/notes/javascript-unicode
     * this is needed to separete UTF-16 encoded characters like emoji's. Could be handy with different languages as well
     *
     * @param {string} string string where symboles should be found
     * @returns {object[]} array of found symboles
     */
    function getSymbols(string) {
        var index = 0;
        var length = string.length;
        var output = [];
        for (; index < length - 1; ++index) {
            var charCode = string.charCodeAt(index);
            if (charCode >= 0xD800 && charCode <= 0xDBFF) {
                charCode = string.charCodeAt(index + 1);
                if (charCode >= 0xDC00 && charCode <= 0xDFFF) {
                    output.push(string.slice(index, index + 2));
                    ++index;
                    continue;
                }
            }
            output.push(string.charAt(index));
        }
        output.push(string.charAt(index));
        return output;
    }

    /**
     * add letter into current text - This method just adds the given letter to the existing content (of the preview)
     * text. It manages also the position of the inserted letter
     *
     * @param {object} self
     * @param {object} key key to be inserted
     */
    function addLetterIntoCurrentText(self, key) {
        var modValues = evaluateKeyModifiers(self, key),
            letter = modValues.val || key.val,
            oldText = String(self.vars.currentText),
            newText = null,
            index = self.vars.previewContainer.selectionStart;
        if (letter && letter.length === 1) {
            newText = oldText.substr(0, self.vars.previewContainer.selectionStart) + letter + oldText.substr(self.vars.previewContainer.selectionEnd);
            self.vars.currentText = newText;
            updatePreviewTextContent(self);
            self.vars.previewContainer.selectionStart = self.vars.previewContainer.selectionEnd = index + 1;
        }
        if (letter && letter.length === 2) {
            newText = oldText.substr(0, self.vars.previewContainer.selectionStart) + letter + oldText.substr(self.vars.previewContainer.selectionEnd);
            self.vars.currentText = newText;
            updatePreviewTextContent(self);
            self.vars.previewContainer.selectionStart = self.vars.previewContainer.selectionEnd = index + 2;
        }
    }

    /**
     * Updates text content of PreviewTextContainer to current text
     * if password-input is enabled the string is replaced with an equal number of *
     *
     * @param {object} self
     */
    function updatePreviewTextContent(self) {
        if (self.vars.previewContainer) {
            if (self.config["password-input"]) {
                let str = '';
                if (self.vars.showPassword === true) {
                    str = self.vars.currentText;
                } else {
                    for (let i = 0; i < self.vars.currentText.length; i++) {
                        str += '●';
                    }
                }
                self.vars.previewContainer.value = str;
            } else {
                self.vars.previewContainer.value = self.vars.currentText;
            }
        }
    }

    // #        Event Listener      # //

    /**
     * onKeyPressListener
     *
     * @param {object} self
     * @param {number} x x postion of cursor
     * @param {number} y y postion of cursor
     * @param {object} event event that is triggered by mouse
     */
    function onKeyPressListener(self, x, y, event) {
        if (event.type.indexOf('mouse') !== -1 && event.type.indexOf('touch') === -1) {
            shmi.addClass(self.element, 'hover-enabled');
        } else {
            shmi.removeClass(self.element, 'hover-enabled');
        }
        self.vars.keyPressed = event.target.getAttribute('data-key-name');

        if (self.config["password-input"]) {
            event.preventDefault();
        } else {
            shmi.addClass(event.target, 'pressed');
        }
    }

    /**
     * onKeyClickListener
     *
     * @param {object} self
     * @param {number} x x postion of cursor
     * @param {number} y y postion of cursor
     * @param {object} event event that is triggered by mouse
     */
    function onKeyClickListener(self, x, y, event) {
        var keyElement = event.target.parentNode,
            keyElementId = keyElement.getAttribute('data-key-id'),
            key = self.vars.activeKeyMap[keyElementId];
        if (keyElement && key) {
            self.vars.keyPressed = false;
            // click = press and release -> set mouse move to null
            self.vars.mouseMoveKey = null;
            fireKeyboardClick(self, key, event);
            fireTypeListeners(self, key, event); // What the heck is this function doing? HELP!
            // |^| returns the true for continue or false for preventing further actions
            // could be used as if(fireTypeListeners(self, key, event)) { fireKeyboardClick(self, key, event); }

            if (self.config["auto-key-focus"] && document.activeElement !== keyElement) {
                // focus clicked key if auto key focus is enabled and key element has a tabindex attribute
                keyElement.focus();
            }
        }
    }

    /**
     * onKeyReleaseListener
     *
     * @param {object} self
     * @param {number} x x postion of cursor
     * @param {number} y y postion of cursor
     * @param {object} event event that is triggered by mouse
     */
    function onKeyReleaseListener(self, x, y, event) {
        if (self.vars.keyPressed) {
            var el = event.target;
            shmi.removeClass(el, 'pressed');
        }
    }

    /**
     * onKeyMouseDownListener
     *
     * @param {object} self
     * @param {object} event event that is triggered by mouse
     */
    function onKeyMouseDownListener(self, event) {
        shmi.removeClass(self.element, 'key-preview-enabled');
        var keyNameAttr = event.target.getAttribute('data-key-name') || false,
            key = self.vars.activeKeyMap[event.target.parentElement.getAttribute('data-key-id')];
        self.vars.keyPressed = keyNameAttr;
        self.vars.mouseMoveKey = null;

        if (self.vars.keyPressed && key && key.hold && !key.alternatives) {
            setTimeout(function() {
                if (self.vars.keyPressed === keyNameAttr) {
                    self.vars.keyHeld = true;
                    var clickAndHoldIV = setInterval(function() {
                        if (self.vars.keyPressed === keyNameAttr) {
                            fireKeyboardClick(self, key, event);
                            fireTypeListeners(self, key, event);
                        } else {
                            clearInterval(clickAndHoldIV);
                        }
                    }, 120);
                }
            }, 1000);
        }
        if (key && key.alternatives) {
            // if alternative keys are available show them after 500ms
            self.vars.altKeysTimeout = setTimeout(function() {
                if (self.vars.keyPressed === keyNameAttr) {
                    shmi.removeClass(self.element, 'key-preview-enabled');
                    shmi.addClass(event.target, 'show-alt-keys');
                }
            }, 500);
        }
    }

    /**
     * onKeyMouseUpListener
     *
     * @param {object} self
     * @param {object} event event that is triggered by mouse
     */
    function onKeyMouseUpListener(self, event) {
        if (self.vars.keyPressed) {
            self.vars.keyPressed = false;
            var key = event.target;
            shmi.removeClass(event.target, 'pressed');
            self.vars.previewContainer.focus();
            if (self.config["auto-key-focus"] && document.activeElement !== key) {
                key.focus();
            }
            key = self.vars.activeKeyMap[key.parentElement.getAttribute('data-key-id')];
            if (key && self.vars.mouseMoveKey) {
                fireKeyboardClick(self, key, event);
                fireTypeListeners(self, key, event);
                self.vars.mouseMoveKey = null;
            }
        }
        removeAllAltKeys(self); // remove displayed alternatives
    }

    /**
     * onKeyMouseInListener
     *
     * @param {object} self
     * @param {object} event event that is triggered by mouse
     */
    function onKeyMouseInListener(self, event) {
        if (self.vars.keyPressed && !self.config["password-input"]) {
            var el = event.target;
            self.vars.mouseMoveKey = !self.vars.keyHeld ? el : null;
            self.vars.keyPressed = el.getAttribute('data-key-name');
            shmi.addClass(el, 'pressed');
        }
    }

    /**
     * onKeyMouseOutListener
     *
     * @param {object} self
     * @param {object} event event that is triggered by mouse
     */
    function onKeyMouseOutListener(self, event) {
        if (self.vars.keyPressed) {
            var el = event.target;
            self.vars.mouseMoveKey = null;
            shmi.removeClass(el, 'pressed');
        }
    }

    /**
     * onKeyTouchStartListener
     *
     * @param {object} self
     * @param {object} event event that is triggered by mouse
     */
    function onKeyTouchStartListener(self, event) {
        if (self.config['key-preview']) {
            shmi.addClass(self.element, 'key-preview-enabled');
        }
        var keyNameAttr = event.target.getAttribute('data-key-name'),
            key = self.vars.activeKeyMap[event.target.parentElement.getAttribute('data-key-id')];
        self.vars.keyPressed = keyNameAttr;
        self.vars.mouseMoveKey = null;
        self.vars.activePalette.element.addEventListener('touchend', self.vars.eventListeners.touchEnd);
        self.vars.activePalette.element.addEventListener('touchmove', self.vars.eventListeners.touchMove);

        if (keyNameAttr && key && key.hold && !key.alternatives) {
            setTimeout(function() {
                if (self.vars.keyPressed === keyNameAttr) {
                    self.vars.keyHeld = true;
                    var pushAndHoldIV = setInterval(function() {
                        if (self.vars.keyPressed === keyNameAttr) {
                            fireKeyboardClick(self, key, event);
                            fireTypeListeners(self, key, event);
                        } else {
                            clearInterval(pushAndHoldIV);
                        }
                    }, 120);
                }
            }, 1000);
        }
        if (key && key.alternatives) {
            // if alternative keys are available show them after 500ms
            self.vars.altKeysTimeout = setTimeout(function() {
                if (self.vars.keyPressed === keyNameAttr) {
                    shmi.removeClass(self.element, 'key-preview-enabled');
                    shmi.addClass(event.target, 'show-alt-keys');
                }
            }, 500);
        }
    }

    /**
     * onKeyTouchMoveListener
     *
     * @param {object} self
     * @param {object} event event that is triggered by mouse
     */
    function onKeyTouchMoveListener(self, event) {
        event.preventDefault();
        event.stopPropagation();
        if (self.vars.keyPressed) {
            var clientY = event.touches[0].clientY,
                clientX = event.touches[0].clientX,
                keyElement = document.elementFromPoint(clientX, clientY);
            if (keyElement) {
                var keyNameAttr = keyElement.getAttribute('data-key-name');
                if (keyNameAttr && keyElement !== self.vars.dragCurrentKey) {
                    if (!self.vars.keyHeld && keyNameAttr !== self.vars.keyPressed) {
                        removeAllPressedHighlights(self);
                        if (!self.config["password-input"]) {
                            shmi.addClass(keyElement, 'pressed');
                        }
                    }
                    self.vars.dragCurrentKey = keyElement;
                    self.vars.keyPressed = keyNameAttr;
                }
            }
        }
    }

    /**
     * onKeyTouchEndListener
     *
     * @param {object} self
     * @param {object} event event that is triggered by mouse
     */
    function onKeyTouchEndListener(self, event) {
        self.vars.keyPressed = false;
        self.vars.activePalette.element.removeEventListener('touchend', self.vars.eventListeners.touchEnd);
        self.vars.activePalette.element.removeEventListener('touchmove', self.vars.eventListeners.touchMove);

        if (self.vars.dragCurrentKey && !self.vars.keyHeld) {
            removeAllPressedHighlights(self);
            self.vars.previewContainer.focus();
            if (self.config["auto-key-focus"] && document.activeElement !== event.target) {
                self.vars.dragCurrentKey.focus();
            }
            var key = self.vars.activeKeyMap[self.vars.dragCurrentKey.parentElement.getAttribute('data-key-id')];
            if (key) {
                fireKeyboardClick(self, key, event);
                fireTypeListeners(self, key, event);
            }
            self.vars.dragCurrentKey = null;
        } else {
            self.vars.keyHeld = false;
        }
        removeAllAltKeys(self); // remove all displayed alternatives
    }

    /**
     * onKeyDownListener
     *
     * @param {object} self
     * @param {object} event event that is triggered by mouse
     */
    function onKeyDownListener(self, event) {
        if (self.vars.lastKey && self.vars.lastKey === event.key) {
            if (self.config["password-input"]) {
                event.preventDefault();
            }
            return;
        }
        self.vars.lastKey = event.key;
        // do not show the key preview if a hardware keyboard event is triggered
        shmi.removeClass(self.element, 'key-preview-enabled');
        shmi.removeClass(self.element, 'hover-enabled');
        var key = getKeyByValue(self, event);
        if (key && !self.config["password-input"]) {
            // highlighting the pressed key
            shmi.addClass(key.element.firstChild, 'pressed');
            if (self.vars.previewContainer && document.activeElement !== self.vars.previewContainer) {
                if (!self.config['auto-key-focus']) {
                    self.vars.previewContainer.focus();
                }
            }
            // get updated String from keyboard input
            self.vars.currentText = self.vars.previewContainer.value;
        }
        if (event.key === "Enter" && !self.config['show-enter']) {
            shmi.fire('keyboard-submit', {
                "success": true,
                "input": self.vars.currentText
            });
        }

        if (self.config["password-input"]) {
            if ((event.key === "v") && event.ctrlKey === true) {
                navigator.clipboard.readText().then(function(clipText) {
                    for (var i = 0; i < clipText.length; i++) {
                        key = clipText.charAt(i);
                        handleKeyHit(key, null, self);
                    }
                });
                event.preventDefault();
                return;
            } else if (event.ctrlKey === true) {
                event.preventDefault();
                return;
            } else if (event.key !== "ArrowLeft" && event.key !== "ArrowRight" && event.key !== "ArrowUp" && event.key !== "ArrowDown" && event.key !== "Home" && event.key !== "End") {
                event.preventDefault();
            }
            handleKeyHit(event.key, event, self);
        }
        updatePreviewTextContent(self);
    }

    /**
     * handleKeyHit fires keyboard hits for the current key press
     *
     * @param {string} key key
     * @param {object} event event that is triggered by the user
     * @param {object} self
     */
    function handleKeyHit(key, event, self) {
        let hit = false;

        for (var palette in self.vars.activeKeyboard.paletteMap) {
            for (var entry in self.vars.activeKeyboard.paletteMap[palette].keyMap) {
                const qkey = self.vars.activeKeyboard.paletteMap[palette].keyMap[entry];
                if (hit) {
                    break;
                }
                if ((qkey.val === key) && (qkey.type === "letter" || qkey.type === "character" || qkey.type === "number")) {
                    fireKeyboardClick(self, qkey, event);
                    hit = true;
                } else if ((qkey.modifiers && qkey.modifiers["shift:val"] && qkey.modifiers["shift:val"] === key) && (qkey.type === "letter" || qkey.type === "character" || qkey.type === "number")) {
                    setModifier(self, "shift", "clipOnce", "letter", event);
                    fireKeyboardClick(self, qkey, event);
                    unsetModifier(self, "shift", "clipOnce", "letter", event);
                    hit = true;
                } else if (qkey.resolve && qkey.resolve.toLowerCase() === key.toLowerCase()) {
                    fireKeyboardClick(self, qkey, event);
                    hit = true;
                } else if (key === "Delete") {
                    deleteLetterFromCurrentText(self, true);
                    hit = true;
                }
            }
        }
    }

    /**
     * onKeyUpListener
     *
     * @param {object} self
     * @param {object} event event that is triggered by mouse
     */
    function onKeyUpListener(self, event) {
        self.vars.lastKey = null;
        if (event.key.toLowerCase() === 'escape') {
            log("Fire: Close Keyboard");
            shmi.fire('keyboard-cancel', {
                "success": false,
                "input": self.vars.currentText
            });
        }

        var key = getKeyByValue(self, event);
        if (key && !self.config["password-input"]) {
            shmi.removeClass(key.element.firstChild, 'pressed');
        }
        if (self.config["password-input"]) {
            event.preventDefault();
        } else {
            // get updated String from keyboard input
            self.vars.currentText = self.vars.previewContainer.value;
        }
        updatePreviewTextContent(self);
    }

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/* handler for Keyboard slide in */
(function() {
    // this queue handler is copied from the numpad and has initially been a queue for the notification dialog
    var keyboardEvent = null,
        keyboard = null;

    function createKeyboard() {
        var keyboardConfig = {},
            iter = shmi.requires("visuals.tools.iterate.iterateObject"),
            okTok = null,
            closeTok = null,
            enableTok = null;

        iter(keyboardEvent.detail, function(val, key) {
            keyboardConfig[key] = keyboardEvent.detail[key];
        });
        if (keyboardConfig.label) {
            keyboardConfig.label = shmi.evalString(shmi.localize(keyboardConfig.label), keyboardEvent.detail.param);
        }

        keyboard = shmi.createControl("alpha-num-keyboard", document.body, keyboardConfig, 'DIV');

        okTok = shmi.listen('keyboard-submit', function(evt) {
            okTok.unlisten();
            closeTok.unlisten();
            okTok = closeTok = null;
            deleteKeyboard();
            shmi.log("Keyboard closed, changes submited", evt);
            if (keyboardConfig.callback && typeof keyboardConfig.callback === "function") {
                keyboardConfig.callback(/*success:*/ true, /*input:*/ evt.detail.input);
            }
        });
        closeTok = shmi.listen('keyboard-cancel', function(evt) {
            closeTok.unlisten();
            okTok.unlisten();
            closeTok = okTok = null;
            deleteKeyboard();
            shmi.log("Keyboard closed, changes aborted", evt);
            if (keyboardConfig.callback && typeof keyboardConfig.callback === "function") {
                keyboardConfig.callback(/*success:*/ false, /*input:*/ null);
            }
        });

        function onKeyboardEnable() {
            if (enableTok) {
                enableTok.unlisten();
                enableTok = null;
            }
            keyboard.show();
        }

        if (keyboard.isActive()) {
            onKeyboardEnable();
        } else {
            enableTok = keyboard.listen("enable", onKeyboardEnable);
        }

        keyboard.enable();
    }

    function deleteKeyboard() {
        if (keyboard) {
            shmi.deleteControl(keyboard);
            keyboard = null;
        }
        keyboardEvent = null;
    }

    function defaultkeyboardHandler(event) {
        if (keyboard !== null) {
            return;
        }
        try {
            shmi.requires("visuals.handler.default.keyboard");
            keyboardEvent = event;
            if (keyboardEvent) {
                createKeyboard();
            }
        } catch (exc) {
            shmi.log("[Keyboard] - error trying to load controls: Keyboard\n" + exc);
        }
    }

    var module = shmi.pkg("visuals.handler.default.keyboard"),
        keyboardToken = null;

    module.register = function() {
        if (keyboardToken) {
            console.debug("Keyboard handler was active!");
            keyboardToken.unlisten();
            keyboardToken = null;
        }
        keyboardToken = shmi.listen('keyboard-request', defaultkeyboardHandler);
    };

    module.deregister = function() {
        if (!keyboardToken) {
            console.debug("Keyboard handler not active!");
        } else {
            keyboardToken.unlisten();
            keyboardToken = null;
        }
    };
}());

/**
 *
 */
(function() {
    /**
     * Displays an keyboard slide in to the user.
     * paramObj : {
     *      "key-preview": true,        // shown on touch only
     *      "auto-key-focus": false     // set tabindex="0" to key
     *      "select-box-enabled": true, // display selectbox to change language
     *      "value": null,
     *      "callback": function(Boolean success, String input) {
     *          if (success) {
     *              console.log("Keyboard successfully closed", input);
     *          } else {
     *              console.log("Keyboard aborted");
     *          }
     *      }
     * }
     *
     * @param {object} paramObj parameter object
     */
    shmi.keyboard = function(paramObj) {
        var nofListeners = shmi.fire('keyboard-request', paramObj, shmi);
        /* use native alert function if alpha num keyboard are not handled */
        if (nofListeners === 0) {
            shmi.log("[Keyboard] - Keyboard not found");
        }
    };
}());

shmi.pkg("visuals.controls");
/**
 * Button to execute UI actions
 *
 * @constructor
 * @extends shmi.visuals.core.BaseControl
 * @param element - the base element of the control
 * @param config - configuration of the control
 */
shmi.visuals.controls.Button = function(element, config) {
    /* check for required packages */
    shmi.requires("visuals.core.UiAction");
    shmi.requires("visuals.io.MouseListener");
    shmi.requires("visuals.io.TouchListener");

    var self = this;

    self.element = element;

    self.config = config || {};

    /*set default options */
    self.parseAttributes();
    shmi.def(self.config, 'class-name', 'button');
    shmi.def(self.config, 'template', 'default/button');
    shmi.def(self.config, 'name', null);
    shmi.def(self.config, 'icon-src', null);
    shmi.def(self.config, 'icon-title', null);
    shmi.def(self.config, 'show-text', true);
    shmi.def(self.config, 'show-icon', false);
    shmi.def(self.config, 'label-from-item', false);
    shmi.def(self.config, 'icon-class', null);
    shmi.def(self.config, 'precision', -1);
    shmi.def(self.config, 'label', "button");
    shmi.def(self.config, 'type', shmi.c("TYPE_STRING"));
    shmi.def(self.config, 'unit-text', null);
    shmi.def(self.config, 'disable-item-lock', false);
    shmi.def(self.config, 'monoflop', false);
    shmi.def(self.config, 'monoflop-interval', 100);
    shmi.def(self.config, 'monoflop-value', 1);

    shmi.def(self.config, 'decimal-delimiter', ".");
    shmi.def(this.config, 'auto-precision', true);
    shmi.def(this.config, 'auto-label', true);
    shmi.def(this.config, 'auto-unit-text', true);
    shmi.def(this.config, 'auto-type', true);

    self.config['show-text'] = shmi.toBoolean(self.config['show-text']);
    self.config['show-icon'] = shmi.toBoolean(self.config['show-icon']);
    self.config['label-from-item'] = shmi.toBoolean(self.config['label-from-item']);

    self.mouselistener = null;
    self.touchlistener = null;
    self.active = false;
    self.initialized = false;
    self.labelElement = null;
    self.valueElement = null;
    self.rafId = 0;
    self._subscriptionTargetId = null;
    self.vars = {
        tokens: [],
        monoFlopInterval: 0,
        conditional: null
    };

    self.startup();
};

shmi.visuals.controls.Button.prototype = {
    uiType: "button",
    events: ['press', 'release', 'click'],
    tooltipProperties: ['icon-title'],
    getClassName: function() {
        return "Button";
    },
    /**
     * Initializes the Button control.
     *
     */
    onInit: function() {
        var self = this,
            nv = shmi.requires("visuals.tools.numericValues");

        nv.initValueSettings(self);

        var label = shmi.getUiElement('button-label', self.element);
        if (!label) {
            shmi.log('[Button] no button-label element provided', 1);
        } else {
            self.labelElement = label;
        }

        self.valueElement = shmi.getUiElement('button-value', self.element);

        var icon = shmi.getUiElement('button-icon', self.element);
        if (!icon) {
            shmi.log('[Button] no button-icon element provided', 1);
        } else if (self.config['icon-src']) {
            try {
                icon.setAttribute('src', self.config['icon-src']);
            } catch (exc) {
                shmi.log("[Button] Exception setting icon-src: " + exc, 2);
            }
        } else if (self.config['icon-class']) {
            if (icon.tagName === "IMG") {
                /* switch img element for div when icon-class and no icon-src is configured */
                var icnDiv = document.createElement('div');
                shmi.addClass(icnDiv, "button-icon");
                icon.parentNode.insertBefore(icnDiv, icon);
                icon.parentNode.removeChild(icon);
                icon = icnDiv;
            }
            var icon_class = self.config['icon-class'].trim().split(" ");
            icon_class.forEach(function(cls) {
                shmi.addClass(icon, cls);
            });
        }
        /* all required elements found */
        if (label && self.config.label) {
            label.textContent = shmi.localize(self.config.label);
        }
        if (self.config.action) {
            self.action = new shmi.visuals.core.UiAction(self.config.action, self);
        }

        if ((self.config['show-text'] === true) && (self.config['show-icon'] === true)) {
            shmi.addClass(self.element, 'icon-and-text');
        } else if (self.config['show-icon'] === true) {
            shmi.addClass(self.element, 'icon-only');
        }

        var unitText = shmi.getUiElement('unit-text', self.element);
        if (unitText && self.config['unit-text']) {
            unitText.textContent = shmi.localize(self.config['unit-text']);
        }

        var btnFuncs = {},
            ses = shmi.visuals.session;
        btnFuncs.onClick = function(x, y, event) {
            if (ses.FocusElement !== null) {
                ses.FocusElement.blur();
                ses.FocusElement = null;
            }
            if (self.element instanceof HTMLElement) {
                self.element.focus();
                shmi.log("[Button] focused", 1);
            } else {
                shmi.log("[Button] only HTMLElements may be focused, type: " + (self.element.constructor), 1);
            }

            var event_detail = {
                x: x,
                y: y,
                event: event
            };
            self.fire('click', event_detail);

            if (self.action) {
                self.action.execute();
            }

            if (self.onClick) {
                self.onClick(self);
            }
        };
        btnFuncs.onPress = function(x, y, event) {
            shmi.addClass(this.element, 'pressed');
            var event_detail = {
                x: x,
                y: y,
                event: event
            };
            self.fire('press', event_detail);
        }.bind(self);
        btnFuncs.onRelease = function(x, y, event) {
            shmi.removeClass(self.element, 'pressed');
            var event_detail = {
                x: x,
                y: y,
                event: event
            };
            self.fire('release', event_detail);
        };
        var io = shmi.visuals.io;
        self.mouselistener = new io.MouseListener(self.element, btnFuncs);
        self.touchlistener = new io.TouchListener(self.element, btnFuncs);

        self.vars.keyUpListener = function(e) {
            var valid = self.vars.keyDownOnMe;
            if (valid && (e.code === 'Enter' || e.code === 'NumpadEnter') && self.element === document.activeElement) {
                shmi.removeClass(self.element, 'pressed');
                var rect = self.element.getBoundingClientRect();
                btnFuncs.onClick(rect.left + (rect.width / 2), rect.top + (rect.height / 2), e);
                self.vars.keyDownOnMe = false;
            }
        };

        self.vars.keyDownListener = function(e) {
            if ((e.code === 'Enter' || e.code === 'NumpadEnter') && self.element === document.activeElement) {
                shmi.addClass(self.element, 'pressed');
                self.vars.keyDownOnMe = true;
            }
        };
    },
    onEnable: function() {
        var self = this;
        if (self.vars.conditional !== null) {
            if (self.vars.conditional.item !== self.config.item) {
                self.vars.conditional.item = self.config.item;
            }
        } else if (self.config.item) {
            self.vars.conditional = shmi.createConditional(self.element, self.config.item);
        }

        if (self.vars.conditional) {
            self.vars.conditional.enable();
        }

        self.element.setAttribute('tabindex', '0');

        self.mouselistener.enable();
        self.touchlistener.enable();

        var im = shmi.requires("visuals.session.ItemManager");
        if (self.config.item) {
            if (self.config["disable-item-lock"]) {
                var h = im.getItemHandler();
                h.setValue = function(value) {
                    self.setValue(value);
                };
                self._subscriptionTargetId = im.subscribeItem(self.config.item, h);
            } else {
                self._subscriptionTargetId = im.subscribeItem(self.config.item, self);
            }
            if (self.config.monoflop) {
                var pTok = self.listen("press", function() {
                    clearInterval(self.vars.monoFlopInterval);
                    self.vars.monoFlopInterval = setInterval(function() {
                        var itemValues = {};
                        itemValues[self.config.item] = self.config['monoflop-value'];
                        im.writeDirect(itemValues, function(status, data) {
                            if (status !== 0) {
                                console.error("[ItemManager] failed to write item:", self.config.item, status, data);
                            }
                        });
                    }, self.config['monoflop-interval']);
                });
                var rTok = self.listen("release", function() {
                    clearInterval(self.vars.monoFlopInterval);
                });
                self.vars.tokens.push(pTok, rTok);
            }
        }

        self.element.addEventListener('keyup', self.vars.keyUpListener, false);
        self.element.addEventListener('keydown', self.vars.keyDownListener, false);

        shmi.log("[Button] enabled", 1);
    },
    onDisable: function() {
        var self = this;
        if (self.vars.conditional) {
            self.vars.conditional.disable();
        }

        if (self.config.item) {
            shmi.visuals.session.ItemManager.unsubscribeItem(self.config.item, self._subscriptionTargetId);
        }

        self.element.removeAttribute('tabindex');

        self.vars.tokens.forEach(function(t) {
            t.unlisten();
        });
        self.vars.tokens = [];

        self.mouselistener.disable();
        self.touchlistener.disable();

        self.element.removeEventListener('keyup', self.vars.keyUpListener);
        self.element.removeEventListener('keydown', self.vars.keyDownListener);

        shmi.log("[Button] disabled", 1);
    },
    onSetValue: function(value) {
        var self = this,
            nv = shmi.requires("visuals.tools.numericValues");

        if (self.config['label-from-item'] === true) {
            if (self.labelElement) {
                shmi.caf(self.rafId);
                self.rafId = shmi.raf(function() {
                    self.labelElement.textContent = shmi.localize(nv.formatOutput(value, self));
                });
            }
        } else if (self.valueElement) {
            shmi.caf(self.rafId);
            self.rafId = shmi.raf(function() {
                self.valueElement.textContent = nv.formatOutput(value, self);
            });
        }
    },
    /**
     * Sets properties for subscribed data
     *
     * @param min - min value
     * @param max - max value
     * @param step - value stepping
     */
    onSetProperties: function(min, max, step) {
        var self = this,
            nv = shmi.requires("visuals.tools.numericValues");

        nv.setProperties(self, arguments);
    },
    onLock: function() {
        var self = this;
        self.mouselistener.disable();
        self.touchlistener.disable();
        shmi.addClass(self.element, 'locked');
        shmi.log("[Button] locked", 1);
        self.element.removeAttribute('tabindex');
        self.element.removeEventListener('keyup', self.vars.keyUpListener);
        self.element.removeEventListener('keydown', self.vars.keyDownListener);
    },
    onUnlock: function() {
        var self = this;
        self.mouselistener.enable();
        self.touchlistener.enable();
        shmi.removeClass(self.element, 'locked');
        shmi.log("[Button] unlocked", 1);
        self.element.setAttribute('tabindex', 0);
        self.element.addEventListener('keyup', self.vars.keyUpListener, false);
        self.element.addEventListener('keydown', self.vars.keyDownListener, false);
    },
    setLabel: function(labelText) {
        var self = this;
        if (self.config['auto-label'] && self.labelElement) {
            self.labelElement.textContent = shmi.localize(labelText);
        }
    },
    setUnitText: function(unitText) {
        var self = this,
            unitTextElement = shmi.getUiElement('unit-text', self.element);
        if (self.config['auto-unit-text'] && unitTextElement) {
            unitTextElement.textContent = shmi.localize(unitText);
        }
    }
};

shmi.extend(shmi.visuals.controls.Button, shmi.visuals.core.BaseControl);

(function() {
    shmi.pkg("visuals.controls");
    /**
     * Creates a new CheckBox control.
     *
     * @constructor
     * @extends shmi.visuals.core.BaseControl
     * @param element - the base element of this control
     * @param config - configuration of this control
     */
    shmi.visuals.controls.CheckBox = function(element, config) {
        /* check for required packages */
        shmi.requires("visuals.io.MouseListener");
        shmi.requires("visuals.io.TouchListener");

        var self = this;

        self.element = element;
        self.config = config || {};

        self.parseAttributes();

        shmi.def(self.config, 'class-name', 'checkbox');
        shmi.def(self.config, 'template', 'default/checkbox');
        shmi.def(self.config, 'name', null);
        shmi.def(self.config, 'on-value', 1);
        shmi.def(self.config, 'off-value', 0);
        shmi.def(self.config, 'pressedClass', 'pressed');
        shmi.def(self.config, 'icon-src', null);
        shmi.def(self.config, 'icon-title', null);
        shmi.def(self.config, 'icon-class', null);
        shmi.def(self.config, 'label', "checkbox");
        shmi.def(self.config, 'auto-label', true);
        shmi.def(self.config, 'show-icon', false);
        shmi.def(self.config, 'show-text', true);
        shmi.def(this.config, 'confirm-off-text', '${V_CONFIRM_OFF}');
        shmi.def(this.config, 'confirm-on-text', '${V_CONFIRM_ON}');
        shmi.def(this.config, 'confirm-on', false);
        shmi.def(this.config, 'confirm-off', false);

        self.value = 0;
        self.initialized = false;
        self.active = false;
        self._subscriptionTargetId = null;
        self.mouseListener = null;
        self.touchListener = null;

        self.startup();
    };

    function setBoxValue(self, value) {
        var im = shmi.requires("visuals.session.ItemManager");
        if (self.config.item) {
            im.writeValue(self.config.item, value);
        } else {
            self.setValue(value);
        }
    }

    shmi.visuals.controls.CheckBox.prototype = {
        uiType: "checkbox",
        events: ["change"],
        tooltipProperties: ["icon-title"],
        getClassName: function() {
            return "CheckBox";
        },
        /**
         * Initializes the control.
         *
         */
        onInit: function() {
            var self = this,
                icon = shmi.getUiElement('checkbox-icon', self.element);
            if (!icon) {
                shmi.log('[Checkbox] no button-icon element provided', 1);
            } else if (self.config['icon-src']) {
                try {
                    icon.setAttribute('src', self.config['icon-src']);
                } catch (exc) {
                    shmi.log("[Checkbox] Exception setting icon-src: " + exc, 2);
                }
            } else if (self.config['icon-class']) {
                if (icon.tagName === "IMG") {
                    /* switch img element for div when icon-class and no icon-src is configured */
                    var icnDiv = document.createElement('div');
                    shmi.addClass(icnDiv, "checkbox-icon");
                    icon.parentNode.insertBefore(icnDiv, icon);
                    icon.parentNode.removeChild(icon);
                    icon = icnDiv;
                }
                var icon_class = self.config['icon-class'].trim().split(" ");
                icon_class.forEach(function(cls) {
                    shmi.addClass(icon, cls);
                });
            }
            if (!self.element) {
                shmi.log('[CheckBox] no base element provided', 3);
                return;
            }

            if (self.config['show-text'] && self.config['show-icon']) {
                shmi.addClass(self.element, "icon-and-text");
            } else if (self.config['show-icon']) {
                shmi.addClass(self.element, "icon-only");
            }

            self.handleElement = shmi.getUiElement('checkbox-handle', self.element);
            if (!self.handleElement) {
                shmi.log('[CheckBox] no checkbox-handle found in base element', 3);
                return;
            }
            self.backgroundElement = shmi.getUiElement('checkbox-background', self.element);
            if (!self.backgroundElement) {
                shmi.log('[CheckBox] no checkbox-background found in base element', 3);
                return;
            }
            self.labelElement = shmi.getUiElement('checkbox-label', self.element);
            if (self.labelElement && self.config.label) {
                self.labelElement.textContent = shmi.localize(self.config.label);
            }

            shmi.addClass(self.handleElement, 'hidden');

            var ioFuncs = {};
            ioFuncs.onClick = function() {
                if (self.value === self.config['on-value']) {
                    if (self.config['confirm-off']) {
                        shmi.confirm(self.config['confirm-off-text'], function onConfirmed(confirmed) {
                            if (confirmed) {
                                setBoxValue(self, self.config['off-value']);
                            }
                        });
                    } else {
                        setBoxValue(self, self.config['off-value']);
                    }
                } else if (self.config['confirm-on']) {
                    shmi.confirm(self.config['confirm-on-text'], function onConfirmed(confirmed) {
                        if (confirmed) {
                            setBoxValue(self, self.config['on-value']);
                        }
                    });
                } else {
                    setBoxValue(self, self.config['on-value']);
                }
            };
            ioFuncs.onPress = function() {
                shmi.addClass(self.element, self.config.pressedClass);
            };
            ioFuncs.onRelease = function() {
                shmi.removeClass(self.element, self.config.pressedClass);
            };
            self.mouseListener = new shmi.visuals.io.MouseListener(self.element, ioFuncs);
            self.touchListener = new shmi.visuals.io.TouchListener(self.element, ioFuncs);
            shmi.log("[Checkbox] initialized", 1);
        },
        /**
         * Enables the CheckBox
         *
         */
        onEnable: function() {
            var self = this;
            if (self.config.item) {
                self._subscriptionTargetId = shmi.visuals.session.ItemManager.subscribeItem(self.config.item, self);
            }

            self.mouseListener.enable();
            self.touchListener.enable();

            shmi.log("[CheckBox] enabled", 1);
        },
        /**
         * Disables the CheckBox
         *
         */
        onDisable: function() {
            var self = this;
            if (self.config.item) {
                shmi.visuals.session.ItemManager.unsubscribeItem(self.config.item, self._subscriptionTargetId);
            }

            self.mouseListener.disable();
            self.touchListener.disable();

            shmi.log("[CheckBox] disabled", 1);
        },
        /**
         * Locks the CheckBox
         *
         */
        onLock: function() {
            var self = this;
            self.mouseListener.disable();
            self.touchListener.disable();
            shmi.addClass(self.element, 'locked');
            shmi.log("[CheckBox] locked", 1);
        },
        /**
         * Unlocks the CheckBox
         *
         */
        onUnlock: function() {
            var self = this;
            self.mouseListener.enable();
            self.touchListener.enable();
            shmi.removeClass(self.element, 'locked');
            shmi.log("[CheckBox] unlocked", 1);
        },
        /**
         * Sets the underlying value of the CheckBox
         *
         * @param value - the new value to set for the CheckBox
         */
        onSetValue: function(value) {
            var self = this,
                changed = false;
            if (value === self.config['on-value']) {
                self.value = self.config['on-value'];
                changed = true;
                shmi.removeClass(self.handleElement, 'hidden');
            } else {
                self.value = self.config['off-value'];
                changed = true;
                shmi.addClass(self.handleElement, 'hidden');
            }
            if (changed) {
                self.fire("change", {
                    value: self.value
                });
            }
        },
        /**
         * Retrieves the current value of the CheckBox
         *
         * @return value current value of the CheckBox
         */
        getValue: function() {
            var self = this;
            return self.value;
        },
        setLabel: function(labelText) {
            var self = this;
            if (self.labelElement && self.config['auto-label']) {
                self.labelElement.textContent = shmi.localize(labelText);
            }
        }
    };

    shmi.extend(shmi.visuals.controls.CheckBox, shmi.visuals.core.BaseControl);
}());

(function() {
    shmi.pkg("visuals.controls");

    var DIRECTION_UP = -1,
        DIRECTION_DOWN = 1;

    /////////////////////////////////////////

    /**
     * Checks whether the combination of lhs and rhs is in `knownObjects`.
     *
     * @param {*} lhs
     * @param {*} rhs
     * @param {{lhs: any, rhs: any}[]} knownObjects List of known combinations
     *  of lhs and rhs.
     * @returns {boolean} `true` if { lhs, rhs } is in `knownObjects`, `false`
     *  else.
     */
    function isKnown(lhs, rhs, knownObjects) {
        return knownObjects.findIndex((entry) => lhs === entry.lhs && rhs === entry.rhs) !== -1;
    }

    /**
     * Checks if two objects have the same keys.
     *
     * @param {object} lhs
     * @param {object} rhs
     * @returns {boolean} `true` if both objects have the same keys, `false`
     *  else.
     */
    function objectsHaveSameKeys(lhs, rhs) {
        const lhsKeys = Object.keys(lhs),
            rhsKeys = Object.keys(rhs);

        if (lhsKeys.length !== rhsKeys.length) {
            return false;
        }

        return lhsKeys.every((value) => rhsKeys.includes(value));
    }

    /**
     * Checks if the two values are equal, checking arrays and objects as well.
     *
     * Note that "equal" does not mean "the same". Objects and arrays having
     * equal contents are considered equal themselves. The values do not need
     * to reference the same object.
     *
     * @param {*} lhs Left-hand side value for the comparison.
     * @param {*} rhs Right-hand side value for the comparison.
     * @param {{lhs: any, rhs: any}[]} knownObjects List of known combinations
     *  of lhs and rhs.
     * @returns {boolean} `true` if lhs and rhs are considered equal, `false` else.
     */
    function isEqualDeepImpl(lhs, rhs, knownObjects = []) {
        if (lhs === rhs) {
            return true;
        } else if (typeof lhs !== typeof rhs || typeof lhs !== "object") {
            return false;
        } else if (Array.isArray(lhs) !== Array.isArray(rhs)) {
            return false;
        } else if (isKnown(lhs, rhs, knownObjects)) {
            return true;
        } else if (Array.isArray(lhs)) {
            // Arrays

            // Arrays need to have the same length to be considered equal.
            if (lhs.length !== rhs.length) {
                return false;
            }

            knownObjects.push({ lhs, rhs });
            return lhs.every((val, idx) => isEqualDeepImpl(val, rhs[idx], knownObjects));
        } else {
            // Objects
            const keys = Object.keys(lhs);

            // If the objects have different keys then they can't be equal.
            if (!objectsHaveSameKeys(lhs, rhs)) {
                return false;
            }

            knownObjects.push({ lhs, rhs });
            return keys.every((key) => isEqualDeepImpl(lhs[key], rhs[key], knownObjects));
        }
    }

    /**
     * Checks if the two values are equal, checking arrays and objects as well.
     *
     * Note that "equal" does not mean "the same". Objects and arrays having
     * equal contents are considered equal themselves. The values do not need
     * to reference the same object.
     *
     * @param {*} lhs Left-hand side value for the comparison.
     * @param {*} rhs Right-hand side value for the comparison.
     * @returns {boolean} `true` if lhs and rhs are considered equal, `false` else.
     */
    function isEqualDeep(lhs, rhs) {
        return isEqualDeepImpl(lhs, rhs, []);
    }

    /**
     * Checks if two "selection" objects are equal.
     *
     * @param {*} lhs
     * @param {*} rhs
     * @returns {boolean}
     */
    function isEqualSelection(lhs, rhs) {
        const lhsAug = Object.assign({}, lhs),
            rhsAug = Object.assign({}, rhs);

        delete lhsAug.dblClicked;
        delete rhsAug.dblClicked;

        return isEqualDeep(lhsAug, rhsAug);
    }

    /////////////////////////////////////////

    /**
     * getRowOffset - get offset of row within datagrid data
     *
     * @param {object} self control instance reference
     * @param {number} rowId row ID
     * @return {number} row offset or `null` if row ID not found
     */
    function getRowOffset(self, rowId) {
        let offset = null;

        const dgm = shmi.requires("visuals.session.DataGridManager"),
            g = dgm.getGrid(self.cConfig.table),
            subId = self.dgSubscription.id;

        if (rowId >= 0) {
            const gridOffset = self.pageBuf.bufferOffset,
                currentIds = g.getCurrentIDs(subId);
            const rowIndex = currentIds.findIndex((id) => id === rowId);

            if (rowIndex >= 0) {
                offset = gridOffset + rowIndex;
            }
        }

        return offset;
    }

    /**
     * setLastClicked - set info on last clicked row and its offset
     *
     * @param {object} self control instance reference
     * @param {number} rowId row ID
     */
    function setLastClicked(self, rowId) {
        if (rowId >= 0) {
            const rowOffset = getRowOffset(self, rowId);
            if (rowOffset !== null) {
                self.lastClicked.rowId = rowId;
                self.lastClicked.offset = rowOffset;
                return;
            }
        }

        self.lastClicked.rowId = -1;
        self.lastClicked.offset = 0;
    }

    /**
     * loadRange - load row data range
     *
     * @param {object} self control instance reference
     * @param {number} rowId row ID of last clicked row
     * @param {number} start data start offset
     * @param {number} length number of rows to fetch
     * @param {function} callback function called on completion
     */
    function loadRange(self, rowId, start, length, callback) {
        startLoading(self);
        const dgm = shmi.requires("visuals.session.DataGridManager"),
            sub = dgm.subscribePage(self.cConfig.table, start, length, (dgInfo) => {
                const currentIds = dgm.getCurrentIDs(self.cConfig.table, sub.id);
                currentIds.forEach((id) => {
                    self.addRowIdToSelection(id);
                });

                dgm.unsubscribe(self.cConfig.table, sub.id);
                setLastClicked(self, rowId);
                stopLoading(self);

                self.updateSelTypeFromSelRows();
                self.updateDelRowsBtnEnable();
                self.updateSelectionView();
                clearTimeout(self.sel_fire_to);
                const selectionData = shmi.cloneObject(self.selection);
                selectionData.dblClicked = false;
                self.sel_fire_to = setTimeout(function() {
                    self.fireSelChanged(selectionData);
                }, shmi.c("ACTION_RETRY_TIMEOUT"));
                callback();
            });
    }

    /**
     * addSelectionRange - add range from last clicked row to row ID to current selection
     *
     * @param {object} self control instance reference
     * @param {number} rowId row ID
     * @return {promise} promise resolving when selection range has been added
     */
    function addSelectionRange(self, rowId) {
        return new Promise((resolve, reject) => {
            const rowOffset = getRowOffset(self, rowId);
            if (rowOffset === null) {
                reject(new Error("Could not find row offset"));
            } else {
                let start, end;
                if (rowOffset >= self.lastClicked.offset) {
                    end = rowOffset;
                    start = self.lastClicked.offset;
                } else {
                    end = self.lastClicked.offset;
                    start = rowOffset;
                }

                const length = (end - start) + 1;
                loadRange(self, rowId, start, length, resolve);
            }
        });
    }

    /**
     * createTableRow - create individual table row
     *
     * @param {ComplexTable2} self table control instance reference
     * @param {number} rowIdx buffer row index
     * @param {HTMLElement} tbodyElem table body element to generate row into
     * @param {boolean} isFirst true when row is first row of tbody
     *
     * @returns {undefined}
     */
    function createTableRow(self, rowIdx, tbodyElem, isFirst) {
        var nofCols = self.cConfig.resp["nof-cols"],
            trElem = null,
            tdElem = null,
            isMultiSelectMode = (self.cConfig.resp["select-mode"] === "MULTI"),
            isSingleSelectMode = (self.cConfig.resp["select-mode"] === "SINGLE"),
            isItemSelectMode = (self.cConfig.resp["select-mode"] === "ITEM"),
            tabCellElem = null,
            bufferRow = self.pageBuf.rows[rowIdx];

        if (bufferRow) {
            trElem = document.createElement("tr");
            if (bufferRow.rowId === null) {
                trElem.style.visibility = "hidden";
                bufferRow.visible = false;
            }
            tbodyElem.appendChild(trElem);
            bufferRow.trElem = trElem;
            if (((self.pageBuf.offset + rowIdx) % 2) === 0) {
                shmi.addClass(trElem, self.uiCssCl.evenRow);
            }
            if (isMultiSelectMode) {
                if (self.config["show-select-boxes"]) {
                    tdElem = document.createElement("td");
                    tabCellElem = document.createElement("div");
                    bufferRow.selCB = self.createSelCBox(tabCellElem, rowIdx, false, self);
                    bufferRow.selCB.setChangeCallback(function(pageBufRow, state) {
                        if (!self.locked) {
                            self.selectCBoxValChanged(pageBufRow, state, false, false, true, false);
                        }
                    });
                    tdElem.appendChild(tabCellElem);
                    trElem.appendChild(tdElem);
                }

                bufferRow.trSelCtrl = self.createSelRowCtrl(trElem, rowIdx, self);
                bufferRow.trSelCtrl.setChangeCallback(function(pageBufRow, state, shiftPressed, ctrlPressed, dblClicked) {
                    if (!self.locked) {
                        self.selectCBoxValChanged(pageBufRow, state, shiftPressed, ctrlPressed, false, dblClicked);
                    }
                });
            } else if (isSingleSelectMode || isItemSelectMode) {
                bufferRow.trSelCtrl = self.createSelRowCtrl(trElem, rowIdx, self);
                bufferRow.trSelCtrl.setChangeCallback(function(pageBufRow, state, shiftPressed, ctrlPressed, dblClicked) {
                    if (!self.locked) {
                        self.selectRowValChanged(pageBufRow, state, shiftPressed, ctrlPressed, dblClicked);
                    }
                });
            }
            for (var col = 0; col < nofCols; col++) {
                createTableCell(self, rowIdx, col, trElem, isFirst);
            }
        } else {
            console.error("[ComplexTable2]", "missing row descriptor for row ID:", rowIdx);
        }
    }

    /**
     * createTableCell - create individual table cell
     *
     * @param {ComplexTable2} self table control instance reference
     * @param {number} rowIdx buffer row index
     * @param {number} columnIdx table column index
     * @param {HTMLElement} trElem table row element to generate cell into
     * @param {boolean} isFirst true if is first row of tbody
     *
     * @returns {undefined}
     */
    function createTableCell(self, rowIdx, columnIdx, trElem, isFirst) {
        var columnConfig = self.cConfig.resp["column-org"][columnIdx],
            tabCellElem = document.createElement("div"),
            tdElem = document.createElement("td"),
            nofCtrls = columnConfig.fields.length;
        tdElem.appendChild(tabCellElem);
        trElem.appendChild(tdElem);
        shmi.addClass(tabCellElem, self.uiCssCl.tdCont);
        if (isFirst) {
            if (columnConfig.fields[0]["min-width"]) {
                tabCellElem.style.minWidth = columnConfig.fields[0]["min-width"];
            }
        }
        for (var ctrlIdx = 0; ctrlIdx < nofCtrls; ctrlIdx++) {
            createCellControl(self, rowIdx, columnIdx, ctrlIdx, columnConfig, tabCellElem);
        }
    }

    /**
     * createCellControl - create control configured for specific table cell
     *
     * @param {ComplexTable2} self table control instance reference
     * @param {number} rowIdx buffer row index
     * @param {number} columnIdx table column index
     * @param {number} controlIdx column control index
     * @param {object} columnConfig table column configuration
     * @param {HTMLElement} cellElem element to display cell content
     *
     * @returns {undefined}
     */
    function createCellControl(self, rowIdx, columnIdx, controlIdx, columnConfig, cellElem) {
        var controlConfig = shmi.cloneObject(columnConfig.fields[controlIdx].ctrl.config),
            bufferRow = self.pageBuf.rows[rowIdx],
            itemName = (bufferRow.rowId === null) ? null : self.makeItem(bufferRow.rowId, columnConfig.fields[controlIdx].dgCol),
            ia = shmi.requires("visuals.tools.item-adapter"),
            columnAdapter = self.getColumnAdapter(columnConfig.fields[controlIdx].fieldName),
            ctrl = null;

        controlConfig.item = itemName;

        ctrl = shmi.createControl(columnConfig.fields[controlIdx].ctrl["ui-type"], cellElem, controlConfig, "div");
        if (itemName !== null) {
            if (columnAdapter === null) {
                ia.unsetAdapter(itemName);
            } else {
                ia.setAdapter(itemName, columnAdapter);
            }
        }

        bufferRow.cols[columnIdx].ctrls.push(ctrl);
    }

    /**
     * createRowDescriptor - create empty row descriptor for use in page buffer
     *
     * @param {number} datagridId  dotagrid row-ID or `null` when not in use
     * @param {number} columnCount number of table columns
     *
     * @returns {object} row descriptor
     */
    function createRowDescriptor(datagridId, columnCount) {
        var rowDescr = {
            rowId: datagridId,
            trElem: null,
            trSelCtrl: null,
            selCB: null,
            cols: [],
            visible: true
        };

        for (var col = 0; col < columnCount; col++) {
            rowDescr.cols.push({ ctrls: [] });
        }

        return rowDescr;
    }

    /**
     * onQuickSearch - called to perform quicksearch filter update
     *
     * @param  {ComplexTable2} self table control instance reference
     * @param  {HTMLElement} valueElement value element of quicksearch input-field
     * @return {undefined}
     */
    function onQuickSearch(self, valueElement) {
        self.qSearchInterval = self.qSearchInterval || null;
        if (self.qSearchInterval === null) {
            self.qSearchInterval = setTimeout(function() {
                if (valueElement.textContent !== self.qSearchOld) {
                    self.quicksearchChanged(valueElement.textContent);
                }
                self.qSearchOld = valueElement.textContent;
                self.qSearchInterval = null;
            }, 300);
        }
    }

    /**
     * setElementOffset - set partial table offset relative to scrolling area content
     *
     * @param  {HTMLElement} element partial table element
     * @param  {number} offset  top offset in px
     * @return {undefined}
     */
    function setElementOffset(element, offset) {
        var useTranslate = false,
            template = "translate(<%= X %>px, <%= Y %>px)";
        if (useTranslate) {
            element.style.transform = shmi.evalString(template, {
                X: 0,
                Y: offset
            });
        } else {
            element.style.top = offset + "px";
        }
    }

    /**
     * setPagePositions - update position of partial table elements
     *
     * @param  {ComplexTable2} self table control instance reference
     * @return {undefined}
     */
    function setPagePositions(self) {
        var tableParts = [self.tbodyElement[0], self.tbodyElement[1], self.tbodyElement[2]],
            linesPerPage = self.cConfig["nof-buffered-rows"] / 3,
            pageHeight = self.metricsData.lineHeightPx * linesPerPage;
        self.vars.pageOrder.forEach(function(pageIndex) {
            setElementOffset(tableParts[pageIndex], self.vars.pageOffsets[pageIndex] * pageHeight);
        });
    }

    /**
     * reinitPages - reinitialize partial table elements
     *
     * @param  {ComplexTable2} self table control instance reference
     * @param  {number} scrollTop current scrolling offset
     * @return {undefined}
     */
    function reinitPages(self, scrollTop) {
        var lineHeight = self.metricsData.lineHeightPx,
            totalHeight = self.pageBuf.nofTotalRows * lineHeight,
            linesPerPage = self.cConfig["nof-buffered-rows"] / 3,
            pageHeight = self.metricsData.lineHeightPx * linesPerPage,
            numPages = Math.ceil(totalHeight / pageHeight),
            offsetPages = scrollTop / pageHeight,
            curPage = Math.floor(offsetPages);

        if ((curPage >= 1) && (numPages > 3)) {
            self.vars.pageOffsets = [curPage - 1, curPage, curPage + 1];
        } else {
            self.vars.pageOffsets = [0, 1, 2];
        }
        self.vars.pageOrder = [0, 1, 2];
        setPagePositions(self);
    }

    /**
     * turnPage - turn table element paging forwards/backwards
     *
     * @param {object} self control instance reference
     * @param {boolean} [backwards=false] `false` to page forwards, `true` for backwards
     */
    function turnPage(self, backwards = false) {
        const curPage = self.vars.pageOffsets[backwards ? self.vars.pageOrder[0] : self.vars.pageOrder[2]];
        if (backwards) {
            self.vars.pageOrder.unshift(self.vars.pageOrder.pop());
        } else {
            self.vars.pageOrder.push(self.vars.pageOrder.shift());
        }
        self.vars.pageOffsets[self.vars.pageOrder[0]] = curPage - 1;
        self.vars.pageOffsets[self.vars.pageOrder[1]] = curPage;
        self.vars.pageOffsets[self.vars.pageOrder[2]] = curPage + 1;
        setPagePositions(self);
    }

    /**
     * updatePages - update position of partial table elements when scrolling changed
     *
     * @param  {ComplexTable2} self table control instance reference
     * @param  {number} scrollTop scrolling offset
     * @return {undefined}
     */
    function updatePages(self, scrollTop) {
        var lineHeight = self.metricsData.lineHeightPx,
            totalHeight = self.pageBuf.nofTotalRows * lineHeight,
            linesPerPage = self.cConfig["nof-buffered-rows"] / 3,
            viewportHeight = self.metricsData.viewportHeight,
            pageHeight = lineHeight * linesPerPage,
            numPages = Math.ceil(totalHeight / pageHeight),
            topY = scrollTop,
            bottomY = topY + viewportHeight,
            dy = scrollTop - self.vars.scrollTop,
            direction = Math.sign(dy),
            pageThreshold = null,
            pageOffset = null,
            oldOffset = self.pageBuf.offset,
            newOffset = null,
            oldBufferOffset = self.pageBuf.bufferOffset,
            oldPageOrder = null,
            reinited = false;

        if ((numPages > 3) && (Math.abs(dy) < pageHeight / 2)) {
            //turn pages after small scroll
            if (direction === DIRECTION_UP) {
                pageOffset = self.vars.pageOffsets[self.vars.pageOrder[0]] * pageHeight;
                pageThreshold = pageOffset + (pageHeight / 2);
                if ((pageOffset > 0) && (topY < pageThreshold)) {
                    //pageUp
                    turnPage(self, true);
                }
            } else if (direction === DIRECTION_DOWN) {
                pageOffset = self.vars.pageOffsets[self.vars.pageOrder[2]] * pageHeight;
                pageThreshold = pageOffset + (pageHeight / 2);
                if ((pageOffset > 0) && (bottomY > pageThreshold)) {
                    //pageDown
                    turnPage(self, false);
                }
            }
        } else if ((numPages > 3) || self.vars.pageOffsets.some((pOffset) => pOffset > 2)) {
            //reorder after big scroll or old page offsets out of range
            oldPageOrder = self.vars.pageOrder.toString();
            reinitPages(self, scrollTop);
            reinited = true;
        }

        newOffset = Math.floor((self.vars.pageOffsets[self.vars.pageOrder[0]] * pageHeight) / lineHeight);
        self.vars.scrollTop = scrollTop;

        if (newOffset !== oldOffset) {
            self.pageBuf.offset = newOffset;
            self.pageBuf.bufferOffset = calcBufferOffset(self);
            if (oldBufferOffset !== self.pageBuf.bufferOffset) {
                startLoading(self);
                self.dgMan.setOffset(self.cConfig.table, self.pageBuf.bufferOffset, self.dgSubscription.id);
            } else {
                self.onDataGridChange(self.dgSubscription.changeInfo);
            }
        } else if (reinited && (oldPageOrder !== self.vars.pageOrder.toString())) {
            self.onDataGridChange(self.dgSubscription.changeInfo);
        }
    }

    /**
     * calcBufferOffset - calculate new (data-)buffer offset based on current (view-buffer-)offset.
     *
     * @param  {ComplexTable2} self table control instance reference
     * @return {number} new data buffer offset
     */
    function calcBufferOffset(self) {
        var newBufferOffset = self.pageBuf.bufferOffset;
        if (self.pageBuf.offset < self.pageBuf.bufferOffset) {
            newBufferOffset = Math.max(0, (self.pageBuf.offset + self.pageBuf.viewportSize) - self.pageBuf.bufferSize);
        } else if (self.pageBuf.offset + self.pageBuf.viewportSize > self.pageBuf.bufferOffset + self.pageBuf.bufferSize) {
            newBufferOffset = self.pageBuf.offset;
        }
        return newBufferOffset;
    }

    /**
     * getTableId - create unique ID to create instance specific inline stylesheet
     *
     * @param  {ComplexTable2} self table control instance reference
     * @return {string} instance ID
     */
    function getTableId(self) {
        var idPrefix = "ct2-instance-",
            numId = 0;
        while (shmi.visuals.controls.ComplexTable2.prototype._tableIds[idPrefix + numId] !== undefined) {
            numId += 1;
        }
        shmi.visuals.controls.ComplexTable2.prototype._tableIds[idPrefix + numId] = true;
        return idPrefix + numId;
    }

    /**
     * releaseTableId - release registered instance ID
     *
     * @param  {ComplexTable2} self table control instance reference
     * @param  {string} tableId instance ID
     * @return {undfined}
     */
    function releaseTableId(self, tableId) {
        delete shmi.visuals.controls.ComplexTable2.prototype._tableIds[tableId];
    }

    /**
     * applyLineHeight - apply configured line-height
     *
     * @param  {ComplexTable2} self table control instance reference
     * @return {undefined}
     */
    function applyLineHeight(self) {
        var styleSheet = null;
        if (self.cConfig.resp["line-height"]) {
            if (!self.styleSheet) {
                styleSheet = document.createElement("style");
                self.styleSheet = styleSheet;
            }
            self.styleSheet.innerHTML = "#" + self.tableId + " .ct2-tdcont { height: " + self.cConfig.resp["line-height"] + "; }";
            if (styleSheet) {
                self.element.insertBefore(self.styleSheet, self.element.firstChild);
            }
        }
    }

    /**
     * startLoading - start loading state
     *
     * @param {object} self control instance reference
     */
    function startLoading(self) {
        self.loadingCount += 1;

        if (self.loadingCount === 1) {
            shmi.addClass(self.element, "loading");
        }
    }

    /**
     * stopLoading - stop loading state
     *
     * @param object} self control instance reference
     */
    function stopLoading(self) {
        self.loadingCount -= 1;
        if (self.loadingCount <= 0) {
            self.loadingCount = 0;
            shmi.removeClass(self.element, "loading");
        }
    }

    /**
     * Creates a new ComplexTable2
     *
     * @constructor
     * @extends shmi.visuals.core.BaseControl
     * @param {Element} element the base element of the control
     * @param {Object} config control configuration
     */
    shmi.visuals.controls.ComplexTable2 = function(element, config) {
        // generate a unique id for this CT2 instance
        this.instID = Date.now();
        while (shmi.visuals.controls.ComplexTable2.instIDs.indexOf(this.instID) !== -1) {
            this.instID++;
        }
        shmi.visuals.controls.ComplexTable2.instIDs.push(this.instID);

        // short cuts to static "const" definitions for CT2
        this.c = shmi.visuals.controls.ComplexTable2.c;
        this.uiElName = shmi.visuals.controls.ComplexTable2.uiElName;
        this.uiCssCl = shmi.visuals.controls.ComplexTable2.uiCssCl;

        this.element = element;
        this.config = config || {};
        this.loadingCount = 0;
        this.lastClicked = {
            rowId: null,
            offset: 0
        };
        this.parseAttributes();

        shmi.def(this.config, "class-name", "complex-table2");
        shmi.def(this.config, "template", "default/complex-table2");
        shmi.def(this.config, "name", null);
        shmi.def(this.config, "label", "complex-table2");
        shmi.def(this.config, "table", "undefined-table");
        shmi.def(this.config, "item-selected-row", null);
        shmi.def(this.config, "rows-items-from-1", true);
        shmi.def(this.config, "field-datagrid-col-map", []);
        shmi.def(this.config, "default-field-control-map", {});
        shmi.def(this.config, "default-field-headers", {});
        shmi.def(this.config, "default-layout", {});
        shmi.def(this.config, "select-mode", "MULTI");
        shmi.def(this.config, "show-select-boxes", true);
        shmi.def(this.config, "text-mode", "NORMAL");
        shmi.def(this.config, "show-nof-rows", false);
        shmi.def(this.config, "delete-selected-rows", false);
        shmi.def(this.config, "useDgIndex", false);
        shmi.def(this.config, "col-width-changeable", false);
        shmi.def(this.config, "cols-show-hide", false);
        shmi.def(this.config, "cols-movable", false);
        shmi.def(this.config, "rows-movable", false);
        shmi.def(this.config, "sortable-fields", []);
        shmi.def(this.config, "default-text-filter-fields", []);
        shmi.def(this.config, "filters", []);
        shmi.def(this.config, "default-nof-buffered-rows", this.c.MIN_BUF_SIZE);
        shmi.def(this.config, "show-buttons-table-min-width-px", 300);
        shmi.def(this.config, "edit-mode-active", false);
        shmi.def(this.config, "fieldIcons", {});
        shmi.def(this.config, "headerMode", "ICON");
        shmi.def(this.config, "adapterSelectConfig", {
            "class-name": "select-box no-label"
        });
        shmi.def(this.config, "toggle-selection", true);
        shmi.def(this.config, "double-click-events", false);
        shmi.def(this.config, "adapterSelectClass", "adapter-select");

        //using standard "input-field" control for edit-mode
        shmi.def(this.config, "edit-mode-map", {});

        // "browser-dependencies" may be undefined
        // "responsive-layouts" may be undefined

        this.cConfig = {};
        this.filterData = {};
        this.colTextFilterData = {};
        this.ctrlPanelData = {};
        this.header = {};
        this.colTxtFilters = {};
        this.pageBuf = {};
        this.asyncProcData = {};
        this.metricsData = {};
        this.selection = {
            type: this.c.SELECT_TYPE_NOTHING,
            selRows: [],
            selRowIndex: [],
            dblClicked: false
        };
        this.lastSelection = null;
        this.scrollBar = null;
        this.dgMan = null;
        this.dgSubscription = {};
        this.selChgEventListeners = {};

        this.tcElement = null;
        this.bodyViewportElement = null;
        this.tabCCElement = null;
        this.tabHCElement = null;
        this.theadElement = null;
        this.txtFiltersHeadElement = null;
        this.tbodyElement = [];
        this.labelElement = null;
        this.footerElement = null;
        this.ctrlPanelElement = null;

        this.resizeDebounceTimeout = null;
        this.sbSliderDebounceTimeout = null;
        this.vpReleaseDebounceTimeout = null;
        this.swipeVertDebounceTimeout = null;

        this.cLayoutClass = null;
        this.isTableToCleanup = false;
        this.isTouchDevice = false;

        this.lastTableVpWidth = -1; // force init 1st time
        this.lastTableVpHeight = -1; // force init 1st time
        this.lastLayoutId = -100; // force init 1st time
        this.applyFiltersEnable = false;
        this.subscrIdItemSelRow = null;
        this.selRowByItem = 0;

        this.editMode = false;
        this.editRow = -1;

        this.headerMode = "TEXT"; // "TEXT" / "ICON"

        this.controls = [];

        this.vars = {
            lastUpdate: 0,
            updateTimeout: 0,
            scrollTop: 0,
            pageOffsets: [
                0,
                1,
                2
            ],
            pageOrder: [
                0,
                1,
                2
            ]
        };

        this.startup();
    };

    shmi.visuals.controls.ComplexTable2.prototype = {

        /*-- BaseControl method interface --*/
        _tableIds: {},
        uiType: "complex-table2",
        isContainer: true,
        events: ["select"],
        getClassName: function() {
            return "ComplexTable2";
        },
        setEditMode: function(editModeEnabled) {
            var self = this;
            self.editMode = editModeEnabled;
        },
        /**
         * Retrieves information about the currently selected rows.
         *
         * @returns {Object} the selection info, for structure refer clearSelection method
         */
        getSelectedRows: function() {
            return shmi.cloneObject(this.selection);
        },
        /**
         * Gets the datagrid row data for the current selection
         *
         * @param {Object} selection
         * @returns {Object} the datagrid data for the selection
         */
        getSelectionRowData: function(selection) {
            var self = this,
                answer = [];
            if (selection && selection.selRows.length > 0) {
                answer = selection.selRows.map((row) => self.dgMan.getRowData(self.config.table, row));
            }
            return answer;
        },
        /**
         * Sets the current selection
         *
         * @param {Object} selection the selection to set, for structure refer clearSelection method
         */
        setSelectedRows: function(selection) {
            shmi.checkArg("selection", selection, "object");
            shmi.checkArg("selection.type", selection.type, "number");
            shmi.checkArg("selection.selRows", selection.selRows, "array");
            shmi.checkArg("selection.selRowIndex", selection.selRowIndex, "array");
            shmi.checkArg("selection.dblClicked", selection.dblClicked, "boolean", "undefined");
            Object.assign(this.selection, {
                type: this.c.SELECT_TYPE_NOTHING,
                selRows: [],
                selRowIndex: [],
                dblClicked: false
            }, shmi.cloneObject(selection));
            this.updateSelectionView();

            var self = this;
            clearTimeout(self.sel_fire_to);
            self.sel_fire_to = setTimeout(function() {
                self.fireSelChanged(shmi.cloneObject(self.selection));
            }, shmi.c("ACTION_RETRY_TIMEOUT"));
        },
        /**
         * Adds a listener for selection changes
         *
         * @param {Object} listenerObj the listener to add, must implement callback method <listener>.tableSelectionChange(selection);
         * @returns {number} the listener id
         */
        addSelChgEventListener: function(listenerObj) {
            var id = Date.now();
            while (this.selChgEventListeners[id] !== undefined) {
                id++;
            }
            this.selChgEventListeners[id] = {
                listener: listenerObj
            };
            shmi.log("[ComplexTable2] addSelChgEventListener called, listenerObj: " + listenerObj + " - id:" + id, 1);
            return id;
        },
        /**
         * Removes a listener for selection changes.
         *
         * @param {number} id the id of the listener to remove
         */
        removeSelChgEventListener: function(id) {
            if (this.selChgEventListeners[id] !== undefined) {
                delete this.selChgEventListeners[id];
            } else {
                shmi.log("[ComplexTable2] removeSelChgEventListener - id " + id + " does not exist", 2);
            }
            shmi.log("[ComplexTable2] removeSelChgEventListener called, id: " + id, 1);
        },
        onInit: function() {
            var self = this,
                i = 0;
            self.headerMode = (self.config.headerMode === "ICON") ? "ICON" : "TEXT";
            self.tableId = getTableId(self);
            self.element.setAttribute("id", self.tableId);
            if (shmi.toBoolean(self.config["edit-mode-active"]) === true) {
                self.editMode = true;
            }

            this.isTouchDevice = this.detectTouchDevice();
            if (this.isTouchDevice) {
                shmi.addClass(this.element, this.uiCssCl.touchDev);
            } else {
                shmi.addClass(this.element, this.uiCssCl.noTouchDev);
            }

            this.buildCCfgRespIndep();
            this.buildCCfgRespDep();

            /* init all members, create child controls, get references to the necessary elements and ui-controls */
            this.dgMan = shmi.visuals.session.DataGridManager;
            this.tcElement = shmi.getUiElement(this.uiElName.tabContainer, this.element);
            if (!this.tcElement) {
                shmi.log("[ComplexTable2] no " + this.uiElName.tabContainer + " element provided (required)", 3);
                return;
            }

            this.theadElement = shmi.getUiElement(this.uiElName.thead, this.element);
            if (!this.theadElement) {
                shmi.log("[ComplexTable2] no " + this.uiElName.thead + " element provided (required)", 3);
                return;
            }
            this.txtFiltersHeadElement = shmi.getUiElement(this.uiElName.txtFiltersHead, this.element);
            if (!this.txtFiltersHeadElement) {
                shmi.log("[ComplexTable2] no " + this.uiElName.txtFiltersHead + " element provided (optional)", 1);
            }

            this.tableElement = shmi.getUiElement(this.uiElName.table, this.element);
            if (!this.tableElement) {
                shmi.log("[ComplexTable2] no " + this.uiElName.table + " element provided (required)", 3);
                return;
            }
            var tableBodyParts = shmi.getUiElements(this.uiElName.tbody, this.element);
            this.tbodyElement = [];
            for (i = 0; i < tableBodyParts.length; i++) {
                this.tbodyElement.push(tableBodyParts[i]);
            }
            if (!this.tbodyElement[0]) {
                shmi.log("[ComplexTable2] no " + this.uiElName.tbody + " element provided (required)", 3);
                return;
            }
            this.bodyViewportElement = shmi.getUiElement(this.uiElName.tbodyContainer, this.element);
            if (!this.bodyViewportElement) {
                shmi.log("[ComplexTable2] no " + this.uiElName.tbodyContainer + " element provided (required)", 3);
                return;
            }

            this.tabCCElement = shmi.getUiElement(this.uiElName.tabContContainer, this.element);
            if (!this.tabCCElement) {
                shmi.log("[ComplexTable2] no " + this.uiElName.tabContContainer + " element provided (required)", 3);
                return;
            }

            this.labelElement = shmi.getUiElement(this.uiElName.label, this.element);
            if (!this.labelElement) {
                shmi.log("[ComplexTable2] no " + this.uiElName.label + " element provided (optional)", 1);
            } else {
                this.labelElement.textContent = shmi.localize(this.cConfig.label);
            }
            this.footerElement = shmi.getUiElement(this.uiElName.footer, this.element);
            if (!this.footerElement) {
                shmi.log("[ComplexTable2] no " + this.uiElName.footer + " element provided (optional)", 1);
            }
            this.ctrlPanelElement = shmi.getUiElement(this.uiElName.ctrlPanel, this.element);
            if (!this.ctrlPanelElement) {
                shmi.log("[ComplexTable2] no " + this.uiElName.ctrlPanelElement + " element provided (optional)", 1);
            }

            this.initScrollHandler();
            this.updateSize = function() {
                clearTimeout(this.resizeDebounceTimeout);
                this.resizeDebounceTimeout = setTimeout(function() {
                    this.resizeDebounceTimeout = null;
                    this.processEvtOnResize();
                }.bind(this), this.c.DEBOUNCE_TO_RESIZE);
            }.bind(this);
            applyLineHeight(self);
            this.cleanupTabContent();
            this.cleanupAsyncProcessing();
            this.clearSelection();
            this.clearFilterData();
            this.clearColTextFilterData();
            this.isTabSizeChanged(); // init last..
            this.isLayoutChanged(); // init last..

            shmi.log("[ComplexTable2] initialized", 1);

            self = this;
            var tabContCtrls;
            if (shmi.isRegistered(this.element, this.uiType)) {
                tabContCtrls = shmi.parseControls(this.tcElement, false, this.element);
                for (i = 0; i < tabContCtrls.length; i++) {
                    self.controls.push(tabContCtrls[i]);
                }
            } else {
                var l_id = this.listen('register', function() {
                    tabContCtrls = shmi.parseControls(self.tcElement, false, self.element);
                    for (i = 0; i < tabContCtrls.length; i++) {
                        self.controls.push(tabContCtrls[i]);
                    }
                    self.unlisten('register', l_id);
                });
            }

            self.adapterItems = [];
            var iterObj = shmi.requires("visuals.tools.iterate.iterateObject");
            iterObj(self.config.adapterSettings, function(val, prop) {
                var adapterEntry = {
                    field: prop,
                    item: null,
                    gridName: self.config.table,
                    options: val
                };
                //function(name, type, min, max, value, setvalueCallback)
                var iName = "virtual:" + self.getAdapterNS() + ":" + prop;
                adapterEntry.item = shmi.createVirtualItem(iName, shmi.c("TYPE_INT"), 0, 1000, 0, self._getAdapterHandler(self, adapterEntry));
                self.adapterItems.push(adapterEntry);
            });
        },
        _getAdapterHandler: function(self, adapterEntry) {
            return function(value) {
                var adapter = null,
                    im = shmi.visuals.session.ItemManager,
                    ia = shmi.requires("visuals.tools.item-adapter"),
                    filterItems = [],
                    iterObj = shmi.requires("visuals.tools.iterate.iterateObject"),
                    fieldIdx = self.getDataColNo(adapterEntry.field);

                adapterEntry.options.forEach(function(option, idx) {
                    if (option.value === value) {
                        adapter = option.adapter;
                    }
                });

                iterObj(im.items, function(itm, itmName) {
                    if (itmName.indexOf("virtual:grid:" + adapterEntry.gridName) === 0) {
                        var nameParts = itmName.split(":");
                        if (parseInt(nameParts[4]) === fieldIdx) {
                            filterItems.push(itm);
                        }
                    }
                });

                if (adapter === null) {
                    filterItems.forEach(function(itm, idx) {
                        ia.unsetAdapter(itm.name);
                    });
                } else {
                    filterItems.forEach(function(itm, idx) {
                        ia.setAdapter(itm.name, adapter);
                    });
                }
            };
        },
        onEnable: function() {
            const self = this;
            let grid = null,
                sizesCalculated = false;

            this.enableEvents();
            this.updateRespDepContent(true);
            this.cleanupMetrics();

            if (this.cConfig.table) {
                grid = this.dgMan.getGrid(this.cConfig.table);
                if (grid) {
                    grid.clearFilter(-1);

                    if (self.cConfig.quicksearch && self.cConfig.quicksearch.enable) {
                        if (self.cConfig.quicksearch.remember) {
                            if (typeof self.qSearchOld === "string") {
                                self.ctrlPanelData.qSearchInput.setValue(self.qSearchOld);
                                self.quicksearchChanged(self.qSearchOld);
                            }
                        }
                    }
                }
            }

            this.setApplyFiltersEnable(true);
            if (this.cConfig.table && this.dgMan.getGrid(this.cConfig.table)) {
                startLoading(self);
                this.dgSubscription = this.dgMan.subscribePage(
                    this.cConfig.table, 0, this.cConfig["buffer-size"],
                    (dgChgInfo) => {
                        stopLoading(self);
                        if (this.testlog) console.log("***** datagridChange", dgChgInfo);
                        this.dgSubscription.changeInfo = dgChgInfo;
                        this.onDataGridChange(dgChgInfo);
                        if (!sizesCalculated) {
                            sizesCalculated = true;
                            self.processEvtOnResize();
                        }
                    });
                this.checkDgIndexFields();
                if (this.cConfig["item-selected-row"] && (this.cConfig["item-selected-row"] !== "undefined-item")) {
                    this.selRowByItem = 0; // at startup always 0
                    this.subscrIdItemSelRow = shmi.visuals.session.ItemManager.subscribeItem(this.cConfig["item-selected-row"], this);
                }
            }

            window.addEventListener("resize", this.updateSize, true);
            window.addEventListener("visuals-layout-change", this.updateSize, true);

            this.controls.forEach(function(ctrl) {
                ctrl.enable();
            });

            shmi.log("[ComplexTable2] enabled", 1);
        },
        /**
         * resetQuickSearch - reset quick search input
         *
         */
        resetQuickSearch: function() {
            var self = this;
            if (self.cConfig.quicksearch && self.cConfig.quicksearch.enable) {
                self.qSearchOld = null;
                if (self.isActive()) {
                    self.ctrlPanelData.qSearchInput.setValue("");
                    self.quicksearchChanged("");
                }
            }
        },
        /**
         * setQuickSearch - set quick search input
         *
         */
        setQuickSearch: function(value) {
            var self = this,
                setValue = value;

            if (self.cConfig.quicksearch && self.cConfig.quicksearch.enable) {
                if (setValue === undefined || setValue === null) {
                    setValue = "";
                } else if (typeof setValue !== "string") {
                    setValue = String(setValue);
                }
                if (self.isActive() && self.qSearchOld !== setValue) {
                    self.ctrlPanelData.qSearchInput.setValue(setValue);
                    self.quicksearchChanged(setValue);
                }
            }
        },
        onDisable: function() {
            if (this.pageBuf.rows) {
                this.pageBuf.rows.forEach(function(row, idx) {
                    if (row.trSelCtrl && row.trSelCtrl.rsMouseListener) {
                        row.trSelCtrl.rsMouseListener.disable();
                        row.trSelCtrl.rsTouchListener.disable();
                    }
                    if (row.selCB && row.selCB.cbMouseListener) {
                        row.selCB.cbMouseListener.disable();
                        row.selCB.cbTouchListener.disable();
                    }
                });
            }

            this.cleanupAsyncProcessing();
            if (this.dgSubscription) {
                this.dgMan.unsubscribe(this.cConfig.table, this.dgSubscription.id);
                this.dgSubscription = null;
            }
            this.disableEvents();
            this.cleanupTabContent();
            if (this.subscrIdItemSelRow) {
                shmi.visuals.session.ItemManager.unsubscribeItem(this.cConfig["item-selected-row"], this.subscrIdItemSelRow);
            }

            if (this.quicksearchHandler && this.quicksearchValueElement) {
                this.quicksearchValueElement.removeEventListener('keyup', this.quicksearchHandler);
                if (this.vars.qSearchToken) {
                    this.vars.qSearchToken.unlisten();
                }
                this.quicksearchValueElement = null;
                this.quicksearchHandler = null;
            }

            if (!(this.cConfig.quicksearch && this.cConfig.quicksearch.enable && this.cConfig.quicksearch.remember)) {
                this.qSearchOld = null;
            }

            this.selection = {
                type: this.c.SELECT_TYPE_NOTHING,
                selRows: [],
                selRowIndex: []
            };
            this.lastSelection = null;
            setLastClicked(this, -1);

            window.removeEventListener("resize", this.updateSize, true);
            window.removeEventListener("visuals-layout-change", this.updateSize, true);
            shmi.log("[ComplexTable2] disabled", 1);
        },
        onDelete: function() {
            var self = this;
            if (self.header.cols) {
                self.header.cols.forEach(function(col, idx) {
                    if (col.ctrls[0].cbMouseListener) {
                        col.ctrls[0].cbMouseListener.disable();
                        col.ctrls[0].cbTouchListener.disable();
                    }
                });
            }
            if (self.ctrlPanelData.filter) {
                self.ctrlPanelData.filter.items.forEach(function(item, idx) {
                    shmi.visuals.session.ItemManager.removeItem(item.name);
                });
            }
            if (self.onBodyScroll && self.bodyViewportElement) {
                self.bodyViewportElement.removeEventListener('scroll', self.onBodyScroll);
            }
            releaseTableId(self, self.tableId);
            self.element.removeAttribute("id");
        },

        /*-- DataGrid callback method interface --*/

        onDataGridChange: function(dgChgInfo) {
            var self = this;
            if (dgChgInfo && (dgChgInfo.status === "OK")) {
                if (self.isTableToCleanup) {
                    self.cleanupTabContent();
                    self.isTableToCleanup = false;
                }
                var currDgIds = self.dgMan.getCurrentIDs(self.cConfig.table, self.dgSubscription.id);

                if (self.pageBuf.rows.length === self.pageBuf.viewportSize) {
                    if ((currDgIds.length <= self.pageBuf.bufferSize) && (dgChgInfo.offset === self.pageBuf.bufferOffset)) {
                        self.pageBuf.size = currDgIds.length;
                        if (self.pageBuf.nofTotalRows !== dgChgInfo.totalRows) {
                            self.pageBuf.nofTotalRows = dgChgInfo.totalRows;
                            self.tableElement.style.height = (self.pageBuf.nofTotalRows * self.metricsData.lineHeightPx) + "px";
                        }
                        self.updateTable(dgChgInfo, currDgIds); // change item refs only, reconnect
                    } else {
                        self.cleanupTabContent();
                        self.reinitTable(dgChgInfo, currDgIds, true);
                    }
                } else {
                    self.cleanupTabContent();
                    self.reinitTable(dgChgInfo, currDgIds);
                }
                self.findSelectionAfterRefresh();
                self.updateNofRowsDisplay();
                self.updateSelectionView();
            }
        },

        // callback interface for optional items
        onSetValue: function(value, type, name) {
            var rowVal = -1;

            if (name === this.cConfig["item-selected-row"]) {
                if (value !== -1) {
                    if (this.cConfig["rows-items-from-1"]) {
                        rowVal = value - 1;
                    } else {
                        rowVal = value;
                    }
                }
                this.selRowByItem = rowVal;
                this.updateSelectionView();
            }
        },

        /*-- private helper methods - usage within CT2 only !! --*/

        /* config processors */
        /* note: use this.cConfig instead of this.Config only! Some properties are pre-processed for
                 a faster access at runtime. this.cConfig is divided in a responsive-independent and responsive-
                 dependent part, the last one may be changed at runtime!!
          */
        /**
         * @private
         */
        buildCCfgRespIndep: function() {
            this.cConfig.name = this.config.name;
            this.cConfig.label = this.config.label;
            this.cConfig["class-name"] = this.config["class-name"];
            this.cConfig.template = this.config.template;
            this.cConfig.table = this.config.table;
            this.cConfig["item-selected-row"] = this.config["item-selected-row"];
            this.cConfig["rows-items-from-1"] = this.config["rows-items-from-1"];
            if (this.config["field-datagrid-col-map"]) {
                this.cConfig["field-datagrid-col-map"] = this.config["field-datagrid-col-map"];
            } else {
                this.cConfig["field-datagrid-col-map"] = null;
            }
            this.cConfig["delete-selected-rows"] = shmi.toBoolean(this.config["delete-selected-rows"]);
            this.cConfig.filters = this.config.filters;
            this.cConfig.quicksearch = this.config.quicksearch;
            this.cConfig["nof-buffered-rows"] = shmi.toNumber(this.config["default-nof-buffered-rows"]);
            if (this.config["buffer-size"] && this.config["buffer-size"] > this.c.MIN_BUF_SIZE) {
                this.cConfig["buffer-size"] = this.config["buffer-size"];
            } else {
                this.cConfig["buffer-size"] = this.c.MIN_BUF_SIZE;
            }

            if (this.config["delete-selected-rows-button-config"]) {
                this.cConfig["delete-selected-rows-button-config"] = this.config["delete-selected-rows-button-config"];
            }
            if (this.config["default-filter-button-config"]) {
                this.cConfig["default-filter-button-config"] = this.config["default-filter-button-config"];
            }

            if (this.config["browser-dependencies"]) {
                var browserDeps = this.config["browser-dependencies"],
                    userAgent = navigator.userAgent;
                for (var i = 0; i < browserDeps.length; i++) {
                    var regEx = new RegExp(browserDeps[i].userAgentDetectRegEx);
                    if (regEx.test(userAgent)) {
                        if (browserDeps[i]["nof-buffered-rows"]) {
                            this.cConfig["nof-buffered-rows"] = shmi.toNumber(browserDeps[i]["nof-buffered-rows"]);
                        }
                        break;
                    }
                }
            }

            this.cConfig["show-buttons-table-min-width-px"] = shmi.toNumber(this.config["show-buttons-table-min-width-px"]);

            // nof-buffered-rows must be even
            var bufSize = this.cConfig["nof-buffered-rows"],
                sizeTemp = 0;
            if ((bufSize % 3) !== 0) {
                sizeTemp = Math.ceil(bufSize / 3) * 3;
                this.cConfig["nof-buffered-rows"] = sizeTemp;
                if (this.cConfig["nof-buffered-rows"] > this.cConfig["buffer-size"]) {
                    this.cConfig["buffer-size"] = this.cConfig["nof-buffered-rows"];
                }
                console.info("[ComplexTable2] nof-buffered-rows must be a multiple of 3, has been corrected to " + sizeTemp);
            }
        },
        /**
         * @private
         */
        buildCCfgRespDep: function() {
            const iterObj = shmi.requires("visuals.tools.iterate.iterateObject");
            var i;
            // init the default resp config
            this.cConfig.resp = {};
            this.cConfig.resp["v-scroll-options"] = this.config["v-scroll-options"];
            this.cConfig.resp["h-scroll-options"] = this.config["h-scroll-options"];
            if (this.cConfig["item-selected-row"] && (this.cConfig["item-selected-row"] !== "undefined-item")) {
                this.cConfig.resp["select-mode"] = "ITEM";
            } else {
                this.cConfig.resp["select-mode"] = this.config["select-mode"];
            }
            this.cConfig.resp["text-mode"] = this.config["text-mode"];
            this.cConfig.resp["show-nof-rows"] = this.config["show-nof-rows"];
            this.cConfig.resp["col-width-changeable"] = this.config["col-width-changeable"];
            this.cConfig.resp["cols-show-hide"] = this.config["cols-show-hide"];
            this.cConfig.resp["cols-movable"] = this.config["cols-movable"];
            this.cConfig.resp["rows-movable"] = this.config["rows-movable"];
            this.cConfig.resp["sortable-fields"] = this.config["sortable-fields"];
            this.cConfig.resp["text-filter-fields"] = this.config["default-text-filter-fields"];
            this.cConfig.resp["toggle-selection"] = (this.config["toggle-selection"] ?? true) && (this.cConfig.resp["select-mode"] !== "MULTI");
            this.cConfig.resp["double-click-events"] = (this.config["double-click-events"] ?? false) && !this.cConfig.resp["toggle-selection"];
            if (this.cConfig.quicksearch) {
                if (this.cConfig.quicksearch.enable) {
                    this.cConfig.resp["text-filter-fields"] = this.cConfig.quicksearch.fields;
                }
            }
            if (this.config["default-text-filter-input-field-config"]) {
                this.cConfig.resp["text-filter-input-field-config"] = this.config["default-text-filter-input-field-config"];
            }
            if (this.config["default-text-filter-clear-button-config"]) {
                this.cConfig.resp["text-filter-clear-button-config"] = this.config["default-text-filter-clear-button-config"];
            }
            //--
            var defLayout = this.config["default-layout"];
            if (defLayout["class-name"]) {
                this.cConfig.resp["layout-class-name"] = defLayout["class-name"];
            } else {
                this.cConfig.resp["layout-class-name"] = null;
            }

            if (defLayout["line-height"]) {
                this.cConfig.resp["line-height"] = defLayout["line-height"];
            } else {
                this.cConfig.resp["line-height"] = "40px";
            }
            // example of this.cConfig.resp["column-org"]
            //[
            //    {
            //        "fields": [
            //            { "dgCol": 0, "header": "headCol1Fld0", "isSortable": true, "hasTextFilter": true, "ctrl":
            // { "ui-type": "text2", "config": { "config-name": "ct2-text2" } } },
            //            { "dgCol": 2, "header": "headCol1Fld1", "isSortable": false, "hasTextFilter": false, "ctrl":
            // { "ui-type": "text2", "config": { "config-name": "ct2-text2" } } }
            //        ],
            //        "hasTextFilter": true // is true if at least one field of the ct2 column has a text filter
            //    },
            //    {
            //        "fields": [
            //            { "dgCol": 5, "header": "headCol2Fld0", "isSortable": true, "hasTextFilter": false, "ctrl":
            // { "ui-type": "text2", "config": { "config-name": "ct2-text2" } } }
            //        ],
            //        "hasTextFilter": false // is true if at least one field of the ct2 column has a text filter
            //    }
            //]
            this.cConfig.resp["column-org"] = [];
            var cfgSrcColOrg = defLayout["column-org"],
                cCfgDestColOrg = this.cConfig.resp["column-org"],
                nofSrcCols = 0,
                colNo;

            iterObj(cfgSrcColOrg, (srcCol, col) => {
                // assume that col is of format col<number>, e.g. "col1", "col17"
                if (col.substr(0, 3).toLowerCase() === "col") {
                    colNo = parseInt(col.substring(3)) - 1; // 0..nofCols-1
                    const destCol = cCfgDestColOrg[colNo] = {},
                        srcFields = srcCol.fields,
                        nofFields = srcFields.length;
                    let colHasTextFilter = false;

                    destCol.fields = [];

                    for (i = 0; i < nofFields; i++) {
                        destCol.fields[i] = {};

                        destCol.fields[i]["min-width"] = srcCol["min-width"];
                        destCol.fields[i]["column-width"] = srcCol["column-width"];
                        destCol.fields[i].fieldName = srcFields[i];
                        destCol.fields[i].dgCol = this.getDataColNo(srcFields[i]);
                        destCol.fields[i].header = this.getHeaderCaption(srcFields[i]);
                        destCol.fields[i].icon = this.config.fieldIcons[srcFields[i]];
                        destCol.fields[i].isSortable = this.getFldSortable(srcFields[i]);
                        destCol.fields[i].hasTextFilter = this.getFldHasTextFilter(srcFields[i]);
                        if (destCol.fields[i].hasTextFilter) {
                            colHasTextFilter = true;
                        }
                        destCol.fields[i].ctrl = this.getCtrlDef(srcFields[i]);
                    }
                    destCol.hasTextFilter = colHasTextFilter;
                    nofSrcCols++;
                }
            });

            var nofCols = cCfgDestColOrg.length;
            this.cConfig.resp["nof-cols"] = nofCols;
            if (nofCols !== nofSrcCols) {
                shmi.log("[ComplexTable2] buildCCfgRespDep - error in default-layout: there must be defined col1 till coln without gaps and duplicates", 3);
            }
            this.cConfig.resp["layout-id"] = -1; // default layout

            // overwrite the default resp config with "responsive-layouts" settings if necessary
            if (this.config["responsive-layouts"]) {
                var mLayout = null,
                    cWidth = this.getTableWidth(),
                    respLayouts = this.config["responsive-layouts"],
                    nofRespLayouts = respLayouts.length;
                for (i = 0; i < nofRespLayouts; i++) {
                    if (cWidth <= respLayouts[i]["table-max-width-px"]) {
                        mLayout = respLayouts[i];
                        this.cConfig.resp["layout-id"] = i;
                        break;
                    }
                }
                if (mLayout) {
                    if (mLayout.headerMode === "ICON") {
                        //self.headerMode = "ICON";
                        this.cConfig.resp.headerMode = "ICON";
                    } else {
                        //self.headerMode = "TEXT";
                        this.cConfig.resp.headerMode = "TEXT";
                    }

                    if (mLayout["line-height"]) {
                        this.cConfig.resp["line-height"] = mLayout["line-height"];
                    }

                    if (mLayout["v-scroll-options"] !== undefined) {
                        this.cConfig.resp["v-scroll-options"] = mLayout["v-scroll-options"];
                    }
                    if (mLayout["h-scroll-options"] !== undefined) {
                        this.cConfig.resp["h-scroll-options"] = mLayout["h-scroll-options"];
                    }
                    if (mLayout["select-mode"] !== undefined) {
                        this.cConfig.resp["select-mode"] = mLayout["select-mode"];
                    }
                    if (mLayout["text-mode"] !== undefined) {
                        this.cConfig.resp["text-mode"] = mLayout["text-mode"];
                    }
                    if (mLayout["show-nof-rows"] !== undefined) {
                        this.cConfig.resp["show-nof-rows"] = mLayout["show-nof-rows"];
                    }
                    if (mLayout["col-width-changeable"] !== undefined) {
                        this.cConfig.resp["col-width-changeable"] = mLayout["col-width-changeable"];
                    }
                    if (mLayout["cols-show-hide"] !== undefined) {
                        this.cConfig.resp["cols-show-hide"] = mLayout["cols-show-hide"];
                    }
                    if (mLayout["cols-movable"] !== undefined) {
                        this.cConfig.resp["cols-movable"] = mLayout["cols-movable"];
                    }
                    if (mLayout["rows-movable"] !== undefined) {
                        this.cConfig.resp["rows-movable"] = mLayout["rows-movable"];
                    }
                    if (mLayout["sortable-fields"] !== undefined) {
                        this.cConfig.resp["sortable-fields"] = mLayout["sortable-fields"];
                    }
                    if (mLayout["class-name"] !== undefined) {
                        this.cConfig.resp["layout-class-name"] = mLayout["class-name"];
                    }
                    if (mLayout["text-filter-fields"] !== undefined) {
                        this.cConfig.resp["text-filter-fields"] = mLayout["text-filter-fields"];
                    }
                    if (mLayout["text-filter-input-field-config"] !== undefined) {
                        this.cConfig.resp["text-filter-input-field-config"] = mLayout["text-filter-input-field-config"];
                    }
                    if (mLayout["text-filter-clear-button-config"] !== undefined) {
                        this.cConfig.resp["text-filter-clear-button-config"] = mLayout["text-filter-input-field-config"];
                    }

                    cfgSrcColOrg = mLayout["column-org"];
                    cCfgDestColOrg = this.cConfig.resp["column-org"];

                    iterObj(cfgSrcColOrg, (srcCol, col) => {
                        // assume that col is of format col<number>, e.g. "col1", "col17"
                        if (col.substr(0, 3).toLowerCase() === "col") {
                            colNo = parseInt(col.substring(3)) - 1; // 0..nofCols-1
                            srcCol = cfgSrcColOrg[col];
                            if ((typeof srcCol === "string") || (srcCol instanceof String)) {
                                // copy src colNo to dest colNo
                                var srcColNo = parseInt(srcCol.substring(3)) - 1; // 0..nofCols-1
                                cCfgDestColOrg[colNo] = cCfgDestColOrg[srcColNo];
                            } else {
                                const destCol = cCfgDestColOrg[colNo] = {},
                                    srcFields = srcCol.fields,
                                    nofFields = srcFields.length;
                                let colHasTextFilter = false;

                                destCol.fields = [];

                                for (i = 0; i < nofFields; i++) {
                                    destCol.fields[i] = {};
                                    destCol.fields[i].fieldName = srcFields[i];
                                    destCol.fields[i].dgCol = this.getDataColNo(srcFields[i]);
                                    var respHeaders = null;
                                    if (srcCol["field-headers"]) {
                                        respHeaders = srcCol["field-headers"];
                                    }
                                    destCol.fields[i].header = this.getHeaderCaption(srcFields[i], respHeaders);
                                    destCol.fields[i].icon = this.config.fieldIcons[srcFields[i]];
                                    destCol.fields[i].isSortable = this.getFldSortable(srcFields[i]);
                                    destCol.fields[i].hasTextFilter = this.getFldHasTextFilter(srcFields[i]);
                                    if (destCol.fields[i].hasTextFilter) {
                                        colHasTextFilter = true;
                                    }
                                    var respColCtrlMap = null;
                                    if (srcCol["field-control-map"]) {
                                        respColCtrlMap = srcCol["field-control-map"];
                                    }
                                    destCol.fields[i].ctrl = this.getCtrlDef(srcFields[i], respColCtrlMap);
                                }
                                destCol.hasTextFilter = colHasTextFilter;
                            }
                        }
                    });

                    nofCols = colNo + 1;
                    if (nofCols < cCfgDestColOrg.length) {
                        // cleanup col defs for col >= nofCols
                        cCfgDestColOrg.splice(nofCols, cCfgDestColOrg.length - nofCols);
                        this.cConfig.resp["nof-cols"] = nofCols;
                    } else if (nofCols > this.cConfig.resp["nof-cols"]) {
                        shmi.log("[ComplexTable2] buildCCfgRespDep, it is impossible to extent the number of columns in responsive-layouts", 3);
                    }
                }
            }
        },
        /**
         * @private
         */
        checkDgIndexFields: function() {
            var grid = this.dgMan.getGrid(this.cConfig.table);
            if (grid && (typeof grid.getIndexFields === "function")) {
                this.cConfig.indexFields = grid.getIndexFields();
                if (this.cConfig.indexFields.length > 0) {
                    this.cConfig.useDgIndex = true;
                }
            } else {
                shmi.log("[ComplexTable2] No index found in datagrid", 1);
            }
        },
        /**
         * @private
         */
        getDataColNo: function(fldName) {
            var colNo = this.cConfig["field-datagrid-col-map"][fldName];
            if (colNo === undefined) {
                colNo = 0;
                shmi.log("[ComplexTable2] getDataColNo, unknown fldName " + fldName, 3);
            } else {
                colNo = shmi.toNumber(colNo);
            }
            return colNo;
        },
        /**
         * @private
         */
        getFldSortable: function(fldName) {
            var sortable = false;
            if (this.cConfig.resp["sortable-fields"].indexOf(fldName) !== -1) {
                sortable = true;
            }
            return sortable;
        },
        /**
         * @private
         *
         * @param {string} fldName
         * @returns {boolean} has text filter true/false
         */
        getFldHasTextFilter: function(fldName) {
            var hasTxtFilter = false;
            if (this.cConfig.resp["text-filter-fields"].indexOf(fldName) !== -1) {
                hasTxtFilter = true;
            }
            return hasTxtFilter;
        },
        /**
         * @private
         *
         * @param {string} fldName
         * @param {Object} [ctrlMap] may be null. If defined it's checked for fldName at 1st.
         * @returns {Object}
         */
        getCtrlDef: function(fldName, ctrlMap) {
            var ctrlDef;
            if (ctrlMap) {
                ctrlDef = ctrlMap[fldName];
                if (!ctrlDef) {
                    ctrlDef = this.config["default-field-control-map"][fldName];
                }
            } else {
                ctrlDef = this.config["default-field-control-map"][fldName];
            }
            if (!ctrlDef) {
                ctrlDef = {};
                shmi.log("[ComplexTable2] getCtrlDef, unknown fldName " + fldName, 3);
            }
            return ctrlDef;
        },
        /**
         * @private
         *
         * @param {string} fldName
         * @param {Object} [headerMap] may be null. If defined it's checked for fldName at 1st.
         * @returns {string} caption, may be null
         */
        getHeaderCaption: function(fldName, headerMap) {
            var caption;
            if (headerMap) {
                caption = headerMap[fldName];
                if (caption === null) {
                    return null; // legal case: e.g. the default caption is overwritten by null in a responsive layout
                } else if (!caption) {
                    caption = this.config["default-field-headers"][fldName];
                }
            } else {
                caption = this.config["default-field-headers"][fldName];
            }
            if (!caption) {
                caption = "";
                shmi.log("[ComplexTable2] getHeaderCaption, unknown fldName " + fldName, 3);
            }
            return caption;
        },
        /*
         * @private
         */
        getColumnAdapter: function(fldName) {
            var self = this,
                adapter = null;
            if (self.config.adapterSettings && (self.config.adapterSettings[fldName] !== undefined)) {
                /* {
                    field: prop,
                    item: null,
                    gridName: self.config.table,
                    options: val
                } */
                var settings = self.config.adapterSettings[fldName],
                    curVal = null;
                self.adapterItems.forEach(function(aEntry, idx) {
                    if (aEntry.field === fldName) {
                        curVal = aEntry.item.readValue();
                    }
                });
                settings.forEach(function(entry, idx) {
                    if (curVal === entry.value) {
                        adapter = entry.adapter;
                    }
                });
            }
            return adapter;
        },
        // *ConfigVal may be null or undefined
        // returns null if both keys do not exist in config
        /**
         * @private
         */
        getChildCtrlConfig: function(primConfigVal, secConfigVal) {
            var cfg = null;
            if (primConfigVal) {
                cfg = shmi.cloneObject(primConfigVal);
            } else if (secConfigVal) {
                cfg = shmi.cloneObject(secConfigVal);
            }
            return cfg;
        },
        /**
         * @private
         */
        processAsyncMetricsUpdate: function(scrollTop) {
            var self = this,
                tm = shmi.requires("visuals.task"),
                tasks = [],
                tl = null;

            self.controls.forEach(function(ctrl, idx) {
                if (!ctrl.isActive()) {
                    var t = tm.createTask("table-control-#" + (idx + 1));
                    t.run = function() {
                        var enableTok = ctrl.listen("enable", function(evt) {
                            enableTok.unlisten();
                            t.complete();
                        });
                    };
                    tasks.push(t);
                }
            });

            tl = tm.createTaskList(tasks, false);

            tl.onComplete = function() {
                var tHeight = 0;
                if (self.active) {
                    self.updateMetricsData(scrollTop);
                    self.processTabReinitDependencies();

                    if (typeof scrollTop === "number") {
                        tHeight = self.tableElement.getBoundingClientRect().height;
                        if (tHeight >= scrollTop + self.metricsData.viewportHeight) {
                            self.bodyViewportElement.scrollTop = scrollTop;
                        } else {
                            self.bodyViewportElement.scrollTop = Math.max(0, tHeight - self.metricsData.viewportHeight);
                        }
                    }
                }
            };

            if (tasks.length === 0) {
                tl.onComplete();
            } else {
                tl.run();
            }
        },
        /**
         * @private
         */
        processEvtOnResize: function() {
            var self = this;
            if (this.initialized) {
                this.buildCCfgRespDep();
                applyLineHeight(self);
                if (this.isTabSizeChanged()) {
                    // do the same table processing as onInit -> onEnable
                    this.currentScrollEvtType = this.c.SCROLL_EVT_TYPE_RESIZE;
                    this.cleanupTabContent();
                    this.cleanupAsyncProcessing();
                    this.findSelectionAfterRefresh();
                    this.clearFilterData();
                    this.clearColTextFilterData();
                    this.updateRespDepContent(true);
                    this.cleanupMetrics();
                    // force onDataGridChange callback in every case:
                    if (this.dgSubscription && this.dgSubscription.changeInfo) {
                        self.cleanupTabContent();
                        self.reinitTable(this.dgSubscription.changeInfo, self.dgMan.getCurrentIDs(self.cConfig.table, self.dgSubscription.id));
                    }
                }
                //}
            }
        },
        /* select processing */
        /**
         * @private
         */
        clearSelection: function() {
            var self = this;
            this.selection.type = this.c.SELECT_TYPE_NOTHING; // || .._ALL || .._SEL_ROWS
            this.selection.selRows = [];
            this.selection.selRowIndex = [];
            this.selection.dblClicked = false;
            this.updateDelRowsBtnEnable();

            if (self.editRow !== -1) {
                self.resetBufferRow(self.editRow);
                self.editRow = -1;
            }
            clearTimeout(self.sel_fire_to);
            self.sel_fire_to = setTimeout(function() {
                self.fireSelChanged(shmi.cloneObject(self.selection));
            }, shmi.c("ACTION_RETRY_TIMEOUT"));
        },
        /**
         * @private
         */
        findSelectionAfterRefresh: function() {
            const self = this,
                grid = self.dgMan.getGrid(self.cConfig.table);

            if (grid && this.cConfig.useDgIndex) {
                if (this.selection.type === this.c.SELECT_TYPE_SEL_ROWS) {
                    const selectionClone = shmi.cloneObject(this.selection);

                    selectionClone.selRowIndex.forEach(function(el, idx) {
                        const newRow = grid.searchIndexRowId(el);
                        if (newRow === -1) {
                            self.removeRowIdFromSelection(selectionClone.selRows[idx]);
                        } else if (newRow !== selectionClone.selRows[idx]) {
                            const currentSelectionIndex = self.selection.selRows.indexOf(selectionClone.selRows[idx]);
                            if (currentSelectionIndex > -1) {
                                self.selection.selRows[currentSelectionIndex] = newRow;
                            }
                        }
                    });
                }

                if (this.selection && this.selection.selRows.length > 0) {
                    this.selection.type = this.c.SELECT_TYPE_SEL_ROWS;
                } else {
                    this.selection.type = this.c.SELECT_TYPE_NOTHING;
                }
                this.selection.dblClicked = false;

                this.updateDelRowsBtnEnable();
                if (self.editRow !== -1) {
                    self.resetBufferRow(self.editRow);
                    self.editRow = -1;
                }
                clearTimeout(self.sel_fire_to);
                self.sel_fire_to = setTimeout(function() {
                    self.fireSelChanged(shmi.cloneObject(self.selection));
                }, shmi.c("ACTION_RETRY_TIMEOUT"));
            } else {
                this.clearSelection();
            }
        },
        /**
         * @private
         */
        updateSelectionView: function() {
            if (this.testlog) console.log("*** update selection view", this.selection.selRows, this.selection.selRowIndex.toString());
            var isMultiSelectMode = (this.cConfig.resp["select-mode"] === "MULTI"),
                isSingleSelectMode = (this.cConfig.resp["select-mode"] === "SINGLE"),
                isItemSelectMode = (this.cConfig.resp["select-mode"] === "ITEM");
            if (isItemSelectMode) {
                this.setAllRowSelectors(this.c.CB_UNCHK);
                this.setRowSelectorByDgRow(this.selRowByItem, this.c.CB_CHK);
            } else {
                switch (this.selection.type) {
                case this.c.SELECT_TYPE_NOTHING:
                    if (isMultiSelectMode) {
                        this.setAllCheckBoxes(this.c.CB_UNCHK);
                        this.setAllRowSelectors(this.c.CB_UNCHK);
                    } else if (isSingleSelectMode) {
                        this.setAllRowSelectors(this.c.CB_UNCHK);
                    }
                    break;
                case this.c.SELECT_TYPE_ALL:
                    if (isMultiSelectMode) {
                        this.setAllCheckBoxes(this.c.CB_CHK);
                        this.setAllRowSelectors(this.c.CB_CHK);
                    } else {
                        shmi.log("[ComplexTable2] updateSelectionView, it is impossible to select all rows in single select mode", 2);
                    }
                    break;
                default: // .._SEL_ROWS
                    if (isMultiSelectMode) {
                        this.setAllCheckBoxes(this.c.CB_UNCHK);
                        this.setAllRowSelectors(this.c.CB_UNCHK);
                        if (this.header.selAllCBox) {
                            this.header.selAllCBox.setState(this.c.CB_UNDEF);
                        }
                        var currDgIds = this.dgMan.getCurrentIDs(this.cConfig.table, this.dgSubscription.id);
                        for (var i = 0; i < this.selection.selRows.length; i++) {
                            if (currDgIds.indexOf(this.selection.selRows[i]) !== -1) {
                                this.setCheckBox(this.selection.selRows[i], this.c.CB_CHK);
                                this.setRowSelector(this.selection.selRows[i], this.c.CB_CHK);
                            }
                        }
                    } else if (isSingleSelectMode) {
                        this.setAllRowSelectors(this.c.CB_UNCHK);
                        /*if (this.selection.selRows.length > 1) {
                            shmi.log("[ComplexTable2] updateSelectionView, it is impossible to select more than one row in single select mode", 2);
                        }*/
                        /*if (this.selection.selRows.length === 1) {
                            this.setRowSelector(this.selection.selRows[0], this.c.CB_CHK);
                        }*/
                        this.selection.selRows.forEach(function(el) {
                            this.setRowSelector(el, this.c.CB_CHK);
                        }.bind(this));
                    }
                    break;
                }
            }
        },
        /**
         * @private
         */
        setAllCheckBoxes: function(selVal) {
            var self = this;
            if (self.header.selAllCBox) {
                self.header.selAllCBox.setState(selVal);
            }
            self.pageBuf.rows.forEach(function(r) {
                if (r.selCB) {
                    r.selCB.setState(selVal);
                }
            });
        },
        /**
         * @private
         */
        setCheckBox: function(rowId, selVal) {
            var bufRow = this.searchRowId(rowId);
            if (bufRow !== -1) {
                if (this.pageBuf.rows[bufRow] && this.pageBuf.rows[bufRow].selCB) {
                    this.pageBuf.rows[bufRow].selCB.setState(selVal);
                }
            }
        },
        /**
         * @private
         */
        setAllRowSelectors: function(selVal) {
            var self = this;
            self.pageBuf.rows.forEach(function(r) {
                if (r.trSelCtrl) {
                    r.trSelCtrl.setState(selVal);
                }
            });
        },
        /**
         * @private
         */
        setRowSelector: function(rowId, selVal) {
            var self = this,
                bufRow = this.searchRowId(rowId);
            if (bufRow !== -1) {
                if (this.pageBuf.rows[bufRow] && this.pageBuf.rows[bufRow].trSelCtrl) {
                    this.pageBuf.rows[bufRow].trSelCtrl.setState(selVal);
                    if (self.editMode) {
                        var cc = shmi.requires("visuals.tools.control-config");
                        if (self.editRow !== -1) {
                            self.resetBufferRow(self.editRow);
                        }
                        self.editRow = bufRow;
                        this.pageBuf.rows[bufRow].cols.forEach(function(colEntry, idx) {
                            colEntry.ctrls.forEach(function(ctrl, jdx) {
                                var cfgCol = self.cConfig.resp["column-org"][idx],
                                    cConf = cc.getConfig(ctrl),
                                    iterObj = shmi.requires("visuals.tools.iterate.iterateObject");
                                if (self.config['edit-mode-map'][cfgCol.fields[jdx].fieldName] !== undefined) {
                                    var editCfg = shmi.cloneObject(self.config['edit-mode-map'][cfgCol.fields[jdx].fieldName]);
                                    iterObj(editCfg, function(val, prop) {
                                        cConf[prop] = val;
                                    });
                                    var lTok = shmi.listen('control-reconfiguration', function(evt) {
                                        lTok.unlisten();
                                        colEntry.ctrls[jdx] = evt.detail.control;
                                    }, {
                                        "detail.oldControl": ctrl
                                    });
                                    cc.setConfig(ctrl, cConf);
                                }
                            });
                        });
                        self.updateSize();
                    }
                }
            }
        },
        /**
         * @private
         */
        resetBufferRow: function(bufRow) {
            var self = this,
                cc = shmi.requires("visuals.tools.control-config");
            if (self.pageBuf.rows[bufRow] === undefined) {
                self.editRow = -1;
                return;
            }
            self.pageBuf.rows[bufRow].cols.forEach(function(colEntry, idx) {
                var cfgCol = self.cConfig.resp["column-org"][idx];
                colEntry.ctrls.forEach(function(ctrl, jdx) {
                    var oldCfg = cc.getConfig(ctrl),
                        cfg = shmi.cloneObject(cfgCol.fields[jdx].ctrl.config);
                    cfg.item = oldCfg.item;
                    cfg.ui = cfgCol.fields[jdx].ctrl["ui-type"];
                    shmi.removeClass(ctrl.element, oldCfg["class-name"]);
                    var lTok = shmi.listen('control-reconfiguration', function(evt) {
                        lTok.unlisten();
                        colEntry.ctrls[jdx] = evt.detail.control;
                    }, {
                        "detail.oldControl": ctrl
                    });
                    cc.setConfig(ctrl, cfg);
                });
            });
        },
        /**
         * @private
         */
        setRowSelectorByDgRow: function(dgRow, selVal) {
            var bufRow = dgRow - this.pageBuf.offset;
            if ((bufRow >= 0) && (bufRow < this.pageBuf.viewportSize)) {
                if (this.pageBuf.rows[bufRow] && this.pageBuf.rows[bufRow].trSelCtrl) {
                    this.pageBuf.rows[bufRow].trSelCtrl.setState(selVal);
                }
            }
        },
        /**
         * @private
         */
        // ..SEL_ROWS and bufRow === -1 means "do not change selected rows"
        updateSelectionData: function(type, bufRow, state, shiftPressed, ctrlPressed, fromCheckbox, dblClicked) {
            var self = this,
                isMultiSelectMode = (this.cConfig.resp["select-mode"] === "MULTI"),
                isSingleSelectMode = (this.cConfig.resp["select-mode"] === "SINGLE"),
                isItemSelectMode = (this.cConfig.resp["select-mode"] === "ITEM");

            if (isItemSelectMode) {
                // write always - independent of chk/unchk
                var im = shmi.visuals.session.ItemManager;
                var selRowVal = this.pageBuf.offset + bufRow;
                var selRowsCopy = null;
                if (this.cConfig["rows-items-from-1"]) {
                    selRowVal++;
                }
                im.writeValue(this.cConfig["item-selected-row"], selRowVal);
            } else {
                switch (type) {
                case this.c.SELECT_TYPE_ALL:
                    if (!isMultiSelectMode) {
                        console.warn("ComplexTable2", "updateSelectionData called with type SELECT_TYPE_ALL but table not in multi-select mode.");
                    }
                    // fallthrough
                case this.c.SELECT_TYPE_NOTHING:
                    this.selection.type = type;
                    this.selection.selRows = [];
                    this.selection.selRowIndex = [];
                    this.selection.dblClicked = dblClicked;
                    break;
                default: // .._SEL_ROWS
                    if (bufRow !== -1) {
                        if (isMultiSelectMode) {
                            if (shiftPressed && this.lastClicked.rowId > -1) {
                                if (!ctrlPressed) {
                                    this.selection.selRows = [];
                                    this.selection.selRowIndex = [];
                                }
                                addSelectionRange(this, this.pageBuf.rows[bufRow].rowId).then().catch((e) => {
                                    shmi.notify("Error fetching selection data: " + e.message, "${V_ERROR}");
                                });
                                this.selection.lastClickedRow = bufRow;
                                setLastClicked(this, this.pageBuf.rows[bufRow].rowId);
                                return;
                            } else if (fromCheckbox || ctrlPressed) {
                                if (state === this.c.CB_CHK) {
                                    this.addRowIdToSelection(this.pageBuf.rows[bufRow].rowId);
                                } else {
                                    this.removeRowIdFromSelection(this.pageBuf.rows[bufRow].rowId);
                                }
                                this.selection.lastClickedRow = bufRow;
                                setLastClicked(this, this.pageBuf.rows[bufRow].rowId);
                            } else {
                                selRowsCopy = shmi.cloneObject(self.selection.selRows);
                                selRowsCopy.forEach(function(rid) {
                                    self.removeRowIdFromSelection(rid);
                                });
                                if (state === this.c.CB_CHK) {
                                    self.addRowIdToSelection(this.pageBuf.rows[bufRow].rowId);
                                }
                                setLastClicked(this, this.pageBuf.rows[bufRow].rowId);
                            }
                        } else if (isSingleSelectMode) {
                            this.selection.selRows = [];
                            this.selection.selRowIndex = [];

                            if (state === this.c.CB_CHK) {
                                this.addRowIdToSelection(this.pageBuf.rows[bufRow].rowId);
                            } else {
                                this.removeRowIdFromSelection(this.pageBuf.rows[bufRow].rowId);
                            }
                        }
                    }
                    this.selection.dblClicked = dblClicked;
                    this.updateSelTypeFromSelRows();
                    break;
                }
                this.updateDelRowsBtnEnable();

                clearTimeout(self.sel_fire_to);
                self.sel_fire_to = setTimeout(function() {
                    self.fireSelChanged(shmi.cloneObject(self.selection));
                }, shmi.c("ACTION_RETRY_TIMEOUT"));
            }
        },
        /**
         * @private
         */
        getRowIndex: function(rowId) {
            var rowIndex = null,
                grid = this.dgMan.getGrid(this.cConfig.table);

            if (grid) {
                rowIndex = grid.getRowIndex(rowId);
            }

            return rowIndex;
        },
        /**
         * @private
         */
        addRowIdToSelection: function(rowId) {
            if (this.selection.selRows.indexOf(rowId) === -1) {
                this.selection.selRows.push(rowId);
                if (this.cConfig.useDgIndex) {
                    var rowIndex = this.getRowIndex(rowId);
                    this.selection.selRowIndex.push(rowIndex);
                }
            }
        },
        /**
         * @private
         */
        removeRowIdFromSelection: function(rowId) {
            var self = this,
                idx = this.selection.selRows.indexOf(rowId);
            if (idx !== -1) {
                this.selection.selRows.splice(idx, 1);
                if (this.cConfig.useDgIndex) {
                    this.selection.selRowIndex.splice(idx, 1);
                }
            } else if (self.editRow !== -1) {
                self.resetBufferRow(self.editRow);
                self.editRow = -1;
                self.updateSize();
            }
        },
        /**
         * @private
         */
        updateSelTypeFromSelRows: function() {
            if (this.selection.selRows.length === this.pageBuf.nofTotalRows) {
                this.selection.type = this.c.SELECT_TYPE_SEL_ROWS;
            } else if (this.selection.selRows.length === 0) {
                this.selection.type = this.c.SELECT_TYPE_NOTHING;
            } else {
                this.selection.type = this.c.SELECT_TYPE_SEL_ROWS;
            }
        },
        /**
         * @private
         */
        selectAllCBoxValChanged: function(state) {
            var type;
            switch (state) {
            case this.c.CB_UNCHK:
                type = this.c.SELECT_TYPE_NOTHING;
                break;
            case this.c.CB_CHK:
                type = this.c.SELECT_TYPE_ALL;
                loadRange(this, -1, 0, this.dgSubscription.changeInfo.totalRows, () => {
                    /* all rows loaded */
                });
                return;
            default:
                type = this.c.SELECT_TYPE_SEL_ROWS;
                break;
            }
            this.updateSelectionData(type, -1, this.c.CB_UNDEF, false, false, true, false);
            this.updateSelectionView();
        },
        /**
         * @private
         */
        selectCBoxValChanged: function(bufRow, state, shiftPressed, ctrlPressed, fromCheckbox, dblClicked) {
            if (this.c.MULTI_SELECT_MODE === "3-state-clear") {
                if (this.header.selAllCBox && this.header.selAllCBox.getState() === this.c.CB_UNCHK) {
                    this.clearSelection();
                }
            }
            this.updateSelectionData(this.c.SELECT_TYPE_SEL_ROWS, bufRow, state, shiftPressed, ctrlPressed, fromCheckbox, dblClicked);
            this.updateSelectionView();
        },
        /**
         * @private
         */
        selectRowValChanged: function(bufRow, state, shiftPressed, ctrlPressed, dblClicked) {
            this.updateSelectionData(this.c.SELECT_TYPE_SEL_ROWS, bufRow, state, shiftPressed, ctrlPressed, false, dblClicked);
            this.updateSelectionView();
        },
        /**
         * @private
         */
        deleteSelectedRows: function() {
            var self = this;
            shmi.confirm(self.config["msg-confirm-delete"] || "${ct2_confirm_delete_rows}", function(confirmOk) {
                self.deleteSelectedRows_confirm(confirmOk);
            });
        },
        /**
         * @private
         */
        deleteSelectedRows_confirm: function(confirmOk) {
            if (confirmOk) {
                switch (this.selection.type) {
                case this.c.SELECT_TYPE_ALL:
                    setLastClicked(this, -1);
                    this.dgMan.deleteAll(this.cConfig.table);
                    break;
                case this.c.SELECT_TYPE_SEL_ROWS:
                    setLastClicked(this, -1);
                    this.dgMan.deleteRow(this.cConfig.table, this.selection.selRows);
                    break;
                default:
                    break;
                }
                this.clearSelection();
            }
        },
        /**
         * @private
         */
        fireSelChanged: function(selection) {
            const iterObj = shmi.requires("visuals.tools.iterate.iterateObject");

            iterObj(this.selChgEventListeners, (listener) => {
                listener.listener.tableSelectionChange(selection);
            });

            if (!this.lastSelection || selection.dblClicked || !isEqualSelection(this.lastSelection, selection)) {
                this.lastSelection = shmi.cloneObject(selection);
                this.updateNofRowsDisplay();
                this.fire("select", selection);
            }
        },

        /* sort processing */
        /**
         * @private
         */
        sortHeaderChanged: function(dgCol, state) {
            var sortCol,
                dir;
            switch (state) {
            case this.c.SORT_UP:
                sortCol = dgCol;
                dir = "DESC";
                break;
            case this.c.SORT_DOWN:
                sortCol = dgCol;
                dir = "ASC";
                break;
            default:
                sortCol = -1;
                dir = "ASC";
                break;
            }
            setLastClicked(this, -1);
            this.dgMan.sort(this.cConfig.table, sortCol, dir);
            this.clearSortHeaderExcept(dgCol);
            this.clearSelection();
        },
        /**
         * @private
         */
        clearSortHeaderExcept: function(dgCol) {
            var nofCols = this.cConfig.resp["nof-cols"];
            for (var col = 0; col < nofCols; col++) {
                var hCtrls = this.header.cols[col].ctrls;
                for (var i = 0; i < hCtrls.length; i++) {
                    if (hCtrls[i].id !== dgCol) {
                        hCtrls[i].setState(this.c.NO_SORT);
                    }
                }
            }
        },

        /* filter processing */
        /**
         * @private
         */
        txtFiltersUsed: function() {
            var txtFlUsed = false;
            if (this.config["default-text-filter-fields"].length > 0) {
                txtFlUsed = true;
            }
            return txtFlUsed;
        },
        /**
         * @private
         */
        clearFilterData: function() {
            this.filterData = {};
            var nofFilters = this.cConfig.filters.length;
            for (var fId = 0; fId < nofFilters; fId++) {
                var fCfg = this.cConfig.filters[fId],
                    dgColProp = this.getDataColNo(fCfg.field).toString(),
                    fData = this.filterData[dgColProp];
                if (!fData) {
                    fData = {};
                    fData.expr = [];
                    fData.changed = false;
                    this.filterData[dgColProp] = fData;
                }
            }
        },
        /**
         * @private
         */
        clearColTextFilterData: function() {
            this.colTextFilterData = {};
            var colTxtFilterFields = this.cConfig.resp["text-filter-fields"];
            for (var fIdx = 0; fIdx < colTxtFilterFields.length; fIdx++) {
                var dgColProp = this.getDataColNo(colTxtFilterFields[fIdx]).toString(),
                    fData = this.colTextFilterData[dgColProp];
                if (!fData) {
                    fData = {};
                    fData.expr = "";
                    fData.changed = false;
                    this.colTextFilterData[dgColProp] = fData;
                }
            }
        },
        /**
         * @private
         */
        quicksearchChanged: function(value) {
            var fExprVal = (typeof value === "string") ? value.trim() : value;
            var fields = this.cConfig.quicksearch.fields;
            if (fExprVal === "") {
                this.removeQuicksearchFilterExpr(fields);
            } else {
                this.setQuicksearchFilterExpr(fExprVal, fields);
            }
            this.applyFilters();
            this.clearSelection();
            this.resetScroll();
        },
        /**
         * @private
         */
        filterChanged: function(value, type, vItemName) {
            if (value === 1) {
                this.setFilterExpr(vItemName);
            } else {
                this.removeFilterExpr(vItemName);
            }
            this.applyFilters();
            this.clearSelection();
            this.resetScroll();
        },
        /**
         * @private
         */
        colTxtFilterChanged: function(value, type, vItemName) {
            var fExprVal = (typeof value === "string") ? value.trim() : value;
            if (fExprVal === "") {
                var fCol = this.getFilterId(vItemName);
                this.removeColTxtFilterExpr(fCol);
            } else {
                this.setColTxtFilterExpr(vItemName, fExprVal);
            }
            this.applyFilters();
            this.clearSelection();
            this.resetScroll();
        },
        /**
         * @private
         */
        colTxtFilterReset: function(fCol) {
            var colTxtFilItem = this.colTxtFilters.items[fCol];
            if (colTxtFilItem) {
                var im = shmi.visuals.session.ItemManager;
                im.writeValue(colTxtFilItem.name, "");
            }
        },
        /**
         * @private
         */
        setFilterExpr: function(vItemName) { // if not yet set !!
            var fId = this.getFilterId(vItemName),
                fIdInt = Number(fId);
            if (fIdInt !== -1) {
                var fCfg = this.cConfig.filters[fId];
                if (fCfg) {
                    var fCfgExpr = fCfg.expr,
                        dgColProp = this.getDataColNo(fCfg.field).toString(),
                        fData = this.filterData[dgColProp];
                    if (fData.expr.indexOf(fCfgExpr) === -1) {
                        fData.expr.push(fCfgExpr);
                        fData.changed = true;
                    }
                }
            }
        },
        /**
         * @private
         */
        removeFilterExpr: function(vItemName) { // if exists !!
            var fId = this.getFilterId(vItemName),
                fIdInt = Number(fId);
            if (fIdInt !== -1) {
                var fCfg = this.cConfig.filters[fId];
                if (fCfg) {
                    var fCfgExpr = fCfg.expr,
                        dgColProp = this.getDataColNo(fCfg.field).toString(),
                        fData = this.filterData[dgColProp],
                        exprIdx = fData.expr.indexOf(fCfgExpr);
                    if (exprIdx !== -1) {
                        fData.expr.splice(exprIdx, 1);
                        fData.changed = true;
                    }
                }
            }
        },
        /**
         * @private
         */
        setColTxtFilterExpr: function(vItemName, vItemValue) {
            var col = this.getFilterId(vItemName),
                dgColNos = this.getColTxtFilterDgCols(col);
            for (var i = 0; i < dgColNos.length; i++) {
                var dgColProp = dgColNos[i].toString(),
                    fData = this.colTextFilterData[dgColProp];
                if (fData) {
                    fData.expr = "%" + vItemValue + "%";
                    fData.changed = true;
                } else {
                    shmi.log("[ComplexTable2] setColTxtFilterExpr, no filter data available for data grid col: " + dgColProp, 3);
                }
            }
        },
        /**
         * @private
         */
        removeColTxtFilterExpr: function(col) {
            var dgColNos = this.getColTxtFilterDgCols(col);
            for (var i = 0; i < dgColNos.length; i++) {
                var dgColProp = dgColNos[i].toString(),
                    fData = this.colTextFilterData[dgColProp];
                if (fData) {
                    fData.expr = "";
                    fData.changed = true;
                } else {
                    shmi.log("[ComplexTable2] removeColTxtFilterExpr, no filter data available for data grid col: " + dgColProp, 3);
                }
            }
        },
        /**
         * @private
         */
        setQuicksearchFilterExpr: function(value, fields) {
            fields.forEach(function(fname) {
                var dgColNos = this.getColTxtFilterDgCols(this.getDataColIndex(fname));
                for (var i = 0; i < dgColNos.length; i++) {
                    var dgColProp = dgColNos[i].toString(),
                        fData = this.colTextFilterData[dgColProp];
                    if (fData) {
                        fData.expr = "%" + value + "%";
                        fData.changed = true;
                    } else {
                        shmi.log("[ComplexTable2] setQuicksearchFilterExpr, no filter data available for data grid col: " + dgColProp, 3);
                    }
                }
            }.bind(this));
        },
        /**
         * @private
         */
        removeQuicksearchFilterExpr: function(fields) {
            fields.forEach(function(fname) {
                var dgColNos = this.getColTxtFilterDgCols(this.getDataColIndex(fname));
                for (var i = 0; i < dgColNos.length; i++) {
                    var dgColProp = dgColNos[i].toString(),
                        fData = this.colTextFilterData[dgColProp];
                    if (fData) {
                        fData.expr = "";
                        fData.changed = true;
                    } else {
                        shmi.log("[ComplexTable2] removeQuicksearchFilterExpr, no filter data available for data grid col: " + dgColProp, 3);
                    }
                }
            }.bind(this));
        },
        /**
         * @private
         */
        getDataColIndex: function(fname) {
            var i = 0;
            var colNo;

            for (var e in this.cConfig["field-datagrid-col-map"]) {
                if (e === fname) {
                    colNo = i;
                }
                i++;
            }

            if (colNo === undefined) {
                colNo = 0;
                shmi.log("[ComplexTable2] getDataColNo, unknown fldName " + fname, 3);
            } else {
                colNo = shmi.toNumber(colNo);
            }
            return colNo;
        },
        /**
         * @private
         */
        getFilterId: function(vItemName) {
            var id = -1,
                pos = vItemName.lastIndexOf(":"),
                posInt = Number(pos);
            if (posInt !== -1) {
                id = shmi.toNumber(vItemName.substring(pos + 1));
            } else {
                shmi.log("[ComplexTable2] getFilterId, unexpected format of virtual item name: " + vItemName, 3);
            }
            return id;
        },
        /**
         * @private
         */
        getColTxtFilterDgCols: function(col) {
            var fDgCols = [],
                cfgCol = this.cConfig.resp["column-org"][col];
            if (cfgCol) {
                var nofFields = cfgCol.fields.length;
                for (var fldIdx = 0; fldIdx < nofFields; fldIdx++) {
                    if (cfgCol.fields[fldIdx].hasTextFilter) {
                        fDgCols.push(cfgCol.fields[fldIdx].dgCol);
                    }
                }
            }
            return fDgCols;
        },
        /**
         * @private
         */
        applyFilters: function() {
            if (this.isApplyFiltersEnabled()) {
                var dgBtnFilCols = Object.keys(this.filterData),
                    dgTxtFilCols = Object.keys(this.colTextFilterData),
                    dgSumFilCols = dgBtnFilCols.slice(); // clone it
                for (var i = 0; i < dgTxtFilCols.length; i++) {
                    if (dgSumFilCols.indexOf(dgTxtFilCols[i]) === -1) { // not yet contained
                        dgSumFilCols.push(dgTxtFilCols[i]);
                    }
                }

                for (i = 0; i < dgSumFilCols.length; i++) {
                    var btnFilProps = this.filterData[dgSumFilCols[i]],
                        txtFilProps = this.colTextFilterData[dgSumFilCols[i]],
                        processFilterCol = false;
                    if (btnFilProps && btnFilProps.changed) {
                        processFilterCol = true;
                    } else if (txtFilProps && txtFilProps.changed) {
                        processFilterCol = true;
                    }
                    if (processFilterCol) {
                        var fExpr = [];
                        if (btnFilProps) {
                            fExpr = btnFilProps.expr.slice();
                            btnFilProps.changed = false;
                        }
                        if (txtFilProps) {
                            if (txtFilProps.expr !== "") {
                                fExpr.push(txtFilProps.expr);
                            }
                            txtFilProps.changed = false;
                        }
                        var dgCol = shmi.toNumber(dgSumFilCols[i]);
                        if (fExpr.length === 0) {
                            setLastClicked(this, -1);
                            this.dgMan.clearFilter(this.cConfig.table, dgCol);
                        } else {
                            setLastClicked(this, -1);
                            this.dgMan.setFilter(this.cConfig.table, dgCol, fExpr);
                        }
                    }
                }
            }
        },
        /**
         * @private
         */
        isApplyFiltersEnabled: function() {
            return this.applyFiltersEnable;
        },
        /**
         * @private
         */
        setApplyFiltersEnable: function(enable) {
            this.applyFiltersEnable = enable;
        },
        /**
         * @private
         */
        resetFilterChangeFlags: function() {
            var dgBtnFilCols = Object.keys(this.filterData);
            for (var i = 0; i < dgBtnFilCols.length; i++) {
                this.filterData[dgBtnFilCols[i]].changed = false;
            }
        },
        /**
         * @private
         */
        resetColTxtFilterChangeFlags: function() {
            var dgTxtFilCols = Object.keys(this.colTextFilterData);
            for (var i = 0; i < dgTxtFilCols.length; i++) {
                this.colTextFilterData[dgTxtFilCols[i]].changed = false;
            }
        },

        /* other helpers */
        /**
         * @private
         */
        clearCtrlPanelData: function() {
            if (!this.ctrlPanelData.filter) { // 1st time
                this.ctrlPanelData.filter = {};
                this.ctrlPanelData.filter.names = [];
                this.ctrlPanelData.filter.items = [];
            }

            // reset filter items
            var restoreApplyFiltersEnable = this.isApplyFiltersEnabled();
            this.setApplyFiltersEnable(false);
            var im = shmi.visuals.session.ItemManager;
            for (var i = 0; i < this.ctrlPanelData.filter.items.length; i++) {
                im.writeValue(this.ctrlPanelData.filter.items[i].name, 0);
            }
            this.resetFilterChangeFlags();
            this.setApplyFiltersEnable(restoreApplyFiltersEnable);

            if (this.ctrlPanelData.filter.btns) {
                for (i = 0; i < this.ctrlPanelData.filter.btns.length; i++) {
                    shmi.deleteControl(this.ctrlPanelData.filter.btns[i]);
                }
            }
            this.ctrlPanelData.filter.btns = [];

            if (this.ctrlPanelData.delRowsBtn) {
                shmi.deleteControl(this.ctrlPanelData.delRowsBtn);
            }
            this.ctrlPanelData.delRowsBtn = null;
        },
        /**
         * @private
         */
        clearColTextFilterCtrls: function() {
            if (!this.colTxtFilters.items) { // 1st time
                this.colTxtFilters.items = []; // 0..maxCols of this instance - on demand extended
            }

            // reset filter items
            var restoreApplyFiltersEnable = this.isApplyFiltersEnabled();
            this.setApplyFiltersEnable(false);
            var im = shmi.visuals.session.ItemManager;
            for (var i = 0; i < this.colTxtFilters.items.length; i++) {
                if (this.colTxtFilters.items[i]) {
                    im.writeValue(this.colTxtFilters.items[i].name, "");
                }
            }
            this.resetColTxtFilterChangeFlags();
            this.setApplyFiltersEnable(restoreApplyFiltersEnable);

            var ctrls = [];
            if (this.colTxtFilters.cols) {
                for (var col = 0; col < this.colTxtFilters.cols.length; col++) {
                    if (this.colTxtFilters.cols[col]) {
                        ctrls = this.colTxtFilters.cols[col].ctrls;
                        for (i = 0; i < ctrls.length; i++) {
                            shmi.deleteControl(ctrls[i]);
                        }
                    }
                }
            }
            this.colTxtFilters.cols = []; // 0..nofCols of current layout
        },
        /**
         * @private
         */
        clearCtrlPanelDOM: function() {
            var self = this;
            if (self.ctrlPanelElement) {
                if (self.ctrlPanelData && self.ctrlPanelData.qSearchInput) {
                    shmi.deleteControl(self.ctrlPanelData.qSearchInput);
                    self.ctrlPanelData.qSearchInput = null;
                }
                var childEl = self.ctrlPanelElement.firstChild;
                while (childEl) {
                    self.ctrlPanelElement.removeChild(childEl);
                    childEl = self.ctrlPanelElement.firstChild;
                }
            }
        },
        /**
         * @private
         */
        updateCtrlPanel: function() {
            if (this.ctrlPanelElement) {
                var cWidth = this.getTableWidth();
                if (cWidth < this.cConfig["show-buttons-table-min-width-px"]) {
                    shmi.removeClass(this.ctrlPanelElement, this.uiCssCl.ctrlPanelNormal);
                    shmi.addClass(this.ctrlPanelElement, this.uiCssCl.ctrlPanelMin);
                } else {
                    shmi.removeClass(this.ctrlPanelElement, this.uiCssCl.ctrlPanelMin);
                    shmi.addClass(this.ctrlPanelElement, this.uiCssCl.ctrlPanelNormal);
                }

                this.clearCtrlPanelData();
                this.clearCtrlPanelDOM();

                if (this.cConfig.quicksearch && this.cConfig.quicksearch.enable) {
                    this.enableQuickSearch();
                } else {
                    this.enableFilterButtons();
                }

                if (this.cConfig["delete-selected-rows"]) {
                    var delBtnElem = document.createElement("div");
                    this.ctrlPanelElement.appendChild(delBtnElem);
                    var config = this.getChildCtrlConfig(this.cConfig["delete-selected-rows-button-config"]);
                    if (!config) {
                        config = {
                            "label": "${ct2_delete_rows}"
                        };
                    }
                    this.ctrlPanelData.delRowsBtn = shmi.createControl("button", delBtnElem, config, "div");
                    this.ctrlPanelData.delRowsBtn.onClick = function() {
                        this.deleteSelectedRows();
                    }.bind(this);
                    this.updateDelRowsBtnEnable();
                }
            }
        },
        /**
         * @private
         */
        enableFilterButtons: function() {
            var fData = this.ctrlPanelData.filter, // shortcut
                fCfg = this.cConfig.filters, // shortcut
                nofFilters = fCfg.length,
                fId,
                itemNamespace = this.getFilterItemNS();
            if (fData.items.length === 0) { // 1st time
                for (fId = 0; fId < nofFilters; fId++) {
                    var fName = shmi.localize(fCfg[fId].label);
                    fData.names.push(fName);
                    var iName = "virtual:" + itemNamespace + ":" + fId,
                        fItem = shmi.createVirtualItem(iName, 2, 0, 1, 0, function(value, type, vItemName) {
                            this.filterChanged(value, type, vItemName);
                        }.bind(this));
                    fData.items.push(fItem);
                }
            }
            for (fId = 0; fId < nofFilters; fId++) {
                var fBtnElem = document.createElement("div");
                this.ctrlPanelElement.appendChild(fBtnElem);
                var cfg = this.getChildCtrlConfig(fCfg[fId]["filter-button-config"],
                    this.cConfig["default-filter-button-config"]);
                if (!cfg) {
                    cfg = {
                        "off-value": 0,
                        "on-value": 1
                    };
                    cfg["off-label"] = cfg["on-label"] = fData.names[fId];
                    if (fCfg[fId].template) {
                        cfg.template = fCfg[fId].template;
                    }
                    if (fCfg[fId]["class-name"]) {
                        cfg["class-name"] = fCfg[fId]["class-name"];
                    }
                }
                cfg.item = fData.items[fId].name;
                var fBtn = shmi.createControl("toggle-button", fBtnElem, cfg, "div");
                fData.btns.push(fBtn);
            }
        },
        /**
         * @private
         */
        enableQuickSearch: function() {
            var self = this;
            if (self.cConfig.quicksearch.chaining === "AND" || self.cConfig.quicksearch.chaining === "OR") {
                shmi.visuals.session.DataGridManager.setFilterChaining(self.cConfig.table, self.cConfig.quicksearch.chaining);
            } else {
                shmi.visuals.session.DataGridManager.setFilterChaining(self.cConfig.table, "OR");
            }

            if (self.ctrlPanelData.qSearchInput) {
                return; //no need to setup quick search input-field again when it already exists
            }

            var qSearchElem = document.createElement("div"),
                qSearchClearElem = document.createElement("div");

            shmi.addClass(qSearchElem, "qsearch-box");
            self.ctrlPanelElement.appendChild(qSearchElem);
            var conf = {
                "label": "${ct2_quicksearch}",
                "class-name": "input-field label-beside"
            };
            self.ctrlPanelData.qSearchInput = shmi.createControl("input-field", qSearchElem, conf, "div");

            qSearchElem.appendChild(qSearchClearElem);
            shmi.addClass(qSearchClearElem, "ct2-qsearch-clear");

            const qSearchHandler = function() {
                if (self.isActive() && !self.locked) {
                    self.ctrlPanelData.qSearchInput.setValue("");
                    self.qSearchOld = null;
                    self.quicksearchHandler();
                }
            };

            qSearchClearElem.addEventListener('click', qSearchHandler);
            qSearchClearElem.addEventListener('touchstart', qSearchHandler);

            shmi.onActive([self.ctrlPanelData.qSearchInput], function() {
                if (self.isActive()) {
                    var valueElement = self.ctrlPanelData.qSearchInput.valueElement;

                    if (self.qSearchOld !== null) {
                        self.ctrlPanelData.qSearchInput.setValue(self.qSearchOld);
                    }

                    self.quicksearchHandler = onQuickSearch.bind(null, self, valueElement);
                    self.quicksearchValueElement = valueElement;
                    valueElement.addEventListener('keyup', self.quicksearchHandler);
                    self.vars.qSearchToken = self.ctrlPanelData.qSearchInput.listen("change", self.quicksearchHandler);

                    if (self.cConfig.quicksearch && self.cConfig.quicksearch.initialFocus) {
                        self.ctrlPanelData.qSearchInput.valueElement.focus();
                    }
                }
            });

            if (self.isLocked()) {
                self.ctrlPanelData.qSearchInput.lock();
            }
        },
        /**
         * @private
         */
        updateDelRowsBtnEnable: function() {
            var enabled = true;
            if (this.ctrlPanelData.delRowsBtn) {
                if ((this.selection.type === this.c.SELECT_TYPE_NOTHING) || ((this.selection.type === this.c.SELECT_TYPE_SEL_ROWS) && (this.selection.selRows.length === 0))) {
                    enabled = false;
                }
                if (enabled) {
                    this.ctrlPanelData.delRowsBtn.unlock();
                } else {
                    this.ctrlPanelData.delRowsBtn.lock();
                }
            }
        },
        /**
         * @private
         */
        getFilterItemNS: function() {
            return "ct2-filter-" + this.instID;
        },
        /**
         * @private
         */
        getColTxtFilterItemNS: function() {
            return "ct2-col-txt-filter-" + this.instID;
        },
        getAdapterNS: function() {
            return "ct2-adapter-" + this.instID;
        },
        /**
         * @private
         */
        updateHeader: function() {
            var self = this;
            // cleanup this.header
            if (this.header.cols) {
                this.header.cols.forEach(function(col, idx) {
                    if (col.ctrls[0].cbMouseListener) {
                        col.ctrls[0].cbMouseListener.disable();
                        col.ctrls[0].cbTouchListener.disable();
                    }
                });
            }
            this.header.selAllCBox = null;
            this.header.cols = [];
            // cleanup DOM
            var childElem = this.theadElement.firstChild;
            while (childElem) {
                this.theadElement.removeChild(childElem);
                childElem = this.theadElement.firstChild;
            }
            // reinit header table
            var nofCols = this.cConfig.resp["nof-cols"],
                isMultiSelectMode = (this.cConfig.resp["select-mode"] === "MULTI"),
                colGroupElem = this.createColGroup(nofCols, isMultiSelectMode);
            this.theadElement.appendChild(colGroupElem);
            var trElem = document.createElement("tr"),
                thElem = null,
                tabCellElem = null;
            if (isMultiSelectMode && this.config["show-select-boxes"]) {
                thElem = document.createElement("th");
                tabCellElem = document.createElement("div");
                this.header.selAllCBox = this.createSelCBox(tabCellElem, -1, true, this);
                this.header.selAllCBox.setChangeCallback(function(id, state) {
                    if (!self.locked) {
                        self.selectAllCBoxValChanged(state);
                    }
                });
                thElem.appendChild(tabCellElem);
                trElem.appendChild(thElem);
            }
            this.theadElement.appendChild(trElem);
            for (var col = 0; col < nofCols; col++) {
                var cfgCol = this.cConfig.resp["column-org"][col];
                thElem = document.createElement("th");
                tabCellElem = document.createElement("div");
                this.header.cols[col] = {};
                this.header.cols[col].ctrls = this.createHeaderCtrls(tabCellElem, col);
                thElem.appendChild(tabCellElem);
                trElem.appendChild(thElem);

                if (cfgCol.fields[0]["min-width"]) {
                    tabCellElem.style.minWidth = cfgCol.fields[0]["min-width"];
                }
                this.header.cols[col].ctrls.forEach(function(hCtrl, idx) {
                    var fldName = hCtrl.fieldName;
                    if (self.config.adapterSettings && self.config.adapterSettings[fldName]) {
                        var settings = self.config.adapterSettings[fldName],
                            itemName = null;
                        self.adapterItems.forEach(function(aEntry, index) {
                            if (aEntry.field === fldName) {
                                itemName = aEntry.item.name;
                            }
                        });

                        var selectElem = document.createElement('div');
                        thElem.appendChild(selectElem);
                        var selectConfig = shmi.cloneObject(self.config.adapterSelectConfig);
                        selectConfig.item = itemName;
                        selectConfig.options = settings;
                        shmi.createControl("select-box", selectElem, selectConfig, 'DIV', 'from');
                        shmi.addClass(thElem, self.config.adapterSelectClass);
                    }
                });
            }
        },
        /**
         * @private
         */
        updateColTextFilters: function() {
            var self = this;
            if (this.txtFiltersHeadElement && this.txtFiltersUsed()) {
                // cleanup filter data
                this.clearColTextFilterCtrls();
                // cleanup DOM
                var childElem = this.txtFiltersHeadElement.firstChild;
                while (childElem) {
                    this.txtFiltersHeadElement.removeChild(childElem);
                    childElem = this.txtFiltersHeadElement.firstChild;
                }

                // init the virtual items
                var fData = this.colTxtFilters,
                    col;
                if (fData.items.length === 0) { // 1st time
                    var maxCols = Object.keys(this.cConfig["field-datagrid-col-map"]).length, // maxCols === number of fields
                        itemNamespace = this.getColTxtFilterItemNS();
                    for (col = 0; col < maxCols; col++) {
                        var iName = "virtual:" + itemNamespace + ":" + col;
                        this.colTxtFilters.items[col] = shmi.createVirtualItem(iName, 0, 0, 1, "", function(value, type, vItemName) {
                            this.colTxtFilterChanged(value, type, vItemName);
                        }.bind(this));
                    }
                }

                // reinit filter header table
                var nofCols = this.cConfig.resp["nof-cols"],
                    isMultiSelectMode = (this.cConfig.resp["select-mode"] === "MULTI"),
                    colGroupElem = self.createColGroup(nofCols, isMultiSelectMode);

                this.txtFiltersHeadElement.appendChild(colGroupElem);
                var trElem = document.createElement("tr");
                this.txtFiltersHeadElement.appendChild(trElem);
                var thElem = null,
                    tabCellElem = null;
                if (isMultiSelectMode) {
                    thElem = document.createElement("th");
                    trElem.appendChild(thElem);
                }
                for (col = 0; col < nofCols; col++) {
                    var cfgCol = this.cConfig.resp["column-org"][col];
                    thElem = document.createElement("th");
                    tabCellElem = document.createElement("div");
                    thElem.appendChild(tabCellElem);
                    trElem.appendChild(thElem);
                    if (cfgCol.hasTextFilter) {
                        this.colTxtFilters.cols[col] = {};
                        this.colTxtFilters.cols[col].ctrls = this.createColTxtFilterCtrls(tabCellElem, col);
                    }
                }
            }
        },
        /**
         * @private
         */
        getDgFilterSettings: function() {
            // clear all filters and sort for the associated grid - async.
            var self = this;
            clearTimeout(self.reset_sort_filters_to);
            self.reset_sort_filters_to = setTimeout(function() {
                self.resetSortFilters();
            }, shmi.c("ACTION_RETRY_TIMEOUT"));
        },
        /**
         * @private
         */
        getColTxtFilterVal: function(fExpr) {
            var txtFlVal = "",
                exprLen = fExpr.length;
            if ((exprLen >= 3) && (fExpr.charAt(0) === "%") && (fExpr.charAt(exprLen - 1) === "%")) {
                txtFlVal = fExpr.slice(1, exprLen - 1);
            } else {
                shmi.log("[ComplexTable2] getColTxtFilterVal, unexpected format of filter expression: " + fExpr, 2);
            }
            return txtFlVal;
        },
        /**
         * @private
         */
        getBtnFilter: function(fExpr, dgCol) {
            var btnFId = -1,
                fCfg = this.cConfig.filters;
            for (var fId = 0; fId < fCfg.length; fId++) {
                var fDgCol = this.getDataColNo(fCfg[fId].field);
                if ((fDgCol === dgCol) && (fExpr === fCfg[fId].expr)) {
                    btnFId = fId;
                    break;
                }
            }
            return btnFId;
        },
        /**
         * @private
         */
        resetSortFilters: function() {
            setLastClicked(this, -1);
            this.dgMan.sort(this.cConfig.table, -1, "ASC");
            this.dgMan.clearFilter(this.cConfig.table, -1);
        },
        /**
         * @private
         */
        updateNofRowsDisplay: function() {
            if (this.cConfig.resp["show-nof-rows"] && this.footerElement) {
                this.footerElement.textContent = this.pageBuf.nofTotalRows + " " + shmi.localize("${ct2_nof_rows}");
            }
        },
        /**
         * @private
         */
        updateRespDepContent: function(layoutChanged) {
            var self = this;
            if (layoutChanged) {
                if (self.cConfig.resp.headerMode === "ICON") {
                    self.headerMode = "ICON";
                } else {
                    self.headerMode = "TEXT";
                }

                if (this.cLayoutClass) {
                    shmi.removeClass(this.element, this.cLayoutClass);
                }
                this.cLayoutClass = this.cConfig.resp["layout-class-name"];
                if (this.cLayoutClass) {
                    shmi.addClass(this.element, this.cLayoutClass);
                }
                if (this.cConfig.resp["show-nof-rows"]) {
                    shmi.removeClass(this.element, this.uiCssCl.rowDisplInvisible);
                    shmi.addClass(this.element, this.uiCssCl.rowDisplVisible);
                } else {
                    shmi.removeClass(this.element, this.uiCssCl.rowDisplVisible);
                    shmi.addClass(this.element, this.uiCssCl.rowDisplInvisible);
                }
                if (this.cConfig.resp["select-mode"] === "MULTI") {
                    shmi.addClass(this.element, this.uiCssCl.multiselect);
                } else {
                    shmi.removeClass(this.element, this.uiCssCl.multiselect);
                }
            }
            this.updateCtrlPanel();
            this.updateHeader();
            if (this.cConfig.quicksearch) {
                if (!this.cConfig.quicksearch.enable) {
                    this.updateColTextFilters();
                }
            } else {
                this.updateColTextFilters();
            }
        },

        /**
         * @private
         */
        cleanupTabContent: function() {
            this.clearPgBuf();
            this.clearTableDOM();
        },
        /**
         * @private
         */
        clearPgBuf: function() {
            if (this.pageBuf.rows) { // obj is already initialized
                for (var bufRow = 0; bufRow < this.pageBuf.rows.length; bufRow++) {
                    if (this.pageBuf.rows[bufRow].trSelCtrl) {
                        if (this.pageBuf.rows[bufRow].trSelCtrl.rsMouseListener) {
                            this.pageBuf.rows[bufRow].trSelCtrl.rsMouseListener.disable();
                            this.pageBuf.rows[bufRow].trSelCtrl.rsTouchListener.disable();
                        }
                    }
                    for (var col = 0; col < this.pageBuf.nofCols; col++) {
                        var ctrls = this.pageBuf.rows[bufRow].cols[col].ctrls;
                        for (var ctrlIdx = 0; ctrlIdx < ctrls.length; ctrlIdx++) {
                            shmi.deleteControl(ctrls[ctrlIdx]);
                        }
                    }
                }
            }
            this.pageBuf.offset = 0;
            this.pageBuf.nofTotalRows = 0;
            this.pageBuf.size = 0;
            this.pageBuf.nofCols = 0;
            this.pageBuf.firstVpRow = 0; // 0 .. .size-1
            this.pageBuf.avgNofVpLines = 0;
            this.pageBuf.rows = [];
        },
        /**
         * @private
         */
        clearTableDOM: function() {
            this.tbodyElement.forEach(function(el) {
                var childEl = el.firstChild;
                while (childEl) {
                    el.removeChild(childEl);
                    childEl = el.firstChild;
                }
            });
        },
        /**
         * @private
         */
        cleanupMetrics: function() {
            if (this.metricsData.isValid) {
                this.metricsData.last = shmi.cloneObject(this.metricsData);
            } else {
                this.metricsData.last = {};
            }
            this.metricsData.isValid = false;
            this.metricsData.lineHeightPx = 0;
        },
        /**
         * @private
         */
        cleanupAsyncProcessing: function() {
            if (this.asyncProcData.onDgChgRetryTO) {
                clearTimeout(this.asyncProcData.onDgChgRetryTO);
            }
            if (this.asyncProcData.onEvtResizeRetryTO) {
                clearTimeout(this.asyncProcData.onEvtResizeRetryTO);
            }
            if (this.asyncProcData.processMetricsTO) {
                clearTimeout(this.asyncProcData.processMetricsTO);
            }
            this.asyncProcData.onDgChgRetryTO = null;
            this.asyncProcData.onEvtResizeRetryTO = null;
            this.asyncProcData.processMetricsTO = null;
        },

        /**
         * @private
         */
        reinitTable: function(dataGridChgInfo, currentDataGridIds, keepScrollTop) {
            var self = this,
                initialSize = this.pageBuf.rows.length;
            // precondition: table + pageBuf ist empty (cleanupTabContent has already been processed)
            if (initialSize !== 0) {
                console.warn("[ComplexTable2] reinitTable, table is not empty, length:", initialSize);
                return;
            }

            if (dataGridChgInfo.totalRows !== 0) {
                shmi.removeClass(this.element, this.uiCssCl.emptyTab);

                //-- init pageBuf --
                this.pageBuf.bufferOffset = dataGridChgInfo.offset;
                this.pageBuf.nofTotalRows = dataGridChgInfo.totalRows;
                self.tableElement.style.height = (self.pageBuf.nofTotalRows * self.metricsData.lineHeightPx) + "px";
                this.pageBuf.size = currentDataGridIds.length; //number of data rows currently loaded
                this.pageBuf.viewportSize = this.cConfig["nof-buffered-rows"]; //total number of available data rows
                this.pageBuf.bufferSize = this.cConfig["buffer-size"];
                if (this.pageBuf.offset < this.pageBuf.bufferOffset) {
                    this.pageBuf.offset = this.pageBuf.bufferOffset;
                } else if (this.pageBuf.offset > this.pageBuf.bufferOffset + this.pageBuf.bufferSize) {
                    this.pageBuf.offset = this.pageBuf.bufferOffset;
                }
                var nofCols = this.pageBuf.nofCols = this.cConfig.resp["nof-cols"];
                for (var bufRow = initialSize; bufRow < this.pageBuf.viewportSize; bufRow++) {
                    //NOTE - here all row descriptors should be initialized & controls created,
                    // but controls cannot yet be connected to items and rows should be hidden
                    this.pageBuf.rows[bufRow] = createRowDescriptor((typeof currentDataGridIds[bufRow] === "number") ? currentDataGridIds[bufRow] : null, nofCols);
                }

                if (!keepScrollTop) {
                    this.bodyViewportElement.scrollTop = 0; // set scrollTop back to top of table
                }

                //-- create table DOM and cell controls --
                var isMultiSelectMode = (this.cConfig.resp["select-mode"] === "MULTI");

                this.tbodyElement.forEach(function(el, idx) {
                    var rowsPerPage = self.pageBuf.viewportSize / 3,
                        colGroupElem = self.createColGroup(nofCols, isMultiSelectMode);
                    el.appendChild(colGroupElem);
                    for (var i = 0; i < rowsPerPage; i++) {
                        createTableRow(self, (idx * rowsPerPage) + i, el, i === 0);
                    }
                });

                this.processAsyncMetricsUpdate(keepScrollTop ? self.vars.scrollTop : null);
            } else {
                this.pageBuf.nofCols = this.cConfig.resp["nof-cols"];
                shmi.addClass(this.element, this.uiCssCl.emptyTab);
            }
        },
        /**
         * @private
         */
        processTabReinitDependencies: function() {
            this.updateNofRowsDisplay();
            this.updateSelectionView();
        },
        doScrollUpdate: function doScrollUpdate() {
            var self = this,
                scrollTop = self.bodyViewportElement.scrollTop;

            updatePages(self, scrollTop);
        },
        /*
         * @private
         */
        initScrollHandler: function() {
            var self = this,
                scrollUpdateTimeoutLength = 100;
            self.bodyViewportElement.style["overflow-y"] = "scroll";
            self.onBodyScroll = function(evt) {
                clearTimeout(self.vars.updateTimeout);
                self.vars.updateTimeout = 0;

                if (!self.isActive() || self.locked) {
                    //reset scroll - event is not cancelable with preventDefault()
                    self.bodyViewportElement.scrollTop = 0;
                    return;
                }

                if (Date.now() - self.vars.lastUpdate > scrollUpdateTimeoutLength) {
                    self.doScrollUpdate();
                    self.vars.lastUpdate = Date.now();
                } else {
                    self.vars.updateTimeout = setTimeout(function() {
                        self.doScrollUpdate();
                        self.vars.lastUpdate = Date.now();
                    }, scrollUpdateTimeoutLength);
                }
            };
            self.bodyViewportElement.addEventListener('scroll', self.onBodyScroll);
        },
        /*
         * @private
         */
        resetScroll: function() {
            this.bodyViewportElement.scrollTop = 0; // set scrollTop back to top of table
        },
        /*
         * @private
         */
        updateTable: function(dataGridChgInfo, currentDataGridIds) {
            var self = this,
                linesPerPage = self.cConfig["nof-buffered-rows"] / 3,
                usedBufferIndices = [];
            // precondition: pageBuf.size is not changed and !== 0
            if ((self.pageBuf.rows.length === 0) || (self.pageBuf.bufferSize < currentDataGridIds.length)) {
                console.error("[ComplexTable2]", "updateTable, table is empty or buffer size has been changed");
                return;
            }
            //self.pageBuf.offset >= self.pageBuf.bufferOffset
            var indexOffset = self.pageBuf.offset - self.pageBuf.bufferOffset;
            currentDataGridIds.forEach(function(dgId, idx) {
                var currentIndex = idx - indexOffset;
                if ((currentIndex < 0) || (currentIndex >= (linesPerPage * 3))) {
                    return;
                }
                var curPage = Math.floor(currentIndex / linesPerPage),
                    bufIdx = (self.vars.pageOrder[curPage] * linesPerPage) + (currentIndex % linesPerPage),
                    curRow = self.pageBuf.rows[bufIdx];

                if (curRow) {
                    usedBufferIndices.push(bufIdx);
                    if (curRow.rowId !== dgId || !curRow.visible) {
                        self.writeRow(bufIdx, currentDataGridIds, idx);
                    }
                } else {
                    console.error("[ComplexTable2]", "invalid buffer index:", bufIdx);
                }
            });

            self.pageBuf.rows.forEach(function(r, idx) {
                if (usedBufferIndices.indexOf(idx) === -1) {
                    if (r.visible) {
                        r.visible = false;
                        r.trElem.style.visibility = "hidden";
                    }
                } else if (!r.visible) {
                    r.visible = true;
                    r.trElem.style.visibility = "";
                }
            });
        },
        /*
         * @private
         */
        writeRow: function(rId, currentDataGridIds, bufRow) {
            var rowId = 0,
                item = "",
                cfgCol = null;

            if (this.pageBuf.rows[rId]) {
                rowId = this.pageBuf.rows[rId].rowId = currentDataGridIds[bufRow];

                for (var col = 0; col < this.pageBuf.nofCols; col++) {
                    cfgCol = this.cConfig.resp["column-org"][col];
                    var ctrls = this.pageBuf.rows[rId].cols[col].ctrls;
                    for (var ctrlIdx = 0; ctrlIdx < ctrls.length; ctrlIdx++) {
                        item = this.makeItem(rowId, cfgCol.fields[ctrlIdx].dgCol);

                        var ia = shmi.requires("visuals.tools.item-adapter"),
                            columnAdapter = this.getColumnAdapter(cfgCol.fields[ctrlIdx].fieldName);
                        if (shmi.visuals.session.ItemManager.getItem(item) !== null) {
                            if (columnAdapter === null) {
                                ia.unsetAdapter(item);
                            } else {
                                ia.setAdapter(item, columnAdapter);
                            }
                        }
                        ctrls[ctrlIdx].setItem(item);
                    }
                }
            }
        },
        /**
         * @private
         */
        updateMetricsData: function(scrollTop) {
            var self = this;
            if (self.pageBuf.size > 0) {
                self.metricsData.lineHeightPx = self.tbodyElement[0].getBoundingClientRect().height / self.tbodyElement[0].rows.length;

                self.tableElement.style.height = (self.pageBuf.nofTotalRows * self.metricsData.lineHeightPx) + "px";
                self.tableElement.style.overflow = "hidden";
                self.metricsData.viewportHeight = self.bodyViewportElement.getBoundingClientRect().height;
                self.metricsData.viewportWidth = self.bodyViewportElement.getBoundingClientRect().width;

                var minBufRows = Math.ceil(self.metricsData.viewportHeight / self.metricsData.lineHeightPx) * 3;
                if (self.cConfig["nof-buffered-rows"] < minBufRows) {
                    self.cConfig["nof-buffered-rows"] = minBufRows;
                    console.info("[ComplexTable2]", "buffer rows configured too low, corrected to:", minBufRows);
                }
                var nofRows = self.cConfig["nof-buffered-rows"];
                self.pageBuf.nofTotalPages = self.pageBuf.nofTotalRows / (nofRows / 3);
            } else if (self.metricsData.last.isValid) { // empty table
                self.metricsData = shmi.cloneObject(self.metricsData.last);
            } else {
                self.metricsData.lineHeightPx = self.c.DEFAULT_LINE_HEIGHT;
            }
            self.metricsData.isValid = true;
            if (typeof scrollTop === "number") {
                reinitPages(self, scrollTop);
            } else {
                self.vars.scrollTop = 0;
                reinitPages(self, 0);
            }
        },
        /**
         * @private
         */
        disableEvents: function() {
            if (this.header.selAllCBox) {
                this.header.selAllCBox.setEventsEnabled(false);
            }
        },
        /**
         * @private
         */
        enableEvents: function() {
            if (this.header.selAllCBox) {
                this.header.selAllCBox.setEventsEnabled(true);
            }
        },
        /**
         * @private
         */
        // $note: col 0..nofCols-1
        createHeaderCtrls: function(tabCellElem, col) {
            var self = this,
                hCtrls = [];
            shmi.addClass(tabCellElem, this.uiCssCl.thCont);
            var cfgCol = this.cConfig.resp["column-org"][col],
                nofFields = cfgCol.fields.length;
            for (var headIdx = 0; headIdx < nofFields; headIdx++) {
                var hCtrlElem = document.createElement("div"),
                    icon = cfgCol.fields[headIdx].icon,
                    hCaption = cfgCol.fields[headIdx].header || " ";
                if (hCaption) {
                    var lhCaption = shmi.localize(hCaption);
                    if (headIdx > 0) {
                        lhCaption = " / " + lhCaption;
                    }
                    var hCtrl = new shmi.visuals.controls.ComplexTable2.FieldHeader(
                        hCtrlElem,
                        cfgCol.fields[headIdx].dgCol, // id
                        cfgCol.fields[headIdx].isSortable,
                        this.c.NO_SORT,
                        lhCaption,
                        self,
                        cfgCol.fields[headIdx].fieldName, icon);
                    hCtrl.setChangeCallback(function(id, state) {
                        if (!self.locked) {
                            self.sortHeaderChanged(id, state);
                        }
                    });
                    hCtrls.push(hCtrl);
                    tabCellElem.appendChild(hCtrlElem);
                }
            }

            return hCtrls;
        },
        /**
         * @private
         */
        // $note: col 0..nofCols-1
        createColTxtFilterCtrls: function(tabCellElem, col) {
            var cfCtrls = [],
                cfContainerElem = document.createElement("div"),
                cfLineElem = document.createElement("div");
            cfContainerElem.appendChild(cfLineElem);
            tabCellElem.appendChild(cfContainerElem);

            shmi.addClass(tabCellElem, this.uiCssCl.thCont);
            shmi.addClass(cfContainerElem, this.uiCssCl.colTxtFilterCont);
            shmi.addClass(cfLineElem, this.uiCssCl.colTxtFilterLineCont);

            var fImgElem = document.createElement("img");
            cfLineElem.appendChild(fImgElem);
            fImgElem.src = "pics/system/controls/complex-table2/filter.svg";
            shmi.addClass(fImgElem, this.uiCssCl.colTxtFilterIcon);

            var iName = this.colTxtFilters.items[col].name,
                cfg = this.getChildCtrlConfig(this.cConfig.resp["text-filter-input-field-config"]);
            if (!cfg) {
                cfg = {
                    "class-name": "input-field if-filter"
                };
            }
            cfg.item = iName;
            var fInpFld = shmi.createControl("input-field", cfLineElem, cfg, "div");
            cfCtrls.push(fInpFld);

            cfg = this.getChildCtrlConfig(this.cConfig.resp["text-filter-clear-button-config"]);
            if (!cfg) {
                cfg = {
                    "class-name": "button icon-only reset-filter",
                    "label": "X",
                    "icon-src": "pics/system/controls/complex-table2/cross_white.svg"
                };
            }
            var fResetFilterBtn = shmi.createControl("button", cfLineElem, cfg, "div");
            fResetFilterBtn.onClick = function() {
                this.colTxtFilterReset(col);
            }.bind(this);
            cfCtrls.push(fResetFilterBtn);

            return cfCtrls;
        },
        /**
         * @private
         */
        createSelCBox: function(tabCellElem, id, is3state, ct2ref) {
            return new shmi.visuals.controls.ComplexTable2.CheckBox(tabCellElem, id, is3state, this.c.CB_UNCHK, ct2ref);
        },
        /**
         * @private
         */
        createSelRowCtrl: function(trElem, id, ct2ref) {
            return new shmi.visuals.controls.ComplexTable2.RowSelector(trElem, id, this.c.CB_UNCHK, ct2ref, {
                toggleSelection: this.cConfig.resp["toggle-selection"],
                doubleClickEvents: this.cConfig.resp["double-click-events"]
            });
        },
        /**
         * @private
         */
        createColGroup: function(nofCols, isMultiSelectMode) {
            var colGroupElem = document.createElement("colgroup");
            var columns = [];
            if (isMultiSelectMode && this.config["show-select-boxes"]) {
                columns.push(document.createElement("col"));
                columns[columns.length - 1].setAttribute("class", this.uiCssCl.colCb);
            }
            for (var i = 0; i < nofCols; i++) {
                var cfgCol = this.cConfig.resp["column-org"][i];
                columns.push(document.createElement("col"));
                columns[columns.length - 1].setAttribute("class", this.uiCssCl.colPfx + (i + 1));
                if (cfgCol.fields[0]["column-width"]) {
                    columns[columns.length - 1].style.width = cfgCol.fields[0]["column-width"];
                }
            }
            columns.forEach(function(c) {
                colGroupElem.appendChild(c);
            });
            return colGroupElem;
        },
        /**
         * @private
         */
        getTableWidth: function() {
            var width = 0;
            if (this.tcElement) {
                width = this.tcElement.offsetWidth;
            } else {
                var tcElem = shmi.getUiElement(this.uiElName.tabContainer, this.element);
                if (!tcElem) {
                    shmi.log("[ComplexTable2] no " + this.uiElName.tabContainer + " element provided (required)", 3);
                } else {
                    width = tcElem.offsetWidth;
                }
            }
            return width;
        },
        /**
         * @private
         */
        getTableVpWidth: function() {
            return this.bodyViewportElement.offsetWidth;
        },
        /**
         * @private
         */
        isTabSizeChanged: function() {
            var changed = false;
            if ((this.getTableVpWidth() !== this.lastTableVpWidth) || (this.element.offsetHeight !== this.lastTableVpHeight)) {
                changed = true;
                this.lastTableVpWidth = this.getTableVpWidth();
                this.lastTableVpHeight = this.element.offsetHeight;
            }
            return changed;
        },
        /**
         * @private
         */
        isLayoutChanged: function() {
            var changed = false;
            if (this.cConfig.resp["layout-id"] !== this.lastLayoutId) {
                changed = true;
                this.lastLayoutId = this.cConfig.resp["layout-id"];
            }
            return changed;
        },
        /**
         * @private
         */
        makeItem: function(dgRowId, dgCol) {
            return this.dgSubscription.prefix + dgRowId + ":" + dgCol;
        },
        /**
         * @private
         */
        searchRowId: function(rowId) {
            var bufRowFound = -1;
            for (var bufRow = 0; bufRow < this.pageBuf.viewportSize; bufRow++) {
                if (this.pageBuf.rows[bufRow] && this.pageBuf.rows[bufRow].rowId === rowId) {
                    bufRowFound = bufRow;
                    break;
                }
            }
            return bufRowFound;
        },
        /**
         * @private
         */
        detectTouchDevice: function() {
            return !!("ontouchstart" in document.documentElement);
        },
        onLock: function() {
            var self = this;
            shmi.addClass(self.element, "locked");
            if (self.cConfig.quicksearch && self.cConfig.quicksearch.enable && self.ctrlPanelData.qSearchInput) {
                self.ctrlPanelData.qSearchInput.lock();
            }
        },
        onUnlock: function() {
            var self = this;
            shmi.removeClass(self.element, "locked");
            if (self.cConfig.quicksearch && self.cConfig.quicksearch.enable && self.ctrlPanelData.qSearchInput) {
                self.ctrlPanelData.qSearchInput.unlock();
            }
        }
    };

    shmi.extend(shmi.visuals.controls.ComplexTable2, shmi.visuals.core.BaseControl);
    shmi.registerControlType("complex-table2", shmi.visuals.controls.ComplexTable2, true);

    /*-- CT2 helper classes --*/

    /**
     * Creates a new ComplexTable2.CheckBox control.
     *
     * @param {HTMLElement} element <div></div>
     * @param {number} id bufRow 0..n, -1 for summary CB
     * @param {boolean} is3State
     * @param {number} initialState shmi.visuals.controls.ComplexTable2.c.CB_??
     * @param {object} ct2ref ct2 reference
     *
     * @constructor
     */
    shmi.visuals.controls.ComplexTable2.CheckBox = function(element, id, is3State, initialState, ct2ref) {
        this.ct2c = shmi.visuals.controls.ComplexTable2.c; // short cut
        this.css = shmi.visuals.controls.ComplexTable2.CheckBox.uiCssCl; // short cut
        this.element = element;
        this.id = id;
        this.is3tate = is3State;
        this.state = initialState;
        this.chgCallb = null;
        this.cbMouseListener = null;
        this.cbTouchListener = null;

        shmi.addClass(this.element, shmi.visuals.controls.ComplexTable2.uiCssCl.selCb);

        var cbEvents = {};
        cbEvents.onPress = function() {
            if (!ct2ref.locked) {
                shmi.addClass(this.element, this.css.sel);
            }
        }.bind(this);
        cbEvents.onRelease = function() {
            if (!ct2ref.locked) {
                shmi.removeClass(this.element, this.css.sel);
            }
        }.bind(this);
        cbEvents.onClick = function() {
            // process event
            if (!ct2ref.locked) {
                if (this.is3tate) {
                    switch (this.state) {
                    case this.ct2c.CB_UNCHK:
                        this.state = this.ct2c.CB_CHK;
                        break;
                    case this.ct2c.CB_CHK:
                        this.state = this.ct2c.CB_UNDEF;
                        break;
                    default:
                        this.state = this.ct2c.CB_UNCHK;
                        break;
                    }
                } else {
                    this.state = (this.state === this.ct2c.CB_UNCHK) ? this.ct2c.CB_CHK : this.ct2c.CB_UNCHK;
                }
                this.updateState();
                this.fireCbChg();
            }
        }.bind(this);
        this.cbMouseListener = new shmi.visuals.io.MouseListener(this.element, cbEvents);
        this.cbTouchListener = new shmi.visuals.io.TouchListener(this.element, cbEvents);

        this.updateState();
        this.setEventsEnabled(true);
    };
    shmi.visuals.controls.ComplexTable2.CheckBox.prototype = {
        setChangeCallback: function(listener) { // null: no event listening
            this.chgCallb = listener;
        },
        setEventsEnabled: function(enabled) {
            if (enabled) {
                this.cbMouseListener.enable();
                this.cbTouchListener.enable();
            } else {
                this.cbMouseListener.disable();
                this.cbTouchListener.disable();
            }
        },
        setState: function(state) {
            this.state = state;
            this.updateState();
        },
        getState: function() {
            return this.state;
        },
        /**
         * @private
         */
        updateState: function() {
            shmi.removeClass(this.element, this.css.ck);
            shmi.removeClass(this.element, this.css.unck);
            shmi.removeClass(this.element, this.css.undef);
            switch (this.state) {
            case this.ct2c.CB_UNCHK:
                shmi.addClass(this.element, this.css.unck);
                break;
            case this.ct2c.CB_CHK:
                shmi.addClass(this.element, this.css.ck);
                break;
            default:
                shmi.addClass(this.element, this.css.undef);
                break;
            }
        },
        /**
         * @private
         */
        fireCbChg: function() {
            if (this.chgCallb) {
                this.chgCallb(this.id, this.state);
            }
        }
    };

    /**
     * Creates a new ComplexTable2.RowSelector control.
     *
     * @param {HTMLElement} element <tr>...</tr>
     * @param {number} id bufRow 0..n
     * @param {number} initialState shmi.visuals.controls.ComplexTable2.c.CB_??
     * @param {object} ct2ref ct2 reference
     * @param {object} [options] options
     *
     * @constructor
     */
    shmi.visuals.controls.ComplexTable2.RowSelector = function(element, id, initialState, ct2ref, options) {
        var self = this;
        self.ct2c = shmi.visuals.controls.ComplexTable2.c; // short cut
        self.css = shmi.visuals.controls.ComplexTable2.RowSelector.uiCssCl; // short cut
        self.element = element;
        self.id = id;
        self.state = initialState;
        self.chgCallb = null;
        self.rsMouseListener = null;
        self.rsTouchListener = null;
        self.toggleSelection = options?.toggleSelection ?? true;

        var rsEvents = {};
        rsEvents.onPress = function() {
            if (!ct2ref.locked) {
                shmi.addClass(self.element, self.css.sel);
            }
        };
        rsEvents.onRelease = function() {
            if (!ct2ref.locked) {
                shmi.removeClass(self.element, self.css.sel);
            }
        };
        rsEvents.onClick = function(x, y, evt) {
            // process event
            if (!ct2ref.locked) {
                self.shiftPressed = evt.shiftKey;
                self.ctrlPressed = evt.ctrlKey;
                self.dblClicked = false;
                self.state = (!self.toggleSelection || self.state === self.ct2c.CB_UNCHK) ? self.ct2c.CB_CHK : self.ct2c.CB_UNCHK;
                self.updateState();
                self.fireRsChg();
            }
        };

        if (options?.doubleClickEvents ?? true) {
            rsEvents.onDoubleClick = function(x, y, evt) {
                // process event
                if (!ct2ref.locked) {
                    self.shiftPressed = evt.shiftKey;
                    self.ctrlPressed = evt.ctrlKey;
                    self.dblClicked = true;
                    self.state = self.ct2c.CB_CHK;
                    self.updateState();
                    self.fireRsChg();
                }
            };
        }

        self.rsMouseListener = new shmi.visuals.io.MouseListener(self.element, rsEvents);
        self.rsTouchListener = new shmi.visuals.io.TouchListener(self.element, rsEvents);

        self.updateState();
        self.setEventsEnabled(true);
    };
    shmi.visuals.controls.ComplexTable2.RowSelector.prototype = {
        setChangeCallback: function(listener) { // null: no event listening
            this.chgCallb = listener;
        },
        setEventsEnabled: function(enabled) {
            if (enabled) {
                this.rsMouseListener.enable();
                this.rsTouchListener.enable();
            } else {
                this.rsMouseListener.disable();
                this.rsTouchListener.disable();
            }
        },
        setState: function(state) {
            this.state = state;
            this.updateState();
        },
        getState: function() {
            return this.state;
        },
        /**
         * @private
         */
        updateState: function() {
            shmi.removeClass(this.element, this.css.ck);
            shmi.removeClass(this.element, this.css.unck);
            if (this.state === this.ct2c.CB_CHK) {
                shmi.addClass(this.element, this.css.ck);
            } else {
                shmi.addClass(this.element, this.css.unck);
            }
        },
        /**
         * @private
         */
        fireRsChg: function() {
            if (this.chgCallb) {
                this.chgCallb(this.id, this.state, this.shiftPressed, this.ctrlPressed, this.dblClicked);
            }
        }
    };

    /**
     * Creates a new ComplexTable2.FieldHeader control.
     *
     * @param {HTMLElement} element <div></div>
     * @param {number} id
     * @param {boolean} isSortable
     * @param {number} initialState shmi.visuals.controls.ComplexTable2.c.SORT_??
     * @param {string} caption (already localized !!)
     *
     * @constructor
     */
    shmi.visuals.controls.ComplexTable2.FieldHeader = function(element, id, isSortable, initialState, caption, ct2Inst, fldName, icon) {
        this.ct2c = shmi.visuals.controls.ComplexTable2.c; // short cut
        this.css = shmi.visuals.controls.ComplexTable2.FieldHeader.uiCssCl; // short cut
        this.ct2Inst = ct2Inst;
        this.element = element;
        this.id = id;
        this.isSortable = isSortable;
        this.state = initialState;
        this.caption = caption;
        this.chgCallb = null;
        this.cbMouseListener = null;
        this.cbTouchListener = null;
        this.fieldName = (fldName !== undefined) ? fldName : null;
        this.icon = (icon === undefined) ? null : icon;

        shmi.addClass(this.element, shmi.visuals.controls.ComplexTable2.uiCssCl.fldHeader);
        //element.textContent = this.caption;
        if (this.ct2Inst.headerMode === "ICON") {
            if (this.icon !== null) {
                shmi.addClass(element, this.icon);
            }
        } else {
            element.textContent = this.caption;
        }

        if (this.isSortable) {
            shmi.addClass(this.element, this.css.sortable);

            var cbEvents = {};
            cbEvents.onPress = function() {
                if (!this.ct2Inst.locked) {
                    shmi.addClass(this.element, this.css.sel);
                }
            }.bind(this);
            cbEvents.onRelease = function() {
                if (!this.ct2Inst.locked) {
                    shmi.removeClass(this.element, this.css.sel);
                }
            }.bind(this);
            cbEvents.onClick = function() {
                if (!this.ct2Inst.locked) {
                    // process event
                    switch (this.state) {
                    case this.ct2c.SORT_UP:
                        this.state = this.ct2c.SORT_DOWN;
                        break;
                    case this.ct2c.SORT_DOWN:
                        this.state = this.ct2c.NO_SORT;
                        break;
                    default:
                        this.state = this.ct2c.SORT_UP; // in all other cases, it's impossible to reach NO_SORT by event
                        break;
                    }
                    this.updateState();
                    this.fireSortChg();
                }
            }.bind(this);
            this.cbMouseListener = new shmi.visuals.io.MouseListener(this.element, cbEvents);
            this.cbTouchListener = new shmi.visuals.io.TouchListener(this.element, cbEvents);
        }

        this.updateState();
        this.setEventsEnabled(true);
    };
    shmi.visuals.controls.ComplexTable2.FieldHeader.prototype = {
        setChangeCallback: function(listener) { // null: no event listening
            this.chgCallb = listener;
        },
        setEventsEnabled: function(enabled) {
            if (this.cbMouseListener) {
                if (enabled) {
                    this.cbMouseListener.enable();
                    this.cbTouchListener.enable();
                } else {
                    this.cbMouseListener.disable();
                    this.cbTouchListener.disable();
                }
            }
        },
        setState: function(state) {
            this.state = state;
            this.updateState();
        },
        getState: function() {
            return this.state;
        },
        /**
         * @private
         */
        updateState: function() {
            if (this.isSortable) {
                shmi.removeClass(this.element, this.css.noSort);
                shmi.removeClass(this.element, this.css.sortUp);
                shmi.removeClass(this.element, this.css.sortDn);
                switch (this.state) {
                case this.ct2c.SORT_UP:
                    shmi.addClass(this.element, this.css.sortUp);
                    break;
                case this.ct2c.SORT_DOWN:
                    shmi.addClass(this.element, this.css.sortDn);
                    break;
                default: // NO_SORT
                    shmi.addClass(this.element, this.css.noSort);
                    break;
                }
            } else {
                shmi.addClass(this.element, this.css.noSort);
            }
        },
        /**
         * @private
         */
        fireSortChg: function() {
            if (this.chgCallb) {
                this.chgCallb(this.id, this.state);
            }
        }
    };

    shmi.visuals.controls.ComplexTable2.instIDs = [];

    shmi.visuals.controls.ComplexTable2.uiElName = {};
    (function() {
        /* The following ui element names are used in CT2's JS code. Change it here if necessary.
         All other ui element names may be changed in the CT2 template(s).
         */
        var __ct2sc = shmi.visuals.controls.ComplexTable2.uiElName;
        __ct2sc.tabContContainer = "ct2-table-content-container";
        __ct2sc.tabHeadContainer = "ct2-table-header-container";
        __ct2sc.tabContainer = "ct2-table-container";
        __ct2sc.tbodyContainer = "ct2-table-body-container";
        __ct2sc.thead = "ct2-table-header";
        __ct2sc.table = "ct2-table";
        __ct2sc.txtFiltersHead = "ct2-table-filters";
        __ct2sc.tbody = "ct2-table-body";
        __ct2sc.label = "ct2-label";
        __ct2sc.ctrlPanel = "ct2-ctrl-panel";
        __ct2sc.footer = "ct2-footer";

        shmi.visuals.controls.ComplexTable2.uiCssCl = {};
        __ct2sc = shmi.visuals.controls.ComplexTable2.uiCssCl;
        /* The following ui element class names are used in CT2's JS code. Change it here if necessary.
         All other ui element class names be changed in the CT2 template(s).
         */
        // (static) styles for dyn. created elements
        // !!! keep them short !!! because they are used in each table row / cell
        __ct2sc.colCb = "ct2-colcb";
        __ct2sc.colPfx = "ct2-col"; // prefix, the classes are ct2-col1, ct2-col2, ..., ct2-col<n>
        __ct2sc.thCont = "ct2-thcont";
        __ct2sc.tdCont = "ct2-tdcont";
        __ct2sc.selCb = "ct2-selcb";
        __ct2sc.evenRow = "ct2-evenRow";
        __ct2sc.fldHeader = "ct2-fhead";
        __ct2sc.colTxtFilterCont = "ct2-filter";
        __ct2sc.colTxtFilterLineCont = "line";
        __ct2sc.colTxtFilterIcon = "filter-icon";
        //__ct2sc.activeRowMarker = "ct2-active-row-marker";
        // dynamically switched styles
        __ct2sc.multiline = "multiline";
        __ct2sc.multiselect = "ct2-multisel";
        __ct2sc.sbNotUsed = "ct2-sb-not-used";
        __ct2sc.sbMinimized = "ct2-sb-minimized";
        __ct2sc.sbNormal = "ct2-sb-normal";
        __ct2sc.rowDisplVisible = "ct2-row-display-visible";
        __ct2sc.rowDisplInvisible = "ct2-row-display-invisible";
        __ct2sc.emptyTab = "ct2-empty-table";
        __ct2sc.ctrlPanelMin = "ct2-ctrl-panel-minimized";
        __ct2sc.ctrlPanelNormal = "ct2-ctrl-panel-normal";
        __ct2sc.activeRow = "ct2-active-row";
        __ct2sc.touchDev = "touch";
        __ct2sc.noTouchDev = "notouch";

        shmi.visuals.controls.ComplexTable2.c = {};
        __ct2sc = shmi.visuals.controls.ComplexTable2.c;
        //--
        __ct2sc.MIN_BUF_SIZE = 150;
        __ct2sc.CONST_LINE_HEIGHT = false;
        __ct2sc.DEBOUNCE_TO_RESIZE = 150;
        __ct2sc.DEFAULT_LINE_HEIGHT = 20;
        __ct2sc.MULTI_SELECT_MODE = "3-state-clear"; // "3-state", "2-state" (NYI)
        __ct2sc.SCROLL_EVT_TYPE_NONE = -1;
        __ct2sc.SCROLL_EVT_TYPE_TOUCH = 11;
        __ct2sc.SCROLL_EVT_TYPE_RELEASE = 12;
        __ct2sc.SCROLL_EVT_TYPE_SB_BTN_UP = 101;
        __ct2sc.SCROLL_EVT_TYPE_SB_BTN_DOWN = 102;
        __ct2sc.SCROLL_EVT_TYPE_SB_SLIDER_CHG = 103;
        __ct2sc.SCROLL_EVT_TYPE_SB_LARGE_CHG = 104;
        __ct2sc.SCROLL_EVT_TYPE_V_DRAG_UP = 201;
        __ct2sc.SCROLL_EVT_TYPE_V_DRAG_DOWN = 202;
        __ct2sc.SCROLL_EVT_TYPE_V_SWIPE_UP = 301;
        __ct2sc.SCROLL_EVT_TYPE_V_SWIPE_DOWN = 302;
        __ct2sc.SCROLL_EVT_TYPE_H_SWIPE_LEFT = 401;
        __ct2sc.SCROLL_EVT_TYPE_H_SWIPE_RIGHT = 402;
        __ct2sc.SCROLL_EVT_TYPE_RESIZE = 501;
        __ct2sc.SCROLL_EVT_TYPE_RESIZE_LAYOUT_CHG = 502;
        //--
        __ct2sc.SELECT_TYPE_NOTHING = -1;
        __ct2sc.SELECT_TYPE_ALL = 1;
        __ct2sc.SELECT_TYPE_SEL_ROWS = 2;
        //--
        __ct2sc.CB_UNCHK = -1;
        __ct2sc.CB_CHK = 1;
        __ct2sc.CB_UNDEF = 2;
        __ct2sc.NO_SORT = -1;
        __ct2sc.SORT_UP = 1;
        __ct2sc.SORT_DOWN = 2;
        //--

        shmi.visuals.controls.ComplexTable2.CheckBox.uiCssCl = {};
        __ct2sc = shmi.visuals.controls.ComplexTable2.CheckBox.uiCssCl;
        // dyn. created checkbox classes
        // !!! keep them short !!! because they are used in each table row
        __ct2sc.sel = "ct2-cb-sel";
        __ct2sc.ck = "ct2-cb-ck";
        __ct2sc.unck = "ct2-cb-unck";
        __ct2sc.undef = "ct2-cb-undef";

        shmi.visuals.controls.ComplexTable2.RowSelector.uiCssCl = {};
        __ct2sc = shmi.visuals.controls.ComplexTable2.RowSelector.uiCssCl;
        // dyn. created row selector classes
        // !!! keep them short !!! because they are used in each table row
        __ct2sc.sel = "ct2-rs-sel";
        __ct2sc.ck = "ct2-rs-ck";
        __ct2sc.unck = "ct2-rs-unck";

        shmi.visuals.controls.ComplexTable2.FieldHeader.uiCssCl = {};
        __ct2sc = shmi.visuals.controls.ComplexTable2.FieldHeader.uiCssCl;
        // dyn. created (sortable) header field classes
        __ct2sc.sel = "ct2-fh-sel";
        __ct2sc.sortUp = "ct2-fh-sort-up";
        __ct2sc.sortDn = "ct2-fh-sort-down";
        __ct2sc.noSort = "ct2-fh-no-sort";
        __ct2sc.sortable = "ct2-fh-sortable";
    })();
}());

(function() {
    'use strict';

    const modifiers = {
        MODE: {
            FLOAT: null,
            STACKED: "stacked-layout",
            INLINE: "inline-layout",
            FLEX: "flex-layout",
            IQ_FLEX: "iq-flex-layout"
        },
        ALIGN: {
            HORIZONTAL: {
                LEFT: "left-aligned",
                CENTER: "centered",
                RIGHT: "right-aligned"
            },
            VERTICAL: {
                TOP: "top",
                MIDDLE: "middle",
                BOTTOM: "bottom"
            },
            PRIMARY: {
                START: "flex-primary-align-start",
                CENTER: "flex-primary-align-center",
                END: "flex-primary-align-end",
                SPACE_BETWEEN: "flex-primary-align-space-between",
                SPACE_AROUND: "flex-primary-align-space-around",
                SPACE_EVENLY: "flex-primary-align-space-evenly"
            },
            SECONDARY: {
                START: "flex-secondary-align-start",
                CENTER: "flex-secondary-align-center",
                END: "flex-secondary-align-end",
                STRETCH: "flex-secondary-align-stretch"
            },
            LINE: {
                START: "flex-line-align-start",
                CENTER: "flex-line-align-center",
                END: "flex-line-align-end",
                STRETCH: "flex-line-align-stretch",
                SPACE_BETWEEN: "flex-line-align-space-between",
                SPACE_AROUND: "flex-line-align-space-around"
            }
        },
        AUTO: {
            WIDTH: "auto-width",
            MARGIN: "auto-margin"
        },
        FLEX: {
            DISTRIBUTE: "distribute-evenly",
            WRAP: "flex-wrap",
            ALL: "flex-all",
            COLUMN: "column-orientation",
            NONE: "flex-none"
        }
    };

    /**
     * Maps config options to modifiers.
     */
    const modifierMap = {
        "auto-width": [
            { value: true, modifier: modifiers.AUTO.WIDTH }
        ],
        "auto-margin": [
            { value: true, modifier: modifiers.AUTO.MARGIN }
        ],
        "h-alignment": [
            { value: "left", modifier: modifiers.ALIGN.HORIZONTAL.LEFT },
            { value: "center", modifier: modifiers.ALIGN.HORIZONTAL.CENTER },
            { value: "right", modifier: modifiers.ALIGN.HORIZONTAL.RIGHT }
        ],
        "v-alignment": [
            { value: "top", modifier: modifiers.ALIGN.VERTICAL.TOP },
            { value: "middle", modifier: modifiers.ALIGN.VERTICAL.MIDDLE },
            { value: "bottom", modifier: modifiers.ALIGN.VERTICAL.BOTTOM }
        ],
        "flex-orientation": [
            { value: "column", modifier: modifiers.FLEX.COLUMN }
        ],
        "flex-distribute": [
            { value: true, modifier: modifiers.FLEX.DISTRIBUTE }
        ],
        "flex-all": [
            { value: true, modifier: modifiers.FLEX.ALL }
        ],
        "flex-none": [
            { value: true, modifier: modifiers.FLEX.NONE }
        ],
        "flex-wrap": [
            { value: true, modifier: modifiers.FLEX.WRAP }
        ],
        "flex-primary-align": [
            { value: "start", modifier: modifiers.ALIGN.PRIMARY.START },
            { value: "center", modifier: modifiers.ALIGN.PRIMARY.CENTER },
            { value: "end", modifier: modifiers.ALIGN.PRIMARY.END },
            { value: "space-between", modifier: modifiers.ALIGN.PRIMARY.SPACE_BETWEEN },
            { value: "space-around", modifier: modifiers.ALIGN.PRIMARY.SPACE_AROUND },
            { value: "space-evenly", modifier: modifiers.ALIGN.PRIMARY.SPACE_EVENLY }
        ],
        "flex-secondary-align": [
            { value: "start", modifier: modifiers.ALIGN.SECONDARY.START },
            { value: "center", modifier: modifiers.ALIGN.SECONDARY.CENTER },
            { value: "end", modifier: modifiers.ALIGN.SECONDARY.END },
            { value: "stretch", modifier: modifiers.ALIGN.SECONDARY.STRETCH }
        ],
        "flex-line-align": [
            { value: "start", modifier: modifiers.ALIGN.LINE.START },
            { value: "center", modifier: modifiers.ALIGN.LINE.CENTER },
            { value: "end", modifier: modifiers.ALIGN.LINE.END },
            { value: "stretch", modifier: modifiers.ALIGN.LINE.STRETCH },
            { value: "space-between", modifier: modifiers.ALIGN.LINE.SPACE_BETWEEN },
            { value: "space-around", modifier: modifiers.ALIGN.LINE.SPACE_AROUND }
        ]
    };

    const containerVariants = {
        float: {
            modifierSettings: ["auto-width", "auto-margin", "h-alignment"]
        },
        stacked: {
            baseModifier: modifiers.MODE.STACKED,
            modifierSettings: ["auto-margin"]
        },
        inline: {
            baseModifier: modifiers.MODE.INLINE,
            modifierSettings: ["auto-width", "auto-margin", "h-alignment", "v-alignment"]
        },
        flex: {
            baseModifier: modifiers.MODE.FLEX,
            modifierSettings: ["auto-width", "auto-margin", "flex-orientation", "h-alignment", "v-alignment", "flex-distribute", "flex-all", "flex-none"]
        },
        iqflex: {
            baseModifier: modifiers.MODE.IQ_FLEX,
            modifierSettings: ["flex-orientation", "flex-wrap", "flex-primary-align", "flex-secondary-align", "flex-line-align", "flex-none"]
        }
    };

    function clearAllModifiers(element) {
        const allModifiers = [
            ...Object.values(modifiers.MODE),
            ...Object.values(modifiers.ALIGN.HORIZONTAL),
            ...Object.values(modifiers.ALIGN.VERTICAL),
            ...Object.values(modifiers.AUTO),
            ...Object.values(modifiers.FLEX)
        ].filter((val) => val !== null);

        allModifiers.forEach((val) => shmi.removeClass(element, val));
    }

    function applyModifiers(element, config) {
        const variantSettings = containerVariants[config.type];
        if (!variantSettings) {
            console.error("[Container]", "unknown layout type:", config.type);
            return;
        }

        // Add base modifier if required
        if (variantSettings.baseModifier) {
            shmi.addClass(element, variantSettings.baseModifier);
        }

        // Add conditional modifiers
        variantSettings.modifierSettings.forEach((modifierKey) => {
            const modifierConditions = modifierMap[modifierKey];
            if (!modifierConditions || !Array.isArray(modifierConditions)) {
                console.error("[Container]", "missing modifier info:", modifierKey);
                return;
            }

            // Get modifier for the config value.
            const modifierInfo = modifierConditions.find(({ value }) => config[modifierKey] === value);
            if (modifierInfo) {
                shmi.addClass(element, modifierInfo.modifier);
            }
        });
    }

    function getMarginCompensator(self) {
        let marginCompensator = self.element.querySelector("div.margin-compensator"),
            oldHtml = null;

        if (!marginCompensator || (marginCompensator.parentNode !== self.element)) {
            marginCompensator = document.createElement("DIV");
            shmi.addClass(marginCompensator, "margin-compensator");
            oldHtml = self.element.innerHTML;
            self.element.innerHTML = "";
            marginCompensator.innerHTML = oldHtml;
            self.element.appendChild(marginCompensator);
        }

        return marginCompensator;
    }

    shmi.visuals.controls.Container = function(element, config) {
        var self = this;
        self.element = element;
        self.config = config || {};

        self.parseAttributes();

        shmi.def(self.config, 'class-name', "container");
        shmi.def(self.config, 'name', null);
        shmi.def(self.config, 'type', "stacked"); //"stacked", "float", "inline", "flex"
        shmi.def(self.config, 'auto-width', false);
        shmi.def(self.config, 'auto-margin', false);
        shmi.def(self.config, 'h-alignment', "left"); //"left", "center", "right"
        shmi.def(self.config, 'v-alignment', "top"); //"top", "middle", "bottom"
        shmi.def(self.config, 'flex-orientation', "row"); //"row", "column"
        shmi.def(self.config, 'flex-distribute', false);
        shmi.def(self.config, 'flex-all', false);
        shmi.def(self.config, 'flex-none', false);
        shmi.def(self.config, 'flex-wrap', false);
        shmi.def(self.config, 'flex-primary-align', "start");
        shmi.def(self.config, 'flex-secondary-align', "stretch");
        shmi.def(self.config, 'flex-line-align', "start");

        self.controls = [];
        self.marginCompensator = null;

        self.startup();
    };

    shmi.visuals.controls.Container.prototype = {
        uiType: "container",
        isContainer: true,
        getClassName: function() {
            return "Container";
        },
        /**
         * Parses child controls when control is registered in layout
         *
         * @param {function} onDone function to call on completion
         * @returns {undefined}
         */
        onRegister: function(onDone) {
            var self = this;

            //clear active css-modifiers, in case of reconfiguration
            clearAllModifiers(self.element);
            applyModifiers(self.element, self.config);
            self.marginCompensator = getMarginCompensator(self);
            self.parseChildren(self.marginCompensator, onDone);
        },
        onEnable: function() {
            const self = this;
            self.controls.forEach((c) => c.enable());
        },
        onDisable: function() {
            const self = this;
            self.controls.forEach((c) => c.disable());
        },
        onLock: function() {

        },
        onUnlock: function() {

        },
        onAddControl: function(options, callback) {
            const self = this,
                cm = shmi.requires("visuals.tools.controller"),
                controls = [];

            let initToken = null;

            if (!self.initialized) {
                throw new Error("Control not initialized yet.");
            }

            if (!Array.isArray(options)) {
                options = [options];
            }
            options.forEach(function(option) {
                const control = shmi.createControl(option.ui, self.marginCompensator, option.config, "DIV", null, false);
                if (control !== null) {
                    if (option.style && typeof option.style === "object") {
                        Object.assign(control.element.style, option.style);
                    }
                    controls.push(control);
                }
            });
            initToken = shmi.waitOnInit(controls, function() {
                if (initToken) {
                    const idx = self._init_.tokens.indexOf(initToken);
                    if (idx !== -1) {
                        self._init_.tokens.splice(idx, 1);
                    }
                    initToken.unlisten();
                    initToken = null;
                }

                const promises = options.map(function(opt, jdx) {
                    const control = controls[jdx];

                    if (opt.controller) {
                        cm.create(opt.controller.name, opt.controller, control.getName());
                    }

                    if (self.isActive()) {
                        control.enable();
                    }

                    if (!Array.isArray(opt.children) || !opt.children.length) {
                        return Promise.resolve();
                    }

                    return new Promise((resolve) => {
                        control.addControl(opt.children, (childErr) => resolve(childErr));
                    });
                });

                Promise.all(promises).then((result) => {
                    const errors = result.filter((r) => !!r);
                    callback(errors.length ? errors[0] : null, controls);
                });
            });
            if (initToken) {
                self._init_.tokens.push(initToken);
            }
        }
    };

    shmi.extend(shmi.visuals.controls.Container, shmi.visuals.core.BaseControl);
}());

/**
 * datetime-display control.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "datetime-display",
 *     "name": null,
 *     "item": null,
 *     "display-utc": false,
 *     "display-format": "$YYYY-$MM-$DD $HH:$mm:$ss",
 *     "input-format": "$X",
 *     "invalid-text": "---"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * item {string}: Name of the item from which to get the timestamp to display
 * display-utc {boolean}: Whether or not the displayed time should be UTC or not
 * display-format {string}: Format string for the displayed date
 * input-format {string}: Format string to use when parsing the items value
 * invalid-text {string}: String to display if the item does not contain a valid
 *  timestamp.
 *
 * @version 1.0
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "datetime-display", // control name in camel-case
        uiType = "datetime-display", // control keyword (data-ui)
        isContainer = false;

    // example - default configuration
    var defConfig = {
        "class-name": "text2 datetime-display",
        "name": null,
        "item": null,
        "display-utc": false,
        "display-format": "$YYYY-$MM-$DD $HH:$mm:$ss",
        "input-format": "$X",
        "invalid-text": "---"
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    // declare private functions - START

    /**
     * Sets the displayed text and tooltip of the control.
     *
     * @param {*} self Reference to the control.
     * @param {string} str Text to display.
     */
    function setText(self, str) {
        self.element.textContent = str;
    }

    // declare private functions - END

    // definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            subscriptionListener: null,
            localizedDisplayFormat: null,
            localizedInvalidText: null,
            rafRunning: false,
            lastValue: null
        },
        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            dtUtils: "visuals.tools.date"
        },

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                this.vars.localizedDisplayFormat = shmi.localize(this.config["display-format"]);
                this.vars.localizedInvalidText = shmi.localize(this.config["invalid-text"]);
                setText(this, this.vars.localizedInvalidText);
            },
            /* called when control is enabled */
            onEnable: function() {
                if (this.config["item"] && !this.vars.subscriptionListener) {
                    this.vars.subscriptionListener = this.imports.im.subscribeItem(this.config["item"], this);
                }
            },
            /* called when control is disabled */
            onDisable: function() {
                if (this.vars.subscriptionListener) {
                    this.vars.subscriptionListener.unlisten();
                }
            },
            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type) {
                if (type === shmi.c("TYPE_INT") || type === shmi.c("TYPE_FLOAT")) {
                    this.vars.lastValue = this.imports.dtUtils.parseDateTime(String(value), this.config["input-format"] || "$X");
                } else if (!this.config["input-format"]) {
                    this.vars.lastValue = null;
                } else {
                    this.vars.lastValue = this.imports.dtUtils.parseDateTime(String(value), this.config["input-format"]);
                }

                // Only attempt to update the DOM during an animation frame.
                // Prevents floods of redraws and may perform better anyways.
                if (!this.vars.rafRunning) {
                    this.vars.rafRunning = true;
                    shmi.raf(function onDraw() {
                        if (!this.vars.lastValue) {
                            setText(this, this.vars.localizedInvalidText);
                        } else {
                            setText(this, this.imports.dtUtils.formatDateTime(this.vars.lastValue, {
                                datestring: this.vars.localizedDisplayFormat,
                                utc: this.config["display-utc"]
                            }));
                        }

                        this.vars.rafRunning = false;
                    }.bind(this));
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

shmi.pkg("visuals.controls");
/**
 * DialogBox control to display content in an overlay window.
 *
 * @constructor
 * @extends shmi.visuals.core.BaseControl
 * @param element - base element of the DialogBox
 * @param config - configuration of the DialogBox
 */
shmi.visuals.controls.DialogBox = function(element, config) {
    /* check for required packages */
    shmi.requires("visuals.gfx.Movable");
    shmi.requires("visuals.io.MouseListener");
    shmi.requires("visuals.io.TouchListener");
    this.element = element;

    /* control configuration */
    this.config = config || {};

    /* parse option attributes from html element */
    this.parseAttributes();
    /* set default options */
    shmi.def(this.config, 'class-name', 'dialog-box');
    shmi.def(this.config, 'template', 'default/dialog-box');
    shmi.def(this.config, 'name', null);
    shmi.def(this.config, 'class-name', 'dialog-box');
    shmi.def(this.config, 'initial-state', 'hidden');
    shmi.def(this.config, 'collapsible', false);
    shmi.def(this.config, 'draggable', false);
    shmi.def(this.config, 'top-level', false);
    shmi.def(this.config, 'content-template', null);
    shmi.def(this.config, 'tab-limit', true);

    /* internal variables */
    this.collapsed = false;
    this.hidden = true;
    this.active = false;
    this.controls = [];
    this.childrenParsed = false;
    this.bgDiv = null;
    this.dragging = false;
    this.mouselistener = null;
    this.touchlistener = null;
    this.contentElement = null;
    this.originalParent = null;
    this.tokens = [];

    this.element.style.visibility = "hidden";

    /* control startup */
    this.startup();
};

shmi.visuals.controls.DialogBox.prototype = {
    uiType: "dialog-box",
    isContainer: true,
    getClassName: function() {
        return "DialogBox";
    },
    events: ["open", "close"],
    /**
     * Initializes the DialogBox
     *
     * @override BaseControl.init
     */
    onTemplate: function(response, failed, onDone) {
        var self = this;
        if (!failed) {
            self.element.innerHTML = response;
            self.contentElement = shmi.getUiElement("dialog-box-content", self.element);
            if (self.contentElement && self.config["content-template"]) {
                var templateUrl = (self.config["content-template"].indexOf(shmi.c("RES_URL_PREFIX")) === 0) ?
                    self.config["content-template"] :
                    shmi.c("TEMPLATE_PATH") + self.config["content-template"] + shmi.c("TEMPLATE_EXT");
                shmi.loadResource(templateUrl, function(contentResponse, contentFailed) {
                    if (!contentFailed) {
                        self.contentElement.innerHTML = contentResponse;
                    }
                    onDone();
                });
            } else {
                onDone();
            }
        } else {
            onDone();
        }
    },
    onRegister: function(onDone) {
        var self = this;

        if (self.contentElement) {
            self.parseChildren(self.contentElement, onDone);
        } else {
            onDone();
        }
    },
    onInit: function() {
        var self = this,
            inl = shmi.requires("visuals.tools.iterate.iterateNodeList"),
            closeButtons = [];

        if (self.contentElement) {
            closeButtons = shmi.getUiElements("button", self.element);
            inl(closeButtons, function(el) {
                var but = null;
                if (el.parentNode && !shmi.testParentChild(self.contentElement, el)) {
                    but = shmi.createControl("button", el, {}, "DIV", "from");
                    but.onClick = function() {
                        self.hide();
                    };
                }
            });
        }

        this.titleElement = shmi.getUiElement('dialog-box-title', this.element);
        if (!this.titleElement) {
            shmi.log('[DialogBox] no dialog-box-title element provided', 1);
        }
        this.frameElement = shmi.getUiElement('dialog-box-frame', this.element);
        if (!this.frameElement) {
            shmi.log('[DialogBox] no dialog-box-frame element provided', 3);
            return;
        }

        /* all required elements found */
        this.hidden = this.config['initial-state'] !== 'visible';
        if (this.hidden) {
            shmi.addClass(this.element, 'hidden');
        }
        this.element.style.visibility = "";

        this.containerElement = shmi.getUiElement('dialog-content', this.element);
        if (!this.containerElement) {
            shmi.log('[DialogBox] no dialog-content element provided', 3);
            return;
        }

        this.element.setAttribute('tabindex', '-1');
        this.mover = new shmi.visuals.gfx.Movable(this.containerElement);
        this.mover.priority = true;
        var functions = {};
        functions.onClick = function(x, y) {
            if (this.config['close-on-click']) {
                this.hide();
            }
        }.bind(this);
        if (this.config.draggable === true) {
            functions.onDrag = function(dx, dy, event) {
                event.preventDefault();
                this.mover.translate(dx, dy);
                for (var i = 0; i < this.controls.length; i++) {
                    try {
                        this.controls[i].offsetX = this.mover.startX + this.mover.tx;
                    } catch (exc) {
                        shmi.log("[Dialog-Box] - Can not set control offset", 2);
                    }
                }
                if (!this.dragging) {
                    this.dragging = true;
                }
            }.bind(this);
        }
        functions.onRotate = function(rot) {
            if (this.config['rotation-enabled']) {
                this.mover.rotate(rot * 57.3);
            }
        }.bind(this);
        functions.onScale = function(s) {
            if (this.config['scaling-enabled']) {
                this.mover.scale(s);
            }
        }.bind(this);
        if (this.titleElement) {
            if (this.config.title) {
                this.titleElement.textContent = shmi.localize(this.config.title);
            }
        }

        functions.onPress = function(x, y, event) {
            //event.preventDefault();
            shmi.addClass(this.element, 'pressed');
            if (document.activeElement !== this.element) {
                this.element.focus();
            }
            shmi.log("[DialogBox] pressed", 1);
        }.bind(this);
        functions.onRelease = function() {
            this.dragging = false;
            shmi.removeClass(this.element, 'pressed');
            shmi.log("[DialogBox] released", 1);
        }.bind(this);
        this.mouselistener = new shmi.visuals.io.MouseListener(this.containerElement, functions);
        this.touchlistener = new shmi.visuals.io.TouchListener(this.containerElement, functions);
    },
    /**
     * Displays the DialogBox
     *
     */
    show: function() {
        var self = this;
        if (!self.active) {
            shmi.log("[DialogBox] cannot show dialog-box without enabling it first", 3);
            return;
        }
        if (!self.hidden) {
            return;
        }
        self.hidden = false;

        if (self.config['top-level'] === true) {
            self.originalParent = self.element.parentNode;
            self.element = document.body.appendChild(self.element);
        }

        if (self.config['cover-background']) {
            if (self.bgDiv) {
                shmi.removeClass(self.bgDiv, 'hidden');
            } else {
                self.bgDiv = document.createElement('div');
                shmi.addClass(self.bgDiv, 'dialog-box-overlay');
                self.bgDiv.setAttribute('tabindex', '-1');
                document.body.appendChild(self.bgDiv);
                shmi.removeClass(self.bgDiv, 'hidden');
            }
        }

        shmi.removeClass(self.element, 'hidden');

        if (self.config['tab-limit']) {
            // Enable element tabulator
            var tabulator = shmi.requires("visuals.tools.tabulator");
            self.tabLimit = tabulator.setTabParent(self.element);
        }

        self.enableControls();
        self.fire('open', {});
    },
    /**
     * Removes the DialogBox from DOM
     *
     */
    hide: function() {
        var self = this;
        if (this.hidden) {
            return;
        }
        this.hidden = true;
        if (!this.config['leave-controls-enabled']) {
            this.disableControls();
        }
        if (self.config['top-level'] === true) {
            self.element = self.originalParent.appendChild(self.element);
        }
        shmi.addClass(this.element, 'hidden');
        if (self.tabLimit) {
            self.tabLimit.unsetTabParent();
            self.tabLimit = null;
        }
        this.reset();
        if (this.config['cover-background']) {
            if (this.bgDiv) {
                shmi.addClass(this.bgDiv, 'hidden');
            }
        }
        this.fire('close', {});
    },
    reset: function() {
        if (this.active && this.mover) {
            this.mover.tx = 0;
            this.mover.ty = 0;
            this.mover.rot = 0;
            this.mover.s = 1.0;
            this.mover.sx = 1.0;
            this.mover.sy = 1.0;
            this.mover.update();
        }
    },
    /**
     * Enables the DialogBox
     *
     */
    onEnable: function() {
        var self = this;

        self.mouselistener.enable();
        self.touchlistener.enable();
        self.tokens.push(shmi.listen("enable", function(evt) {
            if ((evt.source.getName().indexOf(self.getName()) !== -1) && (evt.source.getName() !== self.getName())) {
                if (evt.source.element.getAttribute("data-function") === "close") {
                    evt.source.onClick = function onClick() {
                        self.hide();
                    };
                }
            }
        }, { "source.uiType": "button" }));

        if (!self.hidden) {
            /* set hidden status to prevent show() to return early */
            self.hidden = true;
            self.show();
        }
    },
    /**
     * Disables the DialogBox
     *
     */
    onDisable: function() {
        var self = this;
        self.hide();
        self.mouselistener.disable();
        self.touchlistener.disable();

        // remove overlay background element if present
        if (this.bgDiv) {
            this.bgDiv.parentNode.removeChild(this.bgDiv);
            this.bgDiv = null;
        }

        self.tokens.forEach(function(t) {
            t.unlisten();
        });
        self.tokens = [];
    },
    /**
     * Enables child controls of the DialogBox
     *
     */
    enableControls: function() {
        for (var i = 0; i < this.controls.length; i++) {
            try {
                this.controls[i].enable();
            } catch (exc) {
                console.error("[DialogBox]", "error enabling control:", this.controls[i], this.getName(), "Exception:", exc);
            }
        }
    },
    /**
     * Disables child controls of the DialogBox
     *
     */
    disableControls: function() {
        for (var i = 0; i < this.controls.length; i++) {
            try {
                this.controls[i].disable();
            } catch (exc) {
                console.error("[DialogBox]", "error disabling control:", this.controls[i], this.getName(), "Exception:", exc);
            }
        }
    },
    onAddControl: function(options, callback) {
        var self = this,
            cm = shmi.requires("visuals.tools.controller"),
            controls = [],
            classLabel = "[" + self.getClassName() + "] ",
            initToken = null;

        if (!self.initialized) {
            throw new Error(classLabel + "Control not initialized yet.");
        }

        if (!Array.isArray(options)) {
            options = [options];
        }

        if (!self.contentElement) {
            callback(new Error(classLabel + "Content element missing from template."), controls);
            return;
        }

        options.forEach(function(option) {
            var control = shmi.createControl(option.ui, self.contentElement, option.config, "DIV", null, false);
            if (control !== null) {
                if (option.style && typeof option.style === "object") {
                    let iter = shmi.requires("visuals.tools.iterate.iterateObject");

                    iter(option.style, (value, name) => {
                        control.element.style[name] = value;
                    });
                }
                controls.push(control);
            }
        });

        initToken = shmi.waitOnInit(controls, function() {
            var idx = -1,
                errors = [],
                tm = shmi.requires("visuals.task"),
                tasks = [],
                tl = null;

            if (initToken) {
                idx = self._init_.tokens.indexOf(initToken);
                if (idx !== -1) {
                    self._init_.tokens.splice(idx, 1);
                }
                initToken.unlisten();
                initToken = null;
            }

            options.forEach(function(opt, jdx) {
                var t = null,
                    control = controls[jdx];

                if (opt.controller) {
                    cm.create(opt.controller.name, opt.controller, control.getName());
                }
                if (self.isActive() && !self.hidden) {
                    control.enable();
                }
                if (Array.isArray(opt.children) && opt.children.length) {
                    t = tm.createTask("child control");
                    t.run = function() {
                        control.addControl(opt.children, function(childErr, childOptions) {
                            if (childErr) {
                                errors.push(childErr);
                            }
                            t.complete();
                        });
                    };
                    tasks.push(t);
                }
            });

            if (tasks.length) {
                tl = tm.createTaskList(tasks, false);
                tl.onComplete = function() {
                    callback(errors.length ? errors[0] : null, controls);
                };
                tl.run();
            } else {
                callback(errors.length ? errors[0] : null, controls);
            }
        });
        if (initToken) {
            self._init_.tokens.push(initToken);
        }
    }
};

shmi.extend(shmi.visuals.controls.DialogBox, shmi.visuals.core.BaseControl);

/**
 * duration-display control.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "duration-display",
 *     "name": null,
 *     "target-ts-item": null,
 *     "target-ts-format": "$X",
 *     "current-ts-item": "Systemzeit",
 *     "current-ts-format": "$X",
 *     "display-preset": "compact",
 *     "invalid-text": "---"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * item {string}: Name of the item from which to get the timestamp to display
 * invalid-text {string}: String to display if the item does not contain a valid
 *  timestamp.
 *
 * @version 1.0
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "duration-display", // control name in camel-case
        uiType = "duration-display", // control keyword (data-ui)
        isContainer = false;

    // example - default configuration
    var defConfig = {
        "class-name": "text2 duration-display",
        "name": null,
        "target-ts-item": null,
        "target-ts-format": "$X",
        "current-ts-item": "Systemzeit",
        "current-ts-format": "$X",
        "display-preset": "compact",
        "invalid-text": "---"
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    // declare private functions - START

    /**
     * Sets the displayed text and tooltip of the control.
     *
     * @param {*} self Reference to the control.
     * @param {string} str Text to display.
     */
    function setText(self, str) {
        self.element.textContent = str;
    }

    function doUpdate(self) {
        if (self.vars.tsTarget === null || self.vars.tsCurrent === null) {
            self.vars.lastValue = null;
        } else {
            self.vars.lastValue = self.vars.tsTarget - self.vars.tsCurrent;
        }

        if (!self.vars.rafRunning) {
            self.vars.rafRunning = true;
            shmi.raf(function onDraw() {
                if (!self.vars.lastValue) {
                    setText(self, self.vars.localizedInvalidText);
                } else {
                    setText(self, self.imports.dtUtils.formatDuration(self.vars.lastValue, self.config["display-preset"] || "compact"));
                }

                self.vars.rafRunning = false;
            });
        }
    }

    function makeItemHandler(self, configFormatName, destVarName) {
        return {
            setValue: function setValue(value, type) {
                if (type === shmi.c("TYPE_INT") || type === shmi.c("TYPE_FLOAT")) {
                    self.vars[destVarName] = self.imports.dtUtils.parseDateTime(String(value), self.config[configFormatName] || "$X");
                } else if (!self.config[configFormatName]) {
                    self.vars[destVarName] = null;
                } else {
                    self.vars[destVarName] = self.imports.dtUtils.parseDateTime(String(value), self.config[configFormatName]);
                }

                doUpdate(self);
            }
        };
    }

    // declare private functions - END

    // definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            subscriptions: [],
            tsTarget: null,
            tsCurrent: null,
            localizedInvalidText: null,
            rafRunning: false,
            lastValue: null
        },
        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            dtUtils: "visuals.tools.date"
        },

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                this.vars.localizedInvalidText = shmi.localize(this.config["invalid-text"]);
                setText(this, this.vars.localizedInvalidText);
            },
            /* called when control is enabled */
            onEnable: function() {
                if (this.config["target-ts-item"] && this.config["current-ts-item"] && this.vars.subscriptions.length === 0) {
                    this.vars.subscriptions.push(this.imports.im.subscribeItem(this.config["target-ts-item"], makeItemHandler(this, "target-ts-format", "tsTarget")));
                    this.vars.subscriptions.push(this.imports.im.subscribeItem(this.config["current-ts-item"], makeItemHandler(this, "current-ts-format", "tsCurrent")));
                }
            },
            /* called when control is disabled */
            onDisable: function() {
                this.vars.subscriptions.forEach(function(sub) {
                    sub.unlisten();
                });
                this.vars.subscriptions = [];
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

shmi.pkg("visuals.controls");
/**
 * Creates a new FlipSwitch
 *
 * @constructor
 * @extends shmi.visuals.core.BaseControl
 * @param element - the base element of the control
 * @param config - configuration of the control
 */
shmi.visuals.controls.FlipSwitch = function(element, config) {
    /* check for required packages */
    shmi.requires("visuals.io.MouseListener");
    shmi.requires("visuals.io.TouchListener");
    shmi.requires("visuals.gfx.Movable");

    this.element = element;
    this.config = config || {};

    this.parseAttributes();

    shmi.def(this.config, 'class-name', 'flip-switch');
    shmi.def(this.config, 'template', 'default/flip-switch');
    shmi.def(this.config, 'name', null);
    shmi.def(this.config, 'label', "flip-switch");
    shmi.def(this.config, 'on-label', "I");
    shmi.def(this.config, 'off-label', "O");
    shmi.def(this.config, 'on-value', 1);
    shmi.def(this.config, 'off-value', 0);
    shmi.def(this.config, 'transitionStyle', 'all .16s linear');
    shmi.def(this.config, 'confirm-off-text', '${V_CONFIRM_OFF}');
    shmi.def(this.config, 'confirm-on-text', '${V_CONFIRM_ON}');
    shmi.def(this.config, 'confirm-on', false);
    shmi.def(this.config, 'confirm-off', false);
    shmi.def(this.config, 'icon-src', null);
    shmi.def(this.config, 'icon-title', null);
    shmi.def(this.config, 'icon-class', null);
    shmi.def(this.config, 'auto-label', true);

    this.config['show-icon'] = shmi.toBoolean(this.config['show-icon']);

    this.value = 0;
    this.active = false;
    this.initialized = false;
    this._subscriptionTargetId = null;
    this.movable = null;
    this._dragging = false;

    this.startup();
};

shmi.visuals.controls.FlipSwitch.prototype = {
    uiType: "flip-switch",
    getClassName: function() {
        return "FlipSwitch";
    },
    events: ["change"],
    tooltipProperties: ["icon-title"],
    onInit: function() {
        var self = this;

        var icon = shmi.getUiElement('flip-switch-icon', self.element);
        if (!icon) {
            shmi.log('[Flip Switch] no button-icon element provided', 1);
        } else if (self.config['icon-src']) {
            try {
                icon.setAttribute('src', self.config['icon-src']);
            } catch (exc) {
                shmi.log("[Flip Switch] Exception setting icon-src: " + exc, 2);
            }
        } else if (self.config['icon-class']) {
            if (icon.tagName === "IMG") {
                /* switch img element for div when icon-class and no icon-src is configured */
                var icnDiv = document.createElement('div');
                shmi.addClass(icnDiv, "flip-switch-icon");
                icon.parentNode.insertBefore(icnDiv, icon);
                icon.parentNode.removeChild(icon);
                icon = icnDiv;
            }
            var icon_class = self.config['icon-class'].trim().split(" ");
            icon_class.forEach(function(cls) {
                shmi.addClass(icon, cls);
            });
        }

        this.handleElement = shmi.getUiElement('flip-switch-handle', this.element);
        if (!this.handleElement) {
            shmi.log('[FlipSwitch] no flip-switch-handle element provided (required)', 3);
            return;
        }
        this.handleBoxElement = shmi.getUiElement('handle-box', this.element);
        if (!this.handleBoxElement) {
            shmi.log('[FlipSwitch] no handle-box element provided (required)', 3);
            return;
        }
        shmi.log('[FlipSwitch] flip-switch width: ' + this.handleBoxElement.offsetWidth, 0);
        shmi.log('[FlipSwitch] handle width: ' + this.handleElement.offsetWidth, 0);

        this.labelElement = shmi.getUiElement('label', this.element);
        if (!this.labelElement) {
            shmi.log("[FlipSwitch] no label element provided (optional)", 1);
        }

        this.onLabelElement = shmi.getUiElement('flip-switch-label-on', this.element);
        if (!this.onLabelElement) {
            shmi.log('[FlipSwitch] no flip-switch-label-on element provided (optional)', 1);
        }

        this.offLabelElement = shmi.getUiElement('flip-switch-label-off', this.element);
        if (!this.offLabelElement) {
            shmi.log('[FlipSwitch] no flip-switch-label-off element provided (optional)', 1);
        }

        /* set on-label if present in template and defined in config */
        if (this.onLabelElement) {
            if (this.config['on-label']) {
                this.onLabelElement.textContent = shmi.localize(this.config['on-label']);
            }
        }
        /* set off-label if present in template and defined in config */
        if (this.offLabelElement) {
            if (this.config['off-label']) {
                this.offLabelElement.textContent = shmi.localize(this.config['off-label']);
            }
        }
        /* set control label if present in template and defined in config */
        if (this.labelElement) {
            if (this.config.label) {
                this.labelElement.textContent = shmi.localize(this.config.label);
            }
        }

        /* IO Functionality */
        this.width = this.handleBoxElement.offsetWidth;
        this.handleWidth = this.handleElement.offsetWidth;
        this.movable = new shmi.visuals.gfx.Movable(this.handleElement);
        this.movable.setTransition(true);
        this.movable.transitionStyle = this.config.transitionStyle;
        var ioFuncs = {};
        ioFuncs.onDrag = function(dx, dy, event) {
            event.preventDefault();
            if (((this.movable.tx + dx) >= 0) && ((this.movable.tx + dx + this.handleWidth) <= this.handleBoxElement.offsetWidth)) {
                if (!this._dragging) {
                    this._dragging = true;
                }
                this.movable.translate(dx, 0);
            }
        }.bind(this);
        ioFuncs.onPress = function() {
            this.movable.setTransition(false);
            shmi.addClass(this.handleElement, 'pressed');
        }.bind(this);
        ioFuncs.onRelease = function() {
            if (this._dragging) {
                this._dragging = false;
                if (this.movable.tx > ((this.handleBoxElement.offsetWidth - this.handleElement.offsetWidth) / 2)) {
                    if (this.config['confirm-on']) {
                        shmi.confirm(this.config['confirm-on-text'], function(conf) {
                            if (conf) {
                                self.movable.tx = self.handleBoxElement.offsetWidth - self.handleElement.offsetWidth;
                                if (self.config.item) {
                                    shmi.visuals.session.ItemManager.writeValue(self.config.item, self.config['on-value']);
                                } else {
                                    self.setValue(self.config['on-value']);
                                }
                            } else {
                                self.setValue(self.value);
                            }
                        });
                    } else {
                        this.movable.tx = this.handleBoxElement.offsetWidth - this.handleElement.offsetWidth;
                        if (this.config.item) {
                            shmi.visuals.session.ItemManager.writeValue(this.config.item, this.config['on-value']);
                        } else {
                            this.setValue(this.config['on-value']);
                        }
                    }
                } else if (this.config['confirm-off']) {
                    shmi.confirm(this.config['confirm-off-text'], function(conf) {
                        if (conf) {
                            self.movable.tx = 0;
                            if (self.config.item) {
                                shmi.visuals.session.ItemManager.writeValue(self.config.item, self.config['off-value']);
                            } else {
                                self.setValue(self.config['off-value']);
                            }
                        } else {
                            self.setValue(self.value);
                        }
                    });
                } else {
                    this.movable.tx = 0;
                    if (this.config.item) {
                        shmi.visuals.session.ItemManager.writeValue(this.config.item, this.config['off-value']);
                    } else {
                        this.setValue(this.config['off-value']);
                    }
                }
                this.movable.update();
            }
            this.movable.setTransition(true);
            shmi.removeClass(this.handleElement, 'pressed');
        }.bind(this);
        ioFuncs.onClick = function() {
            if (this.value === this.config['on-value']) {
                if (this.config['confirm-off']) {
                    shmi.confirm(this.config['confirm-off-text'], function(conf) {
                        if (conf) {
                            if (self.config.item) {
                                shmi.visuals.session.ItemManager.writeValue(self.config.item, self.config['off-value']);
                            } else {
                                self.setValue(self.config['off-value']);
                            }
                        }
                    });
                } else if (this.config.item) {
                    shmi.visuals.session.ItemManager.writeValue(this.config.item, this.config['off-value']);
                } else {
                    this.setValue(this.config['off-value']);
                }
            } else if (this.config['confirm-on']) {
                shmi.confirm(this.config['confirm-on-text'], function(conf) {
                    if (conf) {
                        if (self.config.item) {
                            shmi.visuals.session.ItemManager.writeValue(self.config.item, self.config['on-value']);
                        } else {
                            self.setValue(self.config['on-value']);
                        }
                    }
                });
            } else if (this.config.item) {
                shmi.visuals.session.ItemManager.writeValue(this.config.item, this.config['on-value']);
            } else {
                this.setValue(this.config['on-value']);
            }
        }.bind(this);
        var ioClickFunc = {};
        ioClickFunc.onClick = ioFuncs.onClick;
        this.mouseListener = new shmi.visuals.io.MouseListener(this.handleElement, ioFuncs);
        this.touchListener = new shmi.visuals.io.TouchListener(this.handleElement, ioFuncs);
        this.mouseListener2 = new shmi.visuals.io.MouseListener(this.handleBoxElement, ioClickFunc);
        this.touchListener2 = new shmi.visuals.io.TouchListener(this.handleBoxElement, ioClickFunc);
    },
    /**
     * Enables the FlipSwitch
     *
     */
    onEnable: function() {
        this.width = this.handleBoxElement.offsetWidth;
        this.handleWidth = this.handleElement.offsetWidth;
        if (this.config.item) {
            this._subscriptionTargetId = shmi.visuals.session.ItemManager.subscribeItem(this.config.item, this);
        }
        shmi.log("[FlipSwitch] hwidth: " + this.handleWidth + " width: " + this.handleBoxElement.offsetWidth, 0);
        this.mouseListener.enable();
        this.touchListener.enable();
        this.mouseListener2.enable();
        this.touchListener2.enable();
        shmi.log("[FlipSwitch] enabled", 1);
    },
    /**
     * Disables the FlipSwitch
     *
     */
    onDisable: function() {
        if (this.config.item) {
            shmi.visuals.session.ItemManager.unsubscribeItem(this.config.item, this._subscriptionTargetId);
        }
        this.mouseListener.disable();
        this.touchListener.disable();
        this.mouseListener2.disable();
        this.touchListener2.disable();
        shmi.log("[FlipSwitch] disabled", 1);
    },
    /**
     * Locks the FlipSwitch
     *
     */
    onLock: function() {
        shmi.log("[FlipSwitch] locked", 1);
        this.mouseListener.disable();
        this.touchListener.disable();
        this.mouseListener2.disable();
        this.touchListener2.disable();
        shmi.addClass(this.element, 'locked');
    },
    /**
     * Unlocks the FlipSwitch
     *
     */
    onUnlock: function() {
        shmi.log("[FlipSwitch] unlocked", 1);
        this.mouseListener.enable();
        this.touchListener.enable();
        this.mouseListener2.enable();
        this.touchListener2.enable();
        shmi.removeClass(this.element, 'locked');
    },
    /**
     * Sets a new value for the FlipSwitch
     *
     * @param value - new value to set
     */
    onSetValue: function(value) {
        if (this._dragging) {
            return;
        }
        var on = false,
            old_value = this.value;
        if (value === this.config['on-value']) {
            this.value = this.config['on-value'];
            shmi.addClass(this.element, 'on');
            on = true;
        } else {
            this.value = this.config['off-value'];
            shmi.removeClass(this.element, 'on');
        }
        if (on) {
            this.movable.tx = (this.handleBoxElement.offsetWidth - this.handleWidth);
        } else {
            this.movable.tx = 0;
        }

        if (old_value !== this.value) {
            this.fire("change", {
                value: this.value
            });

            if (this.onChange) {
                this.onChange(this.value);
            }
        }

        this.movable.update();
    },
    /**
     * Retrieves the current value of the FlipSwitch
     *
     * @return value - current value
     */
    getValue: function() {
        return this.value;
    },
    setLabel: function(labelText) {
        var self = this;
        if (self.labelElement && self.config['auto-label']) {
            self.labelElement.textContent = shmi.localize(labelText);
        }
    }
};

shmi.extend(shmi.visuals.controls.FlipSwitch, shmi.visuals.core.BaseControl);

shmi.pkg("visuals.controls");

/**
 * Creates a new Form control
 *
 * @constructor
 * @extends shmi.visuals.core.BaseControl
 * @param element - base element of the control
 * @param config - configuration of the control
 */
shmi.visuals.controls.Form = function(element, config) {
    this.element = element;
    this.config = config || {};

    this.parseAttributes();

    shmi.def(this.config, 'class-name', 'form');
    shmi.def(this.config, 'name', null);
    shmi.def(this.config, 'template', null);
    shmi.def(this.config, 'fields', []);
    shmi.def(this.config, 'id-selector', null);
    shmi.def(this.config, 'datagrid', null);

    this.values = [];
    this.controls = [];
    this.fieldItems = [];
    this.fieldControls = {};
    this.loadedFields = {};
    this.applyCallback = null;
    this.cancelCallback = null;
    this.sel_list_id = null;
    /* selection listener id */

    this.startup();
};

shmi.visuals.controls.Form.prototype = {
    uiType: "form",
    isContainer: true,
    getClassName: function() {
        return "Form";
    },
    onInit: function() {},
    onRegister: function(onDone) {
        var self = this,
            form_id = Date.now(),
            nameToken = null,
            s = shmi.visuals.session;

        s.FormIDs = s.FormIDs || [];
        while (s.FormIDs.indexOf(form_id) !== -1) {
            form_id++;
        }
        s.FormIDs.push(form_id);
        self.form_id = form_id;

        nameToken = shmi.listen("register-name", function(evt) {
            var src_ctrl = evt.source;
            if (shmi.testParentChild(self.element, src_ctrl.element)) {
                if ((self.config.fields.indexOf(src_ctrl.config.field) !== -1) && (!self.loadedFields[src_ctrl.config.field])) {
                    self.fieldControls[src_ctrl.config.field] = src_ctrl;
                    /* create virtual item-name to track changed values */
                    var vItemName = "virtual:form:" + self.form_id + ":" + src_ctrl.config.field;
                    /* set item name on field-control */
                    src_ctrl.setItem(vItemName);
                    self.loadedFields[src_ctrl.config.field] = true;
                }
            }
        });
        self._init_.tokens.push(nameToken);

        self.parseChildren(self.element, function() {
            nameToken.unlisten();
            self._init_.tokens.splice(self._init_.tokens.indexOf(nameToken), 1);
            nameToken = null;
            onDone();
        });
    },
    /**
     * Sets the specified values to the Form controls.
     *
     * @param values - [] values for Form controls
     * @param applyCallback - callback to run when applying the Form
     * @param cancelCallback - callback to run when canceling the Form
     */
    setValues: function(values, applyCallback, cancelCallback, force) {
        /* delay execution until control finished initializing */
        if (!this.initialized) {
            return;
        }

        /* cancel execution and let user confirm that modified values will be lost */
        if (!force && (this.checkModified() !== 0)) {
            shmi.askSave("${V_ASK_SAVE_MSG}", function(save, cont) {
                /* only set new values if user agreed to lose modifications */
                if (save === true) {
                    shmi.log("[Form] applying changes...", shmi.c("LOG_MSG"));
                    this.apply();
                } else {
                    shmi.log("[Form] discarding changes...", shmi.c("LOG_MSG"));
                }
                if (cont === true) {
                    this.setValues(values, applyCallback, cancelCallback, true);
                } else {
                    this.unlock();
                }
            }.bind(this));
            return;
        }

        this.values = values;
        this.applied = false;
        this.canceled = false;
        if (this.values.length !== Object.keys(this.fieldControls).length) {
            shmi.log("[Form] values length (" + this.values.length +
                ") does not match number of controls(" + Object.keys(this.fieldControls).length + ")", 2);
        }

        this.deleteFieldItems();

        for (var i = 0; i < this.values.length; i++) {
            if (Array.isArray(this.values[i])) { // old style
                if (this.values[i].length !== 2) {
                    shmi.log("[Form] invalid value set: " + this.values[i], 2);
                    continue;
                }
                try {
                    this.controls[i].type = parseInt(this.values[i][1]);
                    this.controls[i].setValue(this.values[i][0]);
                    shmi.log("[Form] value set: " + this.values[i][0], 2);
                } catch (exc) {
                    shmi.log("[Form] exception setting form data: " + exc, 3);
                }
            } else { //new style
                var val = this.values[i],
                    vItemName = "virtual:form:" + this.form_id + ":" + val.field;
                if ((val.type === null) || (val.type === undefined)) {
                    val.type = shmi.c("TYPE_STRING");
                }
                var v_item = shmi.createVirtualItem(vItemName, val.type, Number.NaN, Number.NaN, val.value, this.getFieldValueCallback(val.field));
                this.fieldItems.push(v_item);
                v_item.writeValue(val.value);
                val.modified = false;
                v_item.notifyUpdateTargets();
                if (this.fieldControls[val.field] !== undefined) {
                    shmi.removeClass(this.fieldControls[val.field].element, 'modified');
                }
                //this.fieldControls[val.field].type = parseInt(val.type);
                //this.fieldControls[val.field].setValue(val.value);
            }
        }

        if (applyCallback) {
            this.applyCallback = applyCallback;
            var modified = this.checkModified();
            if (modified !== 0) {
                if (this.onChange) {
                    this.onChange(this);
                }
            }
        }
        if (cancelCallback) {
            this.cancelCallback = cancelCallback;
        }

        /* unlock if values are set on form */
        this.unlock();
    },
    deleteFieldItems: function() {
        this.fieldItems = [];
    },
    /* creates callback functions for virtual items used for form-fields */
    getFieldValueCallback: function(field) {
        return function(value, type, name) {
            var f_name = field;
            for (var i = 0; i < this.values.length; i++) {
                if (this.values[i].field === f_name) {
                    if (this.values[i].value !== value) {
                        if (this.fieldControls[f_name] !== undefined) {
                            shmi.addClass(this.fieldControls[f_name].element, 'modified');
                        }
                        this.values[i].modified = true;
                    } else {
                        if (this.fieldControls[f_name] !== undefined) {
                            shmi.removeClass(this.fieldControls[f_name].element, 'modified');
                        }
                        this.values[i].modified = false;
                    }
                    break; // stop if field is found
                }
            }

            if (this.onChange) {
                this.onChange(this);
            }
        }.bind(this);
    },
    checkModified: function() {
        var modified = 0;
        if (this.values === undefined) {
            this.values = [];
        }
        for (var i = 0; i < this.values.length; i++) {
            if (this.values[i].modified === true) {
                modified++;
            }
        }
        return modified;
    },
    /**
     * Applies the Form
     *
     */
    apply: function() {
        var i;
        for (i = 0; i < this.values.length; i++) {
            try {
                this.values[i].value = this.fieldControls[this.values[i].field].getValue();
            } catch (exc) {
                shmi.log("[Form] control has no value: " + exc.toString(), 2);
            }
        }
        this.applied = true;
        if (this.applyCallback) {
            this.applyCallback(this.values);
        } else {
            shmi.log("[Form] no apply callback defined", 2);
        }

        /* reset modified states */
        for (i = 0; i < this.values.length; i++) {
            if (this.fieldControls[this.values[i].field] !== undefined) {
                shmi.removeClass(this.fieldControls[this.values[i].field].element, 'modified');
            }
            this.values[i].modified = false;
        }
    },
    /**
     * Cancels the Form
     *
     */
    cancel: function() {
        this.canceled = true;
        if (this.cancelCallback) {
            this.cancelCallback();
        }
    },
    /**
     * Enables the Form
     *
     */
    onEnable: function() {
        for (var i = 0; i < this.controls.length; i++) {
            this.controls[i].enable();
        }

        if (this.config['id-selector']) {
            this.id_selector = shmi.ctrl(this.config['id-selector']);
            if (this.id_selector) {
                var self = this;
                var selection_handler = {
                    tableSelectionChange: function(selection) {
                        self.setDataGridSelection(selection);
                    }
                };
                /* get current selection */
                selection_handler.tableSelectionChange(self.id_selector.getSelectedRows());
                /* register selection-change listener */
                this.sel_list_id = this.id_selector.addSelChgEventListener(selection_handler);
            }
        }

        shmi.log("[Form] enabled", 1);
    },
    getFormReset: function(data) {
        var self = this;
        return function() {
            self.setValues(data, self.applyCallback, self.cancelCallback, true);
        };
    },
    setDataGridSelection: function(selection) {
        var self = this,
            dgm = shmi.visuals.session.DataGridManager,
            i;
        if ((selection.type === 2) && (selection.selRows.length > 0) && self.config.datagrid) {
            self.unlock();
            var grid = dgm.grids[self.config.datagrid];
            if (grid) {
                var values = [],
                    data;
                if (selection.selRows.length > 1) {
                    var data_null = grid.getRowData(selection.selRows[0]);
                    for (var j = 0; j < data_null.length; j++) {
                        var all_equal = true,
                            last_val = null;
                        for (var k = 0; k < selection.selRows.length; k++) {
                            data = grid.getRowData(selection.selRows[k]);
                            if (k === 0) {
                                last_val = data[j].value;
                            } else if (data[j].value === last_val) {
                                continue;
                            } else {
                                all_equal = false;
                                break;
                            }
                        }
                        if (all_equal) {
                            values.push({
                                value: last_val,
                                field: grid.fields[j]
                            });
                        } else {
                            values.push({
                                value: "",
                                field: grid.fields[j]
                            });
                        }
                    }
                } else {
                    data = grid.getRowData(selection.selRows[0]);
                    for (i = 0; i < data.length; i++) {
                        values.push({
                            value: data[i].value,
                            field: grid.fields[i]
                        });
                    }
                }
                self.setValues(values, self.getDataGridApply(grid, selection.selRows), self.getFormReset(values), false);
            }
        } else {
            var vals = [];
            for (i = 0; i < self.config.fields.length; i++) {
                var val = {
                    type: shmi.c("TYPE_STRING"),
                    field: self.config.fields[i],
                    value: ""
                };
                vals.push(val);
            }
            self.setValues(vals, null, null);
            self.lock();
        }
    },
    getDataGridApply: function(grid, sel_rows) {
        var self = this,
            i;

        var apply_row = function(sel_row_idx) {
            for (i = 0; i < self.config.fields.length; i++) {
                var f_ctrl = self.fieldControls[self.config.fields[i]];
                if (f_ctrl && self.values[i].modified) {
                    var grid_field_index = grid.fields.indexOf(self.config.fields[i]);
                    if (grid_field_index !== -1) {
                        grid.data[sel_rows[sel_row_idx]][grid_field_index].item.writeValue(f_ctrl.getValue());
                    }
                }
            }
        };

        if (sel_rows.length === 1) {
            return function() {
                apply_row(0);
            };
        } else if (sel_rows.length > 1) {
            return function() {
                for (i = 0; i < sel_rows.length; i++) {
                    apply_row(i);
                }
            };
        } else {
            return function() {
                console.trace("NO_SELECT_STUB");
            };
        }
    },
    /**
     * Disables the Form
     *
     */
    onDisable: function() {
        var self = this;
        for (var i = 0; i < self.controls.length; i++) {
            self.controls[i].disable();
        }
        if ((self.sel_list_id !== null) && self.id_selector) {
            self.id_selector.removeSelChgEventListener(self.sel_list_id);
            self.sel_list_id = null;
        }
        if (self.field_ctrl_lid) {
            shmi.unlisten("register-name", self.field_ctrl_lid);
            self.field_ctrl_lid = null;
        }
        shmi.log("[Form] disabled", 1);
    },
    /**
     * Locks the Form
     *
     */
    onLock: function() {
        var self = this;

        shmi.addClass(self.element, 'locked');

        function recLock(ctrl) {
            ctrl.lock();
            if (Array.isArray(ctrl.controls)) {
                ctrl.controls.forEach(recLock);
            }
        }

        self.controls.forEach(function(ctrl) {
            recLock(ctrl);
        });
        shmi.log("[Form] locked", 1);
    },
    /**
     * Unlocks the Form
     *
     */
    onUnlock: function() {
        var self = this;

        shmi.removeClass(self.element, 'locked');

        function recUnlock(ctrl) {
            ctrl.unlock();
            if (Array.isArray(ctrl.controls)) {
                ctrl.controls.forEach(recUnlock);
            }
        }

        self.controls.forEach(function(ctrl) {
            recUnlock(ctrl);
        });
        shmi.log("[Form] unlocked", 1);
    },
    /**
     * Sets value to the Form
     *
     * @param value - new value to set
     */
    onSetValue: function(value) {
        shmi.log("[Form] value set: " + value, 0);
    }
};

shmi.extend(shmi.visuals.controls.Form, shmi.visuals.core.BaseControl);

// version https://88.198.203.47/svn//project/branches/naming_scheme - von Meike am 28.08.2014 (war unver??ndert zu Stand 5_CT2)
// ++ patches, fixes Peer 28.08. ff.

shmi.pkg("visuals.controls");
/**
 * General purpose scroll bar.
 *
 * An instance of this control changes values between min and max and fires the related events.
 *
 * @constructor
 * @extends shmi.visuals.core.BaseControl
 * @param {Element} element the base element of the control
 * @param {Object} config control configuration
 */
shmi.visuals.controls.GpScrollBar = function(element, config) {
    // short cuts to static "const" definitions for GPSB
    this.c = shmi.visuals.controls.GpScrollBar.c;
    this.uiElName = shmi.visuals.controls.GpScrollBar.uiElName;
    this.uiCssCl = shmi.visuals.controls.GpScrollBar.uiCssCl;

    this.element = element;

    this.config = config || {};
    /* set default options */
    shmi.def(this.config, "class-name", "gp-scroll-bar");
    shmi.def(this.config, "template", "default/gp-scroll-bar");
    shmi.def(this.config, "name", "Unnamed GpScrollBar");
    shmi.def(this.config, "label", "Label?"); // NYI
    shmi.def(this.config, "max", 100);
    shmi.def(this.config, "min", 0);
    shmi.def(this.config, "precision", 0); // tbd, NYI
    shmi.def(this.config, "step", 0); //, NYI
    shmi.def(this.config, "vertical", true); // currently is true impl. only !!
    shmi.def(this.config, "buttons", true);
    shmi.def(this.config, "inverse", false);

    this.parseAttributes();

    //this.initialized = false;
    this.isMinimized = false; // true means that GBSB is used as indicator only, events are disabled
    this.isEnabled = false;
    this.min = 0;
    this.max = 0;
    this.sliderVal = 0;
    this.sliderWidth = 0;
    this.sliderTrackElem = null;
    this.sliderHandleElem = null;
    this.sliderHandleVisibleElem = null;
    this.sliderHandleMovable = null;
    this.btnUpElem = null;
    this.btnDownElem = null;
    this.sliderHandleMouseListener = null;
    this.sliderHandleTouchListener = null;
    this.sliderTrackMouseListener = null;
    this.sliderTrackTouchListener = null;
    this.btnUpMouseListener = null;
    this.btnUpTouchListener = null;
    this.btnDownMouseListener = null;
    this.btnDownTouchListener = null;
    this.sbEventListeners = {};
    this.enableTO = null;
    this.btnStartAutoRepeatTO = null;
    this.btnProcessAutoRepeatTO = null;
    this.isBtnAutoRepeatProcessing = false;

    // $test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    this.debug = {};
    this.debug.msgFIFO = [];
    this.testlog_msg("startup");
    // $test !!!!!!!!!!!!!!!

    this.startup();
};

shmi.visuals.controls.GpScrollBar.prototype = {
    /*-- BaseControl method interface --*/
    uiType: "gp-scroll-bar",
    getClassName: function() {
        return "GpScrollBar";
    },
    onInit: function() {
        /* config post processing (boolean and number values) */
        this.config["max"] = shmi.toNumber(this.config["max"]);
        this.config["min"] = shmi.toNumber(this.config["min"]);
        this.config["precision"] = shmi.toNumber(this.config["precision"]);
        this.config["step"] = shmi.toNumber(this.config["step"]);
        this.config["vertical"] = shmi.toBoolean(this.config["vertical"]);
        this.config["buttons"] = shmi.toBoolean(this.config["buttons"]);
        this.config["inverse"] = shmi.toBoolean(this.config["inverse"]);

        /* init all members, get references to the necessary elements */
        this.min = this.config["min"];
        this.max = this.config["max"];
        this.sliderVal = this.min; // 1st valid init
        this.sliderWidth = (this.max - this.min) / 10; // 1st valid init
        this.sliderTrackElem = shmi.getUiElement(this.uiElName.sliderTrack, this.element);
        if (!this.sliderTrackElem) {
            shmi.log("[GpScrollBar] no " + this.uiElName.sliderTrack + " element provided (required)", 3);
            return;
        }
        this.sliderHandleElem = shmi.getUiElement(this.uiElName.sliderHandle, this.element);
        if (!this.sliderHandleElem) {
            shmi.log("[GpScrollBar] no " + this.uiElName.sliderHandle + " provided (required)", 3);
            return;
        }
        this.sliderHandleMovable = new shmi.visuals.gfx.Movable(this.sliderHandleElem);
        shmi.addClass(this.sliderHandleElem, this.uiCssCl.sliderAni);
        this.sliderHandleVisibleElem = shmi.getUiElement(this.uiElName.sliderHandleVisible, this.element);
        if (!this.sliderHandleVisibleElem) {
            shmi.log("[GpScrollBar] no " + this.uiElName.sliderHandleVisibleElem + " provided (required)", 3);
            return;
        }

        if (this.config["buttons"]) {
            shmi.addClass(this.element, this.uiCssCl.withBtns);
            this.btnUpElem = shmi.getUiElement(this.uiElName.btnUp, this.element);
            if (!this.btnUpElem) {
                shmi.log("[GpScrollBar] no " + this.uiElName.btnUp + " element provided (required)", 3);
                return;
            }
            this.btnDownElem = shmi.getUiElement(this.uiElName.btnDown, this.element);
            if (!this.btnDownElem) {
                shmi.log("[GpScrollBar] no " + this.uiElName.btnDown + " element provided (required)", 3);
                return;
            }
        } else {
            shmi.addClass(this.element, this.uiCssCl.withoutBtns);
            var btnContElem = shmi.getUiElement(this.uiElName.btnPanel, this.element);
            if (btnContElem) {
                btnContElem.parentNode.removeChild(btnContElem);
            }
        }

        /* define and route the event handlers, filter events */
        var sliderHandleEvents = {};
        sliderHandleEvents.onDrag = function(dx, dy, event) {
            event.preventDefault();
            // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            //console.log("sliderHandleEvents.onDrag(..) fired - dx:" + dx + " dy:" + dy);
            // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            this.updateVal(this.sliderHandleMovable.ty + dy);
            this.updateSlider();
            this.fireValChanged(this.sliderVal, false);
        }.bind(this);
        sliderHandleEvents.onPress = function() {
            // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            //console.log("sliderHandleEvents.onPress() fired");
            // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            shmi.removeClass(this.sliderHandleElem, this.uiCssCl.sliderAni);
            //shmi.addClass(this.sliderHandleElem, this.uiCssCl.btnSel);
            shmi.addClass(this.sliderHandleVisibleElem, this.uiCssCl.btnSel);
        }.bind(this);
        sliderHandleEvents.onRelease = function() {
            // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            //console.log("sliderHandleEvents.onRelease() fired");
            // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            shmi.addClass(this.sliderHandleElem, this.uiCssCl.sliderAni);
            //shmi.removeClass(this.sliderHandleElem, this.uiCssCl.btnSel);
            shmi.removeClass(this.sliderHandleVisibleElem, this.uiCssCl.btnSel);
        }.bind(this);
        this.sliderHandleMouseListener = new shmi.visuals.io.MouseListener(this.sliderHandleElem, sliderHandleEvents);
        this.sliderHandleTouchListener = new shmi.visuals.io.TouchListener(this.sliderHandleElem, sliderHandleEvents);
        var sliderTrackEvents = {};
        sliderTrackEvents.onPress = function(x, y, event) {
            shmi.addClass(this.sliderTrackElem, this.uiCssCl.trackSel);
        }.bind(this);
        sliderTrackEvents.onRelease = function() {
            // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            //console.log("sliderTrackEvents.onRelease() fired");
            // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            shmi.removeClass(this.sliderTrackElem, this.uiCssCl.trackSel);
        }.bind(this);
        sliderTrackEvents.onClick = function(x, y, event) {
            var pos = shmi.getAbsPosition(this.sliderTrackElem);
            var deltaY = y - pos.y;
            this.updateVal(deltaY);
            this.updateSlider();
            this.fireValChanged(this.sliderVal, true);
        }.bind(this);
        this.sliderTrackMouseListener = new shmi.visuals.io.MouseListener(this.sliderTrackElem, sliderTrackEvents);
        this.sliderTrackTouchListener = new shmi.visuals.io.TouchListener(this.sliderTrackElem, sliderTrackEvents);
        if (this.btnUpElem) {
            var btnUpEvents = {};
            btnUpEvents.onPress = function() {
                // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                //console.log("btnUpEvents.onPress() fired");
                // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                shmi.addClass(this.btnUpElem, this.uiCssCl.btnSel);
                if (this.btnStartAutoRepeatTO) {
                    clearTimeout(this.btnStartAutoRepeatTO);
                }
                this.btnStartAutoRepeatTO = setTimeout(function() {
                    this.startBtnAutoRepeat("up");
                }.bind(this), this.c.START_BTN_AUTO_REPEAT_TO);
                this.fireBtnAction("up");
            }.bind(this);
            btnUpEvents.onRelease = function() {
                // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                //console.log("btnUpEvents.onRelease() fired");
                // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                if (this.btnStartAutoRepeatTO) {
                    clearTimeout(this.btnStartAutoRepeatTO);
                }
                this.stopBtnAutoRepeat();
                shmi.removeClass(this.btnUpElem, this.uiCssCl.btnSel);
            }.bind(this);
            this.btnUpMouseListener = new shmi.visuals.io.MouseListener(this.btnUpElem, btnUpEvents);
            this.btnUpTouchListener = new shmi.visuals.io.TouchListener(this.btnUpElem, btnUpEvents);
            var btnDownEvents = {};
            btnDownEvents.onPress = function() {
                // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                //console.log("btnDownEvents.onPress() fired");
                // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                shmi.addClass(this.btnDownElem, this.uiCssCl.btnSel);
                if (this.btnStartAutoRepeatTO) {
                    clearTimeout(this.btnStartAutoRepeatTO);
                }
                this.btnStartAutoRepeatTO = setTimeout(function() {
                    this.startBtnAutoRepeat("down");
                }.bind(this), this.c.START_BTN_AUTO_REPEAT_TO);
                this.fireBtnAction("down");
            }.bind(this);
            btnDownEvents.onRelease = function() {
                // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                //console.log("btnDownEvents.onRelease() fired");
                // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                if (this.btnStartAutoRepeatTO) {
                    clearTimeout(this.btnStartAutoRepeatTO);
                }
                this.stopBtnAutoRepeat();
                shmi.removeClass(this.btnDownElem, this.uiCssCl.btnSel);
            }.bind(this);
            this.btnDownMouseListener = new shmi.visuals.io.MouseListener(this.btnDownElem, btnDownEvents);
            this.btnDownTouchListener = new shmi.visuals.io.TouchListener(this.btnDownElem, btnDownEvents);
        }
        this.resizeListener = function() {
            this.processEvtOnResize();
        }.bind(this);

        shmi.log("[GpScrollBar] initialized", 1);
    },
    onEnable: function() {
        // test !!!!!!!!!!!!!!!
        this.testlog_msg("onEnable");
        // test !!!!!!!!!!!!!!!
        window.addEventListener("resize", this.resizeListener, true);
        this.updateSlider();
        this.isEnabled = true;
        this.updateEvtEnable();
        shmi.log("[GpScrollBar] enabled", 1);
    },
    onDisable: function() {
        // test !!!!!!!!!!!!!!!
        this.testlog_msg("onEnable");
        // test !!!!!!!!!!!!!!!
        window.removeEventListener("resize", this.resizeListener, true);
        this.isEnabled = false;
        this.updateEvtEnable();
        shmi.log("[GpScrollBar] disabled", 1);
    },
    onDelete: function() {},

    /*-- additional method interface --*/

    /*
     * $todo: noch JSDoc method descriptions add
     *
     * val, width are numbers in the range of [min .. max]
     */
    setSliderVal: function(val) {
        this.sliderVal = val;
        this.updateSlider();
        shmi.log("[GpScrollBar] setSliderVal called, val: " + val, 1);
    },
    setSliderWidth: function(width) {
        this.sliderWidth = width;
        this.updateSlider();
        shmi.log("[GpScrollBar] setSliderWidth called, width: " + width, 1);
    },
    setSliderProps: function(val, width) {
        this.sliderVal = val;
        this.sliderWidth = width;
        this.updateSlider();
        shmi.log("[GpScrollBar] setSliderProps called, val: " + val + " - width: " + width, 1);
    },
    getSliderVal: function() {
        shmi.log("[GpScrollBar] getSliderVal called, return: " + this.sliderVal, 1);
        return this.sliderVal;
    },
    getSliderWidth: function() {
        shmi.log("[GpScrollBar] getSliderVal called, return: " + this.sliderWidth, 1);
        return this.sliderWidth;
    },
    setMin: function(min) {
        this.min = min;
        this.updateSlider();
        shmi.log("[GpScrollBar] setMin called, min: " + min, 1);
    },
    setMax: function(max) {
        this.max = max;
        this.updateSlider();
        shmi.log("[GpScrollBar] setMax called, min: " + max, 1);
    },
    setMinMax: function(min, max) {
        this.min = min;
        this.max = max;
        this.updateSlider();
        shmi.log("[GpScrollBar] setMinMax called, min: " + min + " - max: " + max, 1);
    },
    getMin: function() {
        shmi.log("[GpScrollBar] getMin called, return: " + this.min, 1);
        return this.min;
    },
    getMax: function() {
        shmi.log("[GpScrollBar] getMax called, return: " + this.max, 1);
        return this.max;
    },
    setAllProps: function(val, width, min, max) {
        this.sliderVal = val;
        this.sliderWidth = width;
        this.min = min;
        this.max = max;
        this.updateSlider();
        shmi.log("[GpScrollBar] setAllProps called, , val: " + val + " - width: " + width + " - min: " + min + " - max: " + max, 1);
    },
    setMinimized: function(min) {
        this.isMinimized = min;
        this.updateEvtEnable();
        if (min) {
            shmi.removeClass(this.element, this.uiCssCl.normal);
            shmi.addClass(this.element, this.uiCssCl.minimized);
        } else {
            shmi.removeClass(this.element, this.uiCssCl.minimized);
            shmi.addClass(this.element, this.uiCssCl.normal);
        }
    },
    getMinimized: function() {
        return this.isMinimized;
    },

    /*-- event listener control interface --*/

    /*  $todo: noch JSDoc method descriptions add
     *
     *  note: listenerObj must implement the event handler callbacks
     *   .gpsbOnValChange(val) (val in range [min..max])
     *   .gpsbOnBtnAction(btn)  (btn in range ["up", "down"])
     */
    addEventListener: function(listenerObj) {
        var id = Date.now();
        while (this.sbEventListeners[id] !== undefined) {
            id++;
        }
        this.sbEventListeners[id] = {};
        this.sbEventListeners[id].listener = listenerObj;
        shmi.log("[GpScrollBar] addEventListener called, listenerObj: " + listenerObj + " - id:" + id, 1);
        return id;
    },
    removeEventListener: function(id) {
        if (this.sbEventListeners[id] !== undefined) {
            delete this.sbEventListeners[id];
        } else {
            shmi.log("[GpScrollBar] removeEventListener - id " + id + " does not exist", 2);
        }
        shmi.log("[GpScrollBar] removeEventListener called, id: " + id, 1);
    },

    /*-- private helper methods - usage within GPSB only !! --*/

    /**
     * @private
     */
    startBtnAutoRepeat: function(btn) {
        // $test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        //console.log(">>> startBtnAutoRepeat() - btn: " + btn);
        // $test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        this.isBtnAutoRepeatProcessing = true;
        if (this.btnProcessAutoRepeatTO) {
            clearTimeout(this.btnProcessAutoRepeatTO);
        }
        this.btnProcessAutoRepeatTO = setTimeout(function() {
            this.processBtnAutoRepeat(btn);
        }.bind(this), this.c.PROCESS_BTN_AUTO_REPEAT_TO);
    },
    processBtnAutoRepeat: function(btn) {
        if (this.isBtnAutoRepeatProcessing) {
            // $test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            //console.log(">>> processBtnAutoRepeat() - btn: " + btn);
            // $test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            this.fireBtnAction(btn);
            this.btnProcessAutoRepeatTO = setTimeout(function() {
                this.processBtnAutoRepeat(btn);
            }.bind(this), this.c.PROCESS_BTN_AUTO_REPEAT_TO);
        }
    },
    /**
     * @private
     */
    stopBtnAutoRepeat: function() {
        // $test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        //console.log("<<< stopBtnAutoRepeat()");
        // $test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

        this.isBtnAutoRepeatProcessing = false;
        if (this.btnProcessAutoRepeatTO) {
            clearTimeout(this.btnProcessAutoRepeatTO);
        }
    },
    /**
     * @private
     */
    processEvtOnResize: function() {
        this.updateSlider();
        //console.log("GpScrollBar processEvtOnResize");
    },
    /**
     * @private
     */
    updateSlider: function() {
        if (this.initialized) {
            var slHeight = (this.sliderWidth * this.sliderTrackElem.offsetHeight) / (this.max - this.min);
            if (this.isMinimized) {
                if (slHeight < this.c.MIN_SLIDER_SIZE_PX_MINIMIZED) {
                    slHeight = this.c.MIN_SLIDER_SIZE_PX_MINIMIZED;
                }
            } else if (slHeight < this.c.MIN_SLIDER_SIZE_PX) {
                slHeight = this.c.MIN_SLIDER_SIZE_PX;
            }
            var slAvailTrackHeight = this.sliderTrackElem.offsetHeight - slHeight,
                slTop = ((this.max - this.sliderVal) * slAvailTrackHeight) / (this.max - this.min);

            // $todo: vor Einstellung der neuen Slider-TopPos/-Height -> range it ??
            this.sliderHandleElem.style.height = slHeight + "px";
            if (!this.config["inverse"]) {
                slTop = slAvailTrackHeight - slTop;
            }
            this.sliderHandleMovable.translate(0, slTop - this.sliderHandleMovable.ty);
        }
    },
    /**
     * @private
     */
    updateVal: function(newSlTop) {
        if (this.initialized) {
            var newVal = this.max - (newSlTop * (this.max - this.min)) / (this.sliderTrackElem.offsetHeight - this.sliderHandleElem.offsetHeight);
            if (!this.config["inverse"]) {
                newVal = this.max - newVal;
            }
            if (newVal < this.min) {
                this.sliderVal = this.min;
            } else if (newVal > this.max) {
                this.sliderVal = this.max;
            } else {
                this.sliderVal = newVal;
            }
        }
    },
    /**
     * @private
     */
    updateEvtEnable: function() {
        var enabled = (this.isEnabled && !this.isMinimized);
        if (this.initialized) {
            // test !!!!!!!!!!!!!!!
            this.testlog_msg("init ok - en: " + this.isEnabled + " - min: " + this.isMinimized);
            // test !!!!!!!!!!!!!!!
            if (enabled) {
                this.sliderHandleMouseListener.enable();
                this.sliderHandleTouchListener.enable();
                this.sliderTrackMouseListener.enable();
                this.sliderTrackTouchListener.enable();
                if (this.btnUpElem) {
                    this.btnUpMouseListener.enable();
                    this.btnUpTouchListener.enable();
                    this.btnDownMouseListener.enable();
                    this.btnDownTouchListener.enable();
                }
            } else {
                this.sliderHandleMouseListener.disable();
                this.sliderHandleTouchListener.disable();
                this.sliderTrackMouseListener.disable();
                this.sliderTrackTouchListener.disable();
                if (this.btnUpElem) {
                    this.btnUpMouseListener.disable();
                    this.btnUpTouchListener.disable();
                    this.btnDownMouseListener.disable();
                    this.btnDownTouchListener.disable();
                }
            }
        } else {
            // test !!!!!!!!!!!!!!!
            this.testlog_msg("init ! ok - repeat");
            // test !!!!!!!!!!!!!!!
            if (this.enableTO) {
                clearTimeout(this.enableTO);
            }
            this.enableTO = setTimeout(function() {
                this.updateEvtEnable();
            }.bind(this), this.c.ENABLE_TO);
        }
    },
    /**
     * @private
     */
    fireValChanged: function(val, fromTrack) {
        const iterObj = shmi.requires("visuals.tools.iterate.iterateObject");

        iterObj(this.sbEventListeners, (listener) => {
            listener.listener.gpsbOnValChange(val, fromTrack);
        });
    },
    /**
     * @private
     */
    fireBtnAction: function(btn) {
        const iterObj = shmi.requires("visuals.tools.iterate.iterateObject");

        iterObj(this.sbEventListeners, (listener) => {
            listener.listener.gpsbOnBtnAction(btn);
        });
    },

    // test !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    /**
     * @private
     */
    testlog_msg: function(msg) {
        var msgDisplay = document.getElementById("debugDisplEvents");
        if (msgDisplay) {
            var displInfo = [];
            msgDisplay.innerHTML = "";
            displInfo.push("last debug messages:");
            displInfo.push("---");
            console.log(msg);
            this.debug.msgFIFO.push(msg);
            if (this.debug.msgFIFO.length > 10) {
                this.debug.msgFIFO.shift();
            }

            for (var i = this.debug.msgFIFO.length - 1; i >= 0; i--) {
                displInfo.push(this.debug.msgFIFO[i]);
            }
            displInfo.push("--------------------------------");
            for (i = 0; i < displInfo.length; i++) {
                msgDisplay.innerHTML += displInfo[i] + "<br>";
            }
        }
    }
};

shmi.extend(shmi.visuals.controls.GpScrollBar, shmi.visuals.core.BaseControl);
shmi.registerControlType("gp-scroll-bar", shmi.visuals.controls.GpScrollBar, true);

// $todo Meike begin - ++++++++++++++++
// $todo -> hier ggfs. element names und class names anpassen fuer Vereinheitlichung/ueberarbeitung CSS/Templates der Controls
shmi.visuals.controls.GpScrollBar.uiElName = {};
(function() {
    var __gpsbsc = shmi.visuals.controls.GpScrollBar.uiElName;
    /* The following ui element names are used in CT2's JS code. Change it here if necessary.
     All other ui element names may be changed in the CT2 template(s).
     */
    __gpsbsc.sliderTrack = "gpsb-slider-track";
    __gpsbsc.sliderHandle = "gpsb-slider-handle"; // this is the slider touch area
    __gpsbsc.sliderHandleVisible = "gpsb-slider-handle-visible"; // this is the visible slider button
    __gpsbsc.btnPanel = "gpsb-btn-panel";
    __gpsbsc.btnUp = "gpsb-btn-up";
    __gpsbsc.btnDown = "gpsb-btn-down";

    shmi.visuals.controls.GpScrollBar.uiCssCl = {};
    __gpsbsc = shmi.visuals.controls.GpScrollBar.uiCssCl;
    /* The following ui element class names are used in CT2's JS code. Change it here if necessary.
     All other ui element class names be changed in the CT2 template(s).
     */
    // dynamically switched styles
    __gpsbsc.withBtns = "gpsb-with-btns";
    __gpsbsc.withoutBtns = "gpsb-without-btns";
    __gpsbsc.btnSel = "gpsb-btn-selected";
    __gpsbsc.trackSel = "gpsb-track-selected";
    __gpsbsc.minimized = "gpsb-minimized";
    __gpsbsc.normal = "gpsb-normal";
    // transitions/animations
    __gpsbsc.sliderAni = "gpsb-slider-ani";
    // $todo Meike end - ++++++++++++++++

    shmi.visuals.controls.GpScrollBar.c = {};
    __gpsbsc = shmi.visuals.controls.GpScrollBar.c;
    __gpsbsc.MIN_SLIDER_SIZE_PX = 60;
    __gpsbsc.MIN_SLIDER_SIZE_PX_MINIMIZED = 5;
    __gpsbsc.ENABLE_TO = 500;
    __gpsbsc.START_BTN_AUTO_REPEAT_TO = 1000;
    __gpsbsc.PROCESS_BTN_AUTO_REPEAT_TO = 100;
})();

shmi.pkg("visuals.controls");
/**
 * Creates a new control Group
 *
 * @constructor
 * @extends shmi.visuals.core.BaseControl
 * @param element - base element of the control group
 * @param config - configuration of the control group
 */
shmi.visuals.controls.Group = function(element, config) {
    this.element = element;
    this.config = config || {};

    this.parseAttributes();

    shmi.def(this.config, 'class-name', 'group');
    shmi.def(this.config, 'template', null);
    shmi.def(this.config, 'name', null);
    shmi.def(this.config, 'replacers', {});

    this.initialized = false;
    this.controls = [];

    this.startup();
};

shmi.visuals.controls.Group.prototype = {
    uiType: "group",
    isContainer: true,
    getClassName: function() {
        return "Group";
    },
    /**
     * parses child controls
     */
    onRegister: function(onDone) {
        var self = this;
        self.parseChildren(self.element, onDone);
    },
    /**
     * initializes control
     */
    onInit: function() {},
    /**
     * Enables the Group
     *
     */
    onEnable: function() {
        for (var i = 0; i < this.controls.length; i++) {
            this.controls[i].enable();
        }
        shmi.log("[Group] enabled", 1);
    },
    /**
     * Disables the Group
     *
     */
    onDisable: function() {
        for (var i = 0; i < this.controls.length; i++) {
            this.controls[i].disable();
        }
        shmi.log("[Group] disabled", 1);
    },
    /**
     * Locks the Group
     *
     */
    onLock: function() {
        for (var i = 0; i < this.controls.length; i++) {
            this.controls[i].lock();
        }
        shmi.addClass(this.element, 'locked');
        shmi.log("[Group] locked", 1);
    },
    /**
     * Unlocks the Group
     *
     */
    onUnlock: function() {
        for (var i = 0; i < this.controls.length; i++) {
            this.controls[i].unlock();
        }

        shmi.removeClass(this.element, 'locked');
        shmi.log("[Group] locked", 1);
    },
    /**
     * Sets value to the Group
     *
     * @param value - new value to set
     */
    onSetValue: function(value) {
        shmi.log("[Group] value set: " + value, 0);
    }
};

shmi.extend(shmi.visuals.controls.Group, shmi.visuals.core.BaseControl);

/**
 * WebIQ visuals control template.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "html",
 *     "name": null,
 *     "template": "default/html"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "Html", // control name in camel-case
        uiType = "html", // control keyword (data-ui)
        isContainer = false;

    // example - default configuration
    var defConfig = {
        "class-name": "html",
        "name": null,
        "template": "default/html",
        "use-shadow-dom": false,
        "html": ""
    };

    // declare private functions - START
    /**
     * Parses an html string and creates a fragment which can be attached to
     * another node.
     *
     * @param {string} html HTML string to parse.
     * @returns {DocumentFragment}
     */
    function documentFragmentFromHTML(html) {
        const templateNode = document.createElement("template");

        templateNode.innerHTML = html;

        return templateNode.content.cloneNode(true);
    }
    // declare private functions - END

    // definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            html: null
        },
        /* imports added at runtime */
        imports: {},

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                const self = this,
                    htmlContentNode = shmi.getUiElement("html", self.element);

                if (!htmlContentNode) {
                    console.log('Error: "html" element is missing in template!');
                    return;
                }

                // Get the node to which to append the widgets contents.
                const rootNode = (() => {
                    if (self.config["use-shadow-dom"]) {
                        return htmlContentNode.attachShadow({ mode: "open" });
                    }

                    return htmlContentNode;
                })();

                // Store a reference to the root node to preserve backward
                // compatibility with code that does something it's not
                // supposed to (reading "private" widget state).
                self.vars.html = rootNode;

                // Parse the content and attach it to the root node.
                rootNode.appendChild(documentFragmentFromHTML(self.config.html));
            },

            /**
             * Returns the root of the HTML content.
             *
             * @returns {Node & InnerHTML & ParentNode}
             */
            getRootNode: function() {
                return this.vars.html;
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * WebIQ visuals control template.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iframe",
 *     "name": null,
 *     "template": "default/iframe"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "Iframe", // control name in camel-case
        uiType = "iframe", // control keyword (data-ui)
        isContainer = false;

    // example - default configuration
    var defConfig = {
        "class-name": "iframe",
        "name": null,
        "template": "default/iframe ",
        "src": "about:blank",
        "disable-scrollbar": false
    };

    // declare private functions - START

    // declare private functions - END

    // definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            iframe: null
        },
        /* imports added at runtime */
        imports: {
            /* example - add import via shmi.requires(...) */
            im: "visuals.session.ItemManager",
            /* example - add import via function call */
            qm: function() {
                return shmi.visuals.session.QueryManager;
            }
        },

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;
                self.vars.iframe = self.element.querySelector("iframe");
                if (self.vars.iframe) {
                    self.vars.iframe.onerror = function(e) {
                        shmi.notify("Error loading IFrame: " + e.toString());
                    };
                    self.vars.iframe.setAttribute("src", self.config.src);
                    if (self.config["disable-scrollbar"]) {
                        self.vars.iframe.setAttribute("scrolling", "no");
                        self.vars.iframe.style.overflow = 'hidden';
                    }
                }
            },
            /* called when control is enabled */
            onEnable: function() {

            },
            /* called when control is disabled */
            onDisable: function() {

            },
            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;
                if (self.vars.iframe) {
                    self.vars.iframe.style.pointerEvents = "none";
                }
            },
            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;
                if (self.vars.iframe) {
                    self.vars.iframe.style.pointerEvents = "";
                }
            },
            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {

            },
            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(min, max, step) {

            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * ImageChanger control - displays different images depending on
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "image-changer",
 *     "name": null,
 *     "template": "default/image-changer",
 *     "options": [],
 *     "default-image": "pics/system/icons/placeholder.svg",
 *     "default-title": null
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * options {object[]}: Array of configurable images to display for each value
 * default-image {string}: URL of image to display as default
 * default-title {string}: Default image title
 *
 * @version 1.0.0
 */
(function() {
    'use strict';

    //variables for reference in control definition
    var className = "ImageChanger", //control name in camel-case
        uiType = "image-changer", //control keyword (data-ui)
        isContainer = false;

    //example - default configuration
    var defConfig = {
        "class-name": "image-changer",
        "name": null,
        "template": "default/image-changer",
        "options": [],
        "default-image": "pics/system/icons/placeholder.svg",
        "default-title": null
    };

    //setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    //declare private functions - START

    function getActive(self, value) {
        var opt = self.config.options.find(function(o) {
            return o.value === value;
        });

        if (!opt) {
            opt = {
                "icon-src": self.config["default-image"],
                "label": self.config["default-title"] || null
            };
        }
        self.vars.activeOption = opt;

        self.vars.elements.img.setAttribute("src", opt["icon-src"]);
        self.setTooltip(self.getTooltip());
    }

    //declare private functions - END

    //definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            tokens: [],
            listeners: [],
            value: null,
            activeOption: null,
            elements: {
                img: null
            }
        },
        /* imports added at runtime */
        imports: {
            /* example - add import via shmi.requires(...) */
            im: "visuals.session.ItemManager",
            /* example - add import via function call */
            qm: function() {
                return shmi.visuals.session.QueryManager;
            }
        },
        /* array of custom event types fired by this control */
        events: [],
        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            getTooltip: function() {
                var self = this,
                    superTooltip = shmi.visuals.core.BaseControl.prototype.getTooltip.call(this);

                if (superTooltip) {
                    return superTooltip;
                } else if (self.activeOption && self.activeOption.label) {
                    return self.activeOption.label;
                }

                return null;
            },
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this,
                    elements = self.vars.elements;

                elements.img = shmi.getUiElement("image", self.element);
                getActive(self, null); //initialize default image
            },
            /* called when control is enabled */
            onEnable: function() {
                var self = this,
                    im = shmi.requires("visuals.session.ItemManager");

                if (self.config.item) {
                    self.vars.tokens.push(im.subscribeItem(self.config.item, self));
                }
            },
            /* called when control is disabled */
            onDisable: function() {
                var self = this;
                self.vars.tokens.forEach(function(t) {
                    t.unlisten();
                });
                self.vars.tokens = [];
            },
            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {

            },
            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {

            },
            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                var self = this;
                self.vars.value = value;
                getActive(self, self.vars.value);
            },
            /** Sets min & max values and stepping of subscribed variable **/
            onSetProperties: function(min, max, step) {

            }
        }
    };

    //definition of new control extending BaseControl - END

    //generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * Creates a new Image control.
 *
 * The Image control can be used to execute arbitrary javascript code at
 * predefined places in an hmi-app. The configured script will be run every time
 * the Image control is enabled.
 *
 * @constructor
 * @param {HTMLElement} element root element of control
 * @param {object} config control configuration object
 */
shmi.visuals.controls.Image = function(element, config) {
    this.element = element;
    this.config = config || {};
    this.run = false;
    this.loaded = false;

    shmi.def(this.config, 'image-src', 'pics/system/icons/placeholder.svg');
    shmi.def(this.config, 'image-alt', null);
    shmi.def(this.config, 'image-title', null);
    shmi.def(this.config, 'class-name', 'image');
    shmi.def(this.config, 'template', 'default/image');
    shmi.def(this.config, 'label', "image");

    this.listeners = [];

    this.parseAttributes();

    this.imgElement = null;

    this.startup();
};

shmi.visuals.controls.Image.prototype = {
    uiType: "image",
    isContainer: false,
    events: ["click"],
    tooltipProperties: ["image-title"],
    getClassName: function() {
        return "Image";
    },
    /**
     * loads configured script file
     *
     */
    onInit: function() {
        var self = this,
            io = shmi.visuals.io,
            handler = null,
            ml = null,
            tl = null;

        this.imgElement = shmi.getUiElement("image", this.element);
        this.labelElement = shmi.getUiElement("label", this.element);
        if (this.imgElement) {
            this.imgElement.setAttribute("draggable", "false");
            if (this.config['image-src']) {
                this.imgElement.setAttribute("src", this.config['image-src']);
            }
            if (this.config['image-alt']) {
                this.imgElement.setAttribute("alt", this.config['image-alt']);
            }

            handler = {
                onClick: function(x, y, event) {
                    self.fire("click", {
                        x: x,
                        y: y,
                        event: event
                    });
                }
            };
            ml = new io.MouseListener(self.imgElement, handler);
            tl = new io.TouchListener(self.imgElement, handler);
            self.listeners.push(ml, tl);
        }

        if (this.labelElement) {
            this.labelElement.textContent = shmi.localize(this.config.label);
        }
    },
    /**
     * run when control is enabled. executes configured script.
     *
     */
    onEnable: function() {
        var self = this;
        self.listeners.forEach(function(l) {
            l.enable();
        });
    },
    /**
     * run when control is disabled
     *
     */
    onDisable: function() {
        var self = this;
        self.listeners.forEach(function(l) {
            l.disable();
        });
    },
    onLock: function() {
        var self = this;
        self.listeners.forEach(function(l) {
            l.disable();
        });
        shmi.addClass(self.element, "locked");
    },
    onUnlock: function() {
        var self = this;
        self.listeners.forEach(function(l) {
            l.enable();
        });
        shmi.removeClass(self.element, "locked");
    },
    setLabel: function(labelText) {
        var self = this;
        if (self.labelElement) {
            self.labelElement.textContent = shmi.localize(labelText);
        }
    }
};

shmi.extend(shmi.visuals.controls.Image, shmi.visuals.core.BaseControl);
shmi.registerControlType('image', shmi.visuals.controls.Image, false);

(function() {
    shmi.pkg("visuals.controls");
    /**
     * Creates a new Input Field control
     *
     * @constructor
     * @extends shmi.visuals.core.BaseControl
     * @param element - base element of the control
     * @param config - configuration of the control
     */
    shmi.visuals.controls.InputField = function(element, config) {
        this.element = element;
        this.value = null;
        this.config = config || {};

        this.parseAttributes();

        shmi.def(this.config, 'class-name', 'input-field');
        shmi.def(this.config, 'template', 'default/input-field');
        shmi.def(this.config, 'name', null);
        shmi.def(this.config, 'item', null);
        shmi.def(this.config, 'value-alignment', "left"); //"left" |"right" | "auto"
        shmi.def(this.config, 'numeric-class', 'numeric');
        shmi.def(this.config, 'min', Number.NEGATIVE_INFINITY);
        shmi.def(this.config, 'auto-min', true);
        shmi.def(this.config, 'max', Number.POSITIVE_INFINITY);
        shmi.def(this.config, 'auto-max', true);
        shmi.def(this.config, 'step', 0);
        shmi.def(this.config, 'auto-step', false);
        shmi.def(this.config, 'precision', -1);
        shmi.def(this.config, 'auto-precision', true);
        shmi.def(this.config, 'unit-scale', 1);
        shmi.def(this.config, 'decimal-delimiter', '.');
        shmi.def(this.config, 'label', "input-field");
        shmi.def(this.config, 'auto-label', true);
        shmi.def(this.config, 'unit-text', null);
        shmi.def(this.config, 'auto-unit-text', true);
        shmi.def(this.config, 'type', shmi.c("TYPE_STRING"));
        shmi.def(this.config, 'auto-type', true);
        shmi.def(this.config, 'numpad-enabled', false);
        shmi.def(this.config, 'multiline', false);
        shmi.def(this.config, 'value-as-tooltip', false);

        this.type = shmi.c("TYPE_STRING");

        this.valueElement = null;
        this.unitElement = null;
        this.textareaElement = null;
        this.mouseListener = null;
        this.touchListener = null;

        this.active = false;
        this._subscriptionTargetId = null;
        this._timeout = 0;
        this.vars = {
            to_enter: 0,
            wasClicked: false
        };

        this.startup();
    };

    /**
     * check if element is either an 'INPUT'  or 'TEXTAREA' element
     *
     * @param {HTMLElement} element element to test
     * @returns {boolean} `true` if element is 'INPUT' or 'TEXTAREA'
     */
    function isInputElement(element) {
        return ["INPUT", "TEXTAREA"].includes(element.tagName);
    }

    function insertValue(self, value) {
        if (isInputElement(self.valueElement)) {
            self.valueElement.value = "";
            self.valueElement.value = value;
        } else {
            self.valueElement.textContent = value;
            if (!self.valueElement.firstChild) {
                self.valueElement.appendChild(document.createTextNode(""));
            }
        }

        if (self.config["value-as-tooltip"]) {
            if (value !== null && typeof value !== "undefined" && value !== "") {
                self.setTooltip(String(value));
            } else {
                self.setTooltip(self.getTooltip());
            }
        }
    }

    function retrieveValue(self) {
        if (isInputElement(self.valueElement)) {
            return self.valueElement.value;
        } else {
            return self.valueElement.textContent;
        }
    }

    function selectContent(self) {
        if (isInputElement(self.valueElement)) {
            self.valueElement.select();
        } else {
            shmi.addClass(self.valueElement, "selectableText");
            window.getSelection().removeAllRanges();
            if (self.valueElement.firstChild && self.valueElement.lastChild && self.valueElement.firstChild instanceof Node && self.valueElement.lastChild instanceof Node) {
                const range = document.createRange();
                range.setStart(self.valueElement.firstChild, 0);
                range.setEnd(self.valueElement.lastChild, self.valueElement.lastChild.length);
                window.getSelection().addRange(range);
            }
        }
    }

    /**
     * handleFocus - handle focus of input element when control is clicked or tabbed into
     *
     * @param  {object} self control instance reference
     * @return {undefined}
     */
    function handleFocus(self) {
        /* prevent focussing if control is locked */
        if (self.locked || self.focused) {
            return;
        }

        if ((!self.focused) && (shmi.visuals.session.FocusElement !== null)) {
            shmi.visuals.session.FocusElement.blur();
            shmi.visuals.session.FocusElement = null;
        }
        shmi.log("[InputField] focused", 1);
        if (self.vars.wasClicked && showNumpad(self)) {
            return;
        } else if (self.vars.wasClicked && showKeyboard(self)) {
            return;
        }

        selectContent(self);

        self.focused = true;
        self.valueElement.focus();
        shmi.visuals.session.FocusElement = self.valueElement;

        clearTimeout(self.vars.to_enter);
        self.vars.to_enter = setTimeout(function() {
            self.fire("enter", {
                value: self.value
            });
        }, 250);
    }

    /**
     * @param {object} self
     * @return {boolean} if numpad will be shown
     */
    function showNumpad(self) {
        if (!self.config["numpad-enabled"]) { //does nothing if not enabled
            return false;
        }

        var nv = shmi.requires("visuals.tools.numericValues"),
            vs = null,
            params = null,
            im = shmi.visuals.session.ItemManager;

        if (!(self.vars && self.vars.valueSettings)) {
            nv.initValueSettings(self);
        }

        params = {
            "decimal-delimiter": self.config["decimal-delimiter"],
            "unit": (self.vars.unit !== undefined) ? self.vars.unit : self.config["unit-text"],
            "label": (self.vars.label !== undefined) ? self.vars.label : self.config.label,
            "value": retrieveValue(self),
            "callback": function(res) {
                if (self.config.item) {
                    if (self.config.multiline) res = res.toString();
                    im.writeValue(self.config.item, res);
                } else {
                    self.setValue(res);
                }
            }
        };

        vs = self.vars.valueSettings;
        params.min = vs.min;
        params.max = vs.max;
        params.type = vs.type;
        params.precision = vs.precision;

        shmi.numpad(params);
        return true;
    }

    /**
    * @param {object} self
    * @return {boolean} if keyboard will be shown
    */
    function showKeyboard(self) {
        var appConfig = shmi.requires("visuals.session.config"),
            keyboardEnabled = (appConfig.keyboard && appConfig.keyboard.enabled); // get the keyboard config from `project.json`
        if (!keyboardEnabled) { //does nothing if not enabled
            return false;
        }

        var im = shmi.visuals.session.ItemManager,
            params = {
                "value": retrieveValue(self),
                "select-box-enabled": appConfig.keyboard["language-selection"],
                "password-input": self.valueElement && self.valueElement.type && self.valueElement.type.toLowerCase() === "password",
                "show-enter": self.config.multiline,
                "callback": function(success, input) {
                    if (success) {
                        if (self.config.item) {
                            im.writeValue(self.config.item, input);
                        } else {
                            self.setValue(input);
                        }
                    }
                }
            };

        shmi.keyboard(params);
        return true;
    }

    shmi.visuals.controls.InputField.prototype = {
        uiType: "input-field",
        events: ["change", "enter"],
        getClassName: function() {
            return "InputField";
        },
        /**
         * Initializes control
         *
         * @returns {unresolved}
         */
        onInit: function() {
            var self = this,
                c = shmi.Constants;

            this.valueElement = shmi.getUiElement('input-field-value', this.element);
            if (!this.valueElement) {
                shmi.log('[InputField] no input-field-value element provided', 3);
                return;
            }
            this.unitElement = shmi.getUiElement('input-field-unit', this.element);
            if (!this.unitElement) {
                shmi.log('[InputField] no input-field-unit element provided', 1);
            }

            this.labelElement = shmi.getUiElement('input-field-label', this.element);
            if (!this.labelElement) {
                shmi.log('[InputField] no input-field-label element provided', 1);
            } else if (this.config.label !== undefined) {
                self.vars = self.vars || {};
                self.vars.label = this.config.label;
                this.labelElement.textContent = shmi.localize(this.config.label);
            }
            if (this.config.multiline) {
                this.valueBox = this.element.getElementsByClassName('value-box')[0];
                if (!this.valueBox) {
                    shmi.log('[InputField] no value-box element provided', 3);
                    return;
                }
                this.valueElement.remove();
                var el = document.createElement("textarea");
                el.setAttribute('data-ui', 'input-field-value');
                el.disabled = true;
                this.textareaElement = el;
                shmi.addClass(el, 'input-field-value');
                this.valueElement = this.valueBox.appendChild(el);
                shmi.addClass(self.element, 'textarea');
            } else {
                this.textareaElement = null;
            }
            /* all required elements found */
            this.valueElement.setAttribute('tabindex', '0');
            insertValue(self, "");

            var type = parseInt(this.config.type);
            if (!isNaN(type)) {
                this.type = type;
            }

            if (this.config.action) {
                this.action = new shmi.visuals.core.UiAction(this.config.action, this);
            }

            if (!self.config['auto-type'] && (self.config['value-alignment'] === "auto")) {
                if ([c.TYPE_BOOL, c.TYPE_INT, c.TYPE_FLOAT].indexOf(self.type) !== -1) {
                    shmi.addClass(self.element, self.config['numeric-class']);
                }
            } else if (self.config['value-alignment'] === "right") {
                shmi.addClass(self.element, self.config['numeric-class']);
            }

            var ioFuncs = {};
            ioFuncs.onPress = function(sx, sy, event) {
                if (!isInputElement(this.valueElement) && (!this.focused)) {
                    event.preventDefault();
                } else if (this.focused) {
                    window.getSelection().removeAllRanges();
                }
            }.bind(this);

            ioFuncs.onClick = function() {
                self.vars.wasClicked = true;
                handleFocus(self);
                self.vars.wasClicked = false;
            };
            this.mouseListener = new shmi.visuals.io.MouseListener(this.valueElement, ioFuncs);
            this.touchListener = new shmi.visuals.io.TouchListener(this.valueElement, ioFuncs, true);
            this.valueElement.addEventListener('focus', function() {
                if (self.config.multiline) {
                    self.focused = true;
                } else if (!self.vars.wasClicked) {
                    handleFocus(self);
                }
            });
            this.valueElement.addEventListener('keypress', function(event) {
                if (((event.keyCode === 13) && (!self.config.multiline)) || (event.keyCode === 9)) {
                    event.preventDefault();
                    window.getSelection().removeAllRanges();
                    this.valueElement.blur();
                }
            }.bind(this));

            this.valueElement.addEventListener('blur', function() {
                if (!this.focused) {
                    return;
                }
                shmi.log("[InputField] blur event", 1);
                window.getSelection().removeAllRanges();

                shmi.removeClass(this.valueElement, 'selectableText');
                this.validate(this.valueElement);
                this.focused = false;
                shmi.visuals.session.FocusElement = null;
            }.bind(this));
            // "validation lite" for the delimeters.
            this.valueElement.addEventListener('keydown', function(evt) {
                if (self.type === shmi.c("TYPE_FLOAT")) {
                    var wrongDelimeter = (self.config['decimal-delimiter'] === ".") ? "," : ".";
                    if (evt.key === wrongDelimeter) {
                        evt.preventDefault();
                    }
                }
            });

            if (!isInputElement(this.valueElement)) {
                this.valueElement.addEventListener("paste", async (evt) => {
                    evt.preventDefault();

                    try {
                        const text = await navigator.clipboard.readText(),
                            element = self.valueElement,
                            selection = window.getSelection();

                        if (selection.rangeCount) {
                            const range = selection.getRangeAt(0),
                                startOffset = range.startOffset;

                            if (range.commonAncestorContainer.parentNode === element) {
                                const value = element.textContent,
                                    start = value.substring(0, range.startOffset),
                                    end = value.substring(range.endOffset, value.length);
                                element.textContent = `${start}${text}${end}`;
                            } else if (range.commonAncestorContainer === element) {
                                element.textContent = text;
                            }

                            if (element.firstChild) {
                                const offset = startOffset + text.length;
                                range.setEnd(element.firstChild, offset);
                                range.setStart(element.firstChild, offset);
                            }
                        }
                    } catch (err) {
                        console.error("[InputField] error pasting from clipboard:", err);
                    }
                });
            }

            if (this.config['unit-text']) {
                self.vars = self.vars || {};
                self.vars.unit = this.config['unit-text'];
                if (this.unitElement) this.unitElement.textContent = shmi.localize(this.config['unit-text']);
            }
            if (this.config['unit-scale']) {
                this.config['unit-scale'] = parseFloat(shmi.localize(this.config['unit-scale']));
            }
        },
        /**
         * Validates the content of the specified element
         *
         * @param element - element to validate
         */
        validate: function(element) {
            const self = this,
                im = shmi.visuals.session.ItemManager,
                nv = shmi.requires("visuals.tools.numericValues"),
                inputString = retrieveValue(self),
                oldValue = self.value;

            if (inputString === oldValue) {
                return;
            }

            if (!self.config['auto-type']) {
                // No automatic type inference.
            } else if (self.config.multiline) {
                // Multi-line is always string.
                self.type = shmi.c("TYPE_STRING");
            } else if (self.config.item) {
                const item = im.getItem(self.config.item);
                if (item) {
                    self.type = item.type;
                }
            }

            const newValue = (() => {
                if (self.config.multiline) {
                    // Multi-line is always string, therefore no check and
                    // formatting is performed on it.
                    return inputString;
                } else if (![shmi.c("TYPE_INT"), shmi.c("TYPE_FLOAT")].includes(self.type)) {
                    return inputString;
                } else if (!self.floatRegexp.test(inputString.replace(self.config['decimal-delimiter'], '.'))) {
                    return oldValue;
                }

                return nv.applyInputSettings(inputString, self);
            })();

            self.value = newValue;
            if (typeof newValue === "undefined" || newValue === null) {
                insertValue(self, "");
            } else {
                insertValue(self, nv.formatOutput(newValue, self));
            }

            // Emit change event, write item and call UI actions only when something has changed
            if (newValue !== oldValue) {
                if (self.config.item) {
                    im.writeValue(self.config.item, newValue);
                }

                // UI Action
                if (self.action) {
                    self.action.execute(newValue);
                }

                self.fire("change", {
                    value: self.value
                });

                if (self.onChange) {
                    self.onChange(self.value);
                }
            }
        },
        /**
         * Sets min & max values and stepping of subscribed variable
         *
         * @param min - min value
         * @param max - max value
         * @param step - value stepping
         */
        onSetProperties: function(min, max, step, name, type) {
            var self = this,
                c = shmi.Constants,
                nv = shmi.requires("visuals.tools.numericValues");

            if (self.config['auto-type'] && (self.config['value-alignment'] === "auto")) {
                if ([c.TYPE_BOOL, c.TYPE_INT, c.TYPE_FLOAT].indexOf(type)) {
                    shmi.addClass(self.element, self.config['numeric-class']);
                }
            }

            nv.setProperties(self, arguments);

            shmi.log("[InputField] min: " + min + " max: " + max + " step: " + step, 1);
        },
        /**
         * Sets the current value
         *
         * @param value - new value to set
         */
        onSetValue: function(value) {
            var self = this,
                im = shmi.visuals.session.ItemManager,
                nv = shmi.requires("visuals.tools.numericValues"),
                item = null,
                oldValue = self.value;

            if (self.config.item && self.config['auto-type']) {
                item = im.getItem(self.config.item);
                if (item) {
                    self.type = item.type;
                } else if (shmi.visuals.session.config.debug) {
                    console.debug(self.getClassName(), "configured item not available:", self.config.item);
                }
            }

            self.value = value;

            insertValue(self, nv.formatOutput(self.value, self));
            if (document.activeElement === self.valueElement) {
                selectContent(self);
            }
            shmi.log("[InputField] new value: " + this.value, 0);

            if (this.value !== oldValue) {
                this.fire("change", {
                    value: this.value
                });

                if (this.onChange) {
                    this.onChange(this.value);
                }
            }
        },
        /**
         * Retrieves to current value
         *
         * @return value
         */
        getValue: function() {
            var type = this.type;
            if (this.config.item && this.config['auto-type']) {
                type = shmi.visuals.session.ItemManager.items[this.config.item].type;
            }
            if (type === shmi.c("TYPE_INT")) {
                return Math.round(this.value / this.config['unit-scale']);
            } else if (type === shmi.c("TYPE_FLOAT")) {
                return (this.value / this.config['unit-scale']);
            } else {
                return this.value;
            }
        },
        /**
         * Enables the InputField
         *
         */
        onEnable: function() {
            this.valueElement.setAttribute('contenteditable', true);
            if (!this.config.multiline) {
                this.mouseListener.enable();
                this.touchListener.enable();
            }
            if (this.config.item) {
                this._subscriptionTargetId = shmi.visuals.session.ItemManager.subscribeItem(this.config.item, this);
            }
            if (this.textareaElement) this.textareaElement.disabled = false;
            shmi.log("[InputField] enabled", 1);
        },
        /**
         * Disables the InputField
         *
         */
        onDisable: function() {
            this.valueElement.setAttribute('contenteditable', false);
            shmi.removeClass(this.valueElement, 'selectableText');
            if (this.config.item) {
                shmi.visuals.session.ItemManager.unsubscribeItem(this.config.item, this._subscriptionTargetId);
            }
            if (!this.config.multiline) {
                this.mouseListener.disable();
                this.touchListener.disable();
            }
            shmi.log("[InputField] disabled", 1);
        },
        /**
         * Lockes the InputField
         *
         */
        onLock: function() {
            this.locked = true;
            shmi.log("[InputField] locked", 1);
            if (isInputElement(this.valueElement)) {
                this.valueElement.setAttribute('disabled', true);
            } else {
                this.valueElement.setAttribute('contenteditable', false);
            }
            shmi.removeClass(this.valueElement, 'selectableText');
            if (!this.config.multiline) {
                this.mouseListener.disable();
                this.touchListener.disable();
            }
            shmi.addClass(this.element, 'locked');
        },
        /**
         * Unlockes the InputField
         *
         */
        onUnlock: function() {
            this.locked = false;
            shmi.log("[InputField] unlocked", 1);
            if (isInputElement(this.valueElement)) {
                this.valueElement.removeAttribute('disabled');
            } else {
                this.valueElement.setAttribute('contenteditable', true);
            }
            if (!this.config.multiline) {
                this.mouseListener.enable();
                this.touchListener.enable();
            }
            shmi.removeClass(this.element, 'locked');
        },
        setLabel: function(labelText) {
            var self = this;
            if (self.config['auto-label'] && self.labelElement) {
                self.vars = self.vars || {};
                self.vars.label = labelText;
                self.labelElement.textContent = shmi.localize(labelText);
            }
        },
        setUnitText: function(unitText) {
            var self = this;
            if (self.config['auto-unit-text'] && self.unitElement) {
                self.vars = self.vars || {};
                self.vars.unit = unitText;
                self.unitElement.textContent = shmi.localize(unitText);
            }
        },
        floatRegexp: /(^[+-]?[0-9]([.][0-9]*)?$|^[+-]?[1-9]+[0-9]*([.][0-9]*)?$)/,
        intRegexp: /(^[+-]?[0-9]$|^[+-]?[1-9]+[0-9]*$)/
    };

    shmi.extend(shmi.visuals.controls.InputField, shmi.visuals.core.BaseControl);
}());

/**
 * Control IQ Alarm-Info
 *
 * Configuration options (default):
 *
 * {
        "class-name": uiType,
        "name": null,
        "template": "default/iq-alarm-info",
        "noAlarm": null,
        "action": null,
        "enableCycle": true,
        "cycleInterval": 1000,
        "showAlarm": true,
        "showWarn": true,
        "showInfo": true,
        "groupFilter": null,
        "icon-info": "pics/system/controls/iq-alarm-info/icon-info.svg",
        "icon-warning": "pics/system/controls/iq-alarm-info/icon-warning.svg",
        "icon-alarm": "pics/system/controls/iq-alarm-info/icon-alarm.svg"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 *
 */
(function() {
    'use strict';

    //variables for reference in control definition
    const className = "IQAlarmInfo", //control name in camel-case
        uiType = "iq-alarm-info", //control keyword (data-ui)
        isContainer = false;

    //example - default configuration
    const defConfig = {
        "class-name": uiType,
        "name": null,
        "template": "default/iq-alarm-info",
        "noAlarm": null,
        "action": null,
        "enableCycle": true,
        "cycleInterval": 1000,
        "showAlarm": true,
        "showWarn": true,
        "showInfo": true,
        "groupFilter": null,
        "icon-info": "pics/system/controls/iq-alarm-info/icon-info.svg",
        "icon-warning": "pics/system/controls/iq-alarm-info/icon-warning.svg",
        "icon-alarm": "pics/system/controls/iq-alarm-info/icon-alarm.svg",
        "icon-idle": null
    };

    //setup module-logger
    const ENABLE_LOGGING = true,
        RECORD_LOG = false,
        logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG),
        log = logger.log;

    // Constants and private fields
    const ALARM_STATE_SELECTORS = ["severity-info", "severity-warning", "severity-alarm"];

    //declare private functions - START

    /**
     * parseAlarmGroupRange - parse range of alarm groups
     *
     * @param {string} rangeText input text (e.g. `3-7`)
     * @returns {number[2]|null} range start & end tuple or `null` if none could be parsed
     */
    function parseAlarmGroupRange(rangeText) {
        const tuple = rangeText.split("-");

        if (tuple.length === 2) {
            const start = parseInt(tuple[0]),
                end = parseInt(tuple[1]);

            if (!(isNaN(start) || isNaN(end)) && start >= 0 && end >= start) {
                return [start, end];
            }
        }

        return null;
    }

    /**
     * parseAlarmGroupFilter - parse group ID filter from config string
     *
     * @param {string} groupFilter comma separated group ID filter
     * @returns {array} array of group IDs and ranges
     */
    function parseAlarmGroupFilter(groupFilter) {
        if (typeof groupFilter !== "string") {
            return null;
        }

        let entries = groupFilter.split(",");
        entries = entries.map((entry) => {
            if (entry.includes("-")) {
                return parseAlarmGroupRange(entry);
            }

            const groupId = parseInt(entry);
            if (!isNaN(groupId) && groupId >= 0) {
                return groupId;
            }

            return null;
        });
        entries = entries.filter((entry) => entry !== null);
        const rangeEntries = [];
        entries.forEach((entry) => {
            if (Array.isArray(entry)) {
                let idx = entry[0];
                while (idx <= entry[1]) {
                    if (!(rangeEntries.includes(idx) && entries.includes(idx))) {
                        rangeEntries.push(idx);
                    }
                    idx += 1;
                }
            }
        });
        entries.push(...rangeEntries);

        return entries.length ? entries : null;
    }

    /**
     * testAlarmState - test & update state of displayed alarm
     *
     * @param {object} self instance reference
     */
    function testAlarmState(self) {
        const alarmArray = Object.values(self.imports.am.alarms).map((value) => value.properties);
        if (Array.isArray(alarmArray)) {
            if (alarmArray.length < 1) {
                clearAlarmMsg(self);
            } else {
                clearInt(self);
                if (!self.config.enableCycle) {
                    // standard mode: lookup for last Alarm of highest class
                    self.vars.activeAlarmList = getLastHighest(self, alarmArray);
                } else {
                    // cycleMode: cycle display of all active alarms
                    self.vars.activeAlarmList = getEnabledAlarms(self, alarmArray);
                    self.vars.alarmCount = self.vars.activeAlarmList.length;
                    if (self.vars.alarmCount > 1) {
                        self.vars.interval = setInterval(displayAlarms.bind(null, self), self.config.cycleInterval);
                    }
                }
                displayAlarms(self);
            }
        }
    }

    // helper: displays alarms in activeAlarmList
    function displayAlarms(self) {
        if (self.vars.activeAlarmList.length === 0) {
            clearAlarmMsg(self);
        }
        if (self.vars.currentAlarm >= self.vars.activeAlarmList.length) {
            self.vars.currentAlarm = 0;
        }
        const alarm = self.vars.activeAlarmList[self.vars.currentAlarm++];
        let msg = null,
            details = "";
        if (alarm) { // evaluate context items etc.
            msg = shmi.evalString(shmi.localize(`\${alarm_title_${alarm.index}}`), alarm);
            details = shmi.evalString(shmi.localize(`\${alarm_msg_${alarm.index}}`), alarm);
        } else {
            msg = shmi.localize(self.config.noAlarm);
        }
        setAlarmStyle(alarm ? alarm.severity : -1, self);
        setTextContent(self, "alarmMessage", msg);
        setTextContent(self, "alarmDetails", details);
        setTextContent(self, "alarmCount", self.vars.alarmCount);
    }

    // helper: get alarm object of the last Alarm and highest class (=severity)
    function getLastHighest(self, alarmList) {
        let lastAlarm = null;
        const enabledAlarms = getEnabledAlarms(self, alarmList);
        self.vars.alarmCount = enabledAlarms.length;
        enabledAlarms.forEach(function(alarmInfo) {
            // Find latest alarm with the highest occurring severity
            if (lastAlarm === null) {
                lastAlarm = alarmInfo;
            } else if (alarmInfo.severity >= lastAlarm.severity && alarmInfo.timestamp_in > lastAlarm.timestamp_in) {
                lastAlarm = alarmInfo;
            }
        });

        return lastAlarm === null ? [] : [lastAlarm];
    }

    // helper: select alarms to be displayed dependig on the config.notShowWarn and config.notShowInfo parameters
    function getEnabledAlarms(self, alarmList) {
        const enabledAlarms = alarmList.filter(function(alarmInfo) {
            if (Array.isArray(self.vars.groupFilter)) {
                if (!self.vars.groupFilter.includes(alarmInfo.group)) {
                    return false;
                }
            }

            //filter inactive alarms
            if (!alarmInfo.active && (!alarmInfo.acknowledgeable || alarmInfo.acknowledged)) {
                return false;
            }

            // Filter alarms based on control configuration
            return (alarmInfo.severity === 2 && self.config.showAlarm) ||
                (alarmInfo.severity === 1 && self.config.showWarn) ||
                (alarmInfo.severity === 0 && self.config.showInfo);
        });

        return enabledAlarms;
    }

    // helper: clears the alarm message
    function clearAlarmMsg(self) {
        clearInt(self);
        self.vars.currentAlarm = 0;
        setAlarmStyle(-1, self);
        setTextContent(self, "alarmMessage", shmi.localize(self.config.noAlarm));
    }

    // helper: clears the interval
    function clearInt(self) {
        if (self.vars.interval) {
            clearInterval(self.vars.interval);
            self.vars.interval = null;
        }
    }

    /**
     * setAlarmIcon - set active alarm state icon, hide icon if none set
     *
     * @param {object} self control instance
     * @param {string} optionName icon option name
     */
    function setAlarmIcon(self, optionName) {
        if (self.vars.elements.stateIcon) {
            if (self.config[optionName]) {
                self.vars.elements.stateIcon.style.visibility = "";
                self.vars.elements.stateIcon.style.backgroundImage = `url(${self.config[optionName]})`;
            } else {
                self.vars.elements.stateIcon.style.visibility = "hidden";
                self.vars.elements.stateIcon.style.backgroundImage = "";
            }
        }
    }
    // helper: sets alarm class (state)
    function setAlarmStyle(state, self) {
        const wrapper = self.vars.elements.wrapper;

        switch (state) {
        case 0:
            setAlarmIcon(self, "icon-info");
            break;
        case 1:
            setAlarmIcon(self, "icon-warning");
            break;
        case 2:
            setAlarmIcon(self, "icon-alarm");
            break;
        default:
            setAlarmIcon(self, "icon-idle");
        }
        if (wrapper) {
            shmi.removeClass(wrapper, ALARM_STATE_SELECTORS.join(" "));
            if (ALARM_STATE_SELECTORS[state]) {
                shmi.addClass(wrapper, ALARM_STATE_SELECTORS[state]);
            }
        }
    }

    /**
     * setTextContent - set text content of specified element
     *
     * @param {object} self control instance
     * @param {string} elementName name of element
     * @param {string} text text to set
     */
    function setTextContent(self, elementName, text) {
        if (self.vars.elements[elementName]) {
            self.vars.elements[elementName].textContent = text;
        }
    }
    //declare private functions - END

    //definition of new control extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            elements: {
                stateIcon: null,
                alarmCount: null,
                alarmMessage: null,
                button: null,
                wrapper: null
            },
            alarmCount: 0,
            listeners: [],
            tokens: [],
            activeAlarmList: [],
            currentAlarm: 0,
            interval: 0,
            alarmSubscriber: null,
            groupFilter: null
        },
        /* imports added at runtime */
        imports: {
            /* example - add import via function call */
            am: "visuals.session.AlarmManager"
        },
        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                const self = this;

                self.vars.groupFilter = parseAlarmGroupFilter(self.config.groupFilter);

                self.vars.elements.alarmCount = shmi.getUiElement('alarm-count', self.element);
                if (!self.vars.elements.alarmCount) {
                    log(uiType, "alarm-count not found in template");
                }

                self.vars.elements.wrapper = shmi.getUiElement('alarm-info-wrapper', self.element);
                if (!self.vars.elements.wrapper) {
                    log(uiType, "alarm-info-wrapper not found in template");
                }

                self.vars.elements.stateIcon = shmi.getUiElement('state-icon', self.element);
                if (!self.vars.elements.stateIcon) {
                    log(uiType, "state-icon not found in template");
                }

                self.vars.elements.alarmMessage = shmi.getUiElement('alarm-message', self.element);
                if (!self.vars.elements.alarmMessage) {
                    log(uiType, "alarm-message not found in template");
                } else {
                    setTextContent(self, "alarmMessage", shmi.localize(self.config.noAlarm));
                }

                self.vars.elements.alarmDetails = shmi.getUiElement('alarm-details', self.element);
                if (!self.vars.elements.alarmDetails) {
                    log(uiType, "alarm-details not found in template");
                } else {
                    setTextContent(self, "alarmDetails", "");
                }

                self.vars.elements.button = shmi.getUiElement('alert-list-button', self.element);
                if (self.vars.elements.button && Array.isArray(self.config.action) && self.config.action.length) {
                    const core = shmi.requires("visuals.core"),
                        io = shmi.requires("visuals.io"),
                        action = new core.UiAction(self.config.action),
                        handler = {
                            onClick: () => action.execute()
                        },
                        ml = new io.MouseListener(self.vars.elements.button, handler), //MouseListener
                        tl = new io.TouchListener(self.vars.elements.button, handler); //TouchListener

                    shmi.addClass(self.element, "has-action");
                    self.vars.listeners.push(ml, tl);
                } else if (!self.vars.elements.button) {
                    log(uiType, "click element not found in template");
                }

                setAlarmStyle(-1, self);
            },
            /* called when control is enabled */
            onEnable: function() {
                const self = this;

                self.vars.stateIcons = shmi.getUiElements('state', self.element) || [];
                self.vars.alarmSubscriber = self.imports.am.subscribeAlarms(self, (alarm, isLast) => {
                    if (isLast) {
                        testAlarmState(self);
                    }
                });
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });
            },
            /* called when control is disabled */
            onDisable: function() {
                const self = this;

                self.vars.tokens.forEach(function(t) {
                    t.unlisten();
                });
                self.vars.tokens = [];

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
                self.imports.am.unsubscribeAlarms(self.vars.alarmSubscriber);
                clearInt(self);
            },
            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                const self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
                shmi.addClass(self.element, "locked");
            },
            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                const self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });
                shmi.removeClass(self.element, "locked");
            },
            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {

            },
            /** Sets min & max values and stepping of subscribed variable **/
            onSetProperties: function(min, max, step) {

            }
        }
    };

    //definition of new control extending BaseControl - END

    //generate control constructor & prototype using the control-generator tool
    shmi.requires("visuals.tools.control-generator").generate(definition);
})();

/**
 * iq-alarm-list
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-alarm-list",
 *     "name": null,
 *     "template": "default/iq-alarm-list"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 *
 * @version 1.1
 */

/*eslint-env node */
(function() {
    'use strict';

    // variables for reference in control definition
    const className = "iq-alarm-list", // control name in camel-case
        uiType = "iq-alarm-list", // control keyword (data-ui)
        isContainer = true;

    // default configuration
    const defConfig = {
        "class-name": "iq-alarm-list",
        "name": null,
        "template": "default/iq-alarm-list.variant-01",
        "label": '[Label]',
        "show-text": false,
        "group-filter": null,
        "alarm-image": null,
        "warn-image": null,
        "notify-image": null,
        "numberOfColumns": 7,
        "col1-selectBox": null,
        "col1-width": null,
        "col2-selectBox": null,
        "col2-width": null,
        "col3-selectBox": null,
        "col3-width": null,
        "col4-selectBox": null,
        "col4-width": null,
        "col5-selectBox": null,
        "col5-width": null,
        "col6-selectBox": null,
        "col6-width": null,
        "col7-selectBox": null,
        "col7-width": null,
        "dateformat": "${alarmlist_dateformat}",
        "allow-mode-change": false,
        "allow-group-filter-change": false,
        "display-mode": "all",
        "disable-commit": false,
        "detail-template": null,
        "comment-mode": "disabled"
    };

    // setup module-logger
    const ENABLE_LOGGING = true,
        RECORD_LOG = false;
    const logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    const fLog = logger.fLog,
        log = logger.log;

    //declare private functions - START

    /**
     * parseAlarmGroupRange - parse range of alarm groups
     *
     * @param {string} rangeText input text (e.g. `3-7`)
     * @returns {number[2]|null} range start & end tuple or `null` if none could be parsed
     */
    function parseAlarmGroupRange(rangeText) {
        const tuple = rangeText.split("-");

        if (tuple.length === 2) {
            const start = parseInt(tuple[0]),
                end = parseInt(tuple[1]);

            if (!(isNaN(start) || isNaN(end)) && start >= 0 && end >= start) {
                return [start, end];
            }
        }

        return null;
    }

    /**
     * parseAlarmGroupFilter - parse group ID filter from config string
     *
     * @param {string} groupFilter comma separated group ID filter
     * @returns {array} array of group IDs and ranges
     */
    function parseAlarmGroupFilter(groupFilter) {
        if (typeof groupFilter !== "string") {
            return null;
        }

        const entries = groupFilter.split(",").map((entry) => {
            if (entry.includes("-")) {
                return parseAlarmGroupRange(entry);
            }

            const groupId = parseInt(entry);
            if (!isNaN(groupId) && groupId >= 0) {
                return groupId;
            }

            return null;
        });

        const out = [];
        entries.forEach((entry) => {
            if (Array.isArray(entry)) {
                let idx = entry[0];
                while (idx <= entry[1]) {
                    if (!(out.includes(idx) && entries.includes(idx))) {
                        out.push(idx);
                    }
                    idx += 1;
                }
            } else if (typeof entry === "number") {
                out.push(entry);
            }
        });

        return out.length ? out : null;
    }

    /**
     * Observes startup-sequence of control
     *
     * @param {object} self Control-Obj
     * @returns {bool} Startup succeeded or not
     */
    function controlStartUp(self) {
        let anchorsFound = false,
            controlsCreated = false;
        const io = shmi.requires("visuals.io");

        self.vars.timeFilterLabel = shmi.getUiElement("filter-label", self.element);
        self.vars.activeFilterLabel = shmi.getUiElement("time-frame-label", self.element);
        self.vars.activeFilterFrom = shmi.getUiElement("start-date", self.element);
        self.vars.activeFilterTo = shmi.getUiElement("end-date", self.element);
        self.vars.resetButton = shmi.getUiElement("reset-button", self.element);
        self.vars.detailsRight = shmi.getUiElement("alarmListDetails", self.element);

        if (!self.vars.timeFilterLabel || !self.vars.activeFilterLabel || !self.vars.activeFilterFrom || !self.vars.activeFilterTo || !self.vars.resetButton) {
            return false;
        }

        self.vars.timeFilterLabel.textContent = shmi.localize("${alarmlist_set_timefilter}");

        const resetButtonHandler = {
            onClick: function() {
                clearFilter(self);
            }
        };
        self.vars.listeners.push(
            new io.MouseListener(self.vars.resetButton, resetButtonHandler),
            new io.TouchListener(self.vars.resetButton, resetButtonHandler)
        );

        self.vars.configs.alarmTableConfig["default-field-control-map"]["come"].config["display-format"] = self.config.dateformat;
        initAlarmTableGrid(self);

        anchorsFound = gatherAnchors(self);
        if (anchorsFound) controlsCreated = createChildControls(self);

        if (!self.config['allow-mode-change']) {
            self.vars.anchorEls.modeSelectBox.style.display = "none";
        }

        if (!self.config['allow-group-filter-change'] && self.vars.anchorEls.filterGroupSelectBox) {
            self.vars.anchorEls.filterGroupSelectBox.style.display = "none";
        }

        if (self.config["disable-commit"]) {
            shmi.addClass(self.vars.controls.confirmSelAlarmButton.element, "hidden locked");
            shmi.addClass(self.vars.controls.confirmAllAlarmsButton.element, "hidden locked");
        }

        if (!self.config["comment-mode"] || self.config["comment-mode"] === "disabled") {
            shmi.addClass(self.vars.controls.createThreadButton.element, "hidden locked");
        }

        const groupFilter = parseAlarmGroupFilter(self.config["group-filter"]);
        self.vars.groupFilter = groupFilter;
        if (groupFilter) {
            self.vars.grid.setFilter(7, groupFilter);
        }

        return !(!anchorsFound || !controlsCreated);
    }

    /**
     * generates a random id
     *
     * @returns {string}
     */
    function getRandomId() {
        return Math.random().toString(36).substr(2, 9);
    }

    /**
     * Manages to set or remove Filter on datagrid
     *
     * @param {object} self Control-Obj
     */
    function updateTableFilter(self) {
        const filterSet = self.vars.filterSet,
            valAr = [];

        if (filterSet.error) valAr.push(2);
        if (filterSet.warn) valAr.push(1);
        if (filterSet.advise) valAr.push(0);

        self.vars.controls.alarmList.resetScroll();
        if (valAr.length > 0) {
            self.vars.grid.setFilter(3, valAr);
        } else {
            self.vars.grid.clearFilter(3);
        }
    }

    /**
     * Checks if needed grid already exists, if not grid will be created
     *
     * @param {object} self Control-Obj
     */
    function initAlarmTableGrid(self) {
        do {
            self.vars.dataGridId = "iqalarms-" + getRandomId();
        } while (self.imports.dgm.getGrid(self.vars.dataGridId) !== null);

        self.imports.dgm.grids[self.vars.dataGridId] = new shmi.visuals.core.DataGridIQAlarms(self.config["display-mode"], self.vars.dataGridId);
        self.vars.grid = self.imports.dgm.getGrid(self.vars.dataGridId);
    }

    /**
     * toggle The filter set between true and false
     *
     * @param {object} self Control-Obj
     * @param {string} type severity - 'error' or 'warn' or 'advise'
     */
    function toggleFilterSet(self, type) {
        const filterSet = self.vars.filterSet;
        filterSet[type] = !filterSet[type];
    }

    /**
     * Sets needed listener for all childcontrols etc.
     *
     * @param {object} self Control-Obj
     * @returns {bool}
     */
    function setListener(self) {
        const { controls, tokens: listeners } = self.vars,
            { showAlarmDetails, openCreateThreadDialogForAlarmInfo } = shmi.requires("visuals.tools.alarms.ls.alarmDetailsIQ");

        if (!controls.confirmSelAlarmButton || !controls.confirmAllAlarmsButton || !controls.createThreadButton || !controls.modeSelectBox || !controls.filterCheckBoxError || !controls.filterCheckBoxWarn || !controls.filterCheckBoxAdvise) {
            return false;
        }
        listeners.push(controls.confirmSelAlarmButton.listen("click", function(event) {
            const rowData = self.vars.grid.getRowData(controls.alarmList.selection.selRows[0]);
            if (rowData) {
                const alarmInfo = JSON.parse(rowData[8].value);
                if (alarmInfo.acknowledgeable && !alarmInfo.acknowledged) {
                    self.imports.am.ackAlarm(alarmInfo.real_id);
                }
            }
        }));
        listeners.push(controls.confirmAllAlarmsButton.listen("click", function(event) {
            const filters = self.vars.grid.getFilters();

            if (filters && Array.isArray(filters[7])) {
                self.imports.am.ackAlarmGroups(filters[7]);
            } else {
                self.imports.am.ackAlarm(-1);
            }
        }));
        listeners.push(controls.createThreadButton.listen("click", async function(event) {
            const rowData = self.vars.grid.getRowData(controls.alarmList.selection.selRows[0]);

            if (rowData) {
                const alarmInfo = JSON.parse(rowData[8].value);
                await openCreateThreadDialogForAlarmInfo(self, alarmInfo);

                if (self.vars.detailsRight) {
                    showAlarmDetails(alarmInfo, "inline", self.vars.detailsRight, self);
                }
            }
        }));
        if (controls.filterGroupSelectBox) {
            listeners.push(controls.filterGroupSelectBox.listen("change", function(event) {
                const { groupFilter } = self.vars;

                if (event.detail && event.detail.value !== null) {
                    self.vars.grid.setFilter(7, [event.detail.value]);
                } else if (groupFilter) {
                    self.vars.grid.setFilter(7, groupFilter);
                } else {
                    self.vars.grid.clearFilter(7);
                }
            }));
        }
        listeners.push(controls.modeSelectBox.listen("change", function(event) {
            self.vars.grid.setDisplayMode(event.detail.value);
        }));
        listeners.push(controls.filterCheckBoxError.listen("change", function(event) {
            toggleFilterSet(self, "error");
            updateTableFilter(self);
        }));
        listeners.push(controls.filterCheckBoxWarn.listen("change", function(event) {
            toggleFilterSet(self, "warn");
            updateTableFilter(self);
        }));
        listeners.push(controls.filterCheckBoxAdvise.listen("change", function(event) {
            toggleFilterSet(self, "advise");
            updateTableFilter(self);
        }));
        return true;
    }

    /**
     * Gathers all anchor-elements to append controls
     *
     * @param {object} self Control-Obj
     * @returns {bool}
     */
    function gatherAnchors(self) {
        const anchors = self.vars.anchorEls,
            io = shmi.requires("visuals.io");

        anchors.iconError = shmi.getUiElement('iconError', self.element);
        anchors.filterCheckBoxError = shmi.getUiElement('filterCheckBoxError', self.element);
        anchors.iconWarn = shmi.getUiElement('iconWarn', self.element);
        anchors.filterCheckBoxWarn = shmi.getUiElement('filterCheckBoxWarn', self.element);
        anchors.iconAdvise = shmi.getUiElement('iconAdvise', self.element);
        anchors.filterCheckBoxAdvise = shmi.getUiElement('filterCheckBoxAdvise', self.element);
        anchors.filterGroupSelectBox = shmi.getUiElement('filterGroupSelectBox', self.element);
        anchors.modeSelectBox = shmi.getUiElement("mode-select-box", self.element);
        anchors.alarmList = shmi.getUiElement('alarmList', self.element);
        anchors.confirmButtons = shmi.getUiElement('confirmButtons', self.element);
        anchors.filterButton = shmi.getUiElement("filter-button", self.element);

        if (anchors.filterButton) {
            const filterHandler = {
                    onClick: function() {
                        handleDialog(self);
                    }
                },
                ml = new io.MouseListener(anchors.filterButton, filterHandler),
                tl = new io.TouchListener(anchors.filterButton, filterHandler);
            self.vars.listeners.push(ml, tl);
        }

        return true;
    }
    /**
     * Gathers all anchor-elements to append controls
     *
     * @param {object} self Control-Obj
     * @param {object} anchors anchors object
     * @param {object} configs configuration holder object
     * @returns {bool}
     */
    function configureAlarmList(self, anchors, configs) {
        configs.alarmTableConfig.table = self.vars.dataGridId;

        if (shmi.hasClass(anchors.alarmList, "two-rows")) {
            configs.alarmTableConfig["default-layout"]["line-height"] = "78px";
        }

        if (self.config["notify-image"]) {
            configs.iconAdvise["image-src"] = self.config["notify-image"];
            configs.alarmTableConfig["default-field-control-map"].level.config.options[0]["icon-src"] = self.config["notify-image"];
        }
        if (self.config["warn-image"]) {
            configs.iconWarn["image-src"] = self.config["warn-image"];
            configs.alarmTableConfig["default-field-control-map"].level.config.options[1]["icon-src"] = self.config["warn-image"];
        }
        if (self.config["alarm-image"]) {
            configs.iconError["image-src"] = self.config["alarm-image"];
            configs.alarmTableConfig["default-field-control-map"].level.config.options[2]["icon-src"] = self.config["alarm-image"];
        }

        if (self.config["detail-template"]) {
            configs.alarmTableConfig["default-field-control-map"].json.config["detail-template"] = self.config["detail-template"];
        }

        if (self.config["comment-template"]) {
            configs.alarmTableConfig["default-field-control-map"].json.config["comment-template"] = self.config["comment-template"];
        }

        if (self.config["comment-mode"]) {
            configs.alarmTableConfig["default-field-control-map"].json.config["comment-mode"] = self.config["comment-mode"];
        }

        if (self.config["dateformat"]) {
            configs.alarmTableConfig["default-field-control-map"].json.config["dateformat"] = self.config["dateformat"];
        }

        Object.assign(configs.alarmTableConfig["default-field-control-map"].json.config, {
            iconAdvise: configs.iconAdvise,
            iconWarn: configs.iconWarn,
            iconError: configs.iconError
        });

        for (let i = 1; i <= self.config.numberOfColumns; i++) {
            if (self.config["col" + i + "-width"] !== null && self.config["col" + i + "-selectBox"]) {
                configs.alarmTableConfig["default-layout"]["column-org"]["col" + i]["column-width"] = self.config["col" + i + "-width"] + self.config["col" + i + "-selectBox"];
            }
        }

        if (self.vars.detailsRight) {
            //remove details button from CT2 and show empty detail win
            delete (configs.alarmTableConfig["default-layout"]["column-org"].col7);
            const alarmDetailsIQ = shmi.requires("visuals.tools.alarms.ls.alarmDetailsIQ");
            alarmDetailsIQ.showAlarmDetails(null, "inline", self.vars.detailsRight, self);
        }

        if (self.config["disable-commit"]) {
            //remove alarm commit button from CT2
            if (configs.alarmTableConfig["default-layout"]["column-org"].col7) {
                configs.alarmTableConfig["default-layout"]["column-org"].col6 = configs.alarmTableConfig["default-layout"]["column-org"].col7;
                delete (configs.alarmTableConfig["default-layout"]["column-org"].col7);
            } else {
                delete (configs.alarmTableConfig["default-layout"]["column-org"].col6);
            }
        }
    }
    /**
     * Creates Childcontrols
     *
     * @param {object} self Control-Obj
     * @returns {bool}
     */
    function createChildControls(self) {
        const anchors = self.vars.anchorEls,
            controls = self.vars.controls,
            configs = self.vars.configs;

        configureAlarmList(self, anchors, configs);

        controls.iconError = shmi.createControl('iq-image', anchors.iconError, configs.iconError, 'DIV');
        controls.filterCheckBoxError = shmi.createControl('iq-checkbox', anchors.filterCheckBoxError, configs.filterCheckBoxError, 'DIV');
        controls.iconWarn = shmi.createControl('iq-image', anchors.iconWarn, configs.iconWarn, 'DIV');
        controls.filterCheckBoxWarn = shmi.createControl('iq-checkbox', anchors.filterCheckBoxWarn, configs.filterCheckBoxWarn, 'DIV');
        controls.iconAdvise = shmi.createControl('iq-image', anchors.iconAdvise, configs.iconAdvise, 'DIV');
        controls.filterCheckBoxAdvise = shmi.createControl('iq-checkbox', anchors.filterCheckBoxAdvise, configs.filterCheckBoxAdvise, 'DIV');

        if (anchors.filterGroupSelectBox) {
            controls.filterGroupSelectBox = shmi.createControl('iq-select-box', anchors.filterGroupSelectBox, configs.filterGroupSelectBox, 'DIV');
        }
        controls.modeSelectBox = shmi.createControl('iq-select-box', anchors.modeSelectBox, configs.modeSelectBox, 'DIV');

        controls.confirmSelAlarmButton = shmi.createControl('iq-button', anchors.confirmButtons, configs.confirmSelAlarmButton, 'DIV');
        controls.confirmAllAlarmsButton = shmi.createControl('iq-button', anchors.confirmButtons, configs.confirmAllAlarmsButton, 'DIV');
        controls.createThreadButton = shmi.createControl('iq-button', anchors.confirmButtons, configs.createThreadButton, 'DIV');

        controls.alarmList = shmi.createControl('complex-table2', anchors.alarmList, configs.alarmTableConfig, 'DIV');

        return true;
    }
    /**
     * evaluate new unix timestamp with time data
     *
     * @param {number} dateTimestamp unix timestamp
     * @param {number} hours hours
     * @param {number} minutes minutes
     * @param {number} seconds seconds
     * @returns {number} unix timestamp
     */
    function evaluateUnixTimestamp(dateTimestamp, hours, minutes, seconds) {
        const date = new Date(dateTimestamp * 1000);
        date.setHours(hours, minutes, seconds, 0);

        return date.getTime() / 1000;
    }

    /**
     * evaluate timestamp from date/time controls
     *
     * @param {object} control group
     * @returns {number} unix timestamp
     */
    function evaluateTimestampFromControls(ctrlGroup) {
        return evaluateUnixTimestamp(
            ctrlGroup.date.getValue(),
            ctrlGroup.time.getHours(),
            ctrlGroup.time.getMinutes(),
            ctrlGroup.time.getSeconds()
        );
    }
    /**
     * handler for filter button - creates the dialog
     *
     * @param {object} self Control-Obj
     */
    function handleDialog(self) {
        const oldDialogHolder = shmi.ctrl(".alarmTimeFilterDialogHolder"),
            containerConfig = {
                "name": "alarmTimeFilterDialogHolder",
                "class-name": "container alarm-filter-dialog-holder"
            };

        if (oldDialogHolder) {
            shmi.deleteControl(oldDialogHolder);
        }

        const dialogHolder = shmi.createControl("container", self.element, containerConfig, "DIV");
        dialogHolder.addControl([
            {
                ui: "dialog-box",
                controller: {
                    name: "alarmTimeFilterController",
                    slots: {
                        dialog: {
                            ui: "dialog-box",
                            events: ["close"]
                        },
                        selectDateStart: {
                            ui: "iq-select-date",
                            events: ["change"]
                        },
                        selectTimeStart: {
                            ui: "iq-select-time",
                            events: ["change"]
                        },
                        selectDateEnd: {
                            ui: "iq-select-date",
                            events: ["change"]
                        },
                        selectTimeEnd: {
                            ui: "iq-select-time",
                            events: ["change"]
                        },
                        cancelButton: {
                            ui: "iq-button",
                            events: ["click"]
                        },
                        applyButton: {
                            ui: "iq-button",
                            events: ["click"]
                        }
                    },
                    onChange: function(state) {
                    },
                    onDisable: function(state) {
                    },
                    onEnable: function(state) {
                        const nowTime = Math.floor((new Date(shmi.getServerTime() * 1000)).getTime() / 1000),
                            nowTimeDate = Math.floor((nowTime) / 86400) * 86400;

                        state.getInstance("selectDateStart").setValue(nowTimeDate - 86400);
                        state.getInstance("selectTimeStart").setValue(nowTime);
                        state.getInstance("selectDateEnd").setValue(nowTimeDate);
                        state.getInstance("selectTimeEnd").setValue(nowTime);
                    },
                    onEvent: function(state, slot, type, event) {
                        //...handle events generated by slot controls
                        switch (slot) {
                        case "selectDateStart":
                            state.getInstance("selectDateEnd").setValue(event.detail.value + 86400);
                            state.getInstance("selectTimeEnd").setValue(state.getInstance("selectTimeStart").getValue());
                            break;
                        case "applyButton":
                            setFilter(self, state);
                            state.getInstance("dialog").hide();
                            break;
                        case "cancelButton":
                            state.getInstance("dialog").hide();
                            break;
                        default:
                        }
                    }
                },
                config: {
                    "title": "${alarmlist_select_time_range}",
                    "class-name": "dialog-box iq-alarm-list iq-alarm-list-filter-dialog",
                    "template": "default/dialog-box",
                    "name": "alarmTimeFilterDialog",
                    "top-level": true,
                    "content-template": null,
                    "_controllers_": [
                        {
                            "name": "alarmTimeFilterController",
                            "slot": "dialog"
                        }
                    ]
                },
                children: [
                    {
                        ui: "container",
                        config: {
                            "class-name": "container",
                            "name": "container",
                            "template": null
                        },
                        children: [
                            {
                                ui: "iq-label",
                                config: {
                                    "class-name": "iq-label",
                                    "template": "default/iq-label.iq-variant-01",
                                    "text": "${alarmlist_from}"
                                }
                            },
                            {

                                ui: "iq-select-date",
                                config: {
                                    "label": "",
                                    "template": "default/iq-select-date.iq-variant-01",
                                    "class-name": "iq-select-date iq-variant-02",
                                    "_controllers_": [
                                        {
                                            "name": "alarmTimeFilterController",
                                            "slot": "selectDateStart"
                                        }
                                    ]
                                }
                            },
                            {
                                ui: "iq-select-time",
                                config: {
                                    "label": "",
                                    "class-name": "iq-select-time iq-variant-01 iq-icon-variant-01",
                                    "isUTC": true,
                                    "_controllers_": [
                                        {
                                            "name": "alarmTimeFilterController",
                                            "slot": "selectTimeStart"
                                        }
                                    ]
                                }
                            },
                            {
                                ui: "iq-label",
                                config: {
                                    "class-name": "iq-label",
                                    "template": "default/iq-label.iq-variant-01",
                                    "text": "${alarmlist_to}"
                                }
                            },
                            {
                                ui: "iq-select-date",
                                config: {
                                    "label": "",
                                    "template": "default/iq-select-date.iq-variant-01",
                                    "class-name": "iq-select-date iq-variant-02",
                                    "_controllers_": [
                                        {
                                            "name": "alarmTimeFilterController",
                                            "slot": "selectDateEnd"
                                        }
                                    ]
                                }
                            },
                            {
                                ui: "iq-select-time",
                                config: {
                                    "label": "",
                                    "class-name": "iq-select-time iq-variant-01 iq-icon-variant-01",
                                    "isUTC": true,
                                    "_controllers_": [
                                        {
                                            "name": "alarmTimeFilterController",
                                            "slot": "selectTimeEnd"
                                        }
                                    ]
                                }
                            },
                            {
                                ui: "container",
                                config: {
                                    "class-name": "container footer",
                                    "type": "float",
                                    "auto-width": false,
                                    "auto-margin": false,
                                    "h-alignment": "right",
                                    "v-alignment": "top",
                                    "flex-orientation": "row",
                                    "flex-distribute": false,
                                    "flex-all": false,
                                    "name": "container",
                                    "template": null
                                },
                                children: [
                                    {
                                        ui: "iq-button",
                                        config: {
                                            "class-name": "iq-button cancel-button",
                                            "label": "${cancel}",
                                            "_controllers_": [
                                                {
                                                    "name": "alarmTimeFilterController",
                                                    "slot": "cancelButton"
                                                }
                                            ]
                                        }
                                    },
                                    {
                                        ui: "iq-button",
                                        config: {
                                            "class-name": "iq-button apply-button",
                                            "label": "${okay}",
                                            "_controllers_": [
                                                {
                                                    "name": "alarmTimeFilterController",
                                                    "slot": "applyButton"
                                                }
                                            ]
                                        }
                                    }
                                ]
                            }
                        ]
                    }
                ]
            }
        ], function(err, controls) {
            if (err) {
                console.error("Error creating controls");
            } else {
                controls[0].show();
            }
        });
    }
    /**
     * set time filter
     *
     * @param {object} self Control-Obj
     * @param {object} state controller state
     */
    function setFilter(self, state) {
        const from = evaluateTimestampFromControls({ time: state.getInstance("selectTimeStart"), date: state.getInstance("selectDateStart") }),
            to = evaluateTimestampFromControls({ time: state.getInstance("selectTimeEnd"), date: state.getInstance("selectDateEnd") }),
            dt = shmi.requires("visuals.tools.date");
        shmi.addClass(self.element, "filter-active");
        self.vars.activeFilterLabel.textContent = shmi.localize("${alarmlist_timefilter}");
        self.vars.activeFilterFrom.textContent = dt.formatDateTime(from, { datestring: shmi.localize(self.config.dateformat) });
        self.vars.activeFilterTo.textContent = dt.formatDateTime(to, { datestring: shmi.localize(self.config.dateformat) });
        self.vars.controls.alarmList.resetScroll();
        self.vars.grid.setFilter(4, [from, to]);
    }
    /**
     * clear time filter
     *
     * @param {object} self Control-Obj
     */
    function clearFilter(self) {
        self.vars.controls.alarmList.resetScroll();
        self.vars.grid.clearFilter(4);
        shmi.removeClass(self.element, "filter-active");
    }
    /**
     * attach listener to selection of rows
     *
     * @param {object} self Control-Obj
     * @param {object} ct complex table reference
     */
    function attachSelectListener(self, ct) {
        const controls = [];

        if (self.vars.controls.confirmSelAlarmButton) {
            controls.push(self.vars.controls.confirmSelAlarmButton);
        }

        if (self.vars.controls.createThreadButton) {
            controls.push(self.vars.controls.createThreadButton);
        }

        self.vars.tokens.push(ct.listen("select", function(evt) {
            if (evt.detail.selRows.length === 1) {
                if (self.vars.detailsRight) {
                    const alarmInfoColumn = 8,
                        alarmInfo = JSON.parse(ct.getSelectionRowData(evt.detail)[0][alarmInfoColumn].value),
                        alarmDetailsIQ = shmi.requires("visuals.tools.alarms.ls.alarmDetailsIQ");

                    alarmDetailsIQ.showAlarmDetails(alarmInfo, "inline", self.vars.detailsRight, self);
                }

                controls.forEach((c) => c.unlock());
            } else {
                controls.forEach((c) => c.lock());
            }
        }));
    }
    /**
     * Sets the headline text
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined" || !self.config['show-text']) {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
        }
    }

    /**
     * Updates the options of the group filter select-box with available alarm
     * groups returned from the backend.
     *
     * @param {object} self Reference to the iq-alarm-list widget
     */
    function updateGroupFilterSelectbox(self) {
        const { controls: { filterGroupSelectBox }, groupFilter } = self.vars;

        if (!filterGroupSelectBox) {
            return;
        }

        shmi.visuals.session.ConnectSession.requestPromise("alarm.get_groups", {}).then((groups) => {
            const oldValue = filterGroupSelectBox.getValue();
            filterGroupSelectBox.setOptions(
                [
                    ...self.vars.configs.filterGroupSelectBox.options,
                    ...groups.filter((groupId) => !groupFilter || groupFilter.includes(groupId)).map((groupId) => ({
                        label: `\${alarm_group_${groupId}}`,
                        value: groupId
                    }))
                ]
            );
            filterGroupSelectBox.setValue(oldValue);
        }).catch((e) => {
            console.error(`[${className}] Unable to get list of alarm groups`, e);

            filterGroupSelectBox.setOptions(self.vars.configs.filterGroupSelectBox.options);
            filterGroupSelectBox.setValue(null);
        });
    }

    //declare private functions - END

    // Definition of new control extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            tempAlarmGrid: {},
            alarmSubscription: null,
            alarmLevelTemplate: null,
            grid: null,
            dataGridId: "",
            groupFilter: null,
            cancelable: null,
            filterSet: {
                warn: false,
                error: false,
                advise: false
            },
            anchorEls: {
                iconError: null,
                filterCheckBoxError: null,
                iconWarn: null,
                filterCheckBoxWarn: null,
                iconAdvise: null,
                filterCheckBoxAdvise: null,
                filterGroupSelectBox: null,
                modeSelectBox: null,
                alarmList: null,
                confirmButtons: null,
                filterButton: null,
                filterOverlay: null
            },
            controls: {
                iconError: null,
                filterCheckBoxError: null,
                iconWarn: null,
                filterCheckBoxWarn: null,
                iconAdvise: null,
                filterCheckBoxAdvise: null,
                filterGroupSelectBox: null,
                modeSelectBox: null,
                alarmList: null,
                confirmSelAlarmButton: null,
                confirmAllAlarmsButton: null,
                createThreadButton: null
            },
            configs: {
                alarmLevelConfig: {
                    "alarm-icon": "pics/system/controls/iq-alarm-list/ico_alarm_outline.svg",
                    "warn-icon": "pics/system/controls/iq-alarm-list/ico_warning_outline.svg",
                    "notify-icon": "pics/system/controls/iq-alarm-list/ico_service_outline.svg"
                },
                alarmTableConfig: {
                    "label": "${alarmlist_history_title}",
                    "table": "iqalarms",
                    "name": "alarm-table",
                    "class-name": "complex-table2 alarms",
                    "field-datagrid-col-map": {
                        "level": 3,
                        "id": 1,
                        "description": 9,
                        "come": 4,
                        "group": 10,
                        "isAck": 8,
                        "json": 8
                    },
                    "select-mode": "SINGLE",
                    "default-field-control-map": {
                        "level": {
                            "ui-type": "iq-image-changer",
                            "config": {
                                "class-name": "iq-image-changer iq-variant-01",
                                "template": "default/iq-image-changer.iq-variant-01",
                                "options": [
                                    {
                                        "label": "label",
                                        "value": 0,
                                        "icon-src": "pics/system/controls/iq-alarm-list/ico_service_outline.svg",
                                        "icon-title": null,
                                        "icon-class": null
                                    },
                                    {
                                        "label": "label",
                                        "value": 1,
                                        "icon-src": "pics/system/controls/iq-alarm-list/ico_warning_outline.svg",
                                        "icon-title": null,
                                        "icon-class": null
                                    },
                                    {
                                        "label": "label",
                                        "value": 2,
                                        "icon-src": "pics/system/controls/iq-alarm-list/ico_alarm_outline.svg",
                                        "icon-title": null,
                                        "icon-class": null
                                    }
                                ],
                                "scaling-mode": "fit-height"
                            }
                        },
                        "id": {
                            "ui-type": "iq-label",
                            "config": {
                                "class-name": "iq-label",
                                "template": "default/iq-label.iq-variant-01",
                                "options": [],
                                "pattern": "<%= VALUE %>",
                                "value-as-tooltip": true
                            }
                        },
                        "description": {
                            "ui-type": "iq-label",
                            "config": {
                                "class-name": "iq-label",
                                "template": "default/iq-label.iq-variant-01",
                                "options": [],
                                "pattern": null,
                                "value-as-tooltip": true
                            }
                        },
                        "come": {
                            "ui-type": "iq-date-time",
                            "config": {
                                "class-name": "iq-date-time multiline",
                                "template": "default/iq-date-time.iq-variant-01",
                                "display-format": "$DD.$MM.$YYYY, $HH:$mm:$ss",
                                "value-as-tooltip": true
                            }
                        },
                        "group": {
                            "ui-type": "iq-label",
                            "config": {
                                "class-name": "iq-label",
                                "template": "default/iq-label.iq-variant-01",
                                "options": [],
                                "pattern": null,
                                "value-as-tooltip": true
                            }
                        },
                        "isAck": {
                            "ui-type": "local-script",
                            "config": {
                                "module": "visuals.tools.alarms.ls.alarmCommitButton"
                            }
                        },
                        "json": {
                            "ui-type": "local-script",
                            "config": {
                                "module": "visuals.tools.alarms.ls.alarmDetailsIQ",
                                "detail-template": null,
                                "comment-template": null,
                                "dateformat": null
                            }
                        }
                    },
                    "default-field-headers": {
                        "level": "${alarmlist_header_level}",
                        "id": "${alarmlist_header_id}",
                        "description": "${alarmlist_header_description}",
                        "come": "${alarmlist_header_come}",
                        "group": "${alarmlist_header_group}",
                        "isAck": "${alarmlist_header_isack}",
                        "json": "${alarmlist_header_json}"
                    },
                    "_comment": "expr is passed to expr parameter array of shmi.visuals.core.DataGridManager.setFilter",
                    "filters": [],
                    "default-layout": {
                        "class-name": "layout-std",
                        "_comment": "default == no additional css layout class",
                        "column-org": {
                            "col1": {
                                "fields": ["level"],
                                "column-width": "10%"
                            },
                            "col2": {
                                "fields": ["id"],
                                "column-width": "10%"
                            },
                            "col3": {
                                "fields": ["description"]
                            },
                            "col4": {
                                "fields": ["come"],
                                "column-width": "15%"
                            },
                            "col5": {
                                "fields": ["group"]
                            },
                            "col6": {
                                "fields": ["isAck"],
                                "column-width": "90px"
                            },
                            "col7": {
                                "fields": ["json"],
                                "column-width": "90px"
                            }
                        },
                        "line-height": "39px"
                    },
                    "sortable-fields": [
                        "come",
                        "id",
                        "group"
                    ],
                    "delete-selected-rows": false,
                    "show-nof-rows": true,
                    "show-buttons-table-min-width-px": 400,
                    "text-mode": "MULTILINE",
                    "default-nof-buffered-rows": 60,
                    "buffer-size": 500
                },
                iconError: {
                    'class-name': 'iq-image iq-variant-01',
                    'template': 'default/iq-image.iq-variant-01',
                    'image-src': 'pics/system/controls/iq-alarm-list/ico_alarm_outline.svg'
                },
                filterCheckBoxError: {
                    'class-name': 'iq-checkbox iq-variant-01',
                    'template': 'default/iq-checkbox.iq-variant-01',
                    'label': '${alarmlist_level_errors}'
                },
                iconWarn: {
                    'class-name': 'iq-image iq-variant-01',
                    'template': 'default/iq-image.iq-variant-01',
                    'image-src': 'pics/system/controls/iq-alarm-list/ico_warning_outline.svg'
                },
                filterCheckBoxWarn: {
                    'class-name': 'iq-checkbox iq-variant-01',
                    'template': 'default/iq-checkbox.iq-variant-01',
                    'label': '${alarmlist_level_warnings}'
                },
                iconAdvise: {
                    'class-name': 'iq-image iq-variant-01',
                    'template': 'default/iq-image.iq-variant-01',
                    'image-src': 'pics/system/controls/iq-alarm-list/ico_service_outline.svg'
                },
                filterCheckBoxAdvise: {
                    'class-name': 'iq-checkbox iq-variant-01',
                    'template': 'default/iq-checkbox.iq-variant-01',
                    'label': '${alarmlist_level_notifications}'
                },
                filterGroupSelectBox: {
                    "name": "filter-group-select-box",
                    'class-name': 'iq-select-box',
                    'show-text': false,
                    'options': [
                        {
                            'label': '${alarmlist_filter_group_all}',
                            'value': null
                        }
                    ]
                },
                modeSelectBox: {
                    'name': "mode-select-box",
                    'class-name': 'iq-select-box',
                    'show-text': false,
                    'options': [{
                        'label': '${alarmlist_mode_all}',
                        'value': 'all'
                    },
                    {
                        'label': '${alarmlist_mode_live}',
                        'value': 'live'
                    },
                    {
                        'label': '${alarmlist_mode_historic}',
                        'value': 'historic'
                    }]
                },
                alarmList: {},
                confirmSelAlarmButton: {
                    'label': '${alarmlist_commit}',
                    'name': 'confirm-sel-al-btn',
                    'class-name': 'iq-button icon-and-text',
                    'icon-src': 'pics/system/controls/iq-alarm-list/ButtonIcon_AcknSingle_default.svg'
                },
                confirmAllAlarmsButton: {
                    'label': '${alarmlist_commit_all}',
                    'name': 'confirm-all-al-btn',
                    'class-name': 'iq-button icon-and-text',
                    'icon-src': 'pics/system/controls/iq-alarm-list/ButtonIcon_AcknAll_default.svg'
                },
                createThreadButton: {
                    'label': '${alarmlist_create_thread}',
                    'name': 'create-thread-btn',
                    'class-name': 'iq-button icon-and-text',
                    'icon-src': 'pics/system/controls/iq-alarm-list/ButtonIcon_CreateThread_default.svg'
                }
            },
            listeners: [],
            tokens: [],
            editListeners: []
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            dgm: 'visuals.session.DataGridManager',
            am: 'visuals.session.AlarmManager'
        },

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                const self = this;

                if (!controlStartUp(self)) {
                    shmi.log('Init Failed!');
                } else {
                    shmi.log('Init Succeeded!');
                }
                /***********/
                /*** DOM ***/
                /***********/
                self.vars.labelEl = shmi.getUiElement('label', self.element);

                // Label
                setLabelImpl(self, self.config.label);
            },
            /* called when control is enabled */
            onEnable: function() {
                const self = this;

                self.vars.listeners.forEach((l) => l.enable());

                setListener(self);
                attachSelectListener(self, self.vars.controls.alarmList);

                for (const key in self.vars.controls) {
                    if (self.vars.controls[key] && self.vars.controls[key].enable) {
                        self.vars.controls[key].enable();
                    }
                }

                if (self.vars.controls.confirmSelAlarmButton) {
                    self.vars.controls.confirmSelAlarmButton.lock();
                }

                if (self.vars.controls.createThreadButton) {
                    self.vars.controls.createThreadButton.lock();
                }

                self.vars.cancelable = shmi.onReady({
                    controls: {
                        modeSelectBox: self.getName() + ".mode-select-box",
                        filterGroupSelectBox: self.getName() + ".filter-group-select-box"
                    }
                }, function(ref) {
                    ref.controls.modeSelectBox.setValue(self.config["display-mode"]);
                    ref.controls.filterGroupSelectBox.setValue(null);
                    updateGroupFilterSelectbox(self);
                });

                shmi.log("[IQ:iq-alarm-list] Enabled", 1);
            },
            /* called when control is disabled */
            onDisable: function() {
                const self = this;

                self.vars.listeners.forEach((l) => l.disable());

                for (const key in self.vars.controls) {
                    if (self.vars.controls[key] && self.vars.controls[key].disable) {
                        self.vars.controls[key].disable();
                    }
                }

                self.vars.tokens.forEach((t) => t.unlisten());
                self.vars.tokens = [];

                self.vars.editListeners.forEach((l) => l.disable());
                self.vars.editListeners = [];

                if (self.vars.cancelable) {
                    self.vars.cancelable.cancel();
                    self.vars.cancelable = null;
                }

                shmi.log("[IQ:iq-alarm-list] disabled", 1);
            },
            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                const self = this;

                self.vars.listeners.forEach((l) => l.disable());

                for (const key in self.vars.controls) {
                    if (self.vars.controls[key] && self.vars.controls[key].lock) {
                        self.vars.controls[key].lock();
                    }
                }

                shmi.addClass(self.element, 'locked');

                shmi.log("[IQ:iq-alarm-list] Locked", 1);
            },
            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                const self = this;

                self.vars.listeners.forEach((l) => l.enable());

                for (const key in self.vars.controls) {
                    if (self.vars.controls[key] && self.vars.controls[key].unlock) {
                        self.vars.controls[key].unlock();
                    }
                }

                shmi.removeClass(self.element, 'locked');

                shmi.log("[IQ:iq-alarm-list] unlocked", 1);
            },
            setLabel: function(labelText) {
                const self = this;
                setLabelImpl(self, labelText);
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    const cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

(function() {
    /**
     * module to display alarm-detail messages in alarm-table.
     */

    const MODULE_NAME = "visuals.tools.alarms.ls.alarmCommitButton",
        ENABLE_LOGGING = true,
        RECORD_LOG = false,
        logger = shmi.requires("visuals.tools.logging").createLogger(MODULE_NAME, ENABLE_LOGGING, RECORD_LOG),
        fLog = logger.fLog,
        log = logger.log,
        module = shmi.pkg(MODULE_NAME);

    // MODULE CODE - START

    /* private variables */

    /* private functions */

    /**
     * Implements local-script run function.
     *
     * This function will be called each time a local-script will be enabled.
     *
     * @author Matthias Weber <weber@smart-hmi.de>
     * @param {LocalScript} self instance reference of local-script control
     */
    module.run = function(self) {
        const im = shmi.requires("visuals.session.ItemManager"),
            am = shmi.requires("visuals.session.AlarmManager"),
            itemHandler = im.getItemHandler();
        let alarmInfo = null,
            ackButton = null,
            tokens = [];

        self.vars = self.vars || {};
        itemHandler.setValue = function(value) {
            alarmInfo = JSON.parse(value);
            if (ackButton) shmi.deleteControl(ackButton, true);
            if (alarmInfo.active) {
                self.element.parentElement.parentElement.parentElement.style["font-weight"] = "bold";
            } else {
                self.element.parentElement.parentElement.parentElement.style["font-weight"] = "";
            }
            if (alarmInfo.acknowledgeable) {
                if (alarmInfo.acknowledged) {
                    ackButton = shmi.createControl("iq-button", self.element.parentNode, { "icon-src": "pics/system/controls/iq-alarm-list/ButtonIcon_AcknSingle_default.svg", "label": "", "show-icon": true, "show-text": false }, "DIV");
                    shmi.addClass(ackButton.element, "alarm-ack");
                } else {
                    ackButton = shmi.createControl("iq-button", self.element.parentNode, { label: "${alarmlist_single_commit}" }, "DIV");
                    shmi.addClass(ackButton.element, "alarm-ack");
                    tokens.push(ackButton.listen("click", function() {
                        // ACK ALARM
                        if (alarmInfo.real_id) {
                            am.ackAlarm(alarmInfo.real_id);
                        }
                    }));
                }
                if (!alarmInfo.active) {
                    shmi.addClass(ackButton.element, "locked");
                }
            } else {
                ackButton = shmi.createControl("iq-label", self.element.parentNode, { "text": "---", "template": "default/iq-label.iq-variant-01" }, "DIV");
                shmi.addClass(ackButton.element, "alarm-no-ack");
            }
        };

        if (self.config.item) {
            tokens.push(im.subscribeItem(self.config.item, itemHandler));
        }
        /* called when this local-script is disabled */
        self.onDisable = function() {
            self.run = false; // from original .onDisable function of LocalScript control
            tokens.forEach(function(t) {
                t.unlisten();
            });
            tokens = [];
            if (ackButton) shmi.deleteControl(ackButton, true);
        };
    };
    // MODULE CODE - END

    fLog("module loaded");
})();

(function() {
    /**
     * module to display alarm-detail messages in alarm-table.
     */

    const MODULE_NAME = "visuals.tools.alarms.ls.alarmDetailsIQ",
        ENABLE_LOGGING = true,
        RECORD_LOG = false,
        logger = shmi.requires("visuals.tools.logging").createLogger(MODULE_NAME, ENABLE_LOGGING, RECORD_LOG),
        fLog = logger.fLog,
        log = logger.log,
        module = shmi.pkg(MODULE_NAME);

    // MODULE CODE - START

    /* private variables */

    /* private functions */

    /**
     * format time to display
     *
     * @param {Number} timestamp timestamp
     */
    function formatTime(timestamp) {
        const date = new Date(timestamp * 1000);
        if (timestamp === null) {
            return "---";
        } else {
            return date.toLocaleString();
        }
    }
    /**
     * replace zero with '---''
     *
     * @param {Number} value
     */
    function zeroOutNull(value) {
        if (value === null) {
            return "---";
        } else {
            return value;
        }
    }

    /**
     * Opens the create thread dialog box for the given alarm.
     *
     * @param {number} referenceId
     * @param {"alarm"|"alarm_type"} referenceType
     * @returns {Promise<number|null>}
     */
    function openCreateThreadDialogForAlarm(referenceId, referenceType) {
        const { create: createThread } = shmi.requires("visuals.tools.threads");

        return createThread(referenceType, referenceId, null, {
            dialogTitle: "${alarmlist_create_thread}",
            titleLabel: "${alarmlist_threads_title}",
            messageLabel: "${alarmlist_threads_message}",
            applyLabel: "${alarmlist_threads_create_btn}"
        });
    }

    /**
     * Opens the create thread dialog box for the given alarm info according to
     * the widget configuration.
     *
     * @param {*} self
     * @param {object} alarmInfo
     * @returns {Promise<number|null>}
     */
    function openCreateThreadDialogForAlarmInfo(self, alarmInfo) {
        switch (self.config["comment-mode"]) {
        case "alarm":
            return openCreateThreadDialogForAlarm(alarmInfo.id, "alarm");
        case "alarm_type":
            return openCreateThreadDialogForAlarm(alarmInfo.index, "alarm_type");
        case "disabled":
        default:
            return Promise.resolve(null);
        }
    }

    /**
     * Checks whether comments are enabled or not.
     *
     * @param {*} self
     * @returns {boolean}
     */
    function isCommentModeEnabled(self) {
        return self.config["comment-mode"] === "alarm" || self.config["comment-mode"] === "alarm_type";
    }

    /**
     * @param {*} self
     * @param {HTMLTemplateElement} contentTemplate
     * @param {object} alarmInfo
     */
    function openAlarmDetailsPopup(self, contentTemplate, alarmInfo) {
        const dialogBox = shmi.createControl("dialog-box", self.element.parentNode, {
            "name": "alarmDetailsPopup",
            "class-name": "dialog-box iq-alarm-list",
            "top-level": true,
            "title": "Alarm Details"
        }, "DIV");

        let tokens = [];
        const onDialogBoxActive = () => {
            const container = shmi.createControl("container", dialogBox.contentElement, {}, "DIV"),
                buttonContainer = shmi.createControl("container", dialogBox.contentElement, {
                    "class-name": "iq-container",
                    "type": "iqflex"
                }, "DIV"),
                createThreadButton = isCommentModeEnabled(self) ? shmi.createControl("iq-button", buttonContainer.marginCompensator, {
                    "label": "${alarmlist_create_thread}"
                }, "DIV") : null,
                closeBtn = shmi.createControl("iq-button", buttonContainer.marginCompensator, {
                    "label": "${close}"
                }, "DIV"),
                targetContainer = container.marginCompensator || container.element;

            tokens.push(
                closeBtn.listen("click", () => dialogBox.hide()),
                dialogBox.listen("close", () => {
                    tokens.forEach((token) => token.unlisten());
                    tokens = [];

                    shmi.deleteControl(dialogBox);
                })
            );

            if (createThreadButton) {
                tokens.push(createThreadButton.listen("click", async () => {
                    const newThreadId = await openCreateThreadDialogForAlarmInfo(self, alarmInfo);
                    if (newThreadId !== null) {
                        dialogBox.hide();
                        showAlarmDetails(alarmInfo, "popup", null, self);
                    }
                }));
            }

            const detailContent = contentTemplate.content.cloneNode(true);

            targetContainer.appendChild(detailContent);
            initCommentEditing(self, targetContainer, dialogBox, alarmInfo);
            dialogBox.show();
        };

        if (dialogBox.active) {
            onDialogBoxActive();
        } else {
            tokens.push(dialogBox.listen("enable", onDialogBoxActive));
        }
    }

    /**
     * Loads a HTML template from a url and returns it as template-tag.
     *
     * @param {string} url Url to the HTML template.
     * @param {?object} [placeholders]
     * @param {object} [loadOptions] Options to pass to the resource loader.
     * @returns {Promise<HTMLTemplateElement>}
     */
    async function loadHTMLTemplate(url, placeholders = null, loadOptions = {}) {
        const templateData = await shmi.loadResourcePromise(url, loadOptions),
            template = document.createElement("template");

        if (placeholders) {
            template.innerHTML = shmi.localize(shmi.evalString(templateData, placeholders));
        } else {
            template.innerHTML = shmi.localize(templateData);
        }

        return template;
    }

    /**
     * Loads comments for the given alarm using either the alarm id or alarm
     * index depending on the widget configuration.
     *
     * @param {object} self Control-Obj
     * @param {object?} alarmInfo Alarm info
     * @returns {Promise<object[]>}
     */
    async function loadCommentsForAlarm(self, alarmInfo) {
        const { list } = shmi.requires("visuals.tools.threads");

        if (!alarmInfo) {
            return Promise.resolve([]);
        }

        try {
            switch (self.config["comment-mode"]) {
            case "alarm":
                return (await list("alarm", alarmInfo.id, {
                    loadReplies: true,
                    loadUserDisplayName: true
                })).threads;
            case "alarm_type":
                return (await list("alarm_type", alarmInfo.index, {
                    loadReplies: true,
                    loadUserDisplayName: true
                })).threads;
            case "disabled":
            default:
            }
        } catch (e) {
            console.error(`[${MODULE_NAME}]`, "Failed to load comments:", e);
        }

        return [];
    }

    /**
     * initialize thread-edit element to edit threads created by current user
     *
     * @param {object} self alarmlist or local-script instance depending on alarm-detail mode
     * @param {HTMLElement} element UI element to trigger thread editing
     * @param {object} [dialog=null] dialog instance in case of "popup" alarm-detail-mode
     * @param {object} [alarmInfo=null] alarm info data
     */
    function initEditElement(self, element, dialog = null, alarmInfo = null) {
        const userId = element.getAttribute("data-user-id"),
            threadId = parseInt(element.getAttribute("data-thread-id")),
            um = shmi.requires("visuals.session.UserManager"),
            { MouseListener, TouchListener } = shmi.requires("visuals.io"),
            { currentUser } = um;
        if (!isNaN(threadId) && currentUser && userId === currentUser.id) {
            //
            const { edit } = shmi.requires("visuals.tools.threads"),
                handler = {
                    onClick: async () => {
                        const new_threadId = await edit(threadId, {
                            dialogTitle: "${alarmlist_edit_thread}",
                            titleLabel: "${alarmlist_threads_title}",
                            messageLabel: "${alarmlist_threads_message}",
                            applyLabel: "${alarmlist_threads_edit_btn}",
                            deleteLabel: "${alarmlist_threads_delete_btn}"
                        });
                        if (new_threadId !== null) {
                            if (dialog) {
                                dialog.hide();
                                showAlarmDetails(alarmInfo, "popup", null, self);
                            } else if (self.vars.detailsRight) {
                                showAlarmDetails(alarmInfo, "inline", self.vars.detailsRight, self);
                            }
                        }
                    }
                },
                ml = new MouseListener(element, handler),
                tl = new TouchListener(element, handler);

            ml.enable();
            tl.enable();
            self.vars.editListeners.push(ml, tl);
            shmi.addClass(element, "editable");
        }
    }

    /**
     * initialize thread-edit capabilities
     *
     * @param {object} self alarmlist or local-script instance depending on alarm-detail mode
     * @param {HTMLElement} detailContent parent element of alarm-detail content
     * @param {object} [dialog=null] dialog instance in case of "popup" alarm-detail-mode
     * @param {object} [alarmInfo=null] alarm info data
     */
    async function initCommentEditing(self, detailContent, dialog = null, alarmInfo = null) {
        const session = shmi.requires("visuals.session"),
            request = session.ConnectSession.requestPromise.bind(session.ConnectSession);

        let editElements = [],
            mayEdit = false;

        //workaround - test if user may edit thread
        try {
            await request("thread.modify", { thread_id: -1, title: null, message: "" });
        } catch (err) {
            if (!(err.category === "shmi:connect:api:generic" && err.errc === 4)) {
                mayEdit = true;
            }
        }

        self.vars.editListeners.forEach((l) => {
            l.disable();
        });
        self.vars.editListeners = [];

        if (mayEdit) {
            editElements = [...detailContent.querySelectorAll("[data-ui=thread-edit]")];
            editElements.forEach((e) => {
                initEditElement(self, e, dialog, alarmInfo);
            });
        }
    }

    /**
     * show the alarm details
     *
     * @param {object} alarmInfo alarm info object
     * @param {string} mode display mode for alarm details
     * @param {object} target target element
     * @param {object} self Control-Obj
     */
    async function showAlarmDetails(alarmInfo, mode, target, self) {
        const { renderThreadList } = shmi.requires("visuals.tools.threads.render");
        let templateLocation = "templates/default/iq-alarm-list/detail-win.html";

        const threadRenderOptions = {
            renderReplies: true
        };

        if (self.config["detail-template"]) {
            templateLocation = `templates/${self.config["detail-template"]}.html`;
        }

        if (self.config["comment-template"]) {
            threadRenderOptions.templateUrl = self.config["comment-template"];
        }

        function getSeverityAlarmIconSrc(severity) {
            const cfg = (self.vars && self.vars.configs ? self.vars.configs : self.config);

            switch (severity) {
            case 0:
                return cfg.iconAdvise["image-src"];
            case 1:
                return cfg.iconWarn["image-src"];
            case 2:
                return cfg.iconError["image-src"];
            default:
                return "";
            }
        }

        let alarmObj = {
            ALARM_SEVERITY: -1,
            ALARM_SEVERITY_ICON_SRC: "",
            ALARM_COME: "",
            ALARM_GONE: "",
            ALARM_ACK: "",
            ALARM_ACKBY: "",
            ALARM_INDEX: "",
            ALARM_TITLE: "",
            ALARM_URLS: "",
            ALARM_URLS_MATCHED: false,
            ALARM_TEXT: ""
        };

        if (alarmInfo) {
            const urlAttributes = Object.entries(alarmInfo.attributes).filter(([key, value]) => typeof value === "string" && value.startsWith("[URL]"));
            const alarmUrls = urlAttributes.map(([key, value]) => `${key}: <a href='${value.substr(5)}' target='_blank'>${value.substr(5)}</a><br />`).join("");

            alarmObj = {
                ALARM_SEVERITY: alarmInfo.severity,
                ALARM_SEVERITY_ICON_SRC: getSeverityAlarmIconSrc(alarmInfo.severity),
                ALARM_COME: formatTime(alarmInfo.timestamp_in),
                ALARM_GONE: formatTime(alarmInfo.timestamp_out),
                ALARM_ACK: formatTime(alarmInfo.timestamp_acknowledged),
                ALARM_ACKBY: zeroOutNull(alarmInfo.acknowledged_by),
                ALARM_INDEX: alarmInfo.index,
                ALARM_TITLE: shmi.evalString(shmi.localize("${alarm_title_" + alarmInfo.index + "}"), alarmInfo),
                ALARM_URLS: alarmUrls,
                ALARM_URLS_MATCHED: urlAttributes.length > 0,
                ALARM_TEXT: shmi.evalString(shmi.localize("${alarm_msg_" + alarmInfo.index + "}"), alarmInfo)
            };
        }

        const [template, threads] = await Promise.all([
            loadHTMLTemplate(templateLocation, alarmObj),
            loadCommentsForAlarm(self, alarmInfo)
        ]);

        const commentElement = shmi.getUiElement("comments", template.content);
        if (commentElement) {
            commentElement.append(await renderThreadList(threads, threadRenderOptions));
            if (threads.length) {
                shmi.addClass(commentElement, "has-comments");
            }
        }

        if (!alarmObj.ALARM_URLS_MATCHED) {
            const linkContainer = shmi.getElementsByAttribute("class", "link-container", template.content);
            if (linkContainer[0]) {
                linkContainer[0].style.display = "none";
            }
        }

        if (mode === "popup") {
            openAlarmDetailsPopup(self, template, alarmInfo);
        } else {
            // Clear detail window.
            while (target.lastChild) {
                target.removeChild(target.lastChild);
            }
            const detailContent = template.content.cloneNode(true);
            target.appendChild(detailContent);
            initCommentEditing(self, target, null, alarmInfo);
        }
    }

    /**
     * Implements local-script run function.
     *
     * This function will be called each time a local-script will be enabled.
     *
     * @param {LocalScript} self instance reference of local-script control
     */
    module.run = function(self) {
        const im = shmi.requires("visuals.session.ItemManager"),
            itemHandler = im.getItemHandler();
        let alarmInfo = null,
            tokens = [];

        self.vars = self.vars || {};
        self.vars.editListeners = [];

        const detailsButton = shmi.createControl("iq-button", self.element.parentNode, { label: "${alarmlist_show_detail}" }, "DIV");
        itemHandler.setValue = function(value) {
            try {
                alarmInfo = JSON.parse(value);
            } catch (e) {
                console.error("Error parsing alarmInfo data");
            }
        };

        tokens.push(detailsButton.listen("click", function() {
            showAlarmDetails(alarmInfo, "popup", null, self);
        }));

        if (self.config.item) {
            tokens.push(im.subscribeItem(self.config.item, itemHandler));
        }

        /* called when this local-script is disabled */
        self.onDisable = function() {
            self.run = false; // from original .onDisable function of LocalScript control
            tokens.forEach(function(t) {
                t.unlisten();
            });
            tokens = [];
            self.vars.editListeners.forEach((l) => {
                l.disable();
            });
            self.vars.editListeners = [];
            shmi.deleteControl(detailsButton, true);
        };
    };
    /**
     * @function
     * show the alarm details
     *
     * @param {object} alarmInfo alarm info object
     * @param {string} mode display mode for alarm details
     * @param {object} target target element
     */
    module.showAlarmDetails = showAlarmDetails;

    module.openCreateThreadDialogForAlarmInfo = openCreateThreadDialogForAlarmInfo;

    // MODULE CODE - END

    fLog("module loaded");
})();

/**
 * WIQ Rocker Button
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-button-rocker",
 *     "name": null,
 *     "template": "custom/controls/iq-button-rocker"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "min": Minimum value
 * "max": Maximum value
 * "step": Step width
 * "precision": Precision
 * "decimal-delimiter": Decimal delimiter
 * "type": Type is INT
 * "unit-text": Unit Text
 * "auto-label": Whether to use the auto-label (from item)
 * "auto-unit-text": Whether to use the auto-unit (from item)
 * "auto-min": Whether to use auto-min (from item)
 * "auto-max": Whether to use auto-max (from item)
 * "auto-step": Wheter to use auto-step (from item)
 * "auto-precision": Whether to use the auto-precision (from item)
 * "auto-type": Whether to use the auto-type (from item)
 * "numpad-enabled": Whether the numpad is enabled
 * "icon-src": Default icon source
 * "icon-class": Default icon class
 * "tooltip": Tooltip
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-button-rocker", // control name in camel-case
        uiType = "iq-button-rocker", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-button-rocker",
        "name": null,
        "template": "default/iq-button-rocker.iq-variant-01",
        "label": '[Label]',
        "item": null,
        "min": Number.NEGATIVE_INFINITY,
        "max": Number.POSITIVE_INFINITY,
        "step": 1,
        "precision": -1,
        "decimal-delimiter": ".",
        "type": shmi.c("TYPE_INT"),
        "unit-text": "[Unit]",
        "auto-label": true,
        "auto-unit-text": true,
        "auto-min": true,
        "auto-max": true,
        "auto-step": true,
        "auto-precision": true,
        "auto-type": true,
        "numpad-enabled": false,
        "show-icon": false,
        "icon-src": null,
        "icon-class": null,
        "tooltip": null,
        "auto-repeat": false,
        "auto-repeat-interval": 250
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the unit text and handles toggling the `no-unit` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} unitText Unit text to set
     */
    function setUnitTextImpl(self, unitText) {
        if (!self.vars.unitEl) {
            // Nothing to do.
        } else if (unitText === "" || unitText === null || typeof unitText === "undefined") {
            self.vars.unit = "";
            self.vars.unitEl.textContent = "";
            shmi.addClass(self.element, "no-unit");
        } else {
            self.vars.unit = unitText;
            self.vars.unitEl.textContent = shmi.localize(unitText);
            shmi.removeClass(self.element, "no-unit");
        }
    }

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined") {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    /**
     * getButtonStepping - get step interval for increment/decrement buttons. Defaults to 1 when stepping is unset or <=0
     *
     * @param {object} self control instance
     * @returns {number} increment/decrement stepping
     */
    function getButtonStepping(self) {
        if (self.vars.valueSettings && typeof self.vars.valueSettings.step === "number" && self.vars.valueSettings.step > 0) {
            return self.vars.valueSettings.step;
        }
        return 1;
    }

    /**
     * getValueSettings - retrieve active value-settings with fallback to configuration when not available
     *
     * @param {object} self control instance
     * @returns {object} value settings
     */
    function getValueSettings(self) {
        const config = self.getConfig();

        //remove step setting from value settings to allow input independent of button stepping
        const inputSettings = {
            vars: {
                valueSettings: self.vars.valueSettings ? shmi.cloneObject(self.vars.valueSettings) : config
            },
            config: config
        };
        inputSettings.vars.valueSettings.step = 0;

        return inputSettings;
    }

    /**
     * incValue - increment current value by specified amount
     *
     * @param {object} self control instance
     * @param {number} increment increment
     */
    function incValue(self, increment) {
        if (!self.vars.valueSettings) {
            self.imports.nv.initValueSettings(self);
        }
        self.vars.currentValueEl.blur();
        self.vars.value = self.imports.nv.applyInputSettings(Number(self.vars.value) + increment, getValueSettings(self));
        self.updateValue();
        if (typeof self.vars.value !== 'undefined') {
            self.setCurrentElementValue();
        }
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: 0,
            active: false,

            // DOM Elements
            prevEl: null,
            nextEl: null,
            currentValueEl: null,
            labelEl: null,
            unitEl: null,
            iconEl: null,

            label: null,
            isInputTag: null,
            valueSettings: null,
            buttonInterval: 0
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues"
        },

        /* array of custom event types fired by this control */
        events: ["change"],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.prevEl = shmi.getUiElement('previous', self.element);
                self.vars.nextEl = shmi.getUiElement('next', self.element);
                self.vars.currentValueEl = shmi.getUiElement('current', self.element);
                self.vars.labelEl = shmi.getUiElement('label', self.element);
                self.vars.unitEl = shmi.getUiElement('unit', self.element);
                self.vars.iconEl = shmi.getUiElement('icon', self.element);

                if (self.vars.currentValueEl) {
                    self.vars.isInputTag = self.vars.currentValueEl.tagName === 'INPUT';
                }

                /************/
                /*** ICON ***/
                /************/
                if (!self.vars.iconEl) {
                    shmi.log('[IQ:iq-button-rocker] no button-icon element provided', 1);
                } else if (self.config['icon-src'] && self.config['show-icon']) {
                    self.vars.iconEl.style.backgroundImage = `url(${self.config['icon-src']})`;
                } else if (self.config['icon-class'] && self.config['show-icon']) {
                    const iconClasses = self.config['icon-class'].trim().split(" ");
                    iconClasses.forEach(function(cls) {
                        shmi.addClass(self.vars.iconEl, cls);
                    });
                } else {
                    // No icon configured - hide
                    shmi.addClass(self.element, "no-icon");
                }

                // Label is optional, but prev, next and current are req'd for all variants
                if (self.vars.prevEl === null || self.vars.nextEl === null || self.vars.currentValueEl === null) {
                    // As this should never happen we do a bit more work here in cleaning up than checking above
                    shmi.log("[IQ:iq-button-rocker] One of the elements is missing (previous, next, current)", 3);
                    self.vars.prevEl = null;
                    self.vars.nextEl = null;
                    self.vars.currentValueEl = null;
                    self.vars.labelEl = null;
                    self.vars.unitEl = null;
                    self.vars.iconEl = null;
                    return;
                }

                // Unit
                setUnitTextImpl(self, self.config['unit-text']);

                // Label
                setLabelImpl(self, self.config.label);

                // Current value
                if (typeof self.vars.value !== 'undefined') {
                    self.setCurrentElementValue();
                }
                self.createInputField(self.vars.currentValueEl, self.validate.bind(self));

                /*****************/
                /*** LISTENERS ***/
                /*****************/
                let intervalTriggered = false;

                // PREV BUTTON
                var prevFuncs = {
                    onPress: function(x, y, event) {
                        shmi.addClass(self.vars.prevEl, 'pressed');
                        if (self.config["auto-repeat"] === true) {
                            clearInterval(self.vars.buttonInterval);
                            intervalTriggered = false;
                            self.vars.buttonInterval = setInterval(() => {
                                intervalTriggered = true;
                                incValue(self, -getButtonStepping(self));
                            }, self.config["auto-repeat-interval"]);
                        }
                    },
                    onRelease: function() {
                        shmi.removeClass(self.vars.prevEl, 'pressed');
                        if (self.config["auto-repeat"] === true) {
                            clearInterval(self.vars.buttonInterval);
                        }
                    },
                    onClick: function() {
                        if (!intervalTriggered) {
                            incValue(self, -getButtonStepping(self));
                        }
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.prevEl, prevFuncs));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.prevEl, prevFuncs));

                // NEXT BUTTON
                var nextFuncs = {
                    onPress: function(x, y, event) {
                        shmi.addClass(self.vars.nextEl, 'pressed');
                        if (self.config["auto-repeat"] === true) {
                            clearInterval(self.vars.buttonInterval);
                            intervalTriggered = false;
                            self.vars.buttonInterval = setInterval(() => {
                                intervalTriggered = true;
                                incValue(self, getButtonStepping(self));
                            }, self.config["auto-repeat-interval"]);
                        }
                    },
                    onRelease: function() {
                        shmi.removeClass(self.vars.nextEl, 'pressed');
                        if (self.config["auto-repeat"] === true) {
                            clearInterval(self.vars.buttonInterval);
                        }
                    },
                    onClick: function() {
                        if (!intervalTriggered) {
                            incValue(self, getButtonStepping(self));
                        }
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.nextEl, nextFuncs));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.nextEl, nextFuncs));

                // KEYDOWN
                // "Validation light" for delimiters.
                self.vars.currentValueEl.addEventListener('keydown', function(evt) {
                    if (self.config.type === shmi.c("TYPE_FLOAT")) {
                        var wrongDelimiter = (self.config['decimal-delimiter'] === ".") ? "," : ".";
                        if (evt.key === wrongDelimiter) {
                            evt.preventDefault();
                        }
                    }
                });
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                shmi.log("[IQ:iq-button-rocker] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                if (self.config.item) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                shmi.log("[IQ:iq-button-rocker] disabled", 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;
                self.vars.currentValueEl.blur();
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                shmi.addClass(this.element, 'locked');

                if (self.vars.isInputTag) {
                    self.vars.currentValueEl.disabled = true;
                }

                shmi.log("[IQ:iq-button-rocker] Locked", 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.removeClass(self.element, 'locked');

                if (self.vars.isInputTag) {
                    self.vars.currentValueEl.disabled = false;
                }

                shmi.log("[IQ:iq-button-rocker] unlocked", 1);
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                var self = this;
                self.vars.value = value;
                self.setCurrentElementValue();
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                var self = this;
                return self.vars.value;
            },

            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(min, max, step) {
                var self = this;
                self.imports.nv.setProperties(self, arguments);
            },

            setUnitText: function(unitText) {
                var self = this;
                if (self.vars.unitEl && self.config['auto-unit-text']) {
                    setUnitTextImpl(self, unitText);
                }
            },

            setLabel: function(labelText) {
                var self = this;

                if (self.vars.labelEl && self.config['auto-label']) {
                    setLabelImpl(self, labelText);
                }
            },

            /**
             * Creates the input field for the IQ Rocker Button
             *
             * @param element - base element of input field
             * @param validateFunc - function to use for validation
             */
            createInputField: function(element, validateFunc) {
                var self = this;

                // VALIDATE: BASE ELEMENT
                if (!element) {
                    shmi.log('[IQ:iq-button-rocker] No base element provided', 3);
                    return;
                }

                /*****************/
                /*** LISTENERS ***/
                /*****************/
                // All required elements found
                var fieldFuncs = {
                    onClick: function() {
                        var appConfig = shmi.requires("visuals.session.config"),
                            keyboardEnabled = (appConfig.keyboard && appConfig.keyboard.enabled); // get the keyboard config from `project.json`

                        if (self.config["numpad-enabled"] || keyboardEnabled) {
                            var im = shmi.visuals.session.ItemManager,
                                nv = shmi.requires("visuals.tools.numericValues"),
                                vs = null,
                                params = {
                                    "decimal-delimiter": self.config["decimal-delimiter"],
                                    "unit": (self.vars.unit !== undefined) ? self.vars.unit : self.config["unit-text"],
                                    "label": (self.vars.label !== undefined) ? self.vars.label : self.config.label,
                                    "value": self.vars.value,
                                    "callback": function(val) {
                                        self.vars.value = val;
                                        self.updateValue();
                                        self.setCurrentElementValue();
                                    }
                                };

                            if (!(self.vars && self.vars.valueSettings)) {
                                nv.initValueSettings(self);
                            }

                            vs = self.vars.valueSettings;
                            params.min = vs.min;
                            params.max = vs.max;
                            params.type = vs.type;
                            params.precision = vs.precision;
                            params.item = (typeof self.config.item === "string" && self.config.item.length > 0) ? self.config.item : null;

                            if (self.config.item && im.getItem(self.config.item) && self.config['auto-type']) {
                                self.type = im.items[self.config.item].type;
                            }

                            shmi.numpad(params);
                            return;
                        }

                        // Auto-select all text on click
                        shmi.addClass(element, 'selectableText');
                        if (!self.vars.isInputTag) {
                            element.setAttribute('contenteditable', true);

                            var range = document.createRange();
                            element.focus();
                            if (element.firstChild && (element.firstChild instanceof Text)) {
                                range.setStart(element.firstChild, 0);
                                range.setEnd(element.firstChild, element.firstChild.length);
                                window.getSelection().removeAllRanges();
                                window.getSelection().addRange(range);
                            } else {
                                shmi.log("[IQ:iq-button-rocker] Element not found", 0);
                            }
                        } else {
                            self.vars.currentValueEl.setSelectionRange(0, self.vars.currentValueEl.value.length);
                        }
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(element, fieldFuncs));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(element, fieldFuncs));

                element.addEventListener('keypress', function(event) {
                    if (event.keyCode === 13) {
                        event.preventDefault();
                        window.getSelection().removeAllRanges();
                        element.blur();
                    }
                });

                element.addEventListener('blur', function() {
                    if (!self.vars.isInputTag) {
                        element.setAttribute('contenteditable', false);
                    }
                    window.getSelection().removeAllRanges();
                    shmi.removeClass(element, 'selectableText');
                    shmi.log("[IQ:iq-button-rocker] Blur event", 0);
                    validateFunc(element);
                });
            },

            /**
             * Validation
             *
             * @param element - element to validate the content of
             */
            validate: function(element) {
                var self = this,
                    exp = null,
                    val = null,
                    type = (self.config.item && self.config['auto-type']) ? self.vars.valueSettings.type : self.config.type;

                if (self.vars.isInputTag) {
                    val = self.vars.currentValueEl.value;
                } else {
                    val = self.vars.currentValueEl.textContent;
                }

                var inputString = String(val).replace(self.config['decimal-delimiter'], ".");

                if ([shmi.c("TYPE_INT"), shmi.c("TYPE_FLOAT")].indexOf(type) !== -1) {
                    exp = new RegExp(self.floatRegexp);
                } else {
                    shmi.log("[IQ:iq-button-rocker] Invalid value type '" + self.config.type + "' configured", 3);
                    self.setCurrentElementValue();
                    return;
                }
                if (exp.test(inputString)) {
                    shmi.log("[IQ:iq-button-rocker] Valid value", 1);
                    self.vars.value = self.imports.nv.applyInputSettings(inputString, getValueSettings(self));
                    self.updateValue();
                } else {
                    shmi.log("[IQ:iq-button-rocker] Invalid value", 1);
                }

                self.setCurrentElementValue();
            },

            /**
             * Writes current value to connected data source
             */
            updateValue: function() {
                var self = this;
                self.fire("change", { value: self.vars.value });
                if (self.config.item) {
                    shmi.visuals.session.ItemManager.writeValue(self.config.item, self.vars.value);
                }
            },

            /**
             * Sets the element value
             */
            setCurrentElementValue: function() {
                var self = this;
                if (self.vars.isInputTag) {
                    self.vars.currentValueEl.value = self.imports.nv.formatOutput(self.vars.value, self);
                } else {
                    self.vars.currentValueEl.textContent = self.imports.nv.formatOutput(self.vars.value, self);
                }
            },
            floatRegexp: "(^[+-]?[0-9]([.][0-9]*)?$|^[+-]?[1-9]+[0-9]*([.][0-9]*)?$)"
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();
/**
 * iq-button-toggle
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-button-toggle",
 *     "name": null,
 *     "template": "default/iq-button-toggle.variant-01"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "auto-label": Whether to use the auto-label (from item)
 * "tooltip": Tooltip
 * "on-value": ON value (Item)
 * "off-value": OFF value (Item)
 * "confirm-on": Whether to confirm switching it on
 * "confirm-off": Whether to confirm switching it off
 * "on-label": Label for ON
 * "off-label": Label for OFF
 * "on-icon-src": Icon source for ON
 * "on-icon-class": Icon class for ON
 * "on-icon-title": Icon title for ON
 * "off-icon-src": Icon source for OFF
 * "off-icon-class": Icon class for OFF
 * "off-icon-title": Icon title for OFF
 * "confirm-on-text": ON confirmation question,
 * "confirm-off-text": OFF confirmation question
 * "show-icon": Whether to show an icon
 * "show-text": Whether to show text
 * "on-action": ON UI actions
 * "off-action": OFF UI actions
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-button-toggle", // control name in camel-case
        uiType = "iq-button-toggle", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-button-toggle",
        "name": null,
        "template": "default/iq-button-toggle.variant-01",
        "label": '[Label]',
        "item": null,
        "auto-label": true,
        "tooltip": null,
        "on-value": 1,
        "off-value": 0,
        "confirm-on": false,
        "confirm-off": false,
        "on-label": "ON",
        "off-label": "OFF",
        "on-show-text": true,
        "on-show-icon": false,
        "on-icon-src": null,
        "on-icon-class": null,
        "on-tooltip": null,
        "off-show-text": true,
        "off-show-icon": false,
        "off-icon-src": null,
        "off-icon-class": null,
        "off-tooltip": null,
        "confirm-on-text": "${V_CONFIRM_ON}",
        "confirm-off-text": "${V_CONFIRM_OFF}",
        "show-icon": false,
        "show-text": true,
        "on-action": [],
        "off-action": [],
        "icon-src": null,
        "icon-class": null
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined" || !self.config['show-text']) {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: 0,
            active: false,

            // DOM Elements
            labelEl: null,
            buttonEl: null,
            offStateEl: null,
            onStateEl: null,
            onLabelEl: null,
            offLabelEl: null,
            iconEl: null,
            onIconEl: null,
            offIconEl: null,

            initialized: false,
            subscriptionTargetId: null,
            onAction: null,
            offAction: null
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager"
        },

        /* array of custom event types fired by this control */
        events: ["change"],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this,
                    core = shmi.requires("visuals.core");

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.labelEl = shmi.getUiElement('label', self.element);
                self.vars.buttonEl = shmi.getUiElement('button', self.element);
                self.vars.iconEl = shmi.getUiElement('icon', self.element);

                self.vars.offStateEl = shmi.getUiElement('off-state', self.element);
                if (self.vars.offStateEl) {
                    self.vars.offLabelEl = shmi.getUiElement('label', self.vars.offStateEl);
                    self.vars.offIconEl = shmi.getUiElement('toggle-icon', self.vars.offStateEl);

                    if (!self.vars.offLabelEl || !self.config['off-label'] || !self.config['off-show-text']) {
                        shmi.addClass(self.vars.offStateEl, "no-label");
                    } else {
                        self.vars.offLabelEl.textContent = shmi.localize(self.config['off-label']);
                    }

                    if (!self.vars.offIconEl || !self.config['off-show-icon'] || !(self.config['off-icon-class'] || self.config['off-icon-src'])) {
                        shmi.addClass(self.vars.offStateEl, "no-icon");
                    } else {
                        self.setupIconElement(self.vars.offIconEl, self.config['off-icon-class'], self.config['off-icon-src'], self.config['off-show-icon'], self.config['off-show-text']);
                    }
                }

                self.vars.onStateEl = shmi.getUiElement('on-state', self.element);
                if (self.vars.onStateEl) {
                    self.vars.onLabelEl = shmi.getUiElement('label', self.vars.onStateEl);
                    self.vars.onIconEl = shmi.getUiElement('toggle-icon', self.vars.onStateEl);

                    if (!self.vars.onLabelEl || !self.config['on-label'] || !self.config['on-show-text']) {
                        shmi.addClass(self.vars.onStateEl, "no-label");
                    } else {
                        self.vars.onLabelEl.textContent = shmi.localize(self.config['on-label']);
                    }

                    if (!self.vars.onIconEl || !self.config['on-show-icon'] || !(self.config['on-icon-class'] || self.config['on-icon-src'])) {
                        shmi.addClass(self.vars.onStateEl, "no-icon");
                    } else {
                        self.setupIconElement(self.vars.onIconEl, self.config['on-icon-class'], self.config['on-icon-src'], self.config['on-show-icon'], self.config['on-show-text']);
                    }
                }

                if (!self.vars.offStateEl || !self.vars.onStateEl) {
                    shmi.log("[IQ:iq-button-toggle] At least one state element is missing", 1);
                    return;
                }

                // Label
                setLabelImpl(self, self.config.label);

                /************/
                /*** ICON ***/
                /************/
                if (!self.vars.iconEl) {
                    shmi.addClass(self.element, "no-icon");
                    shmi.log('[IQ:iq-button-rocker] no button-icon element provided', 1);
                } else if (self.config['icon-src'] && self.config['show-icon']) {
                    self.vars.iconEl.style.backgroundImage = `url(${self.config['icon-src']})`;
                } else if (self.config['icon-class'] && self.config['show-icon']) {
                    var iconClasses = self.config['icon-class'].trim().split(" ");
                    iconClasses.forEach(function(cls) {
                        shmi.addClass(self.vars.iconEl, cls);
                    });
                } else {
                    // No icon configured - hide
                    shmi.addClass(self.element, "no-icon");
                }

                /*****************/
                /*** LISTENERS ***/
                /*****************/
                var eventFunctions = {
                    onPress: function() {
                        shmi.log("[IQ:iq-button-toggle] Button pressed", 0);
                        shmi.addClass(self.element, 'pressed');
                    },
                    onRelease: function() {
                        shmi.log("[IQ:iq-button-toggle] Button released", 0);
                        shmi.removeClass(self.element, 'pressed');
                    },
                    onClick: function() {
                        if ((self.vars.value === self.config['on-value'])) {
                            // TURN OFF
                            if (self.config['confirm-off']) {
                                shmi.confirm(self.config['confirm-off-text'], function(conf) {
                                    if (conf) {
                                        self.toggle();
                                    }
                                });
                            } else {
                                self.toggle();
                            }
                            // TURN ON
                        } else if (self.config['confirm-on']) {
                            shmi.confirm(self.config['confirm-on-text'], function(conf) {
                                if (conf) {
                                    self.toggle();
                                }
                            });
                        } else {
                            self.toggle();
                        }
                    }
                };

                if (!self.vars.buttonEl) {
                    shmi.log('[IQ:iq-button-rocker] no button element provided', 1);
                } else {
                    self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.buttonEl, eventFunctions));
                    self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.buttonEl, eventFunctions));
                }

                // Reset on- and off-values in case they are set and only an event is set
                if (!self.config.item && self.config.event) {
                    self.config['on-value'] = 1;
                    self.config['off-value'] = 0;
                }

                // Initialize with OFF value
                self.vars.value = self.config['off-value'];

                /******************/
                /*** UI Actions ***/
                /******************/
                if (Array.isArray(self.config["on-action"]) && self.config["on-action"].length) {
                    self.vars.onAction = new core.UiAction(self.config["on-action"], self);
                }
                if (Array.isArray(self.config["off-action"]) && self.config["off-action"].length) {
                    self.vars.offAction = new core.UiAction(self.config["off-action"], self);
                }
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                shmi.log("[IQ:iq-button-toggle] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                if (self.config.item) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                shmi.log("[IQ:iq-button-toggle] disabled", 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                shmi.addClass(self.element, 'locked');

                shmi.log("[IQ:iq-button-toggle] Locked", 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.removeClass(self.element, 'locked');

                shmi.log("[IQ:iq-button-toggle] unlocked", 1);
            },

            /**
             * Toggles the state of the Toggle Button
             */
            toggle: function() {
                var self = this,
                    turnOn = (self.vars.value !== self.config["on-value"]);

                // Item / value
                if (self.config.item) {
                    if (turnOn) {
                        self.imports.im.writeValue(self.config.item, self.config['on-value']);
                    } else {
                        self.imports.im.writeValue(self.config.item, self.config['off-value']);
                    }
                } else if (!turnOn) {
                    self.setValue(self.config['off-value']);
                } else {
                    self.setValue(self.config['on-value']);
                }

                // UI Actions
                if (turnOn && self.vars.onAction) {
                    self.vars.onAction.execute();
                } else if (!turnOn && self.vars.offAction) {
                    self.vars.offAction.execute();
                }
            },

            /**
             * Gets the best tooltip value for the buttons current state.
             *
             * @override
             */
            getTooltip: function() {
                var self = this,
                    superTooltip = shmi.visuals.core.BaseControl.prototype.getTooltip.call(this);

                if (superTooltip) {
                    return superTooltip;
                } else if (self.vars.value === self.config['off-value']) {
                    if (shmi.objectHasOwnProperty(self.config, "off-tooltip")) {
                        return self.config['off-tooltip'];
                    }
                } else if (self.vars.value === self.config['on-value']) {
                    if (shmi.objectHasOwnProperty(self.config, "on-tooltip")) {
                        return self.config['on-tooltip'];
                    }
                }

                return null;
            },

            /**
             * Sets the current value of the Toggle Button
             *
             * @param {string|int|float} value - new value to set
             * @param {string} type
             * @param {string} name
             */
            onSetValue: function(value, type, name) {
                var self = this,
                    oldValue = self.vars.value;

                self.vars.value = parseFloat(value);

                // Show/hide elements
                if (self.vars.value === self.config['off-value']) {
                    shmi.removeClass(self.vars.buttonEl, "iq-on-state");
                    shmi.addClass(self.vars.buttonEl, "iq-off-state");
                } else {
                    shmi.removeClass(self.vars.buttonEl, "iq-off-state");
                    shmi.addClass(self.vars.buttonEl, "iq-on-state");
                }

                // Has changed -> fire events
                if (self.vars.value !== oldValue) {
                    self.setTooltip(self.getTooltip());
                    self.fire("change", {
                        value: self.vars.value
                    });
                }
            },

            /**
             * Retrieves the current value ot the Toggle Button
             *
             * @return value - current value
             */
            getValue: function() {
                return this.vars.value;
            },

            setLabel: function(labelText) {
                var self = this;

                if (self.vars.labelEl && self.config['auto-label']) {
                    setLabelImpl(self, labelText);
                }
            },

            /**
             * Sets up the icon depending on the configuration
             *
             * @param {object} iconEl
             * @param {string} iconClass
             * @param {string} iconSrc
             */
            setupIconElement: function(iconEl, iconClass, iconSrc) {
                if (iconClass) {
                    var onIconClass = iconClass.trim().split(" ");
                    onIconClass.forEach(function(cls) {
                        shmi.addClass(iconEl, cls);
                    });
                } else if (iconSrc) {
                    iconEl.style.backgroundImage = `url(${iconSrc})`;
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-button
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-button",
 *     "name": null,
 *     "template": "default/iq-button"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "min": Minimum value
 * "max": Maximum value
 * "precision": Precision
 * "decimal-delimiter": Decimal delimiter
 * "type": Type is INT
 * "auto-label": Whether to use the auto-label (from item)
 * "auto-min": Whether to use auto-min (from item)
 * "auto-max": Whether to use auto-max (from item)
 * "auto-precision": Whether to use the auto-precision (from item)
 * "auto-type": Whether to use the auto-type (from item)
 * "numpad-enabled": Whether the numpad is enabled
 * "icon-src": Default icon source
 * "icon-class": Default icon class
 * "tooltip": Tooltip
 * "action-release" UI Action on button release
 * "action-while-pressed": UI Action while pressed
 * "interval-while-pressed": UI Action interval while pressed
 * "disable-alarms": Whether to prevent the default alarm styling when alarm/prewarn limits are reached. By disabling you can manually set the classes warn/preWarn via a condition on the widget.
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-button", // control name in camel-case
        uiType = "iq-button", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-button",
        "name": null,
        "template": "default/iq-button.iq-variant-01",

        "item": null,

        "label": '[Label]',
        "auto-label": true,
        "label-from-item": false,

        "icon-src": null,
        "icon-class": null,

        "action": null,
        "onClick": null,

        "tooltip": null,

        "write-bool": false,
        "on-value": 1,
        "off-value": 0,

        "action-release": null,
        "action-while-pressed": null,
        "interval-while-pressed": null,
        "action-pressed": null,

        "disable-alarms": false,

        "show-icon": false
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined") {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        tooltipProperties: ['tooltip'],

        /* instance variables */
        vars: {
            listeners: [],
            value: 0,
            active: false,

            // DOM Elements
            buttonEl: null,
            labelEl: null,
            iconEl: null,

            label: null,
            action: null,
            actionPress: null,
            actionRelease: null,
            actionWhilePressed: null,
            actionWhilePressedTimer: null,

            // For handling keyboard input on button (ENTER)
            keyDownListener: null,
            keyUpListener: null,
            keyDownOnMe: null,

            mouseListener: null,
            touchListener: null,
            initialized: false,
            rafId: 0,
            subscriptionTargetId: null,

            monoFlopInterval: 0,

            // Conditional Control class
            conditional: null,

            tokens: []
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues"
        },

        /* array of custom event types fired by this control */
        events: ['press', 'release', 'click'],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /***************/
            /*** ON INIT ***/
            /***************/
            onInit: function() {
                var self = this;

                self.imports.nv.initValueSettings(self);

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.buttonEl = shmi.getUiElement('button', self.element);
                self.vars.labelEl = shmi.getUiElement('label', self.element);
                self.vars.iconEl = shmi.getUiElement('icon', self.element);

                /************/
                /*** ICON ***/
                /************/
                if (!self.vars.iconEl) {
                    shmi.log('[IQ:iq-button] no button-icon element provided', 1);
                } else if (self.config['icon-src'] && self.config['show-icon']) {
                    self.vars.iconEl.style.backgroundImage = `url(${self.config['icon-src']})`;
                } else if (self.config['icon-class'] && self.config['show-icon']) {
                    var iconClasses = self.config['icon-class'].trim().split(" ");
                    iconClasses.forEach(shmi.addClass.bind(shmi, self.vars.iconEl));
                } else {
                    // No icon configured - hide
                    shmi.addClass(self.element, "no-icon");
                }

                // Label and icon are optional, but button element is required
                if (!self.vars.buttonEl) {
                    shmi.log("[IQ:iq-button] template is missing button element", 3);
                    self.vars.buttonEl = null;
                    self.vars.labelEl = null;
                    self.vars.iconEl = null;
                    return;
                }

                // Label
                setLabelImpl(self, self.config.label);

                // UI Actions
                if (self.config.action) {
                    self.vars.action = new shmi.visuals.core.UiAction(self.config.action, self);
                }
                if (self.config['action-press']) {
                    self.vars.actionPress = new shmi.visuals.core.UiAction(self.config['action-press'], self);
                }
                if (self.config['action-release']) {
                    self.vars.actionRelease = new shmi.visuals.core.UiAction(self.config['action-release'], self);
                }
                if (self.config['action-while-pressed']) {
                    self.vars.actionWhilePressed = new shmi.visuals.core.UiAction(self.config['action-while-pressed'], self);
                }

                /*****************/
                /*** LISTENERS ***/
                /*****************/
                // PREV BUTTON
                var session = shmi.visuals.session;
                var btnFuncs = {
                    onClick: function(x, y, e) {
                        if (session.FocusElement !== null) {
                            session.FocusElement.blur();
                            session.FocusElement = null;
                        }
                        if (self.vars.buttonEl instanceof HTMLElement) {
                            self.vars.buttonEl.focus();
                            session.FocusElement = self.vars.buttonEl;
                            shmi.log("[IQ:iq-button] focused", 1);
                        } else {
                            shmi.log("[IQ:iq-button] only HTMLElements may be focused, type: " + (self.vars.buttonEl.constructor), 1);
                        }

                        self.fire('click', {
                            x: x,
                            y: y,
                            event: e
                        });

                        if (self.vars.action) {
                            self.vars.action.execute();
                        }

                        if (self.onClick) {
                            self.onClick(self);
                        }
                    },
                    onRelease: function(x, y, e) {
                        shmi.removeClass(self.vars.buttonEl, 'pressed');
                        self.fire('release', {
                            x: x,
                            y: y,
                            event: e
                        });
                        if (self.config['write-bool'] && self.config.item) {
                            self.writeValue(self.config["off-value"]);
                        }

                        // Stop any running timer on release
                        if (self.vars.actionWhilePressedTimer) {
                            clearInterval(self.vars.actionWhilePressedTimer);
                        }

                        if (self.vars.actionRelease) {
                            self.vars.actionRelease.execute();
                        }
                    },
                    onPress: function(x, y, e) {
                        shmi.addClass(self.vars.buttonEl, 'pressed');
                        self.fire('press', {
                            x: x,
                            y: y,
                            event: e
                        });

                        if (self.config['write-bool'] && self.config.item) {
                            self.writeValue(self.config["on-value"]);
                        }

                        // UI Action: onPress
                        if (self.vars.actionPress) {
                            self.vars.actionPress.execute();
                        }

                        // UI Action while pressed
                        if (self.vars.actionWhilePressed && self.config['interval-while-pressed']) {
                            if (self.vars.actionWhilePressedTimer) {
                                clearInterval(self.vars.actionWhilePressedTimer);
                            }

                            // While pressed timer
                            self.vars.actionWhilePressedTimer = setInterval(function() {
                                self.vars.actionWhilePressed.execute();
                            }, self.config['interval-while-pressed']);
                        }
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.buttonEl, btnFuncs));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.buttonEl, btnFuncs));

                // Key-up listener
                self.vars.keyUpListener = function(e) {
                    var valid = self.vars.keyDownOnMe;
                    if (valid && (e.code === 'Enter' || e.code === 'NumpadEnter') && self.element === document.activeElement) {
                        shmi.removeClass(self.element, 'pressed');
                        var rect = self.element.getBoundingClientRect();
                        btnFuncs.onClick(rect.left + (rect.width / 2), rect.top + (rect.height / 2), e);
                        self.vars.keyDownOnMe = false;
                    }
                };

                // Key-down listener
                self.vars.keyDownListener = function(e) {
                    if ((e.code === 'Enter' || e.code === 'NumpadEnter') && self.element === document.activeElement) {
                        shmi.addClass(self.element, 'pressed');
                        self.vars.keyDownOnMe = true;
                    }
                };
            },

            /*****************/
            /*** ON ENABLE ***/
            /*****************/
            onEnable: function() {
                var self = this;

                // Conditional
                if (self.vars.conditional !== null) {
                    if (self.vars.conditional.item !== self.config.item) {
                        self.vars.conditional.item = self.config.item;
                    }
                } else if (self.config.item && !self.config['disable-alarms']) {
                    self.vars.conditional = shmi.createConditional(self.element, self.config.item);
                }

                if (self.vars.conditional) {
                    self.vars.conditional.enable();
                }

                self.element.setAttribute('tabindex', '0');

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                // Has Item
                if (self.config.item) {
                    // Item Lock
                    if (self.config["disable-item-lock"]) {
                        var h = self.imports.im.getItemHandler();
                        h.setValue = function(value) {
                            self.setValue(value);
                        };
                        self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, h);
                    } else {
                        self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                    }

                    // Monoflop
                    if (self.config.monoflop) {
                        var pTok = self.listen("press", function() {
                            clearInterval(self.vars.monoFlopInterval);
                            self.vars.monoFlopInterval = setInterval(function() {
                                self.imports.im.writeValue(self.config.item, self.config['monoflop-value'], { skipSameValueCheck: true });
                            }, self.config['monoflop-interval']);
                        });
                        var rTok = self.listen("release", function() {
                            clearInterval(self.vars.monoFlopInterval);
                        });
                        self.vars.tokens.push(pTok, rTok);
                    }
                }

                // Key listeners
                self.element.addEventListener('keyup', self.vars.keyUpListener, false);
                self.element.addEventListener('keydown', self.vars.keyDownListener, false);

                shmi.log("[IQ:iq-button] Enabled", 1);
            },

            /******************/
            /*** ON DISABLE ***/
            /******************/
            onDisable: function() {
                var self = this;

                // Conditional
                if (self.vars.conditional) {
                    self.vars.conditional.disable();
                }

                // Unsubscribe Item
                if (self.config.item) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                // Restore tabindex
                self.element.removeAttribute('tabindex');

                self.vars.tokens.forEach(function(t) {
                    t.unlisten();
                });
                self.vars.tokens = [];

                // Disable listeners
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
                self.element.removeEventListener('keyup', self.vars.keyUpListener);
                self.element.removeEventListener('keydown', self.vars.keyDownListener);

                // Disable timer
                if (self.vars.actionWhilePressedTimer) {
                    clearInterval(self.vars.actionWhilePressedTimer);
                    self.vars.actionWhilePressedTimer = null;
                }

                shmi.log("[IQ:iq-button] disabled", 1);
            },

            /***************/
            /*** ON LOCK ***/
            /***************/
            onLock: function() {
                var self = this;

                // Disable listeners
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                if (self.vars.buttonEl) {
                    self.vars.buttonEl.removeAttribute('tabindex');
                    self.vars.buttonEl.removeEventListener('keyup', self.vars.keyUpListener);
                    self.vars.buttonEl.removeEventListener('keydown', self.vars.keyDownListener);
                    self.vars.buttonEl.blur();
                }

                // CSS
                shmi.addClass(self.element, 'locked');

                shmi.log("[IQ:iq-button] Locked", 1);
            },

            /*****************/
            /*** ON UNLOCK ***/
            /*****************/
            onUnlock: function() {
                var self = this;

                // Enable listeners
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                // CSS
                shmi.removeClass(self.element, 'locked');

                if (self.vars.buttonEl) {
                    self.vars.buttonEl.addEventListener('keyup', self.vars.keyUpListener, false);
                    self.vars.buttonEl.addEventListener('keydown', self.vars.keyDownListener, false);
                    self.vars.buttonEl.setAttribute('tabindex', 0);
                }

                shmi.log("[IQ:iq-button] unlocked", 1);
            },

            /*******************/
            /** ON SET VALUE ***/
            /*******************/
            onSetValue: function(value, type, name) {
                var self = this;
                self.vars.value = value;
                if (self.config['label-from-item'] === true) {
                    shmi.caf(self.rafId);
                    self.rafId = shmi.raf(function() {
                        setLabelImpl(self, shmi.localize(self.imports.nv.formatOutput(value, self)));
                    });
                }
            },

            /*****************/
            /*** GET VALUE ***/
            /*****************/
            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                var self = this;
                return self.vars.value;
            },

            /*************************/
            /*** ON SET PROPERTIES ***/
            /*************************/
            onSetProperties: function(min, max, step) {
                var self = this;
                self.imports.nv.setProperties(self, arguments);
            },

            /********************/
            /*** ON SET LABEL ***/
            /********************/
            setLabel: function(labelText) {
                var self = this;

                if (self.vars.labelEl && self.config['auto-label']) {
                    setLabelImpl(self, labelText);
                }
            },

            /**
             * Writes 1/0 to connected data source
             */
            writeValue: function(value) {
                var self = this;
                if (self.config.item) {
                    self.imports.im.writeValue(self.config.item, value, { skipSameValueCheck: true });
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-checkbox
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-checkbox",
 *     "name": null,
 *     "template": "default/iq-checkbox"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "type": Type is INT
 * "auto-label": Whether to use the auto-label (from item)
 * "icon-src": Default icon source
 * "icon-class": Default icon class
 * "tooltip": Tooltip
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-checkbox", // control name in camel-case
        uiType = "iq-checkbox", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-checkbox",
        "name": null,
        "template": "default/iq-checkbox.variant-01",
        "label": '[Label]',
        "item": null,
        "type": shmi.c("TYPE_INT"),
        "auto-label": true,
        "show-icon": false,
        "show-text": true,
        "icon-src": null,
        "icon-class": null,
        "tooltip": null,
        "on-value": 1,
        "off-value": 0,
        "pressed-class": "pressed",
        "confirm-off-text": "${V_CONFIRM_OFF}",
        "confirm-on-text": "${V_CONFIRM_ON}",
        "confirm-on": false,
        "confirm-off": false
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined") {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: 0,
            active: false,

            // DOM Elements
            checkboxEl: null,
            labelEl: null,
            iconEl: null,
            containerEl: null,

            label: null
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues"
        },

        /* array of custom event types fired by this control */
        events: ['change'],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.checkboxEl = shmi.getUiElement('checkbox', self.element);
                self.vars.labelEl = shmi.getUiElement('label', self.element);
                self.vars.iconEl = shmi.getUiElement('icon', self.element);
                self.vars.containerEl = shmi.getUiElement('iq-checkbox-container', self.element);

                /************/
                /*** ICON ***/
                /************/
                if (!self.vars.iconEl) {
                    shmi.log('[IQ:iq-checkbox] no button-icon element provided', 1);
                    shmi.addClass(self.element, "no-icon");
                } else if (self.config['icon-src'] && self.config['show-icon']) {
                    self.vars.iconEl.style.backgroundImage = `url(${self.config['icon-src']})`;
                } else if (self.config['icon-class'] && self.config['show-icon']) {
                    shmi.addClass(self.vars.iconEl, self.config['icon-class']);
                } else {
                    // No icon configured - hide
                    shmi.addClass(self.element, "no-icon");
                }

                self.vars.value = self.config["off-value"];

                // Label is optional, but prev, next and current are req'd for all variants
                if (self.vars.checkboxEl === null) {
                    // As this should never happen we do a bit more work here in cleaning up than checking above
                    shmi.log("[IQ:iq-checkbox] Checkbox element is missing", 3);
                    self.vars.checkboxEl = null;
                    self.vars.labelEl = null;
                    self.vars.iconEl = null;
                    return;
                }

                // Label
                if (self.config["show-text"]) {
                    setLabelImpl(self, self.config.label);
                } else {
                    setLabelImpl(self, null);
                }

                /*****************/
                /*** LISTENERS ***/
                /*****************/
                if (self.vars.containerEl) {
                    var fieldFuncs = {
                        onPress: function(x, y, event) {
                            shmi.addClass(self.element, self.config["pressed-class"]);
                        },
                        onRelease: function() {
                            shmi.removeClass(self.element, self.config["pressed-class"]);
                        },
                        onClick: function(x, y, e) {
                            e.preventDefault();
                            if (self.vars.value === self.config['on-value']) {
                                // IS ON -> OFF
                                if (self.config['confirm-off']) {
                                    shmi.confirm(self.config['confirm-off-text'], function onConfirmed(confirmed) {
                                        if (confirmed) {
                                            self.updateValue(self.config['off-value']);
                                        }
                                    });
                                } else {
                                    self.updateValue(self.config['off-value']);
                                }
                            } else if (self.config['confirm-on']) {
                                // IS OFF -> ON
                                shmi.confirm(self.config['confirm-on-text'], function onConfirmed(confirmed) {
                                    if (confirmed) {
                                        self.updateValue(self.config['on-value']);
                                    }
                                });
                            } else {
                                self.updateValue(self.config['on-value']);
                            }
                        }
                    };
                    self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.containerEl, fieldFuncs));
                    self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.containerEl, fieldFuncs));
                } else {
                    console.error("[IQ:iq-checkbox] 'iq-checkbox-container' element is missing, cannot attach listener.");
                }
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                shmi.log("[IQ:iq-checkbox] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                if (self.config.item) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                shmi.log("[IQ:iq-checkbox] disabled", 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;
                self.vars.checkboxEl.blur();
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                shmi.addClass(this.element, 'locked');
                self.vars.checkboxEl.disabled = true;

                shmi.log("[IQ:iq-checkbox] Locked", 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.removeClass(self.element, 'locked');
                self.vars.checkboxEl.disabled = false;

                shmi.log("[IQ:iq-checkbox] unlocked", 1);
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                var self = this,
                    changed = false;

                if (value === self.config['on-value']) {
                    self.vars.value = self.config['on-value'];
                    changed = true;
                    shmi.addClass(self.element, 'checked');
                } else {
                    self.vars.value = self.config['off-value'];
                    changed = true;
                    shmi.removeClass(self.element, 'checked');
                }
                if (changed) {
                    self.fire("change", {
                        value: self.vars.value
                    });
                }
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                return this.vars.value;
            },

            setLabel: function(labelText) {
                var self = this;

                if (self.vars.labelEl && self.config['auto-label'] && self.config['show-text']) {
                    setLabelImpl(self, labelText);
                }
            },

            /**
             * Writes current value to connected data source
             */
            updateValue: function(value) {
                var self = this;
                if (self.config.item) {
                    self.imports.im.writeValue(self.config.item, value);
                } else {
                    self.setValue(value);
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();
/**
 * iq-date-time
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-date-time",
 *     "name": null,
 *     "template": "custom/controls/iq-date-time"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * display-utc {boolean}: Whether or not the displayed time should be UTC or not
 * display-format {string}: Format string for the displayed date
 * input-format {string}: Format string to use when parsing the items value
 * invalid-text {string}: String to display if the item does not contain a valid timestamp.
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iqDateTime", // control name in camel-case
        uiType = "iq-date-time", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-date-time",
        "name": null,
        "template": "default/iq-date-time.variant-01",
        "item": null,
        "display-utc": false,
        "display-format": "$YYYY-$MM-$DD $HH:$mm:$ss",
        "input-format": "$X",
        "invalid-text": "${date-time.invalid-time}",
        "tooltip": null,
        "value-as-tooltip": false
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: 0,
            active: false,

            // DOM Elements
            domEl: null,

            subscriptionTargetId: null,
            localizedDisplayFormat: null,
            localizedInvalidText: null,
            rafRunning: false,
            lastValue: null
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            dt: "visuals.tools.date"
        },

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.domEl = shmi.getUiElement('datetime', self.element);
                if (!self.vars.domEl) {
                    shmi.log("[IQ:iq-date-time] Element is missing", 3);
                    return;
                }

                self.vars.localizedDisplayFormat = shmi.localize(self.config["display-format"]);
                self.vars.localizedInvalidText = shmi.localize(self.config["invalid-text"]);

                self.vars.domEl.textContent = self.vars.localizedInvalidText;
                if (self.config["value-as-tooltip"]) {
                    self.setTooltip(self.vars.localizedInvalidText);
                }
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                shmi.log("[IQ:iq-date-time] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                if (self.vars.subscriptionTargetId) {
                    self.vars.subscriptionTargetId.unlisten();
                    self.vars.subscriptionTargetId = null;
                }

                shmi.log("[IQ:iq-date-time] disabled", 1);
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type) {
                var self = this;

                if (type === shmi.c("TYPE_INT") || type === shmi.c("TYPE_FLOAT")) {
                    self.vars.lastValue = self.imports.dt.parseDateTime(String(value), self.config["input-format"] || "$X");
                } else if (!self.config["input-format"]) {
                    self.vars.lastValue = null;
                } else {
                    self.vars.lastValue = self.imports.dt.parseDateTime(String(value), self.config["input-format"]);
                }

                // Only attempt to update the DOM during an animation frame.
                // Prevents floods of redraws and may perform better anyway
                if (!self.vars.rafRunning) {
                    self.vars.rafRunning = true;
                    shmi.raf(function onDraw() {
                        let localizedText = null;
                        if (!self.vars.lastValue) {
                            localizedText = self.vars.localizedInvalidText;
                        } else {
                            localizedText = self.imports.dt.formatDateTime(self.vars.lastValue, {
                                datestring: self.vars.localizedDisplayFormat,
                                utc: self.config["display-utc"]
                            });
                        }

                        self.vars.domEl.textContent = localizedText;
                        self.vars.domEl.innerHTML = self.vars.domEl.innerHTML.replace(/\n/g, "<br>");
                        if (self.config["value-as-tooltip"]) {
                            self.setTooltip(localizedText.replace(/(\\n)+/g, " "));
                        }

                        self.vars.rafRunning = false;
                    });
                }
            },
            onLock: function() {
                shmi.addClass(this.element, 'locked');
            },
            onUnlock: function() {
                shmi.removeClass(this.element, 'locked');
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-duration-display
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-duration-display",
 *     "name": null,
 *     "template": "custom/controls/iq-duration-display"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "target-ts-item": The target time item
 * "target-ts-format": The date format of the target time item
 * "current-ts-item": The current time item
 * "current-ts-format": The date format of the current time item
 * "display-preset": Different display formats
 * "invalid-text": Text shown when an invalid time occurs
 * "tooltip": Tooltip
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-duration-display", // control name in camel-case
        uiType = "iq-duration-display", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-duration-display",
        "name": null,
        "template": "default/iq-duration-display.variant-01",
        "label": '[Label]',
        "target-ts-item": null,
        "target-ts-format": "$X",
        "current-ts-item": "Systemzeit",
        "current-ts-format": "$X",
        "display-preset": "compact",
        "invalid-text": "${duration-display.invalid-duration}",
        "tooltip": null,
        "value-as-tooltip": false
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the displayed text and tooltip of the control.
     *
     * @param {*} self Reference to the control.
     * @param {string} str Text to display.
     */
    function setText(self, str) {
        if (self.vars.durationEl) {
            self.vars.durationEl.textContent = str;
            if (self.config["value-as-tooltip"]) {
                self.setTooltip(str);
            }
        }
    }

    function doUpdate(self) {
        if (self.vars.tsTarget === null || self.vars.tsCurrent === null) {
            self.vars.lastValue = null;
        } else {
            self.vars.lastValue = self.vars.tsTarget - self.vars.tsCurrent;
        }

        if (!self.vars.rafRunning) {
            self.vars.rafRunning = true;
            shmi.raf(function onDraw() {
                if (!self.vars.lastValue) {
                    setText(self, self.vars.localizedInvalidText);
                } else {
                    setText(self, self.imports.dt.formatDuration(self.vars.lastValue, self.config["display-preset"] || "compact"));
                }

                self.vars.rafRunning = false;
            });
        }
    }

    function makeItemHandler(self, configFormatName, destVarName) {
        return {
            setValue: function setValue(value, type) {
                if (type === shmi.c("TYPE_INT") || type === shmi.c("TYPE_FLOAT")) {
                    self.vars[destVarName] = self.imports.dt.parseDateTime(String(value), self.config[configFormatName] || "$X");
                } else if (!self.config[configFormatName]) {
                    self.vars[destVarName] = null;
                } else {
                    self.vars[destVarName] = self.imports.dt.parseDateTime(String(value), self.config[configFormatName]);
                }

                doUpdate(self);
            }
        };
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            tsTarget: null,
            tsCurrent: null,
            localizedInvalidText: null,
            rafRunning: false,
            lastValue: null,

            // DOM elements
            durationEl: null
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            dt: "visuals.tools.date"
        },

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                self.vars.durationEl = shmi.getUiElement('duration', self.element);
                if (!self.vars.durationEl) {
                    shmi.log("[IQ:iq-duration] Element is missing", 3);
                    return;
                }

                self.vars.localizedInvalidText = shmi.localize(self.config["invalid-text"]);
                setText(self, self.vars.localizedInvalidText);
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;
                if (self.config["target-ts-item"] && self.config["current-ts-item"] && self.vars.listeners.length === 0) {
                    self.vars.listeners.push(self.imports.im.subscribeItem(self.config["target-ts-item"], makeItemHandler(self, "target-ts-format", "tsTarget")));
                    self.vars.listeners.push(self.imports.im.subscribeItem(self.config["current-ts-item"], makeItemHandler(self, "current-ts-format", "tsCurrent")));
                }
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;
                self.vars.listeners.forEach(function(sub) {
                    sub.unlisten();
                });
                self.vars.listeners = [];
            },
            onLock: function() {
                shmi.addClass(this.element, 'locked');
            },
            onUnlock: function() {
                shmi.removeClass(this.element, 'locked');
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();
/**
 * iq-flip-switch
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-flip-switch",
 *     "name": null,
 *     "template": "default/iq-flip-switch"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "icon-src": Default icon source
 * "icon-class": Default icon class
 * "tooltip": Tooltip
 * "on-value": The ON value
 * "off-value": The OFF value
 * "transitionStyle": Legacy
 * "confirm-off-text": OFF confirmation text
 * "confirm-on-text": ON confirmation text
 * "confirm-on": Whether to confirm ON
 * "confirm-off": Whether to confirm OFF
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-flip-switch", // control name in camel-case
        uiType = "iq-flip-switch", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-flip-switch",
        "name": null,
        "template": "default/iq-flip-switch.variant-01",
        "label": '[Label]',
        "auto-label": true,
        "on-label": "${flip-switch.label-on}",
        "off-label": "${flip-switch.label-off}",
        "item": null,
        "show-icon": false,
        "icon-src": null,
        "icon-class": null,
        "tooltip": null,
        "on-value": 1,
        "off-value": 0,
        "transitionStyle": "all .16s linear",
        "confirm-off-text": "${V_CONFIRM_OFF}",
        "confirm-on-text": "${V_CONFIRM_ON}",
        "confirm-on": false,
        "confirm-off": false
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            shmi.addClass(self.element, "no-label");
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined") {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    /**
     * Updates the handle position based on the widgets current value.
     *
     * @param {*} self Reference to the widget
     */
    function updateHandle(self) {
        const value = self.getValue(),
            handleWidth = self.vars.handleEl ? Math.ceil(self.vars.handleEl.getBoundingClientRect().width) : 0,
            boxWidth = self.vars.handleBoxEl ? self.vars.handleBoxEl.clientWidth : 0;

        if (self.vars.isDragging || !self.vars.handleBoxEl) {
            return;
        }

        if (value === self.config['on-value']) {
            self.vars.handleDragOffset = (boxWidth - handleWidth);
        } else {
            self.vars.handleDragOffset = 0;
        }
    }

    /**
     * Creates a listener for the `window.resize` event.
     *
     * @param {*} self Reference to the widget
     * @param {function} handlerFunc Event handler
     * @returns {object}
     */
    function makeResizeListener(self, handlerFunc) {
        return {
            disable: () => window.removeEventListener("resize", handlerFunc),
            enable: () => window.addEventListener("resize", handlerFunc)
        };
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: 0,
            active: false,

            // DOM Elements
            iconEl: null,
            labelEl: null,
            labelOnEl: null,
            labelOffEl: null,
            handleEl: null,
            handleBoxEl: null,

            initialized: false,
            subscriptionTargetId: null,
            isDragging: false,
            width: null,
            handleWidth: null,
            handleDragOffset: 0
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues"
        },

        /* array of custom event types fired by this control */
        events: ["change"],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.iconEl = shmi.getUiElement('icon', self.element);
                self.vars.labelEl = shmi.getUiElement('label', self.element);
                self.vars.labelOnEl = shmi.getUiElement('label-on', self.element);
                self.vars.labelOffEl = shmi.getUiElement('label-off', self.element);
                self.vars.handleEl = shmi.getUiElement('handle', self.element);
                self.vars.handleBoxEl = shmi.getUiElement('handle-box', self.element);

                // Validate req'd elements
                if (!self.vars.handleEl || !self.vars.handleBoxEl) {
                    // As this should never happen we do a bit more work here in cleaning up than checking above
                    shmi.log("[IQ:iq-flip-switch] Handle or handle box element is missing!", 3);
                    self.vars.labelEl = null;
                    self.vars.labelOnEl = null;
                    self.vars.labelOffEl = null;
                    self.vars.handleEl = null;
                    self.vars.handleBoxEl = null;
                    self.vars.iconEl = null;
                    return;
                }

                /************/
                /*** ICON ***/
                /************/
                if (!self.vars.iconEl) {
                    shmi.addClass(self.element, "no-icon");
                    shmi.log('[IQ:iq-flip-switch] no button-icon element provided', 1);
                } else if (self.config['icon-src'] && self.config['show-icon']) {
                    self.vars.iconEl.style.backgroundImage = `url(${self.config['icon-src']})`;
                } else if (self.config['icon-class'] && self.config['show-icon']) {
                    var iconClasses = self.config['icon-class'].trim().split(" ");
                    iconClasses.forEach(function(cls) {
                        shmi.addClass(self.vars.iconEl, cls);
                    });
                } else {
                    // No icon configured - hide
                    shmi.addClass(self.element, "no-icon");
                }

                /**************/
                /*** LABELS ***/
                /**************/
                setLabelImpl(self, self.config.label);

                if (self.vars.labelOnEl) {
                    if (self.config['on-label']) {
                        self.vars.labelOnEl.textContent = shmi.localize(self.config['on-label']);
                    }
                }

                if (self.vars.labelOffEl) {
                    if (self.config['on-label']) {
                        self.vars.labelOffEl.textContent = shmi.localize(self.config['off-label']);
                    }
                }

                /*****************/
                /*** LISTENERS ***/
                /*****************/
                let handleDragAfId = null;
                // Handle element listeners
                var handleEventFunctions = {
                    onDrag: function(dx, dy, e) {
                        const handleWidth = self.vars.handleEl ? Math.ceil(self.vars.handleEl.getBoundingClientRect().width) : 0,
                            boxWidth = self.vars.handleBoxEl ? self.vars.handleBoxEl.clientWidth : 0;

                        e.preventDefault();
                        if (self.vars.handleDragOffset + dx < 0) {
                            self.vars.handleDragOffset = 0;
                        } else if (self.vars.handleDragOffset + dx + handleWidth > boxWidth) {
                            self.vars.handleDragOffset = boxWidth - handleWidth;
                        } else {
                            self.vars.handleDragOffset += dx;
                        }

                        if (handleDragAfId) {
                            shmi.caf(handleDragAfId);
                        }

                        handleDragAfId = shmi.raf(() => {
                            self.vars.handleEl.style.transform = `translate3d(${self.vars.handleDragOffset}px, 0, 0)`;
                        });

                        if (!self.vars.isDragging) {
                            self.vars.isDragging = true;
                        }
                    },
                    onPress: function() {
                        shmi.addClass(self.vars.handleEl, 'pressed');
                    },
                    onRelease: function() {
                        if (self.vars.isDragging) {
                            self.vars.isDragging = false;
                            if (self.vars.handleDragOffset > ((self.vars.handleBoxEl.offsetWidth - self.vars.handleEl.offsetWidth) / 2)) {
                                if (self.config['confirm-on']) {
                                    shmi.confirm(self.config['confirm-on-text'], function(confirmed) {
                                        self.handleOnOff(true, confirmed);
                                    });
                                } else {
                                    self.handleOnOff(true, null);
                                }
                            } else if (self.config['confirm-off']) {
                                shmi.confirm(self.config['confirm-off-text'], function(confirmed) {
                                    self.handleOnOff(false, confirmed);
                                });
                            } else {
                                self.handleOnOff(false, null);
                            }
                        }

                        if (handleDragAfId) {
                            shmi.caf(handleDragAfId);
                        }
                        self.vars.handleEl.style.transform = '';
                        shmi.removeClass(self.vars.handleEl, 'pressed');
                    },
                    onClick: function() {
                        if (self.vars.value === self.config['on-value']) {
                            if (self.config['confirm-off']) {
                                shmi.confirm(self.config['confirm-off-text'], function(conf) {
                                    if (conf) {
                                        self.turnOnOff(false);
                                    }
                                });
                            } else {
                                self.turnOnOff(false);
                            }
                        } else if (self.config['confirm-on']) {
                            shmi.confirm(self.config['confirm-on-text'], function(conf) {
                                if (conf) {
                                    self.turnOnOff(true);
                                }
                            });
                        } else {
                            self.turnOnOff(true);
                        }
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.handleEl, handleEventFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.handleEl, handleEventFunctions));

                // Handle box listeners
                var handleBoxEventFunctions = {
                    onClick: handleEventFunctions.onClick
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.handleBoxEl, handleBoxEventFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.handleBoxEl, handleBoxEventFunctions));

                self.vars.listeners.push(makeResizeListener(self, updateHandle.bind(null, self)));
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.log("[IQ:iq-flip-switch] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                if (self.vars.subscriptionTargetId) {
                    self.vars.subscriptionTargetId.unlisten();
                    self.vars.subscriptionTargetId = null;
                }

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                shmi.log("[IQ:iq-flip-switch] disabled", 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                shmi.addClass(self.element, 'locked');

                shmi.log("[IQ:iq-flip-switch] Locked", 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.removeClass(self.element, 'locked');

                shmi.log("[IQ:iq-flip-switch] unlocked", 1);
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                const self = this,
                    oldValue = self.vars.value;

                if (self.vars.isDragging || !self.vars.handleBoxEl) {
                    return;
                }

                if (value === self.config['on-value']) {
                    self.vars.value = self.config['on-value'];
                    shmi.addClass(self.element, 'on');
                } else {
                    self.vars.value = self.config['off-value'];
                    shmi.removeClass(self.element, 'on');
                }

                if (oldValue !== self.vars.value) {
                    self.fire("change", {
                        value: self.vars.value
                    });

                    if (self.onChange) {
                        self.onChange(self.vars.value);
                    }
                }

                updateHandle(self);
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                return this.vars.value;
            },

            setLabel: function(labelText) {
                var self = this;

                if (self.vars.labelEl && self.config['auto-label']) {
                    setLabelImpl(self, labelText);
                }
            },

            /* Handles ON/OFF dragging */
            handleOnOff: function(turnOn, confirmed) {
                var self = this,
                    newValue = turnOn ? self.config['on-value'] : self.config['off-value'];

                // Only do this when it has been confirmed or no confirmation was enabled
                if (confirmed === true || confirmed === null) {
                    updateHandle(self);

                    if (self.config.item) {
                        shmi.visuals.session.ItemManager.writeValue(self.config.item, newValue);
                    } else {
                        self.setValue(newValue);
                    }
                } else {
                    // Reset to previous value as not confirmed
                    self.setValue(self.vars.value);
                }
            },

            /* Handles ON/OFF click */
            turnOnOff: function(turnOn) {
                var self = this,
                    newValue = turnOn ? self.config['on-value'] : self.config['off-value'];

                if (self.config.item) {
                    self.imports.im.writeValue(self.config.item, newValue);
                } else {
                    self.setValue(newValue);
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-heartbeat
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-heartbeat",
 *     "name": null,
 *     "template": "custom/controls/iq-heartbeat"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "item": The item
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iqHeartbeat", // control name in camel-case
        uiType = "iq-heartbeat", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-heartbeat",
        "name": null,
        "template": "default/iq-heartbeat.variant-01",
        "item": null,
        "item-plc": null,
        "timeout-plc": null,
        "image-ok-src": null,
        "image-ok-title": null,
        "image-fail-src": null,
        "image-fail-title": null,
        "action-fail": null,
        "action-ok": null,
        "item-hmi": null,
        "timeout-hmi": null,
        "scaling-mode": "fit-height"
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Handles the set scaling mode
     *
     * @param {object} self
     */
    function applyScalingMode(self) {
        let scalingMode = "fit-height";

        switch (self.config["scaling-mode"]) {
        case "fit-height":
        case "fit-width":
        case "stretch":
        case "contain":
            scalingMode = self.config["scaling-mode"];
            break;
        default:
        }

        shmi.addClass(self.element, scalingMode);
    }

    /**
     * Tries as long as required until the update is successful
     *
     * @param self
     * @param oldVal
     * @param newVal
     * @param callback
     */
    function performCompareExchange(self, oldVal, newVal, callback) {
        if (self.vars.isEnabled) {
            self.imports.im.compareExchange(self.config['item-hmi'], oldVal, newVal, function(expectedValue, retry, error) {
                if (error && retry) {
                    newVal = expectedValue + 1;
                    if (newVal > Number.MAX_SAFE_INTEGER) {
                        newVal = 0;
                    }

                    performCompareExchange(self, expectedValue, newVal, callback);
                } else if (typeof callback === 'function') {
                    callback(!error);
                }
            });
        }
    }

    /**
     * called if plc item timer elapses
     *
     * @param {object} self control instance
     */
    function onPlcTimerElapsed(self) {
        self.vars.isDisconnected = true;
        self.vars.imgEl.src = self.vars.imagePathFail;
        self.setTooltip(self.config['image-fail-title']);
        if (self.vars.actionFail) {
            self.vars.actionFail.execute();
        }
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            active: false,

            // DOM Elements
            imgEl: null,

            tokens: [],
            plcItemValue: null,
            plcItemTimer: null,
            hmiItemValue: null,
            hmiItemTimer: null,
            hmiItemTimerActive: false,
            isDisconnected: false,
            actionOk: null,
            actionFail: null,

            // Images
            imagePathOk: null,
            imagePathFail: null,

            isEnabled: false
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager"
        },

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.imgEl = shmi.getUiElement('image', self.element);
                if (!self.vars.imgEl) {
                    self.log('Image element is missing!', 3);
                    return;
                }

                // Images from layout template - allows us to use different images per layout
                if (self.vars.imgEl.dataset.imgOk && self.vars.imgEl.dataset.imgFail) {
                    self.vars.imagePathOk = self.vars.imgEl.dataset.imgOk;
                    self.vars.imagePathFail = self.vars.imgEl.dataset.imgFail;
                }

                // Override images if defined by user
                if (self.config['image-ok-src']) {
                    self.vars.imagePathOk = self.config['image-ok-src'];
                }
                if (self.config['image-fail-src']) {
                    self.vars.imagePathFail = self.config['image-fail-src'];
                    self.setTooltip(self.config['image-fail-title']);
                }

                self.vars.imgEl.src = self.vars.imagePathFail; // Initial status

                // UI Actions
                if (self.config['action-fail']) {
                    self.vars.actionFail = new shmi.visuals.core.UiAction(self.config['action-fail'], self);
                }
                if (self.config['action-ok']) {
                    self.vars.actionOk = new shmi.visuals.core.UiAction(self.config['action-ok'], self);
                }

                applyScalingMode(self);
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                self.vars.isEnabled = true;

                /**************************/
                /*** PLC ITEM (MONITOR) ***/
                /**************************/
                if (self.config['item-plc']) {
                    self.vars.tokens.push(self.imports.im.subscribeItem(self.config["item-plc"], self));

                    // Start timer
                    self.vars.plcItemTimer = setTimeout(onPlcTimerElapsed.bind(null, self), self.config['timeout-plc']);
                }

                /************************/
                /*** HMI ITEM (WRITE) ***/
                /************************/
                if (self.config['item-hmi'] && self.config['timeout-hmi']) {
                    let hmiHandler = self.imports.im.getItemHandler();
                    hmiHandler.setValue = function(value) {
                        self.vars.hmiItemValue = parseInt(value);
                    };
                    self.vars.tokens.push(self.imports.im.subscribeItem(self.config["item-hmi"], hmiHandler));

                    // Start timer for HMI item update
                    self.vars.hmiItemTimer = setInterval(function() {
                        if (self.vars.hmiItemTimerActive) {
                            return;
                        }
                        self.vars.hmiItemTimerActive = true;

                        // Increment the value
                        let newVal = self.vars.hmiItemValue;
                        if (newVal === null) {
                            newVal = 0;
                        }
                        if (newVal === Number.MAX_VALUE) {
                            newVal = 0;
                        }

                        newVal += 1;

                        // We intentionally do not monitor the result as we have no way to do anything with it
                        performCompareExchange(self, self.vars.hmiItemValue, newVal, function(success) {
                            self.vars.hmiItemTimerActive = false;
                        });
                    }, self.config['timeout-hmi']);
                }

                self.log('Enabled', 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                self.vars.isEnabled = false;

                self.vars.tokens.forEach(function(t) {
                    t.unlisten();
                });
                self.vars.tokens = [];

                // HMI -> PLC
                if (self.vars.hmiItemTimer) {
                    clearInterval(self.vars.hmiItemTimer);
                    self.vars.hmiItemTimer = null;
                }

                // PLC -> HMI
                if (self.vars.plcItemTimer) {
                    clearTimeout(self.vars.plcItemTimer);
                    self.vars.plcItemTimer = null;
                }

                self.log('Disabled', 1);
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value) {
                var self = this;

                // Don't do anything if no plc timeout is set.
                if (!self.config['timeout-plc']) {
                    return;
                }

                // Stop the running timer as we just got an update!
                if (self.vars.plcItemTimer) {
                    clearTimeout(self.vars.plcItemTimer);
                    self.vars.plcItemTimer = null;
                }

                // Have we been disconnected before?
                if (self.vars.isDisconnected) {
                    self.vars.isDisconnected = false;
                    if (self.vars.actionOk) {
                        self.vars.actionOk.execute();
                    }
                }

                if (self.vars.imgEl.src !== self.vars.imagePathOk) {
                    self.setTooltip(self.config['image-ok-title']);
                    self.vars.imgEl.src = self.vars.imagePathOk;
                }

                //restart timer
                self.vars.plcItemTimer = setTimeout(onPlcTimerElapsed.bind(null, self), self.config['timeout-plc']);
            },

            log: function(msg, level) {
                shmi.log('[IQ:iq-heartbeat] ' + msg, level);
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-image-changer
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-image-changer",
 *     "name": null,
 *     "template": "custom/controls/iq-image-changer"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "item": The item
 * "tooltip": Tooltip
 * "options": Array of options
 * "default-image": The default image to show when none has been configured yet
 * "default-title": The default title
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-image-changer", // control name in camel-case
        uiType = "iq-image-changer", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-image-changer",
        "name": null,
        "template": "default/iq-image-changer.variant-01",
        "item": null,
        "tooltip": null,
        "options": [],
        "default-image": "pics/system/icons/placeholder.svg",
        "default-title": null,
        "scaling-mode": "fit-height",
        "action": null,
        "action-release": null,
        "action-while-pressed": null,
        "interval-while-pressed": null,
        "action-pressed": null,
        "write-bool": false,
        "on-value": 1,
        "off-value": 0
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    function applyScalingMode(self) {
        let scalingMode = "fit-height";

        switch (self.config["scaling-mode"]) {
        case "fit-height":
        case "fit-width":
        case "stretch":
        case "contain":
            scalingMode = self.config["scaling-mode"];
            break;
        default:
        }

        shmi.addClass(self.element, scalingMode);
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: 0,
            active: false,

            // DOM Elements
            imageEl: null,
            action: null,
            actionRelease: null,
            actionWhilePressed: null,
            actionWhilePressedTimer: null,
            mouseListener: null,
            touchListener: null,

            activeOption: null,
            subscriptionTargetId: null
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager"
        },

        /* array of custom event types fired by this control */
        events: ['press', 'release', 'click'],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.imageEl = shmi.getUiElement('image', self.element);

                // UI Actions
                if (self.config.action) {
                    self.vars.action = new shmi.visuals.core.UiAction(self.config.action, self);
                }
                if (self.config['action-press']) {
                    self.vars.actionPress = new shmi.visuals.core.UiAction(self.config['action-press'], self);
                }
                if (self.config['action-release']) {
                    self.vars.actionRelease = new shmi.visuals.core.UiAction(self.config['action-release'], self);
                }
                if (self.config['action-while-pressed']) {
                    self.vars.actionWhilePressed = new shmi.visuals.core.UiAction(self.config['action-while-pressed'], self);
                }

                if (Array.isArray(self.config.action) && self.config.action.length > 0) {
                    shmi.addClass(self.vars.imageEl, 'clickable');
                } else {
                    shmi.removeClass(self.vars.imageEl, 'clickable');
                }

                applyScalingMode(self);

                self.getActive(null);

                /*****************/
                /*** LISTENERS ***/
                /*****************/
                const session = shmi.visuals.session;
                var eventFunctions = {
                    onClick: function(x, y, e) {
                        if (session.FocusElement !== null) {
                            session.FocusElement.blur();
                            session.FocusElement = null;
                        }
                        if (self.vars.imageEl instanceof HTMLElement) {
                            self.vars.imageEl.focus();
                            session.FocusElement = self.vars.imageEl;
                            shmi.log("[IQ:iq-image-changer] focused", 1);
                        } else {
                            shmi.log("[IQ:iq-image-changer] only HTMLElements may be focused, type: " + (self.vars.imageEl.constructor), 1);
                        }

                        self.fire("click", {
                            x: x,
                            y: y,
                            event: e
                        });

                        if (self.vars.action) {
                            self.vars.action.execute();
                        }
                    },
                    onRelease: function(x, y, e) {
                        self.fire('release', {
                            x: x,
                            y: y,
                            event: e
                        });
                        if (self.config['write-bool'] && self.config.item) {
                            self.writeValue(self.config["off-value"]);
                        }

                        // Stop any running timer on release
                        if (self.vars.actionWhilePressedTimer) {
                            clearInterval(self.vars.actionWhilePressedTimer);
                        }

                        if (self.vars.actionRelease) {
                            self.vars.actionRelease.execute();
                        }
                    },
                    onPress: function(x, y, e) {
                        self.fire('press', {
                            x: x,
                            y: y,
                            event: e
                        });

                        if (self.config['write-bool'] && self.config.item) {
                            self.writeValue(self.config["on-value"]);
                        }

                        // UI Action: onPress
                        if (self.vars.actionPress) {
                            self.vars.actionPress.execute();
                        }

                        // UI Action while pressed
                        if (self.vars.actionWhilePressed && self.config['interval-while-pressed']) {
                            if (self.vars.actionWhilePressedTimer) {
                                clearInterval(self.vars.actionWhilePressedTimer);
                            }

                            // While pressed timer
                            self.vars.actionWhilePressedTimer = setInterval(function() {
                                self.vars.actionWhilePressed.execute();
                            }, self.config['interval-while-pressed']);
                        }
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.imageEl, eventFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.imageEl, eventFunctions));
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.log("[IQ:iq-image-changer] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                if (self.config.item) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                // Disable timer
                if (self.vars.actionWhilePressedTimer) {
                    clearInterval(self.vars.actionWhilePressedTimer);
                    self.vars.actionWhilePressedTimer = null;
                }

                shmi.log("[IQ:iq-image-changer] disabled", 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                shmi.addClass(self.element, 'locked');

                // Disable timer
                if (self.vars.actionWhilePressedTimer) {
                    clearInterval(self.vars.actionWhilePressedTimer);
                    self.vars.actionWhilePressedTimer = null;
                }

                shmi.log("[IQ:iq-image-changer] Locked", 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.removeClass(self.element, 'locked');

                shmi.log("[IQ:iq-image-changer] unlocked", 1);
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                var self = this;
                self.vars.value = value;
                self.getActive(self.vars.value);
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                return this.value;
            },

            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(min, max, step, name, type, warnmin, warnmax, prewarnmin, prewarnmax, precision) {
            },

            /**
             * Custom function
             *
             * @param {object} value
             */
            getActive: function(value) {
                const self = this;
                let opt = value !== null ? self.config.options.find((o) => {
                    if (Number.isSafeInteger(value) && Number.isSafeInteger(o.mask) && Number.isSafeInteger(o.value)) {
                        return (BigInt(value) & BigInt(o.mask)) === BigInt(o.value);
                    }

                    return o.value === value;
                }) : null;

                if (!opt) {
                    opt = {
                        "icon-src": self.config["default-image"],
                        "label": self.config["default-title"] || null
                    };
                }
                self.vars.activeOption = opt;

                self.vars.imageEl.setAttribute("src", opt["icon-src"]);
                self.setTooltip(self.getTooltip());
            },

            /**
             * Custom function
             *
             * @returns {null|definition.prototypeExtensions.activeOption.label}
             */
            getTooltip: function() {
                var self = this,
                    superTooltip = shmi.visuals.core.BaseControl.prototype.getTooltip.call(this);

                if (superTooltip) {
                    return superTooltip;
                } else if (self.vars.activeOption && self.vars.activeOption.label) {
                    return self.vars.activeOption.label;
                }

                return null;
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-image
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-image",
 *     "name": null,
 *     "template": "custom/controls/iq-image"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "tooltip": Tooltip
 * "image-src": The image source
 *  "image-title": The title
 *  "mg-alt": The alt text
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-image", // control name in camel-case
        uiType = "iq-image", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-image",
        "name": null,
        "template": "default/iq-image.variant-01",
        "tooltip": null,
        "image-src": "pkg://iq-image/placeholder.svg",
        "image-alt": null,
        "image-title": null,
        "scaling-mode": "fit-height",
        "action": null,
        "action-release": null,
        "action-while-pressed": null,
        "interval-while-pressed": null,
        "action-pressed": null,
        "write-bool": false,
        "on-value": 1,
        "off-value": 0
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    function applyScalingMode(self) {
        let scalingMode = "fit-height";

        switch (self.config["scaling-mode"]) {
        case "fit-height":
        case "fit-width":
        case "stretch":
        case "contain":
            scalingMode = self.config["scaling-mode"];
            break;
        default:
        }

        shmi.addClass(self.element, scalingMode);
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            imageEl: null,
            action: null,
            actionRelease: null,
            actionWhilePressed: null,
            actionWhilePressedTimer: null,
            mouseListener: null,
            touchListener: null
        },

        /* imports added at runtime */
        imports: {
        },

        /* array of custom event types fired by this control */
        events: ['press', 'release', 'click'],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.imageEl = shmi.getUiElement('image', self.element);

                if (!self.vars.imageEl) {
                    // As this should never happen we do a bit more work here in cleaning up than checking above
                    shmi.log("[IQ:iq-image] Missing image element", 3);
                    return;
                }

                self.vars.imageEl.setAttribute('draggable', 'false');
                if (self.config['image-src']) {
                    self.vars.imageEl.setAttribute('src', self.config['image-src']);
                }
                if (self.config['image-alt']) {
                    self.vars.imageEl.setAttribute('alt', self.config['image-alt']);
                }

                // UI Actions
                if (self.config.action) {
                    self.vars.action = new shmi.visuals.core.UiAction(self.config.action, self);
                }
                if (self.config['action-press']) {
                    self.vars.actionPress = new shmi.visuals.core.UiAction(self.config['action-press'], self);
                }
                if (self.config['action-release']) {
                    self.vars.actionRelease = new shmi.visuals.core.UiAction(self.config['action-release'], self);
                }
                if (self.config['action-while-pressed']) {
                    self.vars.actionWhilePressed = new shmi.visuals.core.UiAction(self.config['action-while-pressed'], self);
                }

                if (Array.isArray(self.config.action) && self.config.action.length > 0) {
                    shmi.addClass(self.vars.imageEl, 'clickable');
                } else {
                    shmi.removeClass(self.vars.imageEl, 'clickable');
                }

                applyScalingMode(self);

                /*****************/
                /*** LISTENERS ***/
                /*****************/
                const session = shmi.visuals.session;
                var eventFunctions = {
                    onClick: function(x, y, e) {
                        if (session.FocusElement !== null) {
                            session.FocusElement.blur();
                            session.FocusElement = null;
                        }
                        if (self.vars.imageEl instanceof HTMLElement) {
                            self.vars.imageEl.focus();
                            session.FocusElement = self.vars.imageEl;
                            shmi.log("[IQ:iq-image] focused", 1);
                        } else {
                            shmi.log("[IQ:iq-image] only HTMLElements may be focused, type: " + (self.vars.imageEl.constructor), 1);
                        }

                        self.fire("click", {
                            x: x,
                            y: y,
                            event: e
                        });

                        if (self.vars.action) {
                            self.vars.action.execute();
                        }
                    },
                    onRelease: function(x, y, e) {
                        self.fire('release', {
                            x: x,
                            y: y,
                            event: e
                        });
                        if (self.config['write-bool'] && self.config.item) {
                            self.writeValue(self.config["off-value"]);
                        }

                        // Stop any running timer on release
                        if (self.vars.actionWhilePressedTimer) {
                            clearInterval(self.vars.actionWhilePressedTimer);
                        }

                        if (self.vars.actionRelease) {
                            self.vars.actionRelease.execute();
                        }
                    },
                    onPress: function(x, y, e) {
                        self.fire('press', {
                            x: x,
                            y: y,
                            event: e
                        });

                        if (self.config['write-bool'] && self.config.item) {
                            self.writeValue(self.config["on-value"]);
                        }

                        // UI Action: onPress
                        if (self.vars.actionPress) {
                            self.vars.actionPress.execute();
                        }

                        // UI Action while pressed
                        if (self.vars.actionWhilePressed && self.config['interval-while-pressed']) {
                            if (self.vars.actionWhilePressedTimer) {
                                clearInterval(self.vars.actionWhilePressedTimer);
                            }

                            // While pressed timer
                            self.vars.actionWhilePressedTimer = setInterval(function() {
                                self.vars.actionWhilePressed.execute();
                            }, self.config['interval-while-pressed']);
                        }
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.imageEl, eventFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.imageEl, eventFunctions));
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.log("[IQ:iq-image] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                // Disable timer
                if (self.vars.actionWhilePressedTimer) {
                    clearInterval(self.vars.actionWhilePressedTimer);
                    self.vars.actionWhilePressedTimer = null;
                }

                shmi.log("[IQ:iq-image] disabled", 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                shmi.addClass(self.element, 'locked');

                // Disable timer
                if (self.vars.actionWhilePressedTimer) {
                    clearInterval(self.vars.actionWhilePressedTimer);
                    self.vars.actionWhilePressedTimer = null;
                }

                shmi.log("[IQ:iq-image] Locked", 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.removeClass(self.element, 'locked');

                shmi.log("[IQ:iq-image] unlocked", 1);
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-input-field
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-input-field",
 *     "name": null,
 *     "template": "default/iq-input-field"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "min": Minimum value
 * "max": Maximum value
 * "precision": Precision
 * "decimal-delimiter": Decimal delimiter
 * "type": Type is INT
 * "unit-text": Unit Text
 * "auto-label": Whether to use the auto-label (from item)
 * "auto-unit-text": Whether to use the auto-unit (from item)
 * "auto-min": Whether to use auto-min (from item)
 * "auto-max": Whether to use auto-max (from item)
 * "auto-precision": Whether to use the auto-precision (from item)
 * "auto-type": Whether to use the auto-type (from item)
 * "numpad-enabled": Whether the numpad is enabled
 * "icon-src": Default icon source
 * "icon-class": Default icon class
 * "tooltip": Tooltip
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "IqInputField", // control name in camel-case
        uiType = "iq-input-field", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": uiType,
        "name": null,
        "template": "default/iq-input-field.variant-01",
        "label": '[Label]',
        "item": null,
        "numeric-class": "numeric",
        "min": Number.NEGATIVE_INFINITY,
        "max": Number.POSITIVE_INFINITY,
        "step": 1,
        "precision": -1,
        "decimal-delimiter": ".",
        "type": shmi.c("TYPE_STRING"),
        "unit-text": "[Unit]",
        "auto-label": true,
        "auto-unit-text": true,
        "unit-scale": 1,
        "auto-min": true,
        "auto-max": true,
        "auto-step": true,
        "auto-precision": true,
        "auto-type": true,
        "numpad-enabled": false,
        "show-icon": false,
        "icon-src": null,
        "icon-class": null,
        "tooltip": null,
        "multiline": false,
        "notResizable": false
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the unit text and handles toggling the `no-unit` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} unitText Unit text to set
     */
    function setUnitTextImpl(self, unitText) {
        if (!self.vars.unitEl) {
            // Nothing to do.
        } else if (unitText === "" || unitText === null || typeof unitText === "undefined") {
            self.vars.unit = "";
            self.vars.unitEl.textContent = "";
            shmi.addClass(self.element, "no-unit");
        } else {
            self.vars.unit = unitText;
            self.vars.unitEl.textContent = shmi.localize(unitText);
            shmi.removeClass(self.element, "no-unit");
        }
    }

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined") {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: null,
            active: false,

            // DOM Elements
            currentValueEl: null,
            labelEl: null,
            unitEl: null,
            iconEl: null,
            inputEl: null, // Either currentValueEl || textAreaEl depending on the config
            textAreaEl: null,

            label: null,

            // Legacy
            domListeners: [],
            subscriptionTargetId: null,
            toEnterTmr: 0,
            wasClicked: false,
            valueSettings: null,
            uiAction: null,
            focused: false,
            locked: false,
            floatRegexp: "(^[+-]?[0-9]([.][0-9]*)?$|^[+-]?[1-9]+[0-9]*([.][0-9]*)?$)",
            intRegexp: "(^[+-]?[0-9]$|^[+-]?[1-9]+[0-9]*$)",
            type: null
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues"
        },

        /* array of custom event types fired by this control */
        events: ["change", "enter"],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this,
                    c = shmi.Constants;

                self.vars.floatRegexp = new RegExp(self.vars.floatRegexp);

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.currentValueEl = shmi.getUiElement('input', self.element);
                self.vars.textareaEl = shmi.getUiElement('textarea', self.element);
                self.vars.labelEl = shmi.getUiElement('label', self.element);
                self.vars.unitEl = shmi.getUiElement('unit', self.element);
                self.vars.iconEl = shmi.getUiElement('icon', self.element);

                if (!self.vars.currentValueEl || !self.vars.textareaEl) {
                    shmi.log('[IQ:iq-input-field] no input and/or element in HTML', 3);
                    return;
                }

                // Multiline handling
                if (self.config.multiline) {
                    self.vars.inputEl = self.vars.textareaEl;
                    self.vars.currentValueEl.remove();
                } else {
                    self.vars.inputEl = self.vars.currentValueEl;
                    self.vars.textareaEl.remove();
                }
                self.vars.inputEl.style.display = 'block';

                // Resizable handling
                if (self.config.notResizable) {
                    shmi.addClass(self.element, "not-resizable");
                } else {
                    shmi.removeClass(self.element, "not-resizable");
                }

                /* all required elements found */
                self.vars.inputEl.setAttribute('tabindex', '0');
                self.insertValue("");

                var type = parseInt(self.config.type);
                if (!isNaN(type)) {
                    self.vars.type = type;
                }

                if (self.config.action) {
                    self.vars.uiAction = new shmi.visuals.core.UiAction(self.config.action, this);
                }

                if (!self.config['auto-type']) {
                    if ([c.TYPE_BOOL, c.TYPE_INT, c.TYPE_FLOAT].indexOf(self.vars.type) !== -1) {
                        shmi.addClass(self.element, self.config['numeric-class']);
                    }
                }

                /*****************/
                /*** LISTENERS ***/
                /*****************/
                var eventFuncs = {
                    onPress: function(sx, sy, e) {
                        if (self.vars.focused) {
                            window.getSelection().removeAllRanges();
                        }
                    },
                    onClick: function() {
                        self.vars.wasClicked = true;
                        self.handleFocus();
                        self.vars.wasClicked = false;
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.inputEl, eventFuncs));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.inputEl, eventFuncs));

                // Custom DOM Listeners
                self.vars.domListeners.push(self.createEventListener(self.vars.inputEl, 'focus', function(e) {
                    if (self.config.multiline) {
                        self.vars.focused = true;
                    } else if (!self.vars.wasClicked) {
                        self.handleFocus();
                    }
                }));
                self.vars.domListeners.push(self.createEventListener(self.vars.inputEl, 'keypress', function(e) {
                    if (((e.keyCode === 13) && (!self.config.multiline)) || (e.keyCode === 9)) {
                        e.preventDefault();
                        window.getSelection().removeAllRanges();
                        self.vars.currentValueEl.blur();
                    }
                }));
                self.vars.domListeners.push(self.createEventListener(self.vars.inputEl, 'keydown', function(e) {
                    if (self.vars.type === shmi.c("TYPE_FLOAT")) {
                        var wrongDelimeter = (self.config['decimal-delimiter'] === ".") ? "," : ".";
                        if (e.key === wrongDelimeter) {
                            e.preventDefault();
                        }
                    }
                }));
                self.vars.domListeners.push(self.createEventListener(self.vars.inputEl, 'blur', function(e) {
                    if (!self.vars.focused) {
                        return;
                    }
                    shmi.log("[IQ:iq-input-field] blur event", 1);
                    window.getSelection().removeAllRanges();

                    self.validateAndSet(self.vars.inputEl);
                    self.vars.focused = false;
                    shmi.visuals.session.FocusElement = null;
                }));

                /************/
                /*** ICON ***/
                /************/
                if (!self.vars.iconEl) {
                    shmi.log('[IQ:iq-input-field] no button-icon element provided', 1);
                } else if (self.config['icon-src'] && self.config['show-icon']) {
                    self.vars.iconEl.style.backgroundImage = `url(${self.config['icon-src']})`;
                } else if (self.config['icon-class'] && self.config['show-icon']) {
                    var iconClasses = self.config['icon-class'].trim().split(" ");
                    iconClasses.forEach(function(cls) {
                        shmi.addClass(self.vars.iconEl, cls);
                    });
                } else {
                    // No icon configured - hide
                    shmi.addClass(self.element, "no-icon");
                }

                // Unit
                setUnitTextImpl(self, self.config['unit-text']);
                if (self.config['unit-scale']) {
                    self.config['unit-scale'] = parseFloat(shmi.localize(self.config['unit-scale']));
                }

                // Label
                setLabelImpl(self, self.config.label);
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                self.vars.inputEl.disabled = false;
                self.enableAllListeners();

                if (document.activeElement === self.vars.inputEl) {
                    self.vars.focused = true;
                }

                shmi.log("[IQ:iq-input-field] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                if (self.config.item) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                self.disableAllListeners();

                shmi.log("[IQ:iq-input-field] disabled", 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;

                self.vars.locked = true;
                self.vars.inputEl.disabled = true;

                shmi.addClass(self.element, 'locked');

                self.vars.inputEl.blur();
                self.disableAllListeners();

                shmi.log("[IQ:iq-input-field] Locked", 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;

                self.vars.locked = false;
                self.vars.inputEl.disabled = false;

                shmi.removeClass(self.element, 'locked');

                self.enableAllListeners();

                shmi.log("[IQ:iq-input-field] unlocked", 1);
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                var self = this,
                    oldValue = self.vars.value;

                self.vars.value = value;

                self.insertValue(self.imports.nv.formatOutput(self.vars.value, self));
                if (document.activeElement === self.vars.inputEl) {
                    self.selectContent();
                }
                shmi.log("[IQ:iq-input-field] new value: " + self.vars.value, 0);
                if (self.vars.value !== oldValue) {
                    self.fire("change", {
                        value: self.vars.value
                    });
                }
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                var self = this,
                    type = self.vars.type;

                if (self.config.item && self.config['auto-type']) {
                    type = self.imports.im.items[self.config.item].type;
                }

                if (type === shmi.c("TYPE_INT")) {
                    return Math.round(self.vars.value / self.config['unit-scale']);
                } else if (type === shmi.c("TYPE_FLOAT")) {
                    return (self.vars.value / self.config['unit-scale']);
                } else {
                    return self.vars.value;
                }
            },

            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(min, max, step, name, type, warnmin, warnmax, prewarnmin, prewarnmax, precision) {
                var self = this,
                    c = shmi.Constants;

                if (self.config['auto-type']) {
                    self.vars.type = type;
                    if ([c.TYPE_BOOL, c.TYPE_INT, c.TYPE_FLOAT].indexOf(type)) {
                        shmi.addClass(self.element, self.config['numeric-class']);
                    }
                }

                self.imports.nv.setProperties(self, arguments);
            },

            setUnitText: function(unitText) {
                var self = this;

                if (self.vars.unitEl && self.config['auto-unit-text']) {
                    setUnitTextImpl(self, unitText, false);
                }
            },

            setLabel: function(labelText) {
                var self = this;

                if (self.vars.labelEl && self.config['auto-label']) {
                    setLabelImpl(self, labelText, false);
                }
            },

            /**
             * Writes current value to connected data source
             *
             */
            updateValue: function() {
                if (this.config.item) {
                    shmi.visuals.session.ItemManager.writeValue(this.config.item, this.vars.value);
                }
            },

            /**
             * Legacy
             *
             * @param value
             */
            insertValue: function(value) {
                var self = this;
                self.vars.inputEl.value = value;
            },

            /**
             * Legacy
             *
             * @returns {string|*|string|*}
             */
            retrieveValue: function() {
                var self = this;
                return self.vars.inputEl.value;
            },

            /**
             * Legacy
             */
            selectContent: function() {
                var self = this;
                self.vars.inputEl.select();
            },

            /**
             * Legacy
             *
             * handleFocus - handle focus of input element when control is clicked or tabbed into
             */
            handleFocus: function() {
                var self = this;

                /* prevent focusing if control is locked */
                if (self.vars.locked) {
                    return;
                }

                if ((!self.vars.focused) && (shmi.visuals.session.FocusElement !== null)) {
                    shmi.visuals.session.FocusElement.blur();
                    shmi.visuals.session.FocusElement = null;
                }
                shmi.log("[IQ:iq-input-field] focused", 1);

                if (!self.vars.wasClicked) {
                    self.selectContent();
                } else if (self.showNumpad()) {
                    return;
                } else if (self.showKeyboard()) {
                    return;
                }

                self.vars.focused = true;
                self.vars.inputEl.focus();
                shmi.visuals.session.FocusElement = self.vars.inputEl;

                if (self.vars.toEnterTmr) {
                    clearTimeout(self.vars.toEnterTmr);
                }

                self.vars.toEnterTmr = setTimeout(function() {
                    self.fire("enter", {
                        value: self.vars.value
                    });
                }, 250);
            },

            /**
             * Legacy
             *
             * @return {boolean} if numpad will be shown
             */
            showNumpad: function() {
                const self = this,
                    nv = shmi.requires("visuals.tools.numericValues");

                if (!self.config["numpad-enabled"]) { //does nothing if not enabled
                    return false;
                }

                if (!(self.vars && self.vars.valueSettings)) {
                    //initialize value settings in case no item is configured
                    nv.initValueSettings(self);
                }

                shmi.numpad(
                    {
                        "decimal-delimiter": self.config["decimal-delimiter"],
                        "unit": self.vars.unitEl ? self.vars.unitEl.textContent : self.config["unit-text"],
                        "label": self.vars.labelEl ? self.vars.labelEl.textContent : self.config.label,
                        "value": self.retrieveValue(),
                        "callback": function(res) {
                            if (self.config.item) {
                                if (self.config.multiline) {
                                    res = res.toString();
                                }
                                self.imports.im.writeValue(self.config.item, res);
                            } else {
                                self.setValue(res);
                            }
                        },
                        "min": self.vars.valueSettings.min,
                        "max": self.vars.valueSettings.max,
                        "type": self.vars.valueSettings.type,
                        "precision": self.vars.valueSettings.precision,
                        "item": (typeof self.config.item === "string" && self.config.item.length > 0) ? self.config.item : null
                    }
                );

                return true;
            },

            /**
             * Legacy
             *
             * @return {boolean} if keyboard will be shown
             */
            showKeyboard: function() {
                var self = this,
                    appConfig = shmi.requires("visuals.session.config"),
                    keyboardEnabled = (appConfig.keyboard && appConfig.keyboard.enabled); // get the keyboard config from `project.json`

                if (!keyboardEnabled) {
                    return false;
                }

                shmi.keyboard(
                    {
                        "value": self.retrieveValue(),
                        "select-box-enabled": appConfig.keyboard["language-selection"],
                        "password-input": self.vars.currentValueEl && self.vars.currentValueEl.type && self.vars.currentValueEl.type.toLowerCase() === "password",
                        "show-enter": self.config.multiline,
                        "callback": function(success, input) {
                            if (success) {
                                if (self.config.item) {
                                    self.imports.im.writeValue(self.config.item, input);
                                } else {
                                    self.setValue(input);
                                }
                            }
                        }
                    });

                return true;
            },

            /**
             * Validates the content of the element and sets the value accordingly
             */
            validateAndSet: function() {
                const self = this,
                    inputString = self.retrieveValue(),
                    oldValue = self.vars.value;

                if (inputString === oldValue) {
                    return;
                }

                const newValue = (() => {
                    if (self.config.multiline) {
                        // Multi-line is always string, therefore no check and
                        // formatting is performed on it.
                        return inputString;
                    } else if (![shmi.c("TYPE_INT"), shmi.c("TYPE_FLOAT")].includes(self.vars.type)) {
                        return inputString;
                    } else if (!self.vars.floatRegexp.test(inputString.replace(self.config['decimal-delimiter'], '.'))) {
                        return oldValue;
                    }

                    return self.imports.nv.applyInputSettings(inputString, self);
                })();

                self.vars.value = newValue;
                if (typeof newValue === "undefined" || newValue === null) {
                    self.insertValue("");
                } else {
                    self.insertValue(self.imports.nv.formatOutput(newValue, self));
                }

                // Emit change event, write item and call UI actions only when something has changed
                if (newValue !== oldValue) {
                    if (self.config.item) {
                        self.imports.im.writeValue(self.config.item, newValue);
                    }

                    // UI Action
                    if (self.vars.uiAction) {
                        self.vars.uiAction.execute(newValue);
                    }

                    self.fire("change", {
                        value: self.vars.value
                    });
                }
            },

            /**
             * Helper functions
             */
            enableAllListeners: function() {
                var self = this;
                self.vars.domListeners.forEach(function(obj) {
                    obj.element.addEventListener(obj.event, obj.callback);
                });
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });
            },
            disableAllListeners: function() {
                var self = this;
                self.vars.domListeners.forEach(function(obj) {
                    obj.element.removeEventListener(obj.event, obj.callback);
                });
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
            },
            createEventListener: function(element, event, callback) {
                return {
                    'event': event,
                    'callback': callback,
                    'element': element
                };
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-label
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-label",
 *     "name": null,
 *     "template": "custom/controls/iq-label"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "precision": Precision
 * "decimal-delimiter": Decimal delimiter
 * "type": Type is INT
 * "auto-precision": Whether to use the auto-precision (from item)
 * "auto-type": Whether to use the auto-type (from item)
 * "tooltip": Tooltip
 * "options": The options
 * "pattern": The pattern
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-label", // control name in camel-case
        uiType = "iq-label", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-label",
        "name": null,
        "template": "default/iq-label.variant-01",
        "item": null,
        "precision": -1,
        "decimal-delimiter": ".",
        "type": shmi.c("TYPE_INT"),
        "auto-precision": true,
        "auto-type": true,
        "tooltip": null,
        "options": [],
        "pattern": null,
        "value-as-tooltip": false,
        "text": "${label.no-value}"
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            value: 0,
            active: false,

            // DOM Elements
            labelEl: null,

            rafId: null,
            subscriptionTargetId: null
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues"
        },

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                // Configure NV
                self.imports.nv.initValueSettings(self);

                // DOM Element
                self.vars.labelEl = shmi.getUiElement('label', self.element);

                if (self.config.text) {
                    self.updateText(shmi.localize(self.config.text));
                }
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                shmi.log("[IQ:iq-label] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                if (self.vars.subscriptionTargetId) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                shmi.log("[IQ:iq-label] disabled", 1);
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                var self = this;
                return self.vars.value;
            },

            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(min, max, step, name, type, warnmin, warnmax, prewarnmin, prewarnmax, precision) {
                var self = this;
                self.imports.nv.setProperties(self, arguments);
            },

            /* Called by ItemManager when subscribed item changes and once on initial subscription
             *
             * "options" : [
             *  { "label" : "True",  "value" : 0 },
             *  { "label" : "False", "value" : 1 },
             *  { "label" : "=",     "value" : 2 },
             *  { "label" : ">",     "value" : 3 },
             *  { "label" : ">=",    "value" : 4 },
             *  { "label" : "<",     "value" : 5 },
             *  { "label" : "<=",    "value" : 6 },
             *  { "label" : "&",     "value" : 7 },
             *  { "label" : "|",     "value" : 8 },
             *  { "label" : "Null",  "value" : 9 }
             * ]
            **/
            onSetValue: function(value, type, name) {
                var self = this;

                shmi.caf(self.vars.rafId);
                self.vars.rafId = shmi.raf(function() {
                    var newValue = value;

                    if (self.config.options.length) {
                        const selectedOption = self.config.options.find((o) => {
                            if (Number.isSafeInteger(value) && Number.isSafeInteger(o.mask) && Number.isSafeInteger(o.value)) {
                                return (BigInt(value) & BigInt(o.mask)) === BigInt(o.value);
                            }

                            return o.value === value;
                        });

                        if (selectedOption) {
                            newValue = shmi.localize(selectedOption.label);
                        }
                    } else if ((typeof self.config.pattern === "string") && self.config.pattern.trim().length) {
                        newValue = shmi.localize(shmi.evalString(self.config.pattern, { VALUE: value }));
                    } else {
                        newValue = self.imports.nv.formatOutput(value, self);
                    }

                    self.updateText(newValue);
                });
            },

            /**
             * Updates the text
             *
             * @param newValue
             */
            updateText: function(newValue) {
                var self = this;

                self.vars.labelEl.textContent = newValue;
                if (self.config["value-as-tooltip"]) {
                    self.setTooltip(newValue);
                }
            },
            onLock: function() {
                shmi.addClass(this.element, 'locked');
            },
            onUnlock: function() {
                shmi.removeClass(this.element, 'locked');
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-linear-gauge
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-linear-gauge",
 *     "name": null,
 *     "template": "custom/controls/iq-linear-gauge"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "tooltip": Tooltip
 * "text": The text
 * "item": Items to be used inside
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    const className = "IqLinearGauge", // control name in camel-case
        uiType = "iq-linear-gauge", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    const defConfig = {
        "class-name": uiType,
        "name": null,
        "template": "default/iq-linear-gauge.variant-01",
        "item": null,
        "tooltip": null,
        "type": shmi.c("TYPE_INT"),
        "precision": 0,
        "min": Number.NEGATIVE_INFINITY,
        "max": Number.POSITIVE_INFINITY,
        "label": "[Label]",
        "unit-text": "[Unit]",
        "decimal-delimiter": ".",
        "default-value": "---",
        "show-text": true,
        "show-icon": false,
        "icon-src": null,
        "icon-class": null,

        "auto-min": true,
        "auto-max": true,
        "auto-label": true,
        "auto-unit-text": true,
        "auto-precision": true,

        "animation-duration": 100,
        "fill-reverse": false,
        "fill-inverse": false
    };

    // setup module-logger
    const ENABLE_LOGGING = true,
        RECORD_LOG = false;
    const logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    const fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.dom.label) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined" || !self.config["show-text"]) {
            self.vars.label = "";
            self.vars.dom.label.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.dom.label.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    /**
     * Sets the unit text and handles toggling the `no-unit` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} unitText Unit text to set
     */
    function setUnitTextImpl(self, unitText) {
        if (!self.vars.dom.unit) {
            // Nothing to do.
        } else if (unitText === "" || unitText === null || typeof unitText === "undefined" || !self.config["show-text"]) {
            self.vars.unit = "";
            self.vars.dom.unit.textContent = "";
            shmi.addClass(self.element, "no-unit");
        } else {
            self.vars.unit = unitText;
            self.vars.dom.unit.textContent = shmi.localize(unitText);
            shmi.removeClass(self.element, "no-unit");
        }
    }

    /**
     * Returns the bar fill percentage between 0 and 1 for the given value. If
     * no value is given, the widgets current value is used instead.
     *
     * @param {*} self
     * @param {number} [value]
     * @returns {number} Bar fill percentage between 0 and 1.
     */
    function getFillPercentage(self, value) {
        const { min, max } = self.vars.valueSettings,
            percentage = (value - min) / (max - min);

        if (isNaN(percentage) || min === max) {
            return 0;
        }

        if (self.config["fill-inverse"]) {
            return 1 - Math.min(Math.max(percentage, 0), 1);
        }

        return Math.min(Math.max(percentage, 0), 1);
    }

    /**
     * Updates bar.
     *
     * @param {*} self
     * @param {?number} param1.actValue
     */
    function updateWidgetBar(self, { actValue }) {
        if (self.vars.dom.bar) {
            self.vars.dom.bar.style.setProperty("--internal-fill-level", `${getFillPercentage(self, actValue) * 100}%`);
        }
    }

    /**
     * Updates value display.
     *
     * @param {*} self
     */
    function updateWidgetText(self) {
        if (self.vars.dom.value) {
            if (self.getValue() === null) {
                self.vars.dom.value.textContent = shmi.localize(self.config["default-value"]);
            } else {
                self.vars.dom.value.textContent = self.imports.nv.formatOutput(self.getValue(), self);
            }
        }
    }

    /**
     * Queues a widget update for the next animation frame. Will only queue one
     * update per frame.
     *
     * @param {*} self
     */
    function queueWidgetTextUpdate(self) {
        if (self.vars.rafId) {
            return;
        }

        self.vars.rafId = shmi.raf(() => {
            updateWidgetText(self);

            self.vars.rafId = null;
        });
    }

    // Definition of new control extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            dom: {
                value: null,
                bar: null,
                unit: null,
                label: null,
                icon: null
            },
            valueSettings: {
                min: 0,
                max: 100
            },
            animationBundle: null,
            rafId: null,
            value: null,
            tokens: []
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues",
            gfx: "visuals.gfx"
        },

        /* Array of custom event types fired by this control */
        events: [],

        /* Functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                const self = this;

                // DOM
                Object.assign(self.vars.dom, {
                    value: shmi.getUiElement("value", self.element),
                    bar: shmi.getUiElement("bar", self.element),
                    unit: shmi.getUiElement("unit", self.element),
                    label: shmi.getUiElement("label", self.element),
                    icon: shmi.getUiElement("icon", self.element)
                });

                self.imports.nv.initValueSettings(self);

                // Icon
                if (self.vars.dom.icon && self.config['icon-src'] && self.config['show-icon']) {
                    self.vars.dom.icon.style.backgroundImage = `url(${self.config['icon-src']})`;
                } else if (self.vars.dom.icon && self.config['icon-class'] && self.config['show-icon']) {
                    const iconClasses = self.config['icon-class'].trim().split(" ");
                    iconClasses.forEach((cls) => shmi.addClass(self.vars.dom.icon, cls));
                } else {
                    // No icon configured - hide
                    shmi.addClass(self.element, "no-icon");
                }

                // Unit
                setUnitTextImpl(self, self.config['unit-text']);
                if (self.config['unit-scale']) {
                    self.config['unit-scale'] = parseFloat(shmi.localize(self.config['unit-scale']));
                }

                // Label
                setLabelImpl(self, self.config.label);

                // Fill orientation
                if (self.vars.dom.bar) {
                    if (self.config["fill-reverse"]) {
                        shmi.addClass(self.vars.dom.bar, "reversed");
                    }
                }

                self.vars.animationBundle = new self.imports.gfx.AnimationBundle(updateWidgetBar.bind(null, self));
                self.vars.animationBundle.prepare("actValue", null, "linear");

                updateWidgetText(self);
            },

            /* Called when control is enabled */
            onEnable: function() {
                const self = this;

                if (self.config.item) {
                    self.vars.tokens.push(self.imports.im.subscribeItem(self.config.item, self));
                }
            },

            /* Called when control is disabled */
            onDisable: function() {
                const self = this;

                if (self.vars.rafId) {
                    shmi.caf(self.vars.rafId);
                    self.vars.rafId = null;
                }

                self.vars.tokens.forEach((t) => t.unlisten());
                self.vars.tokens = [];
                self.vars.animationBundle.cancel("actValue");
            },

            /* Called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                shmi.addClass(this.element, 'locked');
            },

            /* Called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                shmi.removeClass(this.element, 'locked');
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                const self = this;

                return self.vars.value;
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                const self = this,
                    supportedTypes = [shmi.c("TYPE_BOOL"), shmi.c("TYPE_INT"), shmi.c("TYPE_FLOAT")],
                    newValue = (supportedTypes.includes(type) ? value : Number.NaN);

                if (self.vars.value === newValue) {
                    return;
                }

                self.vars.value = newValue;
                self.vars.animationBundle.start(newValue, self.config["animation-duration"], "actValue");
                queueWidgetTextUpdate(self);
            },

            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(...args) {
                const self = this;
                //const [min, max, step, name, type, warnmin, warnmax, prewarnmin, prewarnmax, precision] = args;

                self.imports.nv.setProperties(self, args);
                self.vars.animationBundle.refresh();
                queueWidgetTextUpdate(self);
            },

            setUnitText: function(unitText) {
                const self = this;

                if (self.config['auto-unit-text']) {
                    setUnitTextImpl(self, unitText);
                }
            },

            setLabel: function(labelText) {
                const self = this;

                if (self.config['auto-label']) {
                    setLabelImpl(self, labelText);
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-progress-info
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-progress-info",
 *     "name": null,
 *     "template": "custom/controls/iq-progress-info"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "tooltip": Tooltip
 * "text": The text
 * "item": Items to be used inside
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    const className = "IqProgressInfo", // control name in camel-case
        uiType = "iq-progress-info", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    const defConfig = {
        "class-name": uiType,
        "name": null,
        "template": "default/iq-progress-info.variant-01",
        "displayFormat": "($baseItem / $comparisonItem) - $percentValue%",
        "baseItem": null,
        "comparisonItem": null,
        "tooltip": null,
        "precision": -1,
        "decimal-delimiter": ".",
        "default-value": "---",

        "auto-precision": true,

        "animation-duration": 100,
        "fill-reverse": false,
        "fill-inverse": false
    };

    // setup module-logger
    const ENABLE_LOGGING = true,
        RECORD_LOG = false;
    const logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    const fLog = logger.fLog,
        log = logger.log;

    /**
     * Creates placeholder elements in the value element for the format string.
     *
     * @param {*} self
     * @param {string} displayFormat
     */
    function setupValueElements(self, displayFormat) {
        const { value: valueEl } = self.vars.dom;

        if (!valueEl) {
            return;
        }

        valueEl.innerHTML = displayFormat.
            replace(/[$]baseItem/g, `<span class="baseValue"></span>`).
            replace(/[$]comparisonItem/g, `<span class="compareValue"></span>`).
            replace(/[$]percentValue/g, `<span class="percentValue"></span>`);

        self.vars.placeholders = {
            baseItem: Array.from(valueEl.getElementsByClassName("baseValue")),
            comparisonItem: Array.from(valueEl.getElementsByClassName("compareValue")),
            percentValue: Array.from(valueEl.getElementsByClassName("percentValue"))
        };
    }

    /**
     * Sets values of placeholder elements.
     *
     * @param {*} self
     * @param {number} baseValue
     * @param {number} comparisonValue
     */
    function setPlaceholderValues(self, baseValue, comparisonValue) {
        const {
                nv: { formatOutput },
                iter: { iterateObject }
            } = self.imports,
            data = {
                baseItem: formatOutput(baseValue, self),
                comparisonItem: formatOutput(comparisonValue, self),
                percentValue: Math.round(baseValue / comparisonValue * 100)
            };

        if (!self.vars.placeholders) {
            return;
        }

        iterateObject(data, (value, key) => {
            const placeholders = self.vars.placeholders[key];
            if (Array.isArray(placeholders)) {
                placeholders.forEach((el) => el.textContent = value);
            }
        });
    }

    /**
     * Returns the bar fill percentage between 0 and 1 for the given value.
     *
     * @param {number} value
     * @param {number} comparisonValue
     * @param {boolean} [invert]
     * @returns {number} Bar fill percentage between 0 and 1.
     */
    function getFillPercentage(value, comparisonValue, invert) {
        const percentage = value / comparisonValue;

        if (isNaN(percentage)) {
            return 0;
        } else if (invert) {
            return 1 - Math.min(Math.max(percentage, 0), 1);
        }

        return Math.min(Math.max(percentage, 0), 1);
    }

    /**
     * Updates value display.
     *
     * @param {*} self
     */
    function updateWidgetText(self) {
        if (!self.vars.dom.value) {
            // Nothing to do
        } else if (self.getValue() === null || self.getComparisonValue() === null) {
            self.vars.dom.value.textContent = shmi.localize(self.config["default-value"]);
            self.vars.placeholders = null;
        } else {
            if (!self.vars.placeholders) {
                setupValueElements(self, self.config.displayFormat);
            }

            setPlaceholderValues(self, self.getValue(), self.getComparisonValue());
        }
    }

    /**
     * Updates bar.
     *
     * @param {*} self
     * @param {?number} param1.baseValue
     * @param {?number} param1.compValue
     */
    function updateWidgetBar(self, { baseValue, compValue }) {
        const valuePercentage = getFillPercentage(baseValue, compValue, self.config["fill-inverse"]);

        if (self.vars.dom.bar) {
            self.vars.dom.bar.style.setProperty("--internal-fill-level", `${valuePercentage * 100}%`);
        }
    }

    /**
     * Queues a widget update for the next animation frame. Will only queue one
     * update per frame.
     *
     * @param {*} self
     */
    function queueWidgetTextUpdate(self) {
        if (self.vars.rafId || !self.vars.dom.value) {
            return;
        }

        self.vars.rafId = shmi.raf(() => {
            updateWidgetText(self);

            self.vars.rafId = null;
        });
    }

    // Definition of new control extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            dom: {
                value: null,
                bar: null,
                barContainer: null
            },
            placeholders: null,
            valueSettings: {
                min: 0,
                max: 100
            },
            animationBundle: null,
            rafId: null,
            value: null,
            valueComp: null,
            listeners: [],
            tokens: []
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            iter: "visuals.tools.iterate",
            nv: "visuals.tools.numericValues",
            gfx: "visuals.gfx"
        },

        /* Array of custom event types fired by this control */
        events: [],

        /* Functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                const self = this;

                // DOM
                Object.assign(self.vars.dom, {
                    value: shmi.getUiElement("value", self.element),
                    bar: shmi.getUiElement("bar", self.element),
                    barContainer: shmi.getUiElement("bar-container", self.element)
                });

                self.imports.nv.initValueSettings(self);

                // Fill orientation
                if (self.vars.dom.bar) {
                    if (self.config["fill-reverse"]) {
                        shmi.addClass(self.vars.dom.bar, "reversed");
                    }

                    self.vars.animationBundle = new self.imports.gfx.AnimationBundle(updateWidgetBar.bind(null, self));
                    self.vars.animationBundle.prepare("baseValue", null, "linear");
                    self.vars.animationBundle.prepare("compValue", null, "linear");
                }

                updateWidgetText(self);
            },

            /* Called when control is enabled */
            onEnable: function() {
                const self = this;

                if (self.config.baseItem) {
                    self.vars.tokens.push(self.imports.im.subscribeItem(self.config.baseItem, self));
                }

                if (self.config.comparisonItem) {
                    self.vars.tokens.push(self.imports.im.subscribeItem(self.config.comparisonItem, {
                        setValue: self.onSetCompValue.bind(self)
                    }));
                }
            },

            /* Called when control is disabled */
            onDisable: function() {
                const self = this;

                if (self.vars.rafId) {
                    shmi.caf(self.vars.rafId);
                    self.vars.rafId = null;
                }

                if (self.vars.animationBundle) {
                    self.vars.animationBundle.cancel("baseValue");
                    self.vars.animationBundle.cancel("compValue");
                }

                self.vars.tokens.forEach((t) => t.unlisten());
                self.vars.tokens = [];
            },

            /* Called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                shmi.addClass(this.element, 'locked');
            },

            /* Called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                shmi.removeClass(this.element, 'locked');
            },

            /**
             * Retrieves current value of the progress info
             *
             * @return value - current value
             */
            getValue: function() {
                const self = this;

                return self.vars.value;
            },

            /**
             * Retrieves current comparison value of the progress info
             *
             * @return value - current comparison value
             */
            getComparisonValue: function() {
                const self = this;

                return self.vars.valueComp;
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                const self = this,
                    supportedTypes = [shmi.c("TYPE_BOOL"), shmi.c("TYPE_INT"), shmi.c("TYPE_FLOAT")],
                    newValue = (supportedTypes.includes(type) ? value : Number.NaN);

                if (self.vars.value === newValue) {
                    return;
                }

                self.vars.value = newValue;
                if (self.vars.animationBundle) {
                    self.vars.animationBundle.start(newValue, self.config["animation-duration"], "baseValue");
                }

                queueWidgetTextUpdate(self);
            },

            onSetCompValue: function(value, type) {
                const self = this,
                    supportedTypes = [shmi.c("TYPE_BOOL"), shmi.c("TYPE_INT"), shmi.c("TYPE_FLOAT")],
                    newValue = (supportedTypes.includes(type) ? value : Number.NaN);

                if (self.vars.valueComp === newValue) {
                    return;
                }

                self.vars.valueComp = newValue;
                if (self.vars.animationBundle) {
                    self.vars.animationBundle.start(newValue, self.config["animation-duration"], "compValue");
                }

                queueWidgetTextUpdate(self);
            },

            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(...args) {
                const self = this;
                //const [min, max, step, name, type, warnmin, warnmax, prewarnmin, prewarnmax, precision] = args;

                self.imports.nv.setProperties(self, args);
                queueWidgetTextUpdate(self);
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-quality-display
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-quality-display",
 *     "name": null,
 *     "template": "custom/controls/iq-quality-display"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "tooltip": Tooltip
 * "text": The text
 * "item": Items to be used inside
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    const className = "IqQualityDisplay", // control name in camel-case
        uiType = "iq-quality-display", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    const defConfig = {
        "class-name": uiType,
        "name": null,
        "template": "default/iq-quality-display.variant-01",
        "item": null,
        "item-ctrl": null,
        "tooltip": null,
        "type": shmi.c("TYPE_INT"),
        "precision": 0,
        "min": 0,
        "max": 100,
        "label": "[Label]",
        "unit-text": "[Unit]",
        "decimal-delimiter": ".",
        "default-value": "---",
        "show-icon": false,
        "show-text": true,
        "icon-src": null,
        "icon-class": null,
        "auto-type": true,
        "auto-min": true,
        "auto-max": true,
        "auto-label": true,
        "auto-unit-text": true,
        "auto-precision": true,

        "animation-duration": 100
    };

    // setup module-logger
    const ENABLE_LOGGING = true,
        RECORD_LOG = false;
    const logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    const fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.dom.label) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined" || !self.config["show-text"]) {
            self.vars.label = "";
            self.vars.dom.label.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.dom.label.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    /**
     * Sets the unit text and handles toggling the `no-unit` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} unitText Unit text to set
     */
    function setUnitTextImpl(self, unitText) {
        if (!self.vars.dom.unit) {
            // Nothing to do.
        } else if (unitText === "" || unitText === null || typeof unitText === "undefined" || !self.config["show-text"]) {
            self.vars.unit = "";
            self.vars.dom.unit.textContent = "";
            shmi.addClass(self.element, "no-unit");
        } else {
            self.vars.unit = unitText;
            self.vars.dom.unit.textContent = shmi.localize(unitText);
            shmi.removeClass(self.element, "no-unit");
        }
    }

    /**
     * Returns the bar fill percentage between 0 and 1 for the given value. If
     * no value is given, the widgets current value is used instead.
     *
     * @param {*} self
     * @param {number} value
     * @returns {number} Bar fill percentage between 0 and 1.
     */
    function getFillPercentage(self, value) {
        const { min, max } = self.vars.valueSettings,
            percentage = (value - min) / (max - min);

        if (isNaN(percentage) || max === min) {
            return 0;
        }

        return Math.min(Math.max(percentage, 0), 1);
    }

    /**
     * Updates value display.
     *
     * @param {*} self
     */
    function updateWidgetText(self) {
        if (self.vars.dom.value) {
            if (self.getValue() === null) {
                self.vars.dom.value.textContent = shmi.localize(self.config["default-value"]);
            } else {
                self.vars.dom.value.textContent = self.imports.nv.formatOutput(self.getValue(), self);
            }
        }

        if (self.vars.dom.controlValue) {
            if (self.getControlValue() === null) {
                self.vars.dom.controlValue.textContent = shmi.localize(self.config["default-value"]);
            } else {
                self.vars.dom.controlValue.textContent = self.imports.nv.formatOutput(self.getControlValue(), self);
            }
        }
    }

    /**
     * Updates bar and control value position.
     *
     * @param {*} self
     * @param {?number} param1.actValue
     * @param {?number} param1.ctrlValue
     */
    function updateWidgetBars(self, { actValue, ctrlValue }) {
        const valuePercentage = getFillPercentage(self, actValue),
            valueControlPercentage = getFillPercentage(self, ctrlValue);

        const {
            barContainer: barContainerEl,
            indicator: indicatorEl,
            controlValue: controlValueEl
        } = self.vars.dom;

        if (barContainerEl) {
            barContainerEl.style.setProperty("--internal-fill-level-set", `${valueControlPercentage * 100}%`);
            barContainerEl.style.setProperty("--internal-fill-level", `${valuePercentage * 100}%`);
        }

        if (indicatorEl) {
            indicatorEl.style.setProperty("--internal-fill-level-set", `${valueControlPercentage * 100}%`);
            indicatorEl.style.setProperty("--internal-fill-level", `${valuePercentage * 100}%`);
        }

        if (barContainerEl && controlValueEl) {
            const barBoundingRect = barContainerEl.getBoundingClientRect();

            controlValueEl.style.setProperty("--internal-fill-level-set", `${valueControlPercentage * 100}%`);
            controlValueEl.style.setProperty("--internal-fill-level", `${valuePercentage * 100}%`);
            controlValueEl.style.setProperty("--internal-bar-offset-h", `${barBoundingRect.width * valueControlPercentage}px`);
            controlValueEl.style.setProperty("--internal-bar-offset-v", `${barBoundingRect.height * (1 - valueControlPercentage)}px`);
        }
    }

    /**
     * Queues a widget update for the next animation frame. Will only queue one
     * update per frame.
     *
     * @param {*} self
     */
    function queueWidgetTextUpdate(self) {
        if (self.vars.rafId) {
            return;
        }

        self.vars.rafId = shmi.raf(() => {
            updateWidgetText(self);

            self.vars.rafId = null;
        });
    }

    /**
     * Computes orientation and anchor of the gauge bar.
     *
     * @param {*} self
     * @returns {object}
     */
    function getFillProperties(self) {
        if (!self.vars.dom.bar) {
            return null;
        }

        const computed = getComputedStyle(self.vars.dom.bar),
            [transformOriginX, transformOriginY] = computed.transformOrigin.replace("px", "").split(" ").map(parseFloat),
            isHorizontalScaling = transformOriginY <= parseFloat(computed.height.replace("px", "")) / 2,
            fromLeft = transformOriginX === 0,
            fromTop = transformOriginY === 0;

        return {
            orientation: isHorizontalScaling ? "horizontal" : "vertical",
            reverse: isHorizontalScaling ? !fromTop : !fromLeft,
            anchor: (() => {
                if (isHorizontalScaling) {
                    return fromLeft ? "left" : "right";
                } else {
                    return fromTop ? "top" : "bottom";
                }
            })()
        };
    }

    /**
     * Creates a Visuals token-like observer that listens for changes that
     * require the widget to be updated.
     *
     * @param {*} self
     * @param {function} callback
     */
    function makeUpdateToken(self, callback) {
        let throttled = false;

        const throttledCallback = () => {
                throttled = false;
                Promise.resolve().then(() => {
                    if (!throttled) {
                        callback();
                        throttled = true;
                    }
                });
            },
            // We only use the ResizeObserver if it's supported. Otherwise we
            // have to rely on the `resize` event only :(.
            // eslint-disable-next-line compat/compat
            resizeObserver = window.ResizeObserver ? new ResizeObserver(throttledCallback) : null;

        window.addEventListener("resize", throttledCallback);
        if (resizeObserver && self.vars.dom.barContainer) {
            resizeObserver.observe(self.vars.dom.barContainer);
        }

        return {
            unlisten: () => {
                window.removeEventListener("resize", throttledCallback);
                if (resizeObserver && self.vars.dom.barContainer) {
                    resizeObserver.unobserve(self.vars.dom.barContainer);
                }
            }
        };
    }

    // Definition of new control extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            dom: {
                value: null,
                bar: null,
                barContainer: null,
                unit: null,
                label: null,
                icon: null,
                indicator: null,
                indicatorLine: null,
                controlValue: null
            },
            valueSettings: {
                min: 0,
                max: 100
            },
            animationBundle: null,
            rafId: null,
            value: null,
            valueCtrl: null,
            orientationInfo: null,
            listeners: [],
            tokens: []
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues",
            gfx: "visuals.gfx"
        },

        /* Array of custom event types fired by this control */
        events: [],

        /* Functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                const self = this;

                // DOM
                Object.assign(self.vars.dom, {
                    value: shmi.getUiElement("value", self.element),
                    bar: shmi.getUiElement("bar", self.element),
                    barContainer: shmi.getUiElement("bar-container", self.element),
                    unit: shmi.getUiElement("unit", self.element),
                    label: shmi.getUiElement("label", self.element),
                    icon: shmi.getUiElement("icon", self.element),
                    indicator: shmi.getUiElement("indicator", self.element),
                    indicatorLine: shmi.getUiElement("indicator-line", self.element),
                    controlValue: shmi.getUiElement("control-value", self.element)
                });

                self.imports.nv.initValueSettings(self);

                // Icon
                if (self.vars.dom.icon && self.config['icon-src'] && self.config['show-icon']) {
                    self.vars.dom.icon.style.backgroundImage = `url(${self.config['icon-src']})`;
                } else if (self.vars.dom.icon && self.config['icon-class'] && self.config['show-icon']) {
                    const iconClasses = self.config['icon-class'].trim().split(" ");
                    iconClasses.forEach((cls) => shmi.addClass(self.vars.dom.icon, cls));
                } else {
                    // No icon configured - hide
                    shmi.addClass(self.element, "no-icon");
                }

                // Unit
                setUnitTextImpl(self, self.config['unit-text']);
                if (self.config['unit-scale']) {
                    self.config['unit-scale'] = parseFloat(shmi.localize(self.config['unit-scale']));
                }

                // Label
                setLabelImpl(self, self.config.label);

                // Fill orientation
                if (self.vars.dom.bar) {
                    self.vars.orientationInfo = getFillProperties(self);
                }

                self.vars.animationBundle = new self.imports.gfx.AnimationBundle(updateWidgetBars.bind(null, self));
                self.vars.animationBundle.prepare("actValue", null, "linear");
                self.vars.animationBundle.prepare("ctrlValue", null, "linear");

                updateWidgetText(self);
            },

            /* Called when control is enabled */
            onEnable: function() {
                const self = this;

                self.vars.tokens.push(makeUpdateToken(self, () => {
                    queueWidgetTextUpdate(self);
                    self.vars.animationBundle.refresh();
                }));

                if (self.config.item) {
                    self.vars.tokens.push(self.imports.im.subscribeItem(self.config.item, self));
                }

                if (self.config["item-ctrl"]) {
                    self.vars.tokens.push(self.imports.im.subscribeItem(self.config["item-ctrl"], {
                        setValue: self.onSetCtrlValue.bind(self)
                    }));
                }
            },

            /* Called when control is disabled */
            onDisable: function() {
                const self = this;

                if (self.vars.rafId) {
                    shmi.caf(self.vars.rafId);
                    self.vars.rafId = null;
                }

                self.vars.animationBundle.cancel("actValue");
                self.vars.animationBundle.cancel("ctrlValue");

                self.vars.tokens.forEach((t) => t.unlisten());
                self.vars.tokens = [];
            },

            /* Called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                shmi.addClass(this.element, 'locked');
            },

            /* Called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                shmi.removeClass(this.element, 'locked');
            },

            /**
             * Retrieves current value of the quality display
             *
             * @return value - current value
             */
            getValue: function() {
                const self = this;

                return self.vars.value;
            },

            /**
             * Retrieves current control value of the quality display
             *
             * @return value - current control value
             */
            getControlValue: function() {
                const self = this;

                return self.vars.valueCtrl;
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                const self = this,
                    supportedTypes = [shmi.c("TYPE_BOOL"), shmi.c("TYPE_INT"), shmi.c("TYPE_FLOAT")],
                    newValue = (supportedTypes.includes(type) ? value : Number.NaN);

                if (self.vars.value === newValue) {
                    return;
                }

                self.vars.value = newValue;
                self.vars.animationBundle.start(newValue, self.config["animation-duration"], "actValue");
                queueWidgetTextUpdate(self);
            },

            onSetCtrlValue: function(value, type) {
                const self = this,
                    supportedTypes = [shmi.c("TYPE_BOOL"), shmi.c("TYPE_INT"), shmi.c("TYPE_FLOAT")],
                    newValue = (supportedTypes.includes(type) ? value : Number.NaN);

                if (self.vars.valueCtrl === newValue) {
                    return;
                }

                self.vars.valueCtrl = newValue;
                self.vars.animationBundle.start(newValue, self.config["animation-duration"], "ctrlValue");
                queueWidgetTextUpdate(self);
            },

            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(...args) {
                const self = this;
                //const [min, max, step, name, type, warnmin, warnmax, prewarnmin, prewarnmax, precision] = args;

                self.imports.nv.setProperties(self, args);
                self.vars.animationBundle.refresh();
                queueWidgetTextUpdate(self);
            },

            setUnitText: function(unitText) {
                const self = this;

                if (self.config['auto-unit-text']) {
                    setUnitTextImpl(self, unitText);
                }
            },

            setLabel: function(labelText) {
                const self = this;

                if (self.config['auto-label']) {
                    setLabelImpl(self, labelText);
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-radial-gauge
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-radial-gauge",
 *     "name": null,
 *     "template": "default/iq-radial-gauge"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "min": Minimum value
 * "max": Maximum value
 * "step": Step width
 * "precision": Precision
 * "type": Type is INT
 * "unit-text": Unit Text
 * "auto-label": Whether to use the auto-label (from item)
 * "auto-min": Whether to use auto-min (from item)
 * "auto-max": Whether to use auto-max (from item)
 * "auto-step": Whether to use auto-step (from item)
 * "auto-precision": Whether to use the auto-precision (from item)
 * "auto-type": Whether to use the auto-type (from item)
 * "icon-src": Default icon source
 * "icon-class": Default icon class
 * "tooltip": Tooltip
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "IqRadialGauge", // control name in camel-case
        uiType = "iq-radial-gauge", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": uiType,
        "name": null,
        "template": "default/iq-radial-gauge.variant-01",
        "label": '[Label]',
        "item": null,
        "min": Number.NEGATIVE_INFINITY,
        "max": Number.POSITIVE_INFINITY,
        "step": 1,
        "precision": 0,
        "type": shmi.c("TYPE_INT"),
        "unit-text": "[Unit]",
        "auto-label": true,
        "auto-min": true,
        "auto-max": true,
        "auto-step": true,
        "auto-precision": true,
        "auto-type": true,
        "show-icon": false,
        "icon-src": null,
        "icon-class": null,
        "tooltip": null,

        "animation-duration": 100,
        "arc-fill-reverse": false,
        "arc-fill-inverse": false,
        "default-value": "---"
    };

    // setup module-logger
    const ENABLE_LOGGING = true,
        RECORD_LOG = false;
    const logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG),
        fLog = logger.fLog,
        log = logger.log,

        /*******************/
        /** CSS MAPPINGS ***/
        /*******************/
        /* Defines a CSS to object (self.vars.styles...) mapping for a line style (background arc, foreground arc):
         *
         * propName = CSS property name, i.e. "color" means: .className { color:#c40000; }
         * attribName = The attribute name from the self.vars.styles object
         * conversion = which conversion to use (none, parseInt or special box-shadow mapping
         */
        lineStyleConfig = [
            {
                "propName": "background",
                "attribName": "color",
                "conversion": "parseBackground"
            },
            {
                "propName": "width",
                "attribName": "lineWidth",
                "conversion": "parseInt"
            },
            {
                "propName": "box-shadow",
                "attribName": ["shadowColor", "shadowOffsetX", "shadowOffsetY", "shadowBlur", null],
                "conversion": "boxShadow"
            }
        ],

        // Defines a CSS to object (self.vars.styles...) mapping for a font style (value, label, unit)
        fontStyleConfig = [
            {
                "propName": "color",
                "attribName": "color",
                "conversion": null
            },
            {
                "propName": "font-size",
                "attribName": "fontSize",
                "conversion": "parseInt"
            },
            {
                "propName": "font-family",
                "attribName": "fontFamily",
                "conversion": null
            },
            {
                "propName": "font-weight",
                "attribName": "fontWeight",
                "conversion": null
            },
            {
                "propName": "font-style",
                "attribName": "fontStyle",
                "conversion": null
            },
            {
                "propName": "text-decoration",
                "attribName": "textUnderline",
                "conversion": "parseUnderline"
            }
        ],

        orientationStyleConfig = [
            {
                "propName": "top",
                "attribName": "offsetVertical",
                "conversion": "parsePercent"
            },
            {
                "propName": "left",
                "attribName": "offsetHorizontal",
                "conversion": "parsePercent"
            },
            {
                "propName": "offset-rotate",
                "attribName": "rotation",
                "conversion": "offsetRotate"
            }
        ],

        arcSliceStyleConfig = [
            {
                "propName": "width",
                "attribName": "width",
                "conversion": "parsePercent"
            }
        ],

        percentageSizeStyle = [
            {
                "propName": "width",
                "attribName": "width",
                "conversion": "parsePercent"
            },
            {
                "propName": "height",
                "attribName": "height",
                "conversion": "parsePercent"
            }
        ],

        // Defines all available mappings (background arc, foreground arc, end marker, value, label, unit)
        cssStyleMappings = [
            {
                "data-ui": "arc-bg",
                "objName": "arcBackground",
                "styleConfig": lineStyleConfig
            },
            {
                "data-ui": "arc-fg",
                "objName": "arcForeground",
                "styleConfig": lineStyleConfig.concat(orientationStyleConfig)
            },
            {
                "data-ui": "arc-slice",
                "objName": "slice",
                "styleConfig": arcSliceStyleConfig
            },
            {
                "data-ui": "line-value",
                "objName": "valueLine",
                "styleConfig": lineStyleConfig
            },
            {
                "data-ui": "value",
                "objName": "value",
                "styleConfig": fontStyleConfig.concat(orientationStyleConfig)
            },
            {
                "data-ui": "label",
                "objName": "label",
                "styleConfig": fontStyleConfig.concat(orientationStyleConfig)
            },
            {
                "data-ui": "unit",
                "objName": "unit",
                "styleConfig": fontStyleConfig.concat(orientationStyleConfig)
            },
            {
                "data-ui": "icon-src",
                "objName": "icon",
                "styleConfig": percentageSizeStyle.concat(orientationStyleConfig)
            },
            {
                "data-ui": "icon-class",
                "objName": "iconClass",
                "styleConfig": fontStyleConfig
            }
        ];

    const styleParsers = {
        identity: (inputValue) => [inputValue],
        parseInt: (inputValue) => {
            const parsed = parseInt(inputValue);
            if (isNaN(parsed)) {
                return null;
            }

            return [parsed];
        },
        parsePercent: (inputValue) => {
            if (inputValue === "0px") {
                return [0];
            } else if (!inputValue.endsWith("%")) {
                return null;
            }

            const parsed = parseFloat(inputValue.substr(0, inputValue.length - 1));
            if (isNaN(parsed)) {
                return null;
            }

            return [parsed / 100];
        },
        boxShadow: (inputValue) => {
            if (inputValue === "none") {
                return [0, 0, 0, 0, "transparent"];
            }

            // <color> <offset-x> <offset-y> [blur-radius] [spread-radius]
            const regex = /^(rgba?\([-\d.,\s]+\))\s+(-?[0-9]+(?:\.[0-9]+)?)px\s+(-?[0-9]+(?:\.[0-9]+)?)px\s+(-?[0-9]+(?:\.[0-9]+)?)px\s+(-?[0-9]+(?:\.[0-9]+)?)px$/;
            const match = inputValue.match(regex);
            if (!match) {
                return null;
            }

            return [
                match[1], // color
                match[2], // offset-x
                match[3], // offset-y
                match[4], // blur-radius
                match[5] // spread-radius
            ];
        },
        offsetRotate: (inputValue) => {
            // [auto] <rotation>
            const regex = /^(?:auto\s+)?(-?[0-9]+(?:\.[0-9]+)?)deg$/,
                match = inputValue.match(regex);
            if (!match) {
                return null;
            }

            return [
                Math.PI * parseFloat(match[1]) / 180
            ];
        },
        parseBackground: (inputValue) => {
            const regAll = /(linear-gradient|radial-gradient)\((?:([^,]+), )?((?:rgba?\([-\d.,\s]+\)(?: [-\d.]+(%|px)?)?(?:, )?)+)\)/,
                regToken = /(rgba?\([-\d.,\s]+\))(?: ([-\d.]+)(%|px)?)?(?:, )?/g,
                regColor = /^(rgba?\([-\d.,\s]+\))/;

            const coarseGradientTokens = inputValue.match(regAll);
            if (!coarseGradientTokens) {
                // background is not a gradient.
                const colorTokens = inputValue.match(regColor);
                if (!colorTokens) {
                    return [null];
                }

                return [{
                    type: "color",
                    color: colorTokens[1]
                }];
            }

            const [, gradientType, firstValue, gradientValuesAll] = coarseGradientTokens,
                gradientValues = Array.from(gradientValuesAll.matchAll(regToken));

            switch (gradientType) {
            case "linear-gradient":
                return [{
                    type: gradientType,
                    // It looks like chrome does something wonky with gradients
                    // at 180° but this takes care of that.
                    rotation: typeof firstValue === "undefined" ? Math.PI : (parseFloat(firstValue.substr(0, firstValue.length - 3)) / 180 * Math.PI) || 0,
                    steps: gradientValues.map(([, color, step]) => ({ color, step: parseInt(step) / 100 }))
                }];
            case "radial-gradient":
                return [{
                    type: gradientType,
                    steps: gradientValues.map(([, color, step]) => ({ color, step: parseInt(step) / 100 }))
                }];
            default:
                return [null];
            }
        },
        parseUnderline: (inputValue) => [inputValue.includes("underline")]
    };

    /**
     * @param {object} self
     * @param {object} styleData
     * @param {string} propName
     * @param {string} objName
     * @param {string} attribName
     * @param {string} conversion "parseInt", "boxShadow"
     */
    function setStyleFromCss(self, styleData, propName, objName, attribName, conversion) {
        const parser = styleParsers[conversion || "identity"];
        if (!styleData || !parser) {
            return;
        }

        const styleVal = propName === "background" ? styleData.getPropertyValue("background-image") === "none" ? styleData.getPropertyValue("background-color") : styleData.getPropertyValue("background-image") : styleData.getPropertyValue(propName);
        if (!styleVal) {
            return;
        }

        const parsed = parser(styleVal);
        if (!parsed) {
            return;
        }

        if (!Array.isArray(attribName)) {
            attribName = [attribName];
        }

        parsed.forEach((value, idx) => {
            if (attribName[idx]) {
                self.vars.styles[objName][attribName[idx]] = value;
            }
        });
    }

    /**
     * @param {*} self
     * @param {number} fontSize
     * @param {string} fontFamily
     * @param {string|number} fontWeight
     * @param {string} fontStyle
     * @param {string} text
     * @returns {TextMetrics}
     */
    function calculateTextMeasurement(self, { fontSize, fontFamily, fontWeight, fontStyle }, text) {
        const ctx = self.vars.canvasObj;

        ctx.font = `${fontStyle} ${fontWeight} ${fontSize * (window.devicePixelRatio || 1)}px ${fontFamily}`;
        return ctx.measureText(text);
    }

    /**
     * @param {*} self
     * @returns {{unit: TextMetrics, label: TextMetrics, value: TextMetrics, valueLineHeight: TextMetrics}}
     */
    function calculateTextMeasurements(self) {
        return {
            value: calculateTextMeasurement(self, self.vars.styles.value, self.vars.valueFormatted),
            valueLineHeight: calculateTextMeasurement(self, self.vars.styles.value, `-0${self.config["decimal-delimiter"]}123456789`),
            unit: calculateTextMeasurement(self, self.vars.styles.unit, self.vars.unitLocalized),
            label: calculateTextMeasurement(self, self.vars.styles.label, self.vars.labelLocalized)
        };
    }

    /**
     * Computes the normalized bounding box around a rotated gauge with the
     * specified slice angle.
     *
     * @param {number} rotation Rotation of the gauge
     * @param {number} sliceAngle angle of the cut out.
     * @returns {object}
     */
    function computeNormalizedGaugeBoundingBox(rotation, sliceAngle) {
        const rotationOffset = (2 * Math.PI - sliceAngle) / 2,
            rotationAngle1 = rotation + rotationOffset,
            rotationAngle2 = rotation + rotationOffset + sliceAngle;

        // Set of points to examine:
        // - End-Points of the partial circle.
        // - Intersections with axis
        const points = [
            { x: Math.cos(rotationAngle1), y: Math.sin(rotationAngle1), theta: rotationAngle1 },
            { x: Math.cos(rotationAngle2), y: Math.sin(rotationAngle2), theta: rotationAngle2 }
        ];

        // Compute intersetctions with axis
        for (let i = Math.ceil(2 * rotationAngle1 / Math.PI); i <= Math.floor(2 * rotationAngle2 / Math.PI); ++i) {
            const theta = i * Math.PI / 2;
            points.push({ x: Math.cos(theta), y: Math.sin(theta), theta });
        }

        // Compute bounding box of points
        return points.reduce((boundingBox, point) => ({
            top: Math.min(boundingBox.top, point.y),
            left: Math.min(boundingBox.left, point.x),
            bottom: Math.max(boundingBox.bottom, point.y),
            right: Math.max(boundingBox.right, point.x),
            centerX: boundingBox.centerX,
            centerY: boundingBox.centerY,
            width: boundingBox.width,
            height: boundingBox.height
        }), {
            top: 1,
            left: 1,
            bottom: -1,
            right: -1,
            centerX: 0,
            centerY: 0,
            width: 2,
            height: 2
        });
    }

    /**
     * Takes a normalized gauge bounding box and translates and scales it such
     * that it will fit exactly within the specified dimensions and is
     * centered.
     *
     * @param {object} boundingBox Normalized gauge bounding box computed by
     * @param {number} width Width to fit the gauge bounding box in.
     * @param {number} height Height to fit the gauge bounding box in.
     * @returns {{top: number, left: number, bottom: number, right: number, centerY: number, centerX: number, height: number, width: number, radius: number}}
     */
    function translateGaugeBoundingBox(boundingBox, width, height) {
        // Compute the size of the longest side of the bounding box.
        const longestSide = (() => {
            const scalerHeight = (boundingBox.bottom - boundingBox.top) / boundingBox.height,
                scalerWidth = (boundingBox.right - boundingBox.left) / boundingBox.width,
                ratioVertical = scalerWidth / scalerHeight,
                ratioHorizontal = scalerHeight / scalerWidth;

            if (boundingBox.right - boundingBox.left > boundingBox.bottom - boundingBox.top) {
                return Math.min(ratioVertical * height, width);
            }

            return Math.min(ratioHorizontal * width, height);
        })();

        // Transformations done on the bounding box:
        // 1. Move upper left corner of bounding box to (0, 0)
        // 2. Scale bounding box such that max(width, height) = 1 while keeping
        //    the aspect ratio.
        // 3. Scale bounding box to fit the target width/height.

        const computeSize = (X, offset, scaler) => (X - offset) / Math.max(boundingBox.right - boundingBox.left, boundingBox.bottom - boundingBox.top) * scaler;

        // Compute dimensions of the translated bounding box.
        const translatedWidth = computeSize(boundingBox.right, boundingBox.left, longestSide),
            translatedHeight = computeSize(boundingBox.bottom, boundingBox.top, longestSide);

        return {
            top: (height - translatedHeight) / 2,
            left: (width - translatedWidth) / 2,
            bottom: (height + translatedHeight) / 2,
            right: (width + translatedWidth) / 2,
            centerY: computeSize(boundingBox.centerY, boundingBox.top, longestSide) + (height - translatedHeight) / 2,
            centerX: computeSize(boundingBox.centerX, boundingBox.left, longestSide) + (width - translatedWidth) / 2,
            height: translatedHeight,
            width: translatedWidth,
            radius: computeSize(1, 0, longestSide)
        };
    }

    /**
     *
     * @param {*} self
     */
    function calculateIconData(self, width, height) {
        if (!self.vars.iconEl) {
            return {
                type: "none",
                height
            };
        } else if (self.vars.iconEl.nodeName === "IMG") {
            const imageRatio = self.vars.iconEl.width / self.vars.iconEl.height;

            return {
                type: "image",
                src: self.vars.iconEl.src,
                element: self.vars.iconEl,
                ready: self.vars.iconEl.complete,
                width: Math.min(imageRatio * height, width),
                height: Math.min(width / imageRatio, height),
                drawWidth: width,
                drawHeight: height
            };
        } else if (self.vars.iconEl.nodeName !== "DIV") {
            return {
                type: "invalid",
                drawWidth: width,
                drawHeight: height
            };
        }

        const computedStyles = getComputedStyle(self.vars.iconEl, ":before");
        if (!computedStyles) {
            return {
                type: "invalid",
                drawWidth: width,
                drawHeight: height
            };
        }

        return {
            type: "icon-font",
            str: computedStyles.content,
            fontFamily: computedStyles.fontFamily,
            measurement: calculateTextMeasurement(self, { fontSize: height, fontFamily: computedStyles.fontFamily, fontWeight: 400, fontStyle: "normal" }, computedStyles.content),
            drawWidth: width,
            drawHeight: height
        };
    }

    /**
     * @param {*} self
     */
    function invalidateTextMeasurements(self) {
        if (!self.vars.cache) {
            self.vars.cache = {};
        }
        self.vars.cache.textMeasurement = null;
    }

    /**
     * Resets the style cache flag, causing all styles to be reevaluated on the
     * next draw.
     *
     * @param {object} self
     */
    function invalidateStyles(self) {
        if (!self.vars.cache) {
            self.vars.cache = {};
        }
        self.vars.cache.hasStyles = false;
    }

    /**
     * Invalidates icon data.
     *
     * @param {*} self Reference to an instance of a radial gauge.
     */
    function invalidateIconData(self) {
        if (!self.vars.cache) {
            self.vars.cache = {};
        }
        self.vars.cache.iconData = null;
    }

    /**
     * Invalidates the cached gauge bounding box.
     *
     * @param {*} self
     */
    function invalidateGaugeBoundingBox(self) {
        if (!self.vars.cache) {
            self.vars.cache = {};
        }
        self.vars.cache.boundingBox = null;
    }

    /**
     * Invalidates cached fill styles.
     *
     * @param {*} self
     */
    function invalidateFillStyles(self) {
        if (!self.vars.cache) {
            self.vars.cache = {};
        }
        self.vars.cache.fillStyles = null;
    }

    /**
     * @param {*} self
     * @returns {boolean}
     */
    function haveDisplayPropertiesChanged(self) {
        if (!self.vars.cache) {
            return true;
        } else if (!self.vars.cache.display) {
            return true;
        }

        const cache = self.vars.cache.display,
            current = {
                devicePixelRatio: window.devicePixelRatio || 1,
                widgetWidth: self.element.clientWidth,
                widgetHeight: self.element.clientHeight
            };

        return Object.keys(current).some((key) => current[key] !== cache[key]);
    }

    /**
     * Checks if styles have been evaluated and can be used.
     *
     * @param {object} self
     */
    function haveStylesChanged(self) {
        if (!self.vars.cache) {
            return true;
        }

        return !self.vars.cache.hasStyles;
    }

    /**
     * @param {*} self
     */
    function checkCache(self) {
        if (!self.vars.cache) {
            self.vars.cache = {};
        }

        if (!self.vars.cache.textMeasurement) {
            self.vars.cache.textMeasurement = calculateTextMeasurements(self);
        }
    }

    /**
     * Gets the bounding box from cache. If no bounding box is cached or a
     * a parameter is chagned, the bounding box is recomputed.
     *
     * @param {*} self
     * @param {number} width
     * @param {number} height
     * @param {number} arcRotation
     * @param {number} arcAngle
     * @returns {{width: number, height: number, arcRotation: number, arcAngle: number, boundingBox: object}}
     */
    function getBoundingBox(self, width, height, arcRotation, arcAngle) {
        if (!self.vars.cache) {
            self.vars.cache = {};
        }

        if (
            !self.vars.cache.boundingBox ||
            self.vars.cache.boundingBox.width !== width ||
            self.vars.cache.boundingBox.height !== height ||
            self.vars.cache.boundingBox.arcRotation !== arcRotation ||
            self.vars.cache.boundingBox.arcAngle !== arcAngle
        ) {
            self.vars.cache.boundingBox = {
                width,
                height,
                arcRotation,
                arcAngle,
                boundingBox: translateGaugeBoundingBox(computeNormalizedGaugeBoundingBox(arcRotation, arcAngle), width, height)
            };
        }

        return self.vars.cache.boundingBox.boundingBox;
    }

    /**
     * Gets icon data from cache. If no icon data is cached or a parameter
     * has changed, the icon data is recomputed.
     *
     * @param {*} self
     * @param {number} width
     * @param {number} height
     * @returns {object}
     */
    function getIconData(self, width, height) {
        if (!self.vars.cache) {
            self.vars.cache = {};
        }

        if (
            !self.vars.cache.iconData ||
            self.vars.cache.iconData.drawWidth !== width ||
            self.vars.cache.iconData.drawHeight !== height
        ) {
            self.vars.cache.iconData = calculateIconData(self, width, height);
        }

        return self.vars.cache.iconData;
    }

    /**
     * Computes absolute text coordinates from canvas dimensions and relative
     * coordinates.
     *
     * @param {number} width Width of the canvas
     * @param {number} height Height of the canvas
     * @param {{x: number, y:number}} position
     * @param {TextMetrics} measurement
     * @param {number} [lineHeight]
     * @return {{x: number, y:number}}
     */
    function getTextPosition(width, height, position, measurement, lineHeight) {
        return {
            x: Math.floor((width - measurement.width) * position.x + 0.5),
            y: Math.floor(height * position.y + (lineHeight || measurement.actualBoundingBoxAscent) * (1 - position.y) + 0.5)
        };
    }

    /**
     * Computes gauge offset from canvas dimensions and relative coordinates.
     *
     * @param {number} width
     * @param {number} height
     * @param {{x: number, y:number}} position
     * @param {*} boundingBox
     * @return {{x: number, y:number}}
     */
    function getGaugeOffset(width, height, position, boundingBox) {
        return {
            x: (width - boundingBox.width) * (position.x - 0.5),
            y: (height - boundingBox.height) * (position.y - 0.5)
        };
    }

    /**
     *
     * @param {object} style
     * @returns {{x: number, y: number}}
     */
    function positionFromStyle(style) {
        return {
            x: style.offsetHorizontal,
            y: style.offsetVertical
        };
    }

    /**
     * Rotates coordinates of a box.
     *
     * @param {{x0: number, y0: number, x1: number, y1: number}} param0
     * @param {number} theta angle to rotate the box by (in rad).
     */
    function rotateBox({ x0, y0, x1, y1 }, theta) {
        // Correct for starting top-left.
        theta -= (135 / 180 * Math.PI);

        const centered = {
                x0: (x0 - x1) / 2, // x0 = x0 - (x0 + x1) / 2
                y0: (y0 - y1) / 2, // y0 = y0 - (y0 + y1) / 2
                x1: (x1 - x0) / 2, // x1 = x1 - (x1 + x0) / 2
                y1: (y1 - y0) / 2 // y1 = y1 - (y1 + y0) / 2
            },
            rotated = {
                x0: centered.x0 * Math.cos(theta) - centered.y0 * Math.sin(theta),
                y0: centered.x0 * Math.sin(theta) + centered.y0 * Math.cos(theta),
                x1: centered.x1 * Math.cos(theta) - centered.y1 * Math.sin(theta),
                y1: centered.x1 * Math.sin(theta) + centered.y1 * Math.cos(theta)
            };

        return {
            x0: rotated.x0 + (x1 + x0) / 2,
            y0: rotated.y0 + (y1 + y0) / 2,
            x1: rotated.x1 + (x1 + x0) / 2,
            y1: rotated.y1 + (y1 + y0) / 2
        };
    }

    /**
     * @param {CanvasRenderingContext2D} ctx Canvas context to generate the
     *  style for.
     * @param {number} x x-Position of the object on the canvas
     * @param {number} y y-Position of the object on the canvas
     * @param {number} width Width of the object to style
     * @param {number} height Height of the object to style
     * @param {*} style
     *
     * @returns {null|string|CanvasGradient}
     */
    function makeFillStyle(ctx, x, y, width, height, style) {
        if (typeof style === "string") {
            return style;
        } else if (style === null) {
            return null;
        }

        switch (style.type) {
        case "color":
            return style.color;
        case "linear-gradient": {
            const scaler = Math.max(1, ...style.steps.map(({ step }) => step)),
                gradientBox = rotateBox({ x0: x - width * (scaler - 1), y0: y - height * (scaler - 1), x1: x + width * scaler, y1: y + height * scaler }, style.rotation),
                gradient = ctx.createLinearGradient(gradientBox.x0, gradientBox.y0, gradientBox.x1, gradientBox.y1);

            style.steps.forEach(({ color, step }) => gradient.addColorStop(step / scaler, color));
            return gradient;
        }
        case "radial-gradient": {
            const scaler = Math.max(1, ...style.steps.map(({ step }) => step)),
                gradient = ctx.createRadialGradient(
                    x + width / 2,
                    y + height / 2,
                    0,
                    x + width / 2,
                    y + height / 2,
                    Math.max(width, height) * scaler / 2
                );

            style.steps.forEach(({ color, step }) => gradient.addColorStop(step / scaler, color));
            return gradient;
        }
        default:
            return null;
        }
    }

    /**
     * Generates a fill style and stores it in cache if it doesn't already
     * exist.
     *
     * @param {*} self
     * @param {*} styleName Name of the style to generate
     * @param {CanvasRenderingContext2D} ctx Canvas context to generate the
     *  style for.
     * @param {number} x x-Position of the object on the canvas
     * @param {number} y y-Position of the object on the canvas
     * @param {number} width Width of the object to style
     * @param {number} height Height of the object to style
     */
    function cacheFillStyle(self, styleName, ctx, x, y, width, height) {
        if (!self.vars.cache) {
            self.vars.cache = {};
        }

        if (!self.vars.cache.fillStyles) {
            self.vars.cache.fillStyles = {};
        }

        if (!self.vars.cache.fillStyles[styleName]) {
            self.vars.cache.fillStyles[styleName] = makeFillStyle(ctx, x, y, width, height, self.vars.styles[styleName].color);
        }
    }

    /**
     * Draws a text
     *
     * @param {CanvasRenderingContext2D} ctx Canvas context to draw the text
     *  on.
     * @param {number} width Width of the canvas
     * @param {number} height Height of the canvas
     * @param {number} fontSize
     * @param {string} fontFamily
     * @param {string|number} fontWeight
     * @param {string} fontStyle
     * @param {number} pixelRatio
     * @param {{x: number, y: number}} position
     * @param {string} text The text to draw.
     * @param {TextMetrics} measurement
     * @param {string} color
     * @param {number} [lineHeight]
     * @param {boolean} [underline]
     */
    function drawText(ctx, width, height, fontSize, fontFamily, fontWeight, fontStyle, pixelRatio, position, text, measurement, color, lineHeight, underline) {
        const textCoords = getTextPosition(width, height, position, measurement, lineHeight),
            underlineLineWidth = Math.max(1, Math.floor(fontSize * pixelRatio / 15)),
            // Offset =
            //   1x line width for extra distance +
            // 0.5x line width to properly align the center so we always draw whole pixels.
            underlineLineOffset = underlineLineWidth * 1.5;

        ctx.beginPath();
        ctx.shadowBlur = 0;
        ctx.fillStyle = color;
        ctx.font = `${fontStyle} ${fontWeight} ${fontSize * pixelRatio}px ${fontFamily}`;
        ctx.fillText(text, textCoords.x, textCoords.y);
        ctx.stroke();

        if (underline) {
            ctx.beginPath();
            ctx.strokeStyle = ctx.fillStyle;
            ctx.lineWidth = underlineLineWidth;
            ctx.moveTo(textCoords.x, textCoords.y + underlineLineOffset);
            ctx.lineTo(textCoords.x + measurement.width, textCoords.y + underlineLineOffset);
            ctx.stroke();
        }
    }

    /**
     * Draws the widget icon
     *
     * @param {CanvasRenderingContext2D} ctx Canvas context to draw the icon
     *  on.
     * @param {object} iconData
     * @param {number} width Width of the canvas
     * @param {number} height Height of the canvas
     * @param {{x: number, y: number}} position
     * @param {number} pixelRatio
     * @param {string} color
     */
    function drawIcon(ctx, iconData, width, height, position, pixelRatio, color) {
        if (iconData.type === "image") {
            ctx.drawImage(iconData.element, (width - iconData.width) * position.x, (height - iconData.height) * position.y, iconData.width, iconData.height);
        } else if (iconData.type === "icon-font") {
            const textCoords = getTextPosition(width, height, position, iconData.measurement);
            drawText(ctx, width, textCoords.x, textCoords.y, iconData.fontFamily, "normal", "normal", pixelRatio, position, iconData.str, iconData.measurement, color);
        }
    }

    /**
     * Applies shadow styles to a canvas context.
     *
     * @param {CanvasRenderingContext2D} ctx Canvas context
     * @param {object} styles
     * @param {number} pixelRatio
     */
    function applyShadowStyles(ctx, styles, pixelRatio) {
        ctx.shadowBlur = styles.shadowBlur * pixelRatio;
        ctx.shadowColor = styles.shadowColor;
        ctx.shadowOffsetX = styles.shadowOffsetX * pixelRatio;
        ctx.shadowOffsetY = styles.shadowOffsetY * pixelRatio;
    }

    /**
     * Draws the gauge.
     *
     * @param {object} self
     * @param {CanvasRenderingContext2D} ctx Canvas context
     * @param {number} multiplier A value between 0 and 1 (0 = 0%, 1 = 100%)
     * @param {number} width Width of the canvas
     * @param {number} height Height of the canvas
     * @param {number} rotation Rotation of the gauge arc
     * @param {number} arcAngleMax
     * @param {number} pixelRatio
     * @param {boolean} inverse
     */
    function drawGauge(self, ctx, multiplier, width, height, rotation, arcAngleMax, pixelRatio, inverse) {
        const rotationOffset = (2 * Math.PI - arcAngleMax) / 2,
            lineWidthBgArc = self.vars.styles.arcBackground.lineWidth * pixelRatio,
            shadowBlurBgArc = self.vars.styles.arcBackground.shadowBlur * pixelRatio,
            lineWidthFgArc = self.vars.styles.arcForeground.lineWidth * pixelRatio,
            shadowBlurFgArc = self.vars.styles.arcForeground.shadowBlur * pixelRatio,
            lineWidthValueLine = self.vars.styles.valueLine.lineWidth * pixelRatio,
            gaugeBoundingBox = getBoundingBox(self, width, height, rotation, arcAngleMax),
            radiusBgArc = gaugeBoundingBox.radius - Math.max(lineWidthBgArc, lineWidthFgArc) / 2 - shadowBlurBgArc,
            radiusFgArc = gaugeBoundingBox.radius - Math.max(lineWidthBgArc, lineWidthFgArc) / 2 - shadowBlurFgArc;

        // Calculate angle offset for foreground arc.
        const arcForegroundBorderWidth = (lineWidthBgArc - lineWidthFgArc),
            arcForegroundRotOffset = Math.max(0, Math.atan(arcForegroundBorderWidth / radiusFgArc)),
            arcBackgroundRotOffset = Math.max(0, Math.atan(-arcForegroundBorderWidth / radiusFgArc)),
            gaugeOffset = getGaugeOffset(width, height, positionFromStyle(self.vars.styles.arcForeground), gaugeBoundingBox);

        // Do not allow an invalid radius
        if (radiusBgArc < 0 || radiusFgArc < 0) {
            return;
        }

        checkCache(self);
        cacheFillStyle(
            self, "arcBackground", ctx,
            gaugeBoundingBox.left + gaugeOffset.x,
            gaugeBoundingBox.top + gaugeOffset.y,
            gaugeBoundingBox.right - gaugeBoundingBox.left,
            gaugeBoundingBox.bottom - gaugeBoundingBox.top
        );
        cacheFillStyle(
            self, "arcForeground", ctx,
            gaugeBoundingBox.left + gaugeOffset.x,
            gaugeBoundingBox.top + gaugeOffset.y,
            gaugeBoundingBox.right - gaugeBoundingBox.left,
            gaugeBoundingBox.bottom - gaugeBoundingBox.top
        );
        ctx.clearRect(0, 0, width, height);

        // BACKGROUND LINE
        ctx.beginPath();
        ctx.lineWidth = lineWidthBgArc;
        ctx.strokeStyle = self.vars.cache.fillStyles.arcBackground;
        applyShadowStyles(ctx, self.vars.styles.arcBackground, pixelRatio);
        ctx.arc(
            gaugeBoundingBox.centerX + gaugeOffset.x,
            gaugeBoundingBox.centerY + gaugeOffset.y,
            radiusBgArc,
            rotation + rotationOffset + arcBackgroundRotOffset / 2,
            arcAngleMax + rotation + rotationOffset - arcBackgroundRotOffset / 2
        );
        ctx.stroke();

        // FILLED LINE
        ctx.beginPath();
        const arcAngleStart = !inverse ? rotation + rotationOffset + arcForegroundRotOffset / 2 : (arcAngleMax - arcForegroundRotOffset) * (1 - multiplier) + rotation + rotationOffset + arcForegroundRotOffset / 2,
            arcAngleEnd = !inverse ? (arcAngleMax - arcForegroundRotOffset) * multiplier + rotation + rotationOffset + arcForegroundRotOffset / 2 : rotation - rotationOffset + arcForegroundRotOffset / 2;
        ctx.lineWidth = lineWidthFgArc;
        ctx.strokeStyle = self.vars.cache.fillStyles.arcForeground;
        applyShadowStyles(ctx, self.vars.styles.arcForeground, pixelRatio);
        ctx.arc(
            gaugeBoundingBox.centerX + gaugeOffset.x,
            gaugeBoundingBox.centerY + gaugeOffset.y,
            radiusFgArc,
            arcAngleStart,
            arcAngleEnd
        );
        ctx.stroke();

        // ICON
        drawIcon(ctx, getIconData(self, width * self.vars.styles.icon.width, height * self.vars.styles.icon.height), width, height, positionFromStyle(self.vars.styles.icon), pixelRatio, self.vars.styles.iconClass.color);

        const valueStyles = self.vars.styles.value,
            valueTextMeasurement = self.vars.cache.textMeasurement.value,
            valueTextLineHeight = self.vars.cache.textMeasurement.valueLineHeight.actualBoundingBoxAscent;

        // VALUE LINE
        ctx.beginPath();
        const valueLineCoords = getTextPosition(width, height, positionFromStyle(self.vars.styles.value), valueTextMeasurement, valueTextLineHeight),
            valueLinePoints = {
                x0: valueLineCoords.x,
                y0: valueLineCoords.y + Math.floor(valueLineCoords.y + valueTextMeasurement.actualBoundingBoxAscent / 4) + lineWidthValueLine / 2,
                x1: valueLineCoords.x + valueTextMeasurement.width,
                y1: valueLineCoords.y + Math.floor(valueLineCoords.y + valueTextMeasurement.actualBoundingBoxAscent / 4) + lineWidthValueLine / 2
            };
        ctx.lineWidth = lineWidthValueLine;
        ctx.strokeStyle = makeFillStyle(
            ctx,
            valueLinePoints.x0,
            valueLinePoints.y0,
            valueLinePoints.x1 - valueLinePoints.x0,
            valueLinePoints.y1 - valueLinePoints.y0,
            self.vars.styles.valueLine.color
        );
        applyShadowStyles(ctx, self.vars.styles.valueLine, pixelRatio);
        ctx.moveTo(valueLinePoints.x0, valueLinePoints.y0);
        ctx.lineTo(valueLinePoints.x1, valueLinePoints.y1);
        ctx.stroke();

        // VALUE
        drawText(ctx, width, height, valueStyles.fontSize, valueStyles.fontFamily, valueStyles.fontWeight, valueStyles.fontStyle, pixelRatio, positionFromStyle(self.vars.styles.value), self.vars.valueFormatted, valueTextMeasurement, valueStyles.color, valueTextLineHeight, valueStyles.textUnderline);

        // UNIT
        if (self.vars.unitLocalized) {
            const unitStyles = self.vars.styles.unit,
                unitTextMeasurement = self.vars.cache.textMeasurement.unit;

            drawText(ctx, width, height, unitStyles.fontSize, unitStyles.fontFamily, unitStyles.fontWeight, unitStyles.fontStyle, pixelRatio, positionFromStyle(self.vars.styles.unit), self.vars.unitLocalized, unitTextMeasurement, unitStyles.color, null, unitStyles.textUnderline);
        }

        // Label
        if (self.vars.labelLocalized) {
            const labelStyles = self.vars.styles.label,
                labelTextMeasurement = self.vars.cache.textMeasurement.label;

            drawText(ctx, width, height, labelStyles.fontSize, labelStyles.fontFamily, labelStyles.fontWeight, labelStyles.fontStyle, pixelRatio, positionFromStyle(self.vars.styles.label), self.vars.labelLocalized, labelTextMeasurement, labelStyles.color, null, labelStyles.textUnderline);
        }
    }

    /**
     * Updates the canvas elements resolution to fit the widget dimensions.
     *
     * @param {*} self
     */
    function updateCanvasResolution(self) {
        const { clientWidth, clientHeight } = self.element,
            devicePixelRatio = window.devicePixelRatio || 1;

        if (self.vars.canvasEl.width !== self.vars.canvasEl.clientWidth * devicePixelRatio) {
            self.vars.canvasEl.width = self.vars.canvasEl.clientWidth * devicePixelRatio;
        }

        if (self.vars.canvasEl.height !== self.vars.canvasEl.clientHeight * devicePixelRatio) {
            self.vars.canvasEl.height = self.vars.canvasEl.clientHeight * devicePixelRatio;
        }

        self.vars.cache.display = {
            devicePixelRatio: devicePixelRatio,
            widgetWidth: clientWidth,
            widgetHeight: clientHeight
        };
    }

    /**
     * @param {*} self
     * @returns {boolean}
     */
    function isReady(self) {
        return !!self.vars.canvasObj;
    }

    /**
     * @param {*} self
     * @returns {?number}
     */
    function startDrawing(self) {
        if (self.vars.animationFrameToken) {
            return null;
        }

        return shmi.raf((timestamp) => {
            if (!isReady(self)) {
                return;
            }

            const devicePixelRatio = window.devicePixelRatio || 1,
                { min, max } = self.vars.valueSettings,
                canvas = self.vars.canvasObj;

            if (self.vars.animationTimestamp === null || self.vars.animationValueStep === null || self.vars.animationValueStep === 0) {
                self.vars.animationValue = self.vars.value;
            } else {
                const valueDiff = self.vars.animationValueStep * (timestamp - self.vars.animationTimestamp);
                self.vars.hasChanged = true;
                if (Math.abs(Math.abs(self.vars.animationValue) - Math.abs(self.vars.value)) < Math.abs(valueDiff)) {
                    self.vars.animationValue = self.vars.value;
                    self.vars.animationValueStep = null;
                } else {
                    self.vars.animationValue = self.vars.animationValue + valueDiff;
                }
            }

            self.vars.animationTimestamp = timestamp;

            if (haveDisplayPropertiesChanged(self)) {
                invalidateTextMeasurements(self);
                invalidateGaugeBoundingBox(self);
                invalidateFillStyles(self);
                if (haveStylesChanged(self)) {
                    self.setupStylesFromCss();
                }

                updateCanvasResolution(self);

                self.vars.hasChanged = true;
            } else if (haveStylesChanged(self)) {
                invalidateFillStyles(self);
                self.setupStylesFromCss();
                self.vars.hasChanged = true;
            } else if (self.vars.forceRedraw) {
                self.vars.hasChanged = true;
                self.vars.forceRedraw = false;
            }

            // Safety net: when a user sets invalid min/max values manually the circle would otherwise be overdrawn
            const multiplier = Math.min(1, Math.max((self.vars.animationValue - min) / (max - min), 0));

            if (self.vars.elementIsInViewport && self.vars.hasChanged) {
                const fillPercent = self.config["arc-fill-inverse"] ? 1 - multiplier : multiplier,
                    arcAngle = 2 * Math.PI * (1 - self.vars.styles.slice.width);

                drawGauge(self, canvas, fillPercent, self.vars.canvasEl.width, self.vars.canvasEl.height, self.vars.styles.arcForeground.rotation, arcAngle, devicePixelRatio, self.config["arc-fill-reverse"]);

                self.vars.hasChanged = false;
            }

            self.vars.animationFrameToken = null;
            self.vars.animationFrameToken = startDrawing(self);
        });
    }

    /**
     * @param {*} self
     */
    function stopDrawing(self) {
        if (self.vars.animationFrameToken) {
            shmi.caf(self.vars.animationFrameToken);
            self.vars.animationFrameToken = null;
        }
    }

    /**
     * @param {*} self
     * @param {number} value
     */
    function startAnimation(self, value, animationDuration = 250) {
        if (animationDuration > 0) {
            self.vars.animationValueStep = (value - self.vars.animationValue) / animationDuration;
        } else {
            self.vars.animationValueStep = (value - self.vars.animationValue);
        }
    }

    /**
     * Adapts a DOM event listener to the visuals listener interface.
     *
     * @param {Element} domElement Element to attach the event listener to.
     * @param {string} eventName Name of the event to listen for.
     * @param {function} eventHandler Event handler function.
     *
     * @returns {{enable: function, disable: function}}
     */
    function makeDomEventListener(domElement, eventName, eventHandler) {
        return {
            enable: domElement.addEventListener.bind(domElement, eventName, eventHandler),
            disable: domElement.removeEventListener.bind(domElement, eventName, eventHandler)
        };
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: null,
            valueFormatted: null,
            active: false,
            hasChanged: false,
            forceRedraw: false,
            cache: {},
            valueSettings: null,

            // DOM Elements
            canvasEl: null,
            styleHolderEl: null,
            iconEl: null,

            canvasObj: null,
            intersectionObserver: null,
            mutationObserver: null,
            elementIsInViewport: null,
            unit: null,
            unitLocalized: null,
            label: null,
            labelLocalized: null,
            animationFrameToken: null,
            animationValue: 0,
            animationValueStep: null,
            animationTimestamp: null,

            // Default styles if not overridden by instance config
            // Shadow blur can only be overridden via CSS (see below for details)
            styles: {
                arcBackground: {
                    lineWidth: 5,
                    color: '#dddddd',
                    shadowBlur: 0
                },
                arcForeground: {
                    lineWidth: 5,
                    offsetVertical: 0.5,
                    offsetHorizontal: 0.5,
                    rotation: Math.PI / 2,
                    shadowBlur: 0,
                    shadowColor: 'rgba(180, 200, 59, 1)',
                    color: 'rgba(180, 200, 59, 1)'
                },
                slice: {
                    width: (360 - 84) / 360
                },
                valueLine: {
                    lineWidth: 1,
                    shadowBlur: 0,
                    shadowColor: 'rgba(60, 60, 60, 0.1)',
                    color: 'rgba(60, 60, 60, 0.1)'
                },
                value: {
                    offsetVertical: 0.5,
                    offsetHorizontal: 0.5,
                    fontSize: 36,
                    fontFamily: "RobotoBold",
                    color: '#7c7c7c'
                },
                label: {
                    offsetVertical: 0.95,
                    offsetHorizontal: 0.5,
                    fontSize: 14,
                    fontFamily: "RobotoLight",
                    color: '#7c7c7c'
                },
                unit: {
                    offsetVertical: 0.32,
                    offsetHorizontal: 0.5,
                    fontSize: 18,
                    fontFamily: "RobotoLight",
                    color: '#7c7c7c'
                },
                icon: {
                    width: 1/6,
                    height: 1/6,
                    offsetVertical: 0.1,
                    offsetHorizontal: 0.5
                },
                iconClass: {
                    color: "black"
                }
            }
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues"
        },

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                const self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.canvasEl = shmi.getUiElement('gauge-canvas', self.element);

                if (!self.vars.canvasEl) {
                    self.log('Canvas element is missing!', 3);
                    return;
                }

                // Read style config from hidden HTML element
                self.vars.styleHolderEl = shmi.getUiElement('style-holder', self.element);
                if (!self.vars.styleHolderEl) {
                    self.log('Style Holder element is missing!', 3);
                    return;
                }

                if (!self.config['show-icon']) {
                    // Nothing to do.
                } else if (self.config['icon-src']) {
                    const iconEl = document.createElement('img');
                    iconEl.src = self.config['icon-src'];
                    self.vars.iconEl = iconEl;
                    self.vars.listeners.push(makeDomEventListener(iconEl, "load", () => {
                        invalidateIconData(self);
                        self.vars.forceRedraw = true;
                    }));
                    self.vars.styleHolderEl.appendChild(iconEl);
                } else if (self.config['icon-class']) {
                    const iconEl = document.createElement('div');
                    ["icon"].concat(
                        self.config['icon-class'].split(" ")
                    ).forEach(
                        (name) => shmi.addClass(iconEl, name)
                    );
                    self.vars.iconEl = iconEl;
                    self.vars.styleHolderEl.appendChild(iconEl);
                }

                // Read styles from CSS and store for drawing.
                self.setupStylesFromCss();

                self.vars.canvasObj = self.vars.canvasEl.getContext('2d');

                self.vars.intersectionObserver = new IntersectionObserver((entries) => {
                    if (entries.length > 1) {
                        entries.sort((lhs, rhs) => rhs.time - lhs.time);
                    }
                    self.vars.elementIsInViewport = entries[0].isIntersecting;
                });

                self.vars.mutationObserver = new MutationObserver((records) => invalidateStyles(self));

                self.imports.nv.initValueSettings(self);

                // Unit
                if (typeof self.config['unit-text'] === "string") {
                    self.vars.unit = self.config['unit-text'];
                    self.vars.unitLocalized = shmi.localize(self.config['unit-text']);
                } else {
                    self.vars.unitLocalized = '';
                }

                // Label
                if (typeof self.config.label === "string") {
                    self.vars.label = self.config.label;
                    self.vars.labelLocalized = shmi.localize(self.config.label);
                } else {
                    self.vars.labelLocalized = '';
                }

                // Initialize value with default
                self.vars.valueFormatted = shmi.localize(self.config["default-value"]);

                // Resize canvas to fit widget
                updateCanvasResolution(self);

                // Paint empty gauge
                const arcAngle = 2 * Math.PI * (1 - self.vars.styles.slice.width);
                drawGauge(self, self.vars.canvasObj, 0, self.vars.canvasEl.width, self.vars.canvasEl.height, self.vars.styles.arcForeground.rotation, arcAngle, window.devicePixelRatio || 1);
            },

            /* called when control is enabled */
            onEnable: function() {
                const self = this;

                if (!isReady(self)) {
                    return;
                }

                self.vars.intersectionObserver.observe(self.vars.canvasEl);
                self.vars.mutationObserver.observe(self.element, { attributes: true, attributeFilter: ['style', 'class'] });

                // Register resize handler. Workaround for detecting style
                // changes. Clears cache on every `resize` event, causing
                // styles to be reevaluated.
                self.vars.listeners.push(makeDomEventListener(window, "resize", () => {
                    // Clearing the cache causes `haveDisplayPropertiesChanged` to
                    // return `true`, which in turn causes the CSS to be reevaluated.
                    self.vars.cache = null;
                }));

                checkCache(self);
                startDrawing(self);

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                self.vars.listeners.forEach((l) => l.enable());

                self.log('Enabled', 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                const self = this;

                if (!isReady(self)) {
                    return;
                }

                self.vars.listeners.forEach((l) => l.disable());

                stopDrawing(self);
                self.vars.intersectionObserver.disconnect(self.vars.canvasEl);
                self.vars.mutationObserver.disconnect(self.element);

                if (self.config.item) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                self.log('Disabled', 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                const self = this;

                if (!isReady(self)) {
                    return;
                }

                shmi.addClass(this.element, 'locked');

                self.log('Locked', 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                const self = this;

                if (!isReady(self)) {
                    return;
                }

                shmi.removeClass(self.element, 'locked');

                self.log('Unlocked', 1);
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                const self = this;

                self.vars.value = value;
                self.vars.valueFormatted = self.imports.nv.formatOutput(self.vars.value, self);

                invalidateTextMeasurements(self);
                startAnimation(self, value, parseInt(self.config["animation-duration"]) || 0);

                self.vars.hasChanged = true;
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                const self = this;

                return self.vars.value;
            },

            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(min, max, step, name, type, warnmin, warnmax, prewarnmin, prewarnmax, precision) {
                const self = this;

                self.imports.nv.setProperties(self, arguments);

                self.vars.min = min;
                self.vars.max = max;

                invalidateTextMeasurements(self);
                self.vars.hasChanged = true;
            },

            setUnitText: function(unitText) {
                const self = this;
                if (self.config['auto-unit-text']) {
                    self.vars.unit = unitText;
                    self.vars.unitLocalized = shmi.localize(unitText);
                }
            },

            setLabel: function(labelText) {
                const self = this;
                if (self.config['auto-label']) {
                    self.vars.label = labelText;
                    self.vars.labelLocalized = shmi.localize(labelText);
                }
            },

            log: function(msg, level) {
                shmi.log('[IQ:iq-radial-gauge] ' + msg, level);
            },

            /**
             * Reads instance styles from CSS as computed style and sets them in this object in self.vars.styles...
             */
            setupStylesFromCss: function() {
                const self = this;

                cssStyleMappings.forEach(function(mapping) {
                    const readEl = shmi.getUiElement(mapping["data-ui"], self.vars.styleHolderEl);
                    if (readEl) {
                        const styleData = getComputedStyle(readEl);
                        if (styleData) {
                            mapping.styleConfig.forEach(function(styleConfig) {
                                setStyleFromCss(self, styleData, styleConfig.propName, mapping.objName, styleConfig.attribName, styleConfig.conversion);
                            });
                        }
                    }
                });

                if (!self.vars.cache) {
                    self.vars.cache = {};
                }
                self.vars.cache.hasStyles = true;
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * WebIQ visuals iq-recipe-list control.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-recipe-list",
 *     "name": null,
 *     "template": "default/iq-recipe-list",
 *     "recipe-template-id": null
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * show-results {string}: When to display the results dropdown.
 *  * `never` Never show results.
 *  * `always` Always show results if there are more than 0.
 *  * `two-or-more` Show results if there are two or more.
 *
 * @version 1.0
 */
(function() {
    'use strict';

    // variables for reference in control definition
    const className = "iq-recipe-list", // control name in camel-case
        uiType = "iq-recipe-list", // control keyword (data-ui)
        isContainer = true;

    // example - default configuration
    const defConfig = {
        "class-name": className,
        "name": null,
        "template": "default/iq-recipe-list",
        "label": uiType,
        "recipe-template-id": null
    };

    // setup module-logger
    const ENABLE_LOGGING = true,
        RECORD_LOG = false;
    const logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    const fLog = logger.fLog,
        log = logger.log;

    // declare private functions - START
    function getRandomId() {
        return Math.random().toString(36).substr(2, 9);
    }

    function eventProxy(self, event) {
        self.fire(event.type, event.detail);
    }

    function doSetValue(val, self) {
        const selObj = {
            selRowIndex: [],
            selRows: [],
            type: -1
        };

        for (let i = 0; i < self.vars.dataGrid.getRowCount(); i++) {
            const rowData = self.vars.dataGrid.getRowData(i);
            if (rowData[0].value === parseInt(val)) {
                selObj.selRowIndex = [rowData[0].value];
                selObj.selRows = [i];
                selObj.type = 2;
            }
        }
        self.vars.ct.setSelectedRows(selObj);
    }

    function checkIfSetValue(val, self) {
        if (self.vars.dataGrid && self.vars.ct) {
            if (self.vars.dataGrid.tasksRunning > 0) {
                const tok = (shmi.listen("datagrid-ready", function() {
                    tok.unlisten();
                    doSetValue(val, self);
                }.bind(val, self)));
            } else {
                doSetValue(val, self);
            }
        }
    }

    // declare private functions - END

    // definition of new control extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            domChild: null,
            dataGridId: null,
            dataGrid: null,
            dataGridSubscriptionId: null,
            ct: null,
            eventProxy: null,
            tokens: [],
            recipeTableConfig: {
                "label": "${recipe-list.title}",
                "table": "",
                "name": "",
                "class-name": "complex-table2",
                "field-datagrid-col-map": {
                    "id": 0,
                    "name": 1
                },
                "select-mode": "SINGLE",
                "default-field-control-map": {
                    "id": {
                        "ui-type": "iq-label",
                        "config": {
                            "class-name": "iq-label",
                            "template": "default/iq-label.iq-variant-01",
                            "options": [],
                            "pattern": "<%= VALUE %>",
                            "value-as-tooltip": true
                        }
                    },
                    "name": {
                        "ui-type": "iq-label",
                        "config": {
                            "class-name": "iq-label",
                            "template": "default/iq-label.iq-variant-01",
                            "options": [],
                            "pattern": null,
                            "value-as-tooltip": true
                        }
                    }
                },
                "default-field-headers": {
                    "id": "${recipe-list.table-header.id}",
                    "name": "${recipe-list.table-header.name}"
                },
                "_comment": "expr is passed to expr parameter array of shmi.visuals.core.DataGridManager.setFilter",
                "filters": [],
                "default-layout": {
                    "class-name": "layout-std",
                    "_comment": "default == no additional css layout class",
                    "column-org": {
                        "col1": {
                            "fields": ["id"],
                            "column-width": "15%"
                        },
                        "col2": {
                            "fields": ["name"],
                            "column-width": "85%"
                        }
                    }
                },
                "sortable-fields": [
                    "name",
                    "id"
                ],
                "quicksearch": {
                    "enable": true,
                    "remember": true,
                    "fields": ["name"]
                },
                "text-mode": "SINGLELINE",
                "show-nof-rows": true,
                "v-scroll-options": ["SCROLLBAR", "V_SWIPE"]
            }
        },
        /* imports added at runtime */
        imports: {
            dgm: "visuals.session.DataGridManager"
        },

        /* array of custom event types fired by this control */
        events: [
            "select"
        ],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                if (typeof (this.config["recipe-template-id"]) !== "number") {
                    this.element.textContent = "Invalid or no recipe template id given. This control will not work. Please select a valid recipe template id.";
                    fLog("Invalid or no recipe template id given:", this.config["recipe-template-id"]);
                    return;
                }

                this.vars.domChild = shmi.getUiElement("table-container", this.element);
                if (this.vars.domChild === null) {
                    this.element.textContent = "One or more required DOM elements are missing from the template. This control will not work. Please select a compatible template.";
                    fLog("Some DOM elements are missing from the template.");
                    return;
                }

                do {
                    this.vars.dataGridId = className + "-" + getRandomId();
                } while (this.imports.dgm.getGrid(this.vars.dataGridId) !== null);

                this.imports.dgm.grids[this.vars.dataGridId] = this.vars.dataGrid = new shmi.visuals.core.DataGridRecipe(this.vars.dataGridId, this.config["recipe-template-id"]);
                this.vars.dataGrid.init();
                this.vars.dataGrid.sort(1, "ASC");
            },
            rebuildTable: function() {
                const tableConfig = shmi.cloneObject(this.vars.recipeTableConfig);

                tableConfig.table = this.vars.dataGridId;
                tableConfig.name = this.vars.dataGridId;
                this.vars.dataGrid.sort(1, "ASC");

                if (this.vars.eventProxy) {
                    this.vars.eventProxy.unlisten();
                    this.vars.eventProxy = null;
                }

                if (this.vars.ct) shmi.deleteControl(this.vars.ct);

                this.vars.ct = shmi.createControl("complex-table2", this.vars.domChild, tableConfig);
                if (this.vars.ct === null) {
                    this.element.textContent = "Unable to create table control for recipes. This control will not work.";
                    fLog("Unable to create table control for recipes.");
                } else {
                    this.vars.eventProxy = this.vars.ct.listen("select", eventProxy.bind(null, this));
                }
            },
            /**
             * Returns the contents of the table selection
             *
             * @returns {object} selected recipe info
             */
            getValue: function() {
                if (this.vars.ct) {
                    const selection = this.vars.ct.getSelectionRowData(this.vars.ct.getSelectedRows());
                    if (selection.length > 0) {
                        return {
                            name: selection[0][1].value,
                            recipe_id: isNaN(selection[0][0].value) ? null : selection[0][0].value,
                            template_id: this.config["recipe-template-id"],
                            values_set: (selection[0][3].value === 1)
                        };
                    }
                }
                return null;
            },
            /**
             * Sets the value of the table selection.
             *
             * @param {string} val Value to set
             */
            setValue: function(val) {
                const self = this;
                checkIfSetValue(val, self);
            },
            refreshGrid: function() {
                if (this.vars.dataGrid) {
                    this.vars.dataGrid.refresh();
                }
            },
            /* called when control is enabled */
            onEnable: function() {
                if (this.vars.dataGrid) {
                    this.rebuildTable();
                }
            },
            /* called when control is disabled */
            onDisable: function() {
                if (this.vars.ct) {
                    this.vars.ct.disable();
                }

                if (this.vars.eventProxy) {
                    this.vars.eventProxy.unlisten();
                    this.vars.eventProxy = null;
                }
            },
            onDelete: function() {
                if (this.vars.ct) {
                    shmi.deleteControl(this.vars.ct);
                }

                if (this.vars.dataGrid) {
                    delete this.imports.dgm.grids[this.vars.dataGridId];
                }
                this.vars.tokens.forEach((tok) => {
                    tok.unlisten();
                });
            },
            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                if (this.vars.ct) {
                    this.vars.ct.lock();
                }
            },
            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                if (this.vars.ct) {
                    this.vars.ct.unlock();
                }
            },
            getSelectedRecipe: function() {
                if (this.vars.ct) {
                    return this.vars.ct.getSelectionRowData(this.vars.ct.getSelectedRows());
                }
                return null;
            },
            getTemplate: function() {
                return this.config["recipe-template-id"];
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    const cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * IQ Responsive Menu
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-responsive-menu",
 *     "name": null,
 *     "template": "default/iq-responsive-menu"
 * }
 *
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 *
 * @version 0.2
 */
(function() {
    'use strict';

    //variables for reference in control definition
    const className = "IqResponsiveMenu", //control name in camel-case
        uiType = "iq-responsive-menu", //control keyword (data-ui)
        isContainer = false;

    //example - default configuration
    const defConfig = {
        "class-name": uiType,
        "name": null,
        "template": `default/${uiType}`,
        "menu": [],
        "mobile-layouts": [],
        "overlay-style": {
            "background": "transparent",
            "zIndex": 1
        }
    };

    /* Example for config.menu structure */
    /**
    const exampleMenu = [
        {
            "label": "My Category",
            "items": [
                {
                    "label": "My Sub-Category",
                    "items": [
                        {
                            "label": "Menu Entry",
                            "action": [
                                {
                                    "name": "test-action",
                                    "params": [1, 2, 3]
                                }
                            ],
                            "icon-src": "pics/system/icons/debug_icon.png" // image icon
                        },
                        {
                            "label": "Menu Entry",
                            "action": [
                                {
                                    "name": "test-action",
                                    "params": [1, 2, 3]
                                }
                            ],
                            "icon-class": "icon icon-add" // icon font class
                        }
                    ]
                },
                {
                    "label": "Menu Entry",
                    "action": [
                        {
                            "name": "test-action",
                            "params": [1, 2, 3]
                        },
                        {
                            "name": "test-action",
                            "params": [4, 5, 6]
                        }
                    ]
                }
            ]
        },
        {
            "label": "Menu Entry",
            "action": [
                {
                    "name": "test-action",
                    "params": [1, 2, 3]
                }
            ]
        },
        {
            "label": "Menu Entry",
            "action": [
                {
                    "name": "test-action",
                    "params": [4, 5, 6]
                }
            ]
        }
    ];
    */

    //setup module-logger
    const ENABLE_LOGGING = true,
        RECORD_LOG = false;
    const logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    const fLog = logger.fLog,
        log = logger.log;

    const DEFAULT_ICON = "pics/system/icons/placeholder.svg";

    //declare private functions - START
    /**
     * Creates a compound listener acting both as touch- and mouse-listener.
     * Creates a touch- and mouse-listener internally and enables/disables them
     * both at the same time.
     *
     * @param {HTMLElement} element Element to attach the listener to.
     * @param {object} handlers Object specifying the corresponding handlers
     *  called by the touch- and mouse-listeners.
     * @returns {object} Object with a listener-like interface.
     */
    function createListener(element, handlers) {
        const io = shmi.requires("visuals.io"),
            ml = new io.MouseListener(element, handlers),
            tl = new io.TouchListener(element, handlers);
        return {
            enable: function() {
                ml.enable();
                tl.enable();
            },
            disable: function() {
                ml.disable();
                tl.disable();
            }
        };
    }

    /**
     * Creates a new html element with css classes and data-ui set.
     *
     * @param {string|string[]} cssClasses CSS class or classes to assign to
     *  the new element.
     * @param {?string} setUiType ui type to set.
     * @param {string} [elementType] Type of the html element. Defaults to
     *  "DIV".
     * @returns {HTMLElement}
     */
    function makeElement(cssClasses, setUiType, elementType = "DIV") {
        const elem = document.createElement(elementType || "DIV");
        if (typeof cssClasses === "string") {
            shmi.addClass(elem, cssClasses);
        } else if (Array.isArray(cssClasses)) {
            cssClasses.forEach(function(cc) {
                if (typeof cc === "string") {
                    shmi.addClass(elem, cc);
                }
            });
        }
        if (typeof setUiType === "string") {
            elem.setAttribute("data-ui", setUiType);
        }
        return elem;
    }

    /**
     * Marks the given layer as "selected". Successive layers are unmarked as
     * "selected".
     *
     * @param {*} self Reference to the widget.
     * @param {?HTMLElement} element Element to assign the "selected" CSS class
     *  to. If set to `null`, in addition to the successive layers being
     *  unmarked as "selected", the given one is unmarked as well.
     * @param {number} layer Layer index
     */
    function setSelected(self, element, layer) {
        if (self.vars.selected[layer] !== element) {
            if (self.vars.selected[layer] !== null) {
                shmi.removeClass(self.vars.selected[layer], "selected");
            }
            self.vars.selected[layer] = element;
            if (element !== null) {
                shmi.addClass(element, "selected");
            }
        }

        self.vars.selected.forEach(function(e, i) {
            if (e && (i > layer)) {
                shmi.removeClass(e, "selected");
                self.vars.selected[i] = null;
            }
        });
    }

    /**
     * Creates a menu section according to the provided configuration. Child
     * elements are created recursively.
     *
     * @param {*} self Reference to the widget.
     * @param {object} sectionConfig Configuration of the menu section.
     * @param {number} [layer] Number of the layer the section is located on.
     * @returns {object} Descriptor of the new menu section.
     */
    function makeMenuSection(self, sectionConfig, layer = 0) {
        const baseElement = makeElement("menu-group menu-element", "menu-group", "LI"),
            groupData = makeElement("group-data"),
            iconWrapper = makeElement("icon", "icon"),
            img = makeElement(null, null, "IMG"),
            label = makeElement("menu-group-label", "menu-group-label"),
            triangle = makeElement("triangle"),
            childList = makeElement(`menu-layer-${layer + 1} menu-layer-${layer}-${layer + 1} menu-layer-${layer + 1}-${layer + 2} accent-border`, `menu-layer-${layer + 1}`, "UL");

        const children = sectionConfig.items.map((child) => makeMenuElement(self, child, layer + 1)).filter((child) => !!child);

        // Piece together DOM elements
        iconWrapper.appendChild(img);
        groupData.appendChild(iconWrapper);
        groupData.appendChild(label);
        groupData.appendChild(triangle);

        children.forEach(({ elements: { base } }) => childList.appendChild(base));

        baseElement.appendChild(groupData);
        baseElement.appendChild(childList);

        const listener = createListener(groupData, {
            onClick: function onClick() {
                const crumb = self.vars.crumbs[`layer${layer}`];
                if (shmi.hasClass(baseElement, "selected")) {
                    setLayer(self, layer);
                    setSelected(self, null, layer);
                    if (layer === 0) {
                        removeOverlay(self);
                    }
                } else {
                    setLayer(self, layer + 1);
                    setSelected(self, baseElement, layer);
                    crumb.textContent = shmi.localize(sectionConfig.label);
                    if (layer === 0) {
                        createOverlay(self);
                    }
                }
            }
        });

        // Setup DOM elements
        label.textContent = shmi.localize(sectionConfig.label);

        if (sectionConfig["icon-src"]) {
            img.setAttribute("src", sectionConfig["icon-src"]);
        } else {
            img.setAttribute("src", DEFAULT_ICON);
        }

        if (sectionConfig["icon-mode"] === "no-icon") {
            baseElement.classList.add("no-icon");
        } else if (sectionConfig["icon-mode"] === "hidden") {
            baseElement.classList.add("hidden-icon");
        }

        if (layer === 0) {
            baseElement.classList.add("accent-border");
        }

        return {
            type: "section",
            config: {
                label: sectionConfig["label"] || null,
                iconSrc: sectionConfig["icon-src"] || null,
                iconMode: sectionConfig["icon-mode"] || "show",
                access: shmi.cloneObject(sectionConfig.access)
            },
            listeners: [listener],
            elements: {
                base: baseElement,
                iconImage: img,
                label: label,
                childList: childList
            },
            children: children
        };
    }

    /**
     * Creates a menu entry according to the provided configuration.
     *
     * @param {*} self Reference to the widget.
     * @param {object} entryConfig Configuration of the menu entry.
     * @param {number} [layer] Number of the layer the entry is located on.
     * @returns {object} Descriptor of the new menu entry.
     */
    function makeMenuEntry(self, entryConfig, layer = 0) {
        const baseElement = makeElement("menu-item menu-element", "menu-item", "LI"),
            iconWrapper = makeElement("icon", "icon"),
            img = makeElement(null, null, "IMG"),
            label = makeElement("menu-item-label", "menu-item-label"),
            { UiAction } = shmi.requires("visuals.core"),
            action = entryConfig.action ? new UiAction(entryConfig.action) : null;

        // Piece together DOM elements
        iconWrapper.appendChild(img);
        baseElement.appendChild(iconWrapper);
        baseElement.appendChild(label);

        // Setup DOM elements
        label.textContent = shmi.localize(entryConfig.label);
        img.setAttribute("src", entryConfig["icon-src"] || DEFAULT_ICON);

        if (entryConfig["icon-mode"] === "no-icon") {
            baseElement.classList.add("no-icon");
        } else if (entryConfig["icon-mode"] === "hidden") {
            baseElement.classList.add("hidden-icon");
        }

        if (layer === 0) {
            baseElement.classList.add("accent-border");
        }

        const listener = createListener(baseElement, {
            onClick: function onClick() {
                setSelected(self, baseElement, layer);
                setLayer(self, -1);
                removeOverlay(self);
                if (action) {
                    action.execute(self);
                }
            }
        });

        return {
            type: "entry",
            config: {
                label: entryConfig["label"] || null,
                iconSrc: entryConfig["icon-src"] || null,
                iconMode: entryConfig["icon-mode"] || "show",
                access: shmi.cloneObject(entryConfig.access),
                action: shmi.cloneObject(entryConfig.action)
            },
            listeners: [listener],
            elements: {
                base: baseElement,
                iconImage: img,
                label: label
            }
        };
    }

    /**
     * Creates a menu separator according to the provided configuration.
     *
     * @param {*} self Reference to the widget.
     * @param {object} separatorConfig Configuration of the menu separator.
     * @param {number} [layer] Number of the layer the separator is located on.
     * @returns {object} Descriptor of the new menu separator.
     */
    function makeMenuSeparator(self, separatorConfig, layer = 0) {
        const baseElement = makeElement("menu-item separator", "menu-item", "LI"),
            label = makeElement("menu-item-label", "menu-item-label");

        // Piece together DOM elements
        baseElement.appendChild(label);

        // Setup DOM elements
        label.textContent = shmi.localize(separatorConfig.label);

        if (separatorConfig["show-stroke"]) {
            baseElement.classList.add("show-stroke");
        }

        return {
            type: "separator",
            config: {
                label: separatorConfig["label"] || null,
                showStroke: separatorConfig["show-stroke"] || false
            },
            listeners: [],
            elements: {
                base: baseElement,
                label: label
            }
        };
    }

    /**
     * Creates a menu element based on the element type in the provided
     * configuration. If the configuration doesn't include a type, elements
     * with children are assumed to be sections while everything else is
     * assumed to be entries.
     *
     * @param {*} self Reference to the widget.
     * @param {object} config Configuration of the menu element.
     * @param {number} layer Number of the layer the menu element is located
     *  on.
     * @returns {object} Descriptor of the new menu element.
     */
    function makeMenuElement(self, config, layer) {
        const elementType = config.type || (Array.isArray(config.items) ? "group" : "item");

        switch (elementType) {
        case "group":
            if (layer > 1) {
                console.warn(`[${className}] Creating sections for layers >1 not supported.`);
            }

            return makeMenuSection(self, config, layer);

        case "item":
            return makeMenuEntry(self, config, layer);

        case "separator":
            return makeMenuSeparator(self, config, layer);

        default:
            console.warn(`[${className}] Unknown type for menu element ${elementType}`);
            return null;
        }
    }

    /**
     * Creates the menu root.
     *
     * @param {*} self Reference to the widget.
     * @param {object[]} menu Menu entries for layer 0
     * @returns {object} Descriptor of the new menu.
     */
    function makeMenu(self, menu) {
        const menuBase = shmi.getUiElement("menu-layer-0", self.element),
            burgerButton = shmi.getUiElement("burger-button", self.element);

        const children = menu.map((menuElement) => makeMenuElement(self, menuElement, 0)).filter((element) => !!element);

        // Piece together DOM elements
        children.forEach(({ elements: { base } }) => menuBase.appendChild(base));

        // Setup DOM elements
        const listener = createListener(burgerButton, {
            onClick: function onClick() {
                if (self.vars.activeLayer >= 0) {
                    setLayer(self, -1);
                    removeOverlay(self);
                } else {
                    setLayer(self, 0);
                    createOverlay(self);
                }
            }
        });

        return {
            type: "menu",
            config: {},
            listeners: [listener],
            children: children
        };
    }

    /**
     * Returns an array of all listeners of a menu.
     *
     * @param {object} menu Descriptor of a menu or menu element.
     * @returns {object[]} Array of listeners.
     */
    function collectListeners(menu) {
        if (!Array.isArray(menu.children)) {
            return menu.listeners || [];
        }

        return menu.children.reduce((combined, current) => combined.concat(collectListeners(current)), menu.listeners);
    }

    /**
     * Returns an array of all access condition data of a menu.
     *
     * @param {*} menu Descriptor of a menu or menu element.
     * @returns {object[]} Array of access condition data.
     */
    function collectAccessData(menu) {
        const accessData = [];

        if (menu.config.access && menu.config.access.condition && ["hide", "lock"].includes(menu.config.access.type)) {
            accessData.push({
                base: menu.elements.base,
                type: menu.config.access.type,
                condition: menu.config.access.condition,
                listeners: menu.listeners.slice(),
                menu
            });
        }

        if (!Array.isArray(menu.children)) {
            return accessData;
        }

        return menu.children.reduce((combined, current) => combined.concat(collectAccessData(current)), accessData);
    }

    /**
     * Sets the layer-active css class for the corresponding layer.
     * If `layer` is set to -1, all selections are cleared.
     *
     * @param {*} self Reference to the widget.
     * @param {number} layer Layer index
     */
    function setLayer(self, layer) {
        if (layer !== self.vars.activeLayer) {
            if (self.vars.activeLayer >= 0) {
                shmi.removeClass(self.element, `layer-${self.vars.activeLayer}-active`);
            }
            self.vars.activeLayer = layer;
            if (layer >= 0) {
                shmi.addClass(self.element, `layer-${self.vars.activeLayer}-active`);
            }
        }

        if (layer === -1) {
            self.vars.selected.forEach(function(e, idx) {
                if (e !== null) {
                    shmi.removeClass(e, "selected");
                    self.vars.selected[idx] = null;
                }
            });
        }
    }

    /**
     * Returns a condition callback for the given access data reference.
     *
     * @param {HTMLElement} param0.base HTML element to add/remove the css
     *  modifiers to/from.
     * @param {object[]} param0.listeners Listeners to enable/disable.
     * @param {string} cssClass CSS modifier to add/remove.
     * @returns {function}
     */
    function makeAccessConditionCallback({ base, listeners }, cssClass) {
        return (isActive) => {
            if (!isActive) {
                shmi.addClass(base, cssClass);
                listeners.forEach((l) => l.disable());
            } else {
                shmi.removeClass(base, cssClass);
                listeners.forEach((l) => l.enable());
            }
        };
    }

    function createOverlay(self) {
        if (self.vars.overlay) {
            removeOverlay(self);
        }

        const overlay = {
                element: document.createElement("DIV"),
                listener: null
            },
            handler = { onClick: null },
            elem = overlay.element,
            s = elem.style;
        s.position = "fixed";
        s.top = 0;
        s.bottom = 0;
        s.left = 0;
        s.right = 0;
        s.background = self.config["overlay-style"].background;
        s.zIndex = self.config["overlay-style"].zIndex;

        document.body.insertBefore(overlay.element, document.body.firstChild);

        handler.onClick = function() {
            if (self.vars.activeLayer >= 0) {
                setLayer(self, -1);
                removeOverlay(self);
            }
        };
        overlay.listener = createListener(overlay.element, handler);
        overlay.listener.enable();
        self.vars.overlay = overlay;
    }

    function removeOverlay(self) {
        if (self.vars.overlay) {
            self.vars.overlay.element.parentNode.removeChild(self.vars.overlay.element);
            self.vars.overlay.listener.disable();
            self.vars.overlay = null;
        }
    }

    //declare private functions - END

    //definition of new control extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            listeners: [],
            activeLayer: -1,
            mobile: false,
            selected: [
                null, /* layer 0 */
                null, /* layer 1 */
                null /* layer 2 */
            ],
            crumbs: {
                layer0: null,
                layer1: null
            },
            tokens: [],
            overlay: null
        },
        /* imports added at runtime */
        imports: {
        },
        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                const self = this;

                self.vars.crumbs.layer0 = shmi.getUiElement("selected-entry-layer-0", self.element);
                self.vars.crumbs.layer1 = shmi.getUiElement("selected-entry-layer-1", self.element);
                self.vars.menu = makeMenu(self, self.config.menu);

                shmi.removeClass(self.element, "mobile");
            },
            /* called when control is enabled */
            onEnable: function() {
                const self = this,
                    { ConditionObserver } = shmi.requires("visuals.tools.conditions"),
                    currentLayout = shmi.getCurrentLayout();

                if (self.config["mobile-layouts"].indexOf(currentLayout) !== -1) {
                    shmi.addClass(self.element, "mobile");
                    self.vars.mobile = true;
                } else {
                    shmi.removeClass(self.element, "mobile");
                    self.vars.mobile = false;
                }

                const layoutToken = shmi.listen("layout-change", function(evt) {
                    if (self.config["mobile-layouts"].indexOf(evt.detail.layout) !== -1) {
                        shmi.addClass(self.element, "mobile");
                        if (!self.vars.mobile) {
                            setLayer(self, -1);
                        }
                        self.vars.mobile = true;
                    } else {
                        shmi.removeClass(self.element, "mobile");
                        if (self.vars.mobile) {
                            setLayer(self, 0);
                        }
                        self.vars.mobile = false;
                    }
                    removeOverlay(self);
                });
                self.vars.tokens.push(layoutToken);

                if (!self.vars.mobile) {
                    setLayer(self, 0);
                }

                self.vars.listeners = collectListeners(self.vars.menu);
                self.vars.listeners.forEach((l) => l.enable());
                self.vars.tokens.push(
                    ...collectAccessData(self.vars.menu).map((reference) => {
                        switch (reference.type) {
                        case "hide":
                            return new ConditionObserver(reference.condition, makeAccessConditionCallback(reference, "hidden"));
                        case "lock":
                            return new ConditionObserver(reference.condition, makeAccessConditionCallback(reference, "locked"));
                        default:
                            // Should never happen
                            throw new Error(`Invalid access type ${reference.type}`);
                        }
                    })
                );
            },
            /* called when control is disabled */
            onDisable: function() {
                const self = this;
                self.vars.tokens.forEach((t) => t.unlisten());
                self.vars.listeners.forEach((l) => l.disable());
                self.vars.tokens = [];
                self.vars.listeners = [];
                removeOverlay(self);
            },
            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                const self = this;

                shmi.addClass(self.element, "locked");
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });
            },
            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                const self = this;

                shmi.removeClass(self.element, "locked");
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });
            },
            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {

            },
            /** Sets min & max values and stepping of subscribed variable **/
            onSetProperties: function(min, max, step) {

            }
        }
    };

    //definition of new control extending BaseControl - END

    //generate control constructor & prototype using the control-generator tool
    const cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * WebIQ visuals searchbar control.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-searchbar",
 *     "name": null,
 *     "template": "default/iq-searchbar.iq-variant-01",
 *     "item": null,
 *     "datagrid": null,
 *     "search-column": null,
 *     "show-results": "two-or-more",
 *     "result-columns": null
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * item {string}: Item alias of an item to connect. Can be set to `null` if no
 *  item is required.
 * datagrid {string}: Name of a DataGrid to search in.
 * search-column {string}: Name of the column to search in.
 * show-results {string}: When to display the results dropdown.
 *  * `never` Never show results.
 *  * `always` Always show results if there are more than 0.
 *  * `two-or-more` Show results if there are two or more.
 * result-columns {string[]}: Array of DataGrid column names to display in
 *  the results dropdown. `null` or an empty array will cause all columns to
 *  be included.
 *
 * @version 1.0
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iqSearchbar", // control name in camel-case
        uiType = "iq-searchbar", // control keyword (data-ui)
        isContainer = false;

    // example - default configuration
    var defConfig = {
        "class-name": "iq-searchbar",
        "name": null,
        "template": "default/iq-searchbar.iq-variant-01",
        "label": uiType,
        "auto-label": true,
        "item": null,
        "show-icon": false,
        "icon-src": null,
        "icon-class": null,
        "tooltip": null,
        "datagrid": null,
        "search-column": null,
        "show-results": "two-or-more",
        "max-results": 100,
        "max-results-hint": false,
        "result-columns": null,
        "select-contents-on-focus": true,
        "partial-word-search": false
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    // declare private functions - START

    function getMatchData(self) {
        var gData = self.vars.dataGridData || [],
            hintIdx = self.vars.currentHintIndex || 0;

        return gData[hintIdx] || null;
    }

    /**
     * Checks if all dom elements could be found.
     *
     * @param {object} dom `this.vars.dom`
     * @returns {boolean} `true` if the object did not contain any `null`
     *  values, `false` else.
     */
    function verifyDomElements(dom) {
        var iterObj = shmi.requires("visuals.tools.iterate.iterateObject"),
            ok = true;

        iterObj(dom, function(val) {
            ok = ok && val !== null;
        });

        return ok;
    }

    /**
     * Adds an event listener to the given target and stores data required to
     * remove it again in the given category.
     *
     * @param {*} self Reference to a searchbar control.
     * @param {string} category Name of the category for which to register
     *  the event listener.
     * @param {EventTarget} target Target to add the event listener to.
     * @param {string} type Name of the event.
     * @param {function} listener
     */
    function addRegisteredEventListener(self, category, target, type, listener) {
        var listeners = self.vars.registeredEventListeners[category];

        // If the category doesn't exist, create it.
        if (!listeners) {
            self.vars.registeredEventListeners[category] = [];
            listeners = self.vars.registeredEventListeners[category];
        }

        target.addEventListener(type, listener);

        listeners.push({
            type: type,
            target: target,
            listener: listener
        });
    }

    /**
     * Removes all event listeners for a given category. If no category has
     * been given, event listeners for all categories are removed.
     *
     * @param {*} self Reference to a searchbar control.
     * @param {string} [category] Name of the category for which to remove all
     *  event listeners.
     */
    function removeRegisteredEventListeners(self, category) {
        var iterObj = shmi.requires("visuals.tools.iterate.iterateObject"),
            listeners = [];

        if (category) {
            listeners = self.vars.registeredEventListeners[category] || [];
            delete self.vars.registeredEventListeners[category];
        } else {
            iterObj(self.vars.registeredEventListeners, function(l) {
                listeners = listeners.concat(l || []);
            });
            self.vars.registeredEventListeners = {};
        }

        listeners.forEach(function(listener) {
            listener.target.removeEventListener(listener.type, listener.listener);
        });
    }

    /**
     * Removes the displayed hint.
     *
     * @param {*} self Reference to a searchbar control.
     */
    function clearHint(self) {
        self.vars.dom.suggestion.textContent = "";
        self.vars.currentHint = null;
    }

    /**
     * Sets a hint.
     *
     * @param {*} self Reference to a searchbar control
     * @param {?string} hint The hint
     * @param {boolean} [force] Show the hint text even if the user input field
     *  is empty.
     */
    function setHint(self, hint, force) {
        var val;

        if (!hint) {
            clearHint(self);
        } else {
            val = self.getValueDirect();

            if (val.length === 0 && !force) {
                clearHint(self);
            } else {
                if (!self.config["partial-word-search"]) {
                    self.vars.dom.suggestion.textContent = hint.substr(val.length);
                }
                self.vars.currentHint = hint;
            }
        }
    }

    /**
     * Set a hint at the given index in the current data grid data buffer.
     *
     * @param {*} self Reference to a searchbar control.
     * @param {number} idx Index of the hint to set.
     * @param {boolean} [force] Show the hint text even if the user input field
     *  is empty.
     */
    function setHintByIdx(self, idx, force) {
        if (self.vars.dataGridData.length === 0) {
            clearHint(self);

            return;
        }

        // Make sure the index is in [0, self.vars.dataGridData)
        while (idx < 0) {
            idx += self.vars.dataGridData.length;
        }
        idx = idx % self.vars.dataGridData.length;

        if (self.vars.currentHintIndex !== null) {
            shmi.removeClass(self.vars.dom.resultBox.firstChild.children[self.vars.currentHintIndex], "hint-current");
        }
        shmi.addClass(self.vars.dom.resultBox.firstChild.children[idx], "hint-current");

        setHint(self, self.vars.dataGridData[idx][self.config["search-column"]], force);
        self.vars.currentHintIndex = idx;
    }

    /**
     * Update the displayed hint. If new hints need to be fetched, this is done
     * asynchronously and the hint is cleared in the meantime.
     *
     * @param {*} self Reference to a searchbar control.
     * @param {string} [inputText] String to look (and display) up hints for.
     *  If no string is given, the current string of the input field is used
     *  instead.
     */
    function updateHint(self, inputText) {
        fetchHint(self, inputText || self.getValueDirect());
    }

    /**
     * Callback called when a finished processing a new hint.
     *
     * @param {*} self Reference to a searchbar control.
     * @param {?string} inputText String the looked up hint is meant for. If no
     *  string is given, the current string of the input field is used
     *  instead
     * @param {?string} hint The fetched hint. May be `null` if there is no
     *  hint, in which case `inputText` also isn't a valid input.
     */
    function onHint(self, inputText, hint) {
        inputText = inputText || self.getValueDirect();

        if (!hint) {
            if (inputText.length > 0) {
                shmi.addClass(self.element, "notfound");
            }
            clearHint(self);
            self.vars.lastHintSearchResult = null;
        } else {
            shmi.removeClass(self.element, "notfound");
            setHint(self, hint);
            self.vars.lastHintSearchResult = hint;
        }
    }

    /**
     * Fetches a new hint from the connected datagrid. If the given string is
     * guaranteed to yield in the same hint being found, the last search hit is
     * being used instead of actually querying the datagrid.
     *
     * @param {*} self Reference to a searchbar control.
     * @param {string} inputText String to look up a hint for.
     */
    function fetchHint(self, inputText) {
        var newValueIsMoreSpecific = self.vars.lastHintSearchToken && (inputText.indexOf(self.vars.lastHintSearchToken) === 0);
        var hintMatchesNewValue = self.vars.lastHintSearchResult && (self.vars.lastHintSearchResult.indexOf(inputText) === 0);

        // No need to update the hint
        if (newValueIsMoreSpecific && hintMatchesNewValue) {
            onHint(self, inputText, self.vars.lastHintSearchResult);
            self.vars.dataGridData = self.vars.dataGridData.filter((val) => val[self.config["search-column"]] && val[self.config["search-column"]].indexOf(inputText) !== -1);
            updateResultBox(self, self.vars.dataGridData);

            return;
        }

        clearHint(self);
        if (self.config["partial-word-search"]) {
            self.vars.dataGrid.setFilter(self.vars.dataGridSearchColumnIdx, [ "%"+ inputText + "%" ]);
        } else {
            self.vars.dataGrid.setFilter(self.vars.dataGridSearchColumnIdx, [ inputText + "%" ]);
        }
        self.vars.lastHintSearchToken = inputText;
    }

    /**
     * Removes all nodes from the result box.
     *
     * @param {*} self Reference to a searchbar control.
     */
    function clearResultBox(self) {
        removeRegisteredEventListeners(self, "result-box");
        while (self.vars.dom.resultBox.firstChild) {
            self.vars.dom.resultBox.removeChild(self.vars.dom.resultBox.firstChild);
        }
    }

    /**
     * Creates a new result table and attaches it to the result box.
     *
     * @param {*} self Reference to a searchbar control.
     * @param {object[]} data DataGrid data to display.
     */
    function updateResultBox(self, data) {
        var table = document.createElement("table"),
            resultBoxVisible = false,
            columns,
            mainColumnIdx;

        clearResultBox(self);

        if (self.config["show-results"] === "never") {
            // Control is set up to never show results.
            resultBoxVisible = false;
        } else if (self.config["show-results"] === "two-or-more" && data.length <= 1) {
            // Control is set up to only show the results if there are two or
            // more. We don't have enough so stop here.
            resultBoxVisible = false;
        } else if (data.length !== 0) {
            resultBoxVisible = true;
        }

        if (resultBoxVisible) {
            self.vars.dom.resultBox.style.display = "";
        } else {
            self.vars.dom.resultBox.style.display = "none";
            return; //nothing left to do here
        }

        columns = Object.keys(data[0]);
        if ((self.config["result-columns"] || []).length > 0) {
            columns = columns.filter(function(columnName) {
                return self.config["result-columns"].indexOf(columnName) !== -1;
            });
        }

        // See if there are columns left we can display. If not, stop here.
        if (columns.length === 0) {
            self.vars.dom.resultBox.innerHTML = "No columns to display. Please check <span style=\"font-weight: bold\">result-columns</span> in your control configuration.";
            fLog("No columns to display.", "dataGrid columns =", Object.keys(data[0]), "columns to display =", self.config["result-columns"]);
            return;
        }

        // Make sure to `search-column` is at position 0 in the columns array
        // (= it is displayed first).
        mainColumnIdx = columns.indexOf(self.config["search-column"]);
        if (mainColumnIdx > 0) {
            columns.unshift(columns[mainColumnIdx]);
            columns.splice(mainColumnIdx + 1, 1);
        }

        // Fill the table.
        data.forEach(function(row, idx) {
            var tr = document.createElement("tr");
            columns.forEach(function(columnName) {
                var cell = document.createElement("td");
                cell.textContent = row[columnName] || "";

                tr.appendChild(cell);
            });

            addRegisteredEventListener(self, "result-box", tr, "touchstart", (evt) => {
                evt.stopPropagation();
            });
            addRegisteredEventListener(self, "result-box", tr, "mousedown", function(ev) {
                self.setValue(row[self.config["search-column"]]);
                shmi.removeClass(self.element, "focused");
                self.vars.lastValue = self.getValueDirect();
                self.fire("change", { value: self.vars.lastValue });
            });

            addRegisteredEventListener(self, "result-box", tr, "mouseover", setHintByIdx.bind(null, self, idx, true));
            addRegisteredEventListener(self, "result-box", tr, "mouseleave", shmi.removeClass.bind(shmi, tr, "hint-current"));

            table.appendChild(tr);
        });

        if (self.config["max-results-hint"] && (data.length === self.config["max-results"]) && (self.vars.dataGridResults > data.length)) {
            const row = document.createElement("tr");
            row.classList.add("max-results-notice");
            columns.forEach((colName, idx) => {
                const cell = document.createElement("td");
                if (idx === 0) {
                    const noticeText = shmi.localize("${iqsearchbar.results.moreAvailable}");
                    cell.textContent = noticeText;
                    cell.setAttribute("title", noticeText);
                }

                row.appendChild(cell);
            });
            table.appendChild(row);
        }

        self.vars.dom.resultBox.appendChild(table);
    }

    /**
     * Scrolls the result box to the given element. Will not scroll, if the
     * element is already visible. After a scroll, the element will be top
     * aligned if the result box was scrolled up and down aligned if the result
     * box was scrolled down.
     *
     * @param {*} self Reference to a searchbar control.
     * @param {DOMElement} element Element to scroll the result box to.
     */
    function doResultBoxScroll(self, element) {
        var scrollTop = self.vars.dom.resultBox.scrollTop,
            scrollBottom = scrollTop + self.vars.dom.resultBox.offsetHeight;

        if (element.offsetTop < scrollTop) {
            element.scrollIntoView(true);
        } else if (element.offsetTop + element.offsetHeight > scrollBottom) {
            element.scrollIntoView(false);
        }
    }

    /**
     * Returns a datagrids data in a more convenient format.
     *
     * @param {DataGrid} dg Data grid to get data from.
     * @param {number} subID Subscription id.
     * @returns {object[]} Array of rows where each row is key-, value store.
     */
    function getDataGridDataHelper(dg, subID) {
        var fieldNames = dg.getFields();

        return dg.getCurrentIDs(subID).map(function(id) {
            var row = dg.getRowData(id),
                rowdata = {};

            if (!row) {
                return null;
            }

            row.forEach(function(cell, cellIdx) {
                rowdata[fieldNames[cellIdx]] = String(cell.value);
            });

            return rowdata;
        });
    }

    /**
     * Subscriber function attached to the searchbars datagrid.
     *
     * @param {*} self Reference to a searchbar control.
     * @param {object} dgData DataGrid update notification data.
     */
    function hintFetchCompleteCallback(self, dgData) {
        const inputText = self.getValueDirect(),
            rowData = getDataGridDataHelper(self.vars.dataGrid, self.vars.dataGridSubscriptionId.id);

        let hint;

        // No hint found?
        if (rowData.length > 0) {
            hint = (rowData[0] || {})[self.config["search-column"]];
        }

        self.vars.currentHintIndex = null;
        self.vars.dataGridData = rowData;
        self.vars.dataGridResults = dgData.totalRows;
        onHint(self, inputText, hint || null);
        updateResultBox(self, rowData);
    }

    /**
     * Set the current selection to the end of the given elements content.
     *
     * @param {DOMElement} element
     * @param {boolean} selectContent Selects the entire content instead of
     *  just the end.
     */
    function setSelected(element, selectContent) {
        // Setting textContent reset our cursor position. Set the
        // cursor to the end of the content.
        var range = document.createRange();
        range.selectNodeContents(element);
        if (!selectContent) {
            range.setStart(element, range.endOffset);
        }

        var sel = window.getSelection();
        sel.removeAllRanges();
        sel.addRange(range);
    }

    /**
     * Event listener callback attached to the user input field.
     *
     * @param {*} self Reference to a searchbar control.
     * @param {KeyboardEvent} ev
     */
    function userInputKeyDownHandler(self, ev) {
        if (ev.key === "Tab") {
            // On "Tab" use autocompletion if appropriate. Prevent the event
            // from propagating if autocompletion was used but don't otherwise.
            if (self.vars.currentHint && self.getValueDirect() !== self.vars.currentHint) {
                ev.preventDefault();

                self.setValue(self.vars.currentHint);
                clearHint(self);

                // Setting textContent reset our cursor position. Set the
                // cursor to the end of the content.
                setSelected(self.vars.dom.input);
            }
        } else if (ev.key === "Escape") {
            // On "Escape" toggle the result box.
            if (shmi.hasClass(self.element, "show-results")) {
                shmi.removeClass(self.element, "show-results");
            } else {
                shmi.addClass(self.element, "show-results");
            }
        } else if (ev.key === "ArrowUp") {
            // On "ArrowUp" scroll up by 1.
            if (self.vars.dom.resultBox.firstChild && self.vars.dom.resultBox.firstChild.children) {
                setHintByIdx(self, (self.vars.currentHintIndex || 0) - 1, true);
                doResultBoxScroll(self, self.vars.dom.resultBox.firstChild.children[self.vars.currentHintIndex]);
                ev.preventDefault();
            }
        } else if (ev.key === "PageUp") {
            // On "PageUp" scroll up by 10.
            if (self.vars.dom.resultBox.firstChild && self.vars.dom.resultBox.firstChild.children) {
                setHintByIdx(self, Math.max((self.vars.currentHintIndex || 0) - 10, 0), true);
                doResultBoxScroll(self, self.vars.dom.resultBox.firstChild.children[self.vars.currentHintIndex]);
                ev.preventDefault();
            }
        } else if (ev.key === "ArrowDown") {
            // On "ArrowDown" scroll down by 1.
            if (self.vars.dom.resultBox.firstChild && self.vars.dom.resultBox.firstChild.children) {
                setHintByIdx(self, (self.vars.currentHintIndex || 0) + 1, true);
                doResultBoxScroll(self, self.vars.dom.resultBox.firstChild.children[self.vars.currentHintIndex]);
                ev.preventDefault();
            }
        } else if (ev.key === "PageDown") {
            // On "PageDown" scroll down by 10.
            if (self.vars.dom.resultBox.firstChild && self.vars.dom.resultBox.firstChild.children) {
                setHintByIdx(self, Math.min((self.vars.currentHintIndex || 0) + 10, self.vars.dataGridData.length - 1), true);
                doResultBoxScroll(self, self.vars.dom.resultBox.firstChild.children[self.vars.currentHintIndex]);
                ev.preventDefault();
            }
        } else if (ev.key === "Enter") {
            // On "Return" or "Enter" remove focus and autocomplete.
            if (self.vars.currentHint) {
                self.setValue(self.vars.currentHint);
            }

            ev.target.blur();
        }
    }

    /**
     * Attaches event listeners to the user input field.
     *
     * @param {*} self Reference to a searchbar control.
     */
    function setupDomUserInput(self) {
        // Update the hint whenever the content of the user input field
        // changes.
        addRegisteredEventListener(self, "input", self.vars.dom.input, "input", function onInput() {
            var inputText = self.getValueDirect();

            self.vars.dom.mirror.textContent = inputText;

            updateHint(self, inputText);
        });

        addRegisteredEventListener(self, "input", self.vars.dom.input, "keydown", userInputKeyDownHandler.bind(null, self));

        addRegisteredEventListener(self, "input", self.vars.dom.input, "focus", function onFocus() {
            self.fire("enter", { value: self.getValue() });
            self.vars.userInputActive = true;
            shmi.addClass(self.element, "focused");
            shmi.addClass(self.element, "show-results");
            updateHint(self);

            if (self.config["select-contents-on-focus"]) {
                setSelected(self.vars.dom.input, true);
            }
        });

        addRegisteredEventListener(self, "input", self.vars.dom.input, "blur", function onBlur() {
            const value = self.getValueDirect();

            self.vars.userInputActive = false;
            self.vars.lastValue = value;
            shmi.removeClass(self.element, "focused");
            window.getSelection().removeAllRanges();
            clearHint(self);

            if (self.vars.itemSubscription && self.config.item) {
                self.imports.im.writeValue(self.config.item, value);
            }

            self.fire("change", { value: value });
        });

        addRegisteredEventListener(self, "input", self.vars.dom.input, "touchstart", function onDown(evt) {
            showKeyboard(self);
            evt.stopPropagation();
        });

        addRegisteredEventListener(self, "input", self.vars.dom.input, "click", function onClick() {
            showKeyboard(self);
            shmi.addClass(self.element, "show-results");
        });
    }

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.dom.label) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined") {
            self.vars.label = "";
            self.vars.dom.label.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.dom.label.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    /**
     * Show the alphanumeric keyboard if it is enabled
     *
     * @param {*} self Reference to the widget
     */
    function showKeyboard(self) {
        const appConfig = shmi.requires("visuals.session.config"),
            keyboardEnabled = (appConfig.keyboard && appConfig.keyboard.enabled); // get the keyboard config from `project.json`

        if (!keyboardEnabled) {
            return;
        }

        shmi.keyboard(
            {
                "value": self.getValueDirect(),
                "select-box-enabled": appConfig.keyboard["language-selection"],
                "show-enter": false,
                "callback": function(success, input) {
                    if (success) {
                        self.setValue(input);
                        shmi.addClass(self.element, "focused");
                        self.fire("change", { value: self.getValue() });
                    }
                }
            });
    }

    // declare private functions - END

    // definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            dom: {
                icon: null,
                label: null,
                input: null,
                mirror: null,
                suggestion: null,
                resultBox: null,
                clear: null
            },
            label: null,
            initDone: false,
            registeredEventListeners: {},
            dataGrid: null,
            dataGridSubscriptionId: null,
            dataGridSearchColumnIdx: -1,
            dataGridData: [],
            dataGridResults: 0,
            currentHint: null,
            currentValue: null,
            lastHintSearchToken: null,
            lastHintSearchResult: null,
            lastValue: null,
            userInputActive: false,
            currentHintIndex: null,
            itemSubscription: null,
            listeners: []
        },
        /* imports added at runtime */
        imports: {
            /* example - add import via shmi.requires(...) */
            im: "visuals.session.ItemManager",
            dgm: "visuals.session.DataGridManager"
        },

        /* array of custom event types fired by this control */
        events: [
            "change", "enter"
        ],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                const self = this,
                    io = shmi.requires("visuals.io");

                this.vars.dom.icon = shmi.getUiElement("icon", this.element);
                this.vars.dom.label = shmi.getUiElement("label", this.element);
                this.vars.dom.input = shmi.getUiElement("searchbar-user-input", this.element);
                this.vars.dom.mirror = shmi.getUiElement("searchbar-mirror", this.element);
                this.vars.dom.suggestion = shmi.getUiElement("searchbar-suggestion", this.element);
                this.vars.dom.resultBox = shmi.getUiElement("searchbar-result-box", this.element);
                this.vars.dom.clear = shmi.getUiElement("clear-input", this.element);
                this.vars.dataGrid = this.imports.dgm.getGrid(this.config.datagrid);

                if (!verifyDomElements(this.vars.dom)) {
                    this.element.textContent = "One or more required DOM elements are missing from the template. This control will not work. Please select a compatible template.";
                    fLog("Some DOM elements are missing from the template.");
                    return;
                }

                // Icon
                if (!this.vars.dom.icon) {
                    shmi.log('[IQ:iq-input-field] no button-icon element provided', 1);
                } else if (this.config['icon-src'] && this.config['show-icon']) {
                    this.vars.dom.icon.style.backgroundImage = `url(${this.config['icon-src']})`;
                } else if (this.config['icon-class'] && this.config['show-icon']) {
                    const iconClasses = this.config['icon-class'].trim().split(" ");
                    iconClasses.forEach(shmi.addClass.bind(shmi, this.vars.dom.icon));
                } else {
                    // No icon configured - hide
                    shmi.addClass(this.element, "no-icon");
                }

                // Label
                setLabelImpl(this, this.config.label);

                const domLabel = shmi.getUiElement("searchbar-label", this.element);
                if (domLabel) {
                    if (this.config.label && String(this.config.label).length > 0) {
                        domLabel.textContent = shmi.localize(this.config.label);
                    } else {
                        domLabel.style.display = "none";
                    }
                }

                const clearHandler = {
                    onClick: function() {
                        if (self.isActive() && !self.locked) {
                            self.setValue("");
                            self.fire("change", { value: self.getValue() });
                        }
                    }
                };
                self.vars.listeners.push(new io.MouseListener(self.vars.dom.clear, clearHandler));
                self.vars.listeners.push(new io.TouchListener(self.vars.dom.clear, clearHandler));

                if (this.vars.dataGrid === null) {
                    this.element.textContent = "Unable to find datagrid or no datagrid configured. This control will not work. Please select a valid datagrid.";
                    if (this.config.datagrid) {
                        fLog("Datagrid not found", this.config.datagrid);
                    } else {
                        fLog("No datagrid configured");
                    }

                    return;
                }

                this.vars.initDone = true;

                this.vars.dataGridSearchColumnIdx = this.vars.dataGrid.getFields().indexOf(this.config["search-column"]);
                if (this.vars.dataGridSearchColumnIdx === -1) {
                    this.element.textContent = "Datagrid does not have the configured field to search in. This control will not work. Please select a valid search field.";
                    fLog("Datagrid does not have the configured search field.");
                }

                this.vars.dataGrid.sort(this.vars.dataGridSearchColumnIdx, "ASC");
            },
            /**
             * Returns the value of the user input field.
             *
             * @returns {string}
             */
            getValue: function() {
                return this.vars.lastValue;
            },
            /**
             * Returns the contents of the user input field. Since some
             * browsers add non-breaking-spaces, those spaces are converted to
             * normal ones.
             *
             * @returns {string} Value of the input field.
             */
            getValueDirect: function() {
                if (!this.vars.initDone) {
                    return null;
                }

                const value = this.vars.dom.input.value;

                // Replace non breaking spaces with normal ones
                return value.replace(String.fromCharCode(0xA0), " ");
            },
            getMatchData: function() {
                return getMatchData(this);
            },
            /**
             * Sets the value of the user input field.
             *
             * @param {string} val Value to set
             */
            setValue: function(val) {
                if (!this.vars.initDone) {
                    return;
                }

                this.vars.dom.input.value = val;
                this.vars.dom.mirror.textContent = val;
                this.vars.lastValue = val;
                updateHint(this, val);
            },
            /* called when control is enabled */
            onEnable: function() {
                if (!this.vars.initDone) {
                    return;
                }
                const maxResults = typeof this.config["max-results"] === "number" && this.config["max-results"] > 0 ? this.config["max-results"] : 100;

                setupDomUserInput(this);

                this.vars.dataGridSubscriptionId = this.vars.dataGrid.subscribePage(0, maxResults, hintFetchCompleteCallback.bind(null, this));
                if (this.config.item) {
                    this.vars.itemSubscription = this.vars.im.subscribeItem(this.config.item, this);
                }
                this.vars.listeners.forEach(function(l) {
                    l.enable();
                });
            },
            /* called when control is disabled */
            onDisable: function() {
                if (!this.vars.initDone) {
                    return;
                }

                this.vars.dataGrid.unsubscribe(this.vars.dataGridSubscriptionId.id);
                this.vars.dataGridSubscriptionId = null;

                if (this.vars.itemSubscription) {
                    this.vars.itemSubscription.unlisten();
                    this.vars.itemSubscription = null;
                }

                clearResultBox(this);
                removeRegisteredEventListeners(this);
                this.vars.listeners.forEach(function(l) {
                    l.disable();
                });
            },
            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                this.vars.listeners.forEach(function(l) {
                    l.disable();
                });
                shmi.addClass(this.element, "locked");
            },
            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                this.vars.listeners.forEach(function(l) {
                    l.enable();
                });
                shmi.removeClass(this.element, "locked");
            },
            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                if (this.vars.initDone && !this.vars.userInputActive) {
                    this.setValue(value);
                }
            },
            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(min, max, step) {

            },
            setLabel: function(labelText) {
                if (this.vars.dom.label && this.config['auto-label']) {
                    setLabelImpl(this, labelText, false);
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-select-box
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-select-box",
 *     "name": null,
 *     "template": "default/iq-select-box.iq-variant-01"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "auto-label": Whether to use the auto-label (from item)
 * "icon-src": Default icon source
 * "icon-class": Default icon class
 * "tooltip": Tooltip
 * "no-selection-label": The default "no selection" label
 * "selected": Which option is selected
 * "options": The options
 * "item": The item
 * "show-icon": Whether to show the icon
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-select-box", // control name in camel-case
        uiType = "iq-select-box", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-select-box",
        "name": null,
        "template": "default/iq-select-box.iq-variant-01",
        "label": '[Label]',
        "item": null,
        "auto-label": true,
        "icon-src": null,
        "icon-class": null,
        "tooltip": null,
        "no-selection-label": "---",
        "selected": -1,
        "options": [],
        "show-text": true,
        "show-icon": false
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined") {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            optionListeners: [],
            domListeners: {},
            value: 0,
            active: false,

            // DOM Elements
            templateEl: null,
            labelEl: null,
            iconEl: null,
            containerEl: null,
            anchorEl: null,
            optionEl: null,
            selectedEl: null,
            selectedValEl: null,
            optionEls: [],

            selected: -1,
            clickedInside: false,
            isOpen: false,
            isLocked: false,
            highlightIndex: null,
            subscriptionTargetId: null,

            // Dynamic CSS classes
            iconClass: 'iq-icon',
            showTextAndIconClass: 'iq-icon-and-text',
            showIconOnlyClass: 'iq-icon-only',
            labelAreaClass: 'iq-label-area',
            highlightClass: 'highlighted', // WebIQ default
            selectBoxSelectedClass: 'iq-select-box-selected',
            optionIconAndTextClass: 'iq-option-icon-and-text',
            optionIconOnlyClass: 'iq-option-icon-only',
            optionIconClass: 'iq-option-icon'
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            iter: "visuals.tools.iterate.iterateObject"
        },

        /* array of custom event types fired by this control */
        events: ["change"],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.templateEl = self.element.cloneNode(true);

                self.vars.iqWidgetContent = shmi.getUiElement('iq-widget-content', self.element);
                self.vars.labelEl = shmi.getUiElement('label', self.element);
                self.vars.iconEl = shmi.getUiElement('icon', self.element);

                self.vars.containerEl = shmi.getUiElement('options', self.element);
                self.vars.selectedEl = shmi.getUiElement('selected', self.element);
                self.vars.selectedValEl = shmi.getUiElement('value', self.vars.selectedEl);
                self.vars.optionEl = shmi.getUiElement('option', self.element);
                self.vars.anchorEl = shmi.getUiElement('anchor', self.vars.containerEl);

                if (!self.vars.containerEl || !self.vars.anchorEl || !self.vars.optionEl || !self.vars.selectedEl || !self.vars.selectedValEl) {
                    shmi.log('[IQ:iq-select-box] At least one element is missing: containerEl, anchorEl, optionEl, selectedEl, selectedValEl', 3);
                    return;
                }

                // remove template DOM nodes and remap option elements to template
                self.vars.iqWidgetContent.removeChild(self.vars.selectedEl);
                self.vars.iqWidgetContent.removeChild(self.vars.containerEl);
                self.vars.iqWidgetContent.removeChild(self.vars.optionEl);

                self.vars.containerEl = shmi.getUiElement('options', self.vars.templateEl);
                self.vars.anchorEl = shmi.getUiElement('anchor', self.vars.containerEl);
                self.vars.optionEl = shmi.getUiElement('option', self.vars.templateEl);
                self.vars.selectedEl = shmi.getUiElement('selected', self.vars.templateEl);
                self.vars.selectedValEl = shmi.getUiElement('value', self.vars.selectedEl);

                /************/
                /*** ICON ***/
                /************/
                if (!self.vars.iconEl) {
                    shmi.log('[IQ:iq-select-box] no button-icon element provided', 1);
                } else if (self.config['icon-src'] && self.config['show-icon']) {
                    self.vars.iconEl.style.backgroundImage = `url(${self.config['icon-src']})`;
                } else if (self.config['icon-class'] && self.config['show-icon']) {
                    var iconClasses = self.config['icon-class'].trim().split(" ");
                    iconClasses.forEach(function(cls) {
                        shmi.addClass(self.vars.iconEl, cls);
                    });
                } else {
                    // No icon configured - hide
                    shmi.addClass(self.element, "no-icon");
                }

                // Label
                if (self.config['show-text']) {
                    setLabelImpl(self, self.config.label);
                } else {
                    setLabelImpl(self, null);
                }

                var popupContainerEl = document.createElement("DIV");
                shmi.addClass(popupContainerEl, 'iq-popup-container');
                self.vars.iqWidgetContent.appendChild(popupContainerEl);
                popupContainerEl.appendChild(self.vars.containerEl);
                popupContainerEl.appendChild(self.vars.selectedEl);

                self.rebuildOptions(false);

                self.vars.selected = self.config.selected;
                self.vars.containerEl.setAttribute('tabindex', '-1');

                /*****************/
                /*** LISTENERS ***/
                /*****************/
                self.vars.domListeners = self.createDomListenerObject();
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.element, self.createElementListenerObject()));
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.element, self.createElementListenerObject()));
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.containerEl, self.createContainerListenerObject()));

                // Defaults
                shmi.addClass(self.vars.containerEl, 'hidden');
                if (self.vars.selected === -1) {
                    self.setSelected(null);
                } else {
                    self.setSelected(self.vars.optionEls[self.config.selected]);
                }
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                self.element.setAttribute('tabindex', 0);

                self.lockUnlockListeners(false);

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                shmi.log("[IQ:iq-select-box] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                self.lockUnlockListeners(true);

                if (self.vars.subscriptionTargetId) {
                    self.vars.subscriptionTargetId.unlisten();
                    self.vars.subscriptionTargetId = null;
                }

                shmi.log("[IQ:iq-select-box] disabled", 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;

                self.vars.isLocked = true;
                shmi.addClass(self.vars.containerEl, 'hidden');
                self.vars.highlightIndex = -1;

                self.lockUnlockListeners(true);

                shmi.addClass(self.element, 'locked');

                shmi.log("[IQ:iq-select-box] Locked", 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;

                self.lockUnlockListeners(false);

                shmi.removeClass(self.element, 'locked');

                shmi.log("[IQ:iq-select-box] unlocked", 1);
            },

            setOptions: function(options) {
                const self = this;

                self.config.options = options;
                if (self.initialized) {
                    self.rebuildOptions(self.isActive());
                    self.setValue(self.getValue());
                }
            },

            getOptions: function() {
                var self = this;
                return shmi.cloneObject(self.config.options);
            },

            /**
             * Sets the selected element
             *
             * @param element - element to select
             */
            setSelected: function(element) {
                const self = this,
                    selectedIdx = self.vars.optionEls.findIndex((el) => el === element);

                // Unselect all
                self.vars.optionEls.forEach(function(val, idx) {
                    shmi.removeClass(val, self.vars.selectBoxSelectedClass);
                });

                if (selectedIdx === -1) {
                    self.vars.selectedValEl.textContent = shmi.localize(self.config['no-selection-label']);
                    self.vars.value = null;
                    shmi.removeClass(self.vars.selectedEl, self.vars.showTextAndIconClass);
                    shmi.removeClass(self.vars.selectedEl, self.vars.showIconOnlyClass);
                } else {
                    // Get selected label
                    const selectedLabel = self.config.options[selectedIdx].label;

                    shmi.addClass(element, self.vars.selectBoxSelectedClass);
                    self.vars.selectedValEl.textContent = shmi.localize(selectedLabel);
                    self.vars.value = self.config.options[selectedIdx].value;

                    var selectedOption = self.config.options[selectedIdx],
                        iconEl = shmi.getUiElement("option-icon", self.vars.selectedEl),
                        hasIcon = false;

                    if (selectedOption && iconEl) {
                        hasIcon = self.setIcon(iconEl, selectedOption, true);
                        if (hasIcon && selectedOption.label) {
                            shmi.addClass(self.vars.selectedEl, self.vars.optionIconAndTextClass);
                        } else if (hasIcon) {
                            shmi.addClass(self.vars.selectedEl, self.vars.optionIconOnlyClass);
                        } else {
                            shmi.removeClass(self.vars.selectedEl, self.vars.optionIconAndTextClass);
                            shmi.removeClass(self.vars.selectedEl, self.vars.optionIconOnlyClass);
                        }
                    } else if (iconEl) {
                        shmi.removeClass(self.vars.selectedEl, self.vars.optionIconAndTextClass);
                        shmi.removeClass(self.vars.selectedEl, self.vars.optionIconOnlyClass);
                    }
                }
                self.vars.selected = selectedIdx;
            },

            getCurrentIndex: function() {
                var self = this;

                return self.vars.selected;
            },

            /**
             * Writes the current value to a connected data-source item
             *
             */
            updateValue: function() {
                var self = this;
                if (self.config.item) {
                    self.imports.im.writeValue(self.config.item, self.getValue());
                }
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                const self = this,
                    oldValue = self.getValue();

                shmi.log("[IQ:iq-select-box] Setting value...: " + value, 0);

                const selectedIndex = self.config.options.findIndex((option) => option.value === value);
                self.setSelected(selectedIndex !== -1 ? self.vars.optionEls[selectedIndex] : null);

                if (self.getValue() !== oldValue) {
                    self.fire("change", {
                        value: self.getValue()
                    });
                }
                shmi.log("[IQ:iq-select-box] Value set: " + value, 1);
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                var self = this;

                return self.vars.value;
            },

            /**
             * Called when the label is received
             *
             * @param labelText
             */
            setLabel: function(labelText) {
                var self = this;

                if (self.vars.labelEl && self.config['auto-label'] && self.config['show-text']) {
                    setLabelImpl(self, labelText);
                }
            },

            /**
             * Helper function for locking/unlocking
             *
             * @param {boolean} lock
             */
            lockUnlockListeners: function(lock) {
                var self = this;

                // SHMI listeners
                self.vars.listeners.forEach(function(l) {
                    if (lock) {
                        l.disable();
                    } else {
                        l.enable();
                    }
                });

                self.vars.optionListeners.forEach(function(l) {
                    if (lock) {
                        l.disable();
                    } else {
                        l.enable();
                    }
                });

                // DOM Listeners
                self.imports.iter(self.vars.domListeners, function(val, prop) {
                    if (val.element) {
                        if (lock) {
                            val.element.removeEventListener(val.event, val.callback);
                        } else {
                            val.element.addEventListener(val.event, val.callback);
                        }
                    }
                });
            },

            /**
             * Rebuilds options
             *
             * @param enableListeners
             */
            rebuildOptions: function(enableListeners) {
                var self = this;

                // Remove listeners
                self.vars.optionListeners.forEach(function(listener) {
                    listener.disable();
                });
                self.vars.optionListeners = [];

                // Reset option-elements in case of call to 'setOptions(..)'
                self.vars.optionEls = [];

                // Clear old option elements
                while (self.vars.anchorEl.firstChild) {
                    self.vars.anchorEl.removeChild(self.vars.anchorEl.firstChild);
                }

                self.config.options.forEach(function(option, index) {
                    var label = shmi.localize(option.label),
                        optionEl = self.vars.optionEl.cloneNode(true),
                        valueEl = shmi.getUiElement('value', optionEl),
                        iconEl = shmi.getUiElement('icon', optionEl),
                        hasIcon = false;

                    if (valueEl) {
                        valueEl.textContent = label;
                    } else {
                        optionEl.textContent = label;
                    }

                    if (option["tooltip"] || option["icon-title"]) {
                        optionEl.setAttribute("title", shmi.localize(option.tooltip || option["icon-title"]));
                    }

                    if (iconEl) {
                        hasIcon = self.setIcon(iconEl, option, false);
                        if (hasIcon && label) {
                            shmi.addClass(optionEl, self.vars.optionIconAndTextClass);
                        } else if (hasIcon) {
                            shmi.addClass(optionEl, self.vars.optionIconOnlyClass);
                        }
                    }

                    self.vars.anchorEl.appendChild(optionEl);
                    self.vars.optionEls.push(optionEl);
                });

                var mouseHandler = self.makeOptionMouseHandler(),
                    touchHandler = self.makeOptionTouchHandler();

                self.vars.optionEls.forEach(function(el) {
                    self.vars.optionListeners.push(new shmi.visuals.io.TouchListener(el, touchHandler));
                    self.vars.optionListeners.push(new shmi.visuals.io.MouseListener(el, mouseHandler));
                });

                if (enableListeners) {
                    self.vars.optionListeners.forEach(function(listener) {
                        listener.enable();
                    });
                }
            },

            /**
             * Sets the icon
             *
             * @param {object} element
             * @param {object} option
             * @param {boolean} isSelected
             *
             * @returns {boolean}
             */
            setIcon: function(element, option, isSelected) {
                var self = this,
                    img = null,
                    hasIcon = false;

                if (option["icon-src"]) {
                    if (element.tagName !== "IMG") {
                        img = document.createElement("IMG");
                        element.parentNode.insertBefore(img, element);
                        element.parentNode.removeChild(element);
                    } else {
                        img = element;
                    }

                    shmi.addClass(img, self.vars.optionIconClass);
                    img.setAttribute("src", option["icon-src"]);
                    hasIcon = true;
                } else if (option["icon-class"]) {
                    if (element.tagName !== "DIV") {
                        img = document.createElement("DIV");
                        element.parentNode.insertBefore(img, element);
                        element.parentNode.removeChild(element);
                    } else {
                        img = element;
                    }

                    shmi.addClass(img, self.vars.optionIconClass);
                    shmi.addClass(img, option["icon-class"]);
                    hasIcon = true;
                } else {
                    shmi.addClass(element, self.vars.optionIconClass);
                }

                if (img && hasIcon) {
                    img.dataset.ui = "option-icon";
                } else {
                    element.dataset.ui = "option-icon";
                }

                if (!hasIcon) {
                    shmi.addClass(element, "hidden");
                } else {
                    shmi.removeClass(element, "hidden");
                }

                return hasIcon;
            },

            /**
             * Creates mouse listener functions to be attached to each option element.
             */
            makeOptionMouseHandler: function() {
                var self = this;
                return {
                    onClick: function onClick(x, y, e) {
                        shmi.log("[IQ:iq-select-box] Click on element", 0);
                        const oldValue = self.getValue();

                        self.setSelected(e.currentTarget);
                        self.updateValue();

                        shmi.addClass(self.vars.containerEl, 'hidden');

                        self.vars.isOpen = false;
                        self.vars.highlightIndex = -1;
                        if (self.getValue() !== oldValue) {
                            self.fire("change", {
                                value: self.getValue()
                            });
                        }
                    },

                    onPress: function onPress(x, y, e) {
                        shmi.log("[IQ:iq-select-box] Press", 0);
                        e.preventDefault();
                    },

                    onRelease: function onRelease() {
                        shmi.log("[IQ:iq-select-box] Release", 0);
                    }
                };
            },

            /**
             * Creates touch listener functions to be attached to each option element.
             */
            makeOptionTouchHandler: function() {
                var self = this;
                return {
                    onPress: function onPress() {
                    },
                    onClick: function onClick(x, y, e) {
                        const oldValue = self.getValue();

                        shmi.log("[IQ:iq-select-box] Click on element", 0);

                        self.setSelected(e.currentTarget);
                        self.updateValue();

                        shmi.addClass(self.vars.containerEl, 'hidden');
                        self.vars.isOpen = false;
                        self.vars.highlightIndex = -1;

                        if (self.getValue() !== oldValue) {
                            self.fire("change", {
                                value: self.getValue()
                            });
                        }
                    }
                };
            },

            /**
             * Generates the listener object
             *
             * @returns {object}
             */
            createDomListenerObject: function() {
                var self = this;
                return {
                    containerBlur: {
                        "event": 'blur',
                        "element": self.vars.containerEl,
                        "callback": function(e) {
                            if (self.vars.clickedInside) {
                                setTimeout(function() {
                                    self.vars.containerEl.focus();
                                }, shmi.c("DECOUPLE_TIMEOUT"));
                            } else {
                                shmi.addClass(self.vars.containerEl, 'hidden');
                                setTimeout(function() {
                                    self.vars.isOpen = false;
                                }, shmi.c("DECOUPLE_TIMEOUT"));
                                self.vars.highlightIndex = -1;
                            }
                        }
                    },
                    elementKeydown: {
                        "event": 'keydown',
                        "element": self.element,
                        "callback": function(e) { // Add keyboard listener to select box to be opened on ENTER
                            var key = e.key ? e.key : e.code,
                                focusedNotLocked = (this.element === document.activeElement) && !this.locked;
                            if (focusedNotLocked && (key === 'Enter' || key === 'NumpadEnter')) {
                                shmi.log("[IQ:iq-select-box] click on container", 1);
                                if (self.vars.isOpen) {
                                    self.vars.containerEl.blur();
                                } else {
                                    shmi.removeClass(self.vars.containerEl, 'hidden');
                                    self.vars.isOpen = true;
                                    setTimeout(function() {
                                        self.vars.containerEl.focus();
                                        shmi.visuals.session.FocusElement = self.vars.containerEl;
                                    }, shmi.c("DECOUPLE_TIMEOUT"));
                                }
                            }
                        }
                    },
                    containerKeydown: {
                        "event": "keydown",
                        "element": self.vars.containerEl,
                        "callback": function(e) {
                            var key = e.key ? e.key : e.code;
                            shmi.log("[IQ:iq-select-box] Keydown event: " + key, 1);
                            if (self.vars.highlightIndex < 0) {
                                self.vars.highlightIndex = self.getCurrentIndex();
                            }
                            shmi.log("[IQ:iq-select-box] Highlight index " + self.vars.highlightIndex, 1);
                            if (key === "Escape") {
                                self.vars.optionEls.forEach(function(optionEl) {
                                    shmi.removeClass(optionEl, 'highlighted');
                                });
                                shmi.addClass(self.vars.containerEl, 'hidden');
                                self.vars.isOpen = false;
                                self.vars.highlightIndex = -1;
                            } else if (key === "Enter" || key === "NumpadEnter") {
                                if (self.vars.highlightIndex !== -1) {
                                    shmi.log("[IQ:iq-select-box] Click on element", 0);
                                    var element = self.vars.optionEls[self.vars.highlightIndex],
                                        oldValue = self.getValue();

                                    self.setSelected(element);
                                    self.updateValue();
                                    shmi.addClass(self.vars.containerEl, 'hidden');
                                    self.vars.isOpen = false;
                                    self.vars.highlightIndex = -1;
                                    if (self.getValue() !== oldValue) {
                                        self.fire("change", { value: self.getValue() });
                                    }
                                    self.vars.optionEls.forEach(function(optionEl) {
                                        shmi.removeClass(optionEl, 'highlighted');
                                    });
                                    self.vars.highlightIndex = -1;
                                }
                            } else if (key === "ArrowUp") {
                                if (self.vars.optionEls[self.vars.highlightIndex]) {
                                    shmi.removeClass(self.vars.optionEls[self.vars.highlightIndex], 'highlighted');
                                }
                                if (self.vars.highlightIndex > 0) {
                                    self.vars.highlightIndex--;
                                } else {
                                    self.vars.highlightIndex = self.vars.optionEls.length - 1;
                                }
                                if (self.vars.optionEls[self.vars.highlightIndex]) {
                                    shmi.addClass(self.vars.optionEls[self.vars.highlightIndex], 'highlighted');
                                }
                            } else if (key === "ArrowDown") {
                                if (self.vars.optionEls[self.vars.highlightIndex]) {
                                    shmi.removeClass(self.vars.optionEls[self.vars.highlightIndex], 'highlighted');
                                }
                                if (self.vars.highlightIndex < (self.vars.optionEls.length - 1)) {
                                    self.vars.highlightIndex++;
                                } else {
                                    self.vars.highlightIndex = 0;
                                }
                                if (self.vars.optionEls[self.vars.highlightIndex]) {
                                    shmi.addClass(self.vars.optionEls[self.vars.highlightIndex], 'highlighted');
                                }
                            }
                            return false;
                        }
                    }
                };
            },

            /**
             * Generates SHMI listener object
             *
             * @returns {{onPress: onPress, onClick: onClick}}
             */
            createElementListenerObject: function() {
                var self = this;
                return {
                    onPress: function() {
                        // Left empty
                    },
                    onClick: function(x, y, e) {
                        if (e.target !== self.vars.anchorEl) {
                            shmi.log("[IQ:is-select-box] Click on container", 1);
                            if (self.vars.isOpen) {
                                self.vars.containerEl.blur();
                            } else {
                                shmi.removeClass(self.vars.containerEl, 'hidden');
                                self.vars.isOpen = true;
                                setTimeout(function() {
                                    self.vars.containerEl.focus();
                                    shmi.visuals.session.FocusElement = self.vars.containerEl;
                                }, shmi.c("DECOUPLE_TIMEOUT"));
                            }
                        }
                    }
                };
            },

            /**
             * Creates SHMI listener object
             *
             * @returns {{onPress: onPress, onRelease: onRelease, onLeave: onLeave}}
             */
            createContainerListenerObject: function() {
                var self = this;
                return {
                    onPress: function(x, y, e) {
                        if (e.currentTarget === self.vars.containerEl) {
                            self.vars.clickedInside = true;
                        }
                    },
                    onRelease: function(x, y, e) {
                        if (e.currentTarget === self.vars.containerEl) {
                            self.vars.clickedInside = false;
                        }
                    },
                    onLeave: function() {
                        shmi.log("[IQ:iq-select-box] onLeave container element", 0);
                        self.vars.containerEl.blur();
                        self.vars.highlightIndex = -1;
                    }
                };
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-select-date
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-select-date",
 *     "name": null,
 *     "template": "default/iq-select-date"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "auto-label": Whether to use the auto-label (from item)
 * "tooltip": Tooltip
 * "dateformat": The date format
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-select-date", // control name in camel-case
        uiType = "iq-select-date", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-select-date",
        "name": null,
        "template": "default/iq-select-date.variant-01",
        "label": '[Label]',
        "item": null,
        "auto-label": true,
        "tooltip": null,
        "dateformat": "${V_DATEFORMAT}",
        "daynames": [
            '${V_SUN_SHORT}',
            '${V_MON_SHORT}',
            '${V_TUE_SHORT}',
            '${V_WED_SHORT}',
            '${V_THU_SHORT}',
            '${V_FRI_SHORT}',
            '${V_SAT_SHORT}'
        ],
        "daynames-long": [
            '${V_SUN}',
            '${V_MON}',
            '${V_TUE}',
            '${V_WED}',
            '${V_THU}',
            '${V_FRI}',
            '${V_SAT}'
        ],
        "monthnames": [
            '${V_JAN}',
            '${V_FEB}',
            '${V_MAR}',
            '${V_APR}',
            '${V_MAY}',
            '${V_JUN}',
            '${V_JUL}',
            '${V_AUG}',
            '${V_SEP}',
            '${V_OCT}',
            '${V_NOV}',
            '${V_DEC}'
        ],
        "show-icon": false,
        "show-text": true,
        "icon-src": null,
        "icon-class": null,
        "icon-title": null
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined") {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: 0,
            active: false,
            isOpen: true,

            // DOM Elements
            labelEl: null,
            calendarEl: null,
            daySelectEl: null,
            headerEl: null,
            dayNameEls: null,
            selectedEl: null,
            dayEls: [],

            monthSelectEl: null,
            previousMonthEl: null,
            currentMonthEl: null,
            nextMonthEl: null,

            yearSelectEl: null,
            previousYearEl: null,
            currentYearEl: null,
            nextYearEl: null,

            pressedEl: null,

            // Event Handler
            blurHandler: null,
            subscriptionTargetId: null,

            // Vars
            date: null,
            selectedDate: null
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues",
            dt: "visuals.tools.date"
        },

        /* array of custom event types fired by this control */
        events: ["change"],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                self.initDate();

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.iconEl = shmi.getUiElement('icon', self.element);
                self.vars.labelEl = shmi.getUiElement('label', self.element);
                self.vars.calendarEl = shmi.getUiElement('calendar', self.element);
                self.vars.daySelectEl = shmi.getUiElement('day-select', self.element);
                self.vars.headerEl = shmi.getUiElement('day-header', self.element);
                self.vars.yearSelectEl = shmi.getUiElement('year-select', self.element);

                self.vars.dayNameEls = shmi.getUiElements('day-name', self.vars.headerEl);
                self.vars.dayEls = shmi.getUiElements('day', self.element);
                self.vars.valueEl = shmi.getUiElement('value', self.element);

                self.vars.selectedEl = shmi.getUiElement('selected-date', self.element);
                if (self.vars.valueEl) {
                    self.vars.selectedEl = self.vars.valueEl;
                }

                self.vars.monthSelectEl = shmi.getUiElement('month-select', self.element);
                if (self.vars.monthSelectEl) {
                    self.vars.previousMonthEl = shmi.getUiElement('previous', self.vars.monthSelectEl);
                    self.vars.currentMonthEl = shmi.getUiElement('current', self.vars.monthSelectEl);
                    self.vars.nextMonthEl = shmi.getUiElement('next', self.vars.monthSelectEl);
                }

                if (self.vars.yearSelectEl) {
                    self.vars.previousYearEl = shmi.getUiElement('previous', self.vars.yearSelectEl);
                    self.vars.currentYearEl = shmi.getUiElement('current', self.vars.yearSelectEl);
                    self.vars.nextYearEl = shmi.getUiElement('next', self.vars.yearSelectEl);
                }

                if (!self.vars.calendarEl || !self.vars.daySelectEl || !self.vars.selectedEl || !self.vars.headerEl || !self.vars.monthSelectEl || !self.vars.yearSelectEl) {
                    shmi.log('[IQ:iq-select-date] No calendar/day-select/selected-date/value/day-header/month-select/year-select element provided', 3);
                    return;
                }

                if (!self.vars.dayNameEls || (self.vars.dayNameEls && self.vars.dayNameEls.length !== 7)) {
                    shmi.log('[IQ:iq-select-date] No day-name elements, or not the right count (7) provided', 3);
                    return;
                }

                if (!(self.vars.previousMonthEl && self.vars.currentMonthEl && self.vars.nextMonthEl && self.vars.previousYearEl && self.vars.currentYearEl && self.vars.nextYearEl)) {
                    shmi.log('[IQ:iq-select-date] Not all month & year select elements provided', 3);
                    return;
                }

                self.vars.dayNameEls.forEach(function(el, idx) {
                    el.textContent = shmi.localize(self.config.daynames[(idx === 6) ? 0 : idx + 1]);
                });

                self.vars.calendarEl.setAttribute('tabindex', "0");

                /************/
                /*** ICON ***/
                /************/
                if (!self.vars.iconEl) {
                    shmi.log('[IQ:iq-select-date] no button-icon element provided', 1);
                } else if (self.config['icon-src'] && self.config['show-icon']) {
                    self.vars.iconEl.style.backgroundImage = `url(${self.config['icon-src']})`;
                } else if (self.config['icon-class'] && self.config['show-icon']) {
                    const iconClasses = self.config['icon-class'].trim().split(" ");
                    iconClasses.forEach(function(cls) {
                        shmi.addClass(self.vars.iconEl, cls);
                    });
                } else {
                    // No icon configured - hide
                    shmi.addClass(self.element, 'no-icon');
                }

                // LISTENERS
                self.setupListeners();

                // INTITIAL UPDATE
                self.updateCalendar();

                // Label
                setLabelImpl(self, self.config['show-text'] ? self.config.label : null);
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                if (self.vars.calendarEl) {
                    self.vars.calendarEl.addEventListener("blur", self.vars.blurHandler);
                }

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                shmi.log("[IQ:iq-select-date] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                if (self.vars.calendarEl) {
                    self.vars.calendarEl.removeEventListener("blur", self.vars.blurHandler);
                }

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                if (self.config.item) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                shmi.log("[IQ:iq-select-date] disabled", 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;

                shmi.addClass(self.vars.calendarEl, 'hidden');
                self.isOpen = false;

                if (self.vars.calendarEl) {
                    self.vars.calendarEl.removeEventListener("blur", self.vars.blurHandler);
                }
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                shmi.addClass(self.element, 'locked');

                shmi.log("[IQ:iq-select-date] Locked", 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;

                if (self.vars.calendarEl) {
                    self.vars.calendarEl.addEventListener("blur", self.vars.blurHandler);
                }

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.removeClass(self.element, 'locked');

                shmi.log("[IQ:iq-select-date] unlocked", 1);
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                var self = this,
                    date = new Date(value * 1000),
                    oldValue = self.vars.value;

                value = parseInt(value);

                self.vars.value = value;
                self.vars.date.setFullYear(date.getYear() + 1900, date.getMonth());
                self.vars.selectedDate.setFullYear(date.getYear() + 1900, date.getMonth(), date.getDate());
                self.updateCalendar();

                if (oldValue !== self.vars.value) {
                    self.fire("change", {
                        value: self.vars.value
                    });
                }
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                var self = this;

                return self.vars.value;
            },

            setLabel: function(labelText) {
                var self = this;

                if (self.vars.labelEl && self.config['auto-label'] && self.config['show-text']) {
                    setLabelImpl(self, labelText);
                }
            },

            /**
             * Updates the date with the given month
             *
             * @param {int} newMonth
             * @param {object} el
             */
            updateDateMonth: function(newMonth, el) {
                var self = this;

                self.vars.date.setFullYear(self.vars.date.getYear() + 1900, newMonth);
                self.vars.selectedDate.setFullYear(self.vars.date.getYear() + 1900, newMonth, el.textContent);
            },

            /**
             * Sets up the listeners, moved here to make onInit shorter
             */
            setupListeners: function() {
                var self = this;

                // MONTH: PREVIOUS
                var previousMonthFunctions = {
                    onPress: function() {
                    },
                    onClick: function(x, y, event) {
                        self.vars.date.setFullYear(self.vars.date.getYear() + 1900, self.vars.date.getMonth() - 1);
                        self.updateCalendar();
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.previousMonthEl, previousMonthFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.previousMonthEl, previousMonthFunctions));

                // MONTH: NEXT
                var nextMonthFunctions = {
                    onPress: function() {
                    },
                    onClick: function(x, y, e) {
                        self.vars.date.setFullYear(self.vars.date.getYear() + 1900, self.vars.date.getMonth() + 1);
                        self.updateCalendar();
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.nextMonthEl, nextMonthFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.nextMonthEl, nextMonthFunctions));

                // YEAR: PREVIOUS
                var previousYearFunctions = {
                    onPress: function() {
                    },
                    onClick: function(x, y, e) {
                        self.vars.date.setFullYear(self.vars.date.getYear() + 1900 - 1, self.vars.date.getMonth());
                        self.updateCalendar();
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.previousYearEl, previousYearFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.previousYearEl, previousYearFunctions));

                // YEAR: NEXT
                var nextYearFunctions = {
                    onPress: function() {
                    },
                    onClick: function(x, y, e) {
                        self.vars.date.setFullYear(self.vars.date.getYear() + 1900 + 1, self.vars.date.getMonth());
                        self.updateCalendar();
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.nextYearEl, nextYearFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.nextYearEl, nextYearFunctions));

                // DAY
                self.vars.pressedEl = null;
                var dayFunctions = {
                    onPress: function(x, y, e) {
                        shmi.addClass(e.target, 'pressed');
                        self.vars.pressedEl = e.target;
                    },
                    onRelease: function(x, y, e) {
                        shmi.removeClass(self.vars.pressedEl, 'pressed');
                    },
                    onClick: function(x, y, e) {
                        var dayEl = e.target;
                        if (shmi.hasClass(dayEl, 'iq-previous-month-day')) {
                            self.updateDateMonth(self.vars.date.getMonth() - 1, dayEl);
                        } else if (shmi.hasClass(dayEl, 'iq-current-month-day')) {
                            self.updateDateMonth(self.vars.date.getMonth(), dayEl);
                        } else if (shmi.hasClass(dayEl, 'iq-next-month-day')) {
                            self.updateDateMonth(self.vars.date.getMonth() + 1, dayEl);
                        }

                        self.updateCalendar();

                        if (self.config.item) {
                            self.imports.im.writeValue(self.config.item, self.vars.value);
                        }

                        shmi.addClass(self.vars.calendarEl, 'hidden');

                        self.vars.isOpen = false;
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.daySelectEl, dayFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.daySelectEl, dayFunctions));

                // CALENDAR SELECTOR
                shmi.addClass(self.vars.calendarEl, 'hidden');
                self.vars.isOpen = false;
                var selectedFunctions = {
                    onClick: function() {
                        if (!self.vars.isOpen) {
                            shmi.removeClass(self.vars.calendarEl, 'hidden');
                            self.isOpen = true;
                            shmi.decouple(() => {
                                self.vars.calendarEl.focus();
                                shmi.visuals.session.FocusElement = self.vars.calendarEl;
                            });
                        } else {
                            self.vars.calendarEl.blur();
                        }
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.element, selectedFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.element, selectedFunctions));

                self.vars.blurHandler = function() {
                    shmi.addClass(self.vars.calendarEl, 'hidden');
                    shmi.decouple(() => {
                        self.vars.isOpen = false;
                    });
                };
            },
            /**
             * Updates the calendar element
             *
             */
            updateCalendar: function() {
                var self = this,
                    oldValue = self.vars.value,
                    date = new Date(self.vars.value * 1000);

                self.initDate();

                date.setFullYear(self.vars.selectedDate.getYear() + 1900, self.vars.selectedDate.getMonth(), self.vars.selectedDate.getDate());
                self.vars.value = date.getTime() / 1000;

                shmi.log("[IQ:iq-select-date] selected time: " + self.vars.value, 0);

                var currentMonth = self.vars.date.getMonth(),
                    currentYear = self.vars.date.getYear() + 1900,
                    firstDay = new Date(currentYear, currentMonth, 1).getDay(),
                    start = firstDay - 1;

                self.vars.currentMonthEl.textContent = shmi.localize(self.config.monthnames[currentMonth]);
                self.vars.currentYearEl.textContent = currentYear;

                if (firstDay === 1) {
                    start = 7;
                } else if (firstDay === 0) {
                    start = 6;
                }

                var prevDays = self.getDaysOfMonth(currentMonth, currentYear),
                    curDays = self.getDaysOfMonth(currentMonth + 1, currentYear),
                    tempDayEl = shmi.getUiElements('day', self.element),
                    tempDay = prevDays - start + 1;

                for (var i = 0; i < start; i++) {
                    tempDayEl[i].className = 'iq-previous-month-day';
                    tempDayEl[i].textContent = tempDay;
                    if (new Date(currentYear, currentMonth - 1, tempDay).getTime() === self.vars.selectedDate.getTime()) {
                        shmi.addClass(tempDayEl[i], 'iq-selected');
                    } else {
                        shmi.removeClass(tempDayEl[i], 'iq-selected');
                    }
                    tempDay++;
                }

                tempDay = 1;
                for (i = start; i < curDays + start; i++) {
                    tempDayEl[i].className = 'iq-current-month-day';
                    tempDayEl[i].textContent = tempDay;
                    if (new Date(currentYear, currentMonth, tempDay).getTime() === self.vars.selectedDate.getTime()) {
                        shmi.addClass(tempDayEl[i], 'iq-selected');
                    } else {
                        shmi.removeClass(tempDayEl[i], 'iq-selected');
                    }
                    tempDay++;
                }

                tempDay = 1;
                for (i = curDays + start; i < tempDayEl.length; i++) {
                    tempDayEl[i].className = 'iq-next-month-day';
                    tempDayEl[i].textContent = tempDay;
                    if (new Date(currentYear, currentMonth + 1, tempDay).getTime() === self.vars.selectedDate.getTime()) {
                        shmi.addClass(tempDayEl[i], 'iq-selected');
                    } else {
                        shmi.removeClass(tempDayEl[i], 'iq-selected');
                    }
                    tempDay++;
                }

                var currentDate = shmi.localize(self.config.dateformat);
                if (typeof (currentDate) === 'string') {
                    self.vars.selectedEl.textContent = self.imports.dt.formatDateTime(self.vars.selectedDate, { datestring: currentDate });
                } else {
                    self.vars.selectedEl.innerHTML = '&nbsp;';
                }

                if (oldValue !== self.vars.value) {
                    self.fire("change", {
                        value: self.vars.value
                    });
                }
            },

            /**
             * Returns number of days for month of year
             *
             * @param month - month of year
             * @param year - year
             * @return days days in month
             */
            getDaysOfMonth: function(month, year) {
                var d = new Date(year, month, 0);
                return d.getDate();
            },

            /**
             * Initializes the date if not set. As setValue might be called before onInit has finished we need to do it like this
             */
            initDate: function() {
                var self = this;

                // Initialize date
                if (self.vars.date === null) {
                    self.vars.date = new Date();
                }

                if (self.vars.selectedDate === null) {
                    self.vars.selectedDate = new Date(new Date().setHours(0, 0, 0, 0));
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-select-radio
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-select-radio",
 *     "name": null,
 *     "template": "default/iq-select-radio.iq-variant-01"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "auto-label": Whether to use the auto-label (from item)
 * "icon-src": Default icon source
 * "icon-class": Default icon class
 * "tooltip": Tooltip
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-select-radio", // control name in camel-case
        uiType = "iq-select-radio", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-select-radio",
        "name": null,
        "template": "default/iq-select-radio.variant-01",
        "label": '[Label]',
        "item": null,
        "auto-label": true,
        "show-icon": false,
        "icon-src": null,
        "icon-class": null,
        "tooltip": null,
        "options": [],
        "selected": -1
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined") {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: -1,
            active: false,

            // DOM Elements
            optionContainerEl: null,
            optionEls: [],
            labelEl: null,
            iconEl: null,
            iconClassEl: null,
            label: null,
            selectedEl: null
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager",
            nv: "visuals.tools.numericValues"
        },

        /* array of custom event types fired by this control */
        events: [
            "change"
        ],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            // Note: onTemplate has not been implemented in this version anymore as variants should be used instead

            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.labelEl = shmi.getUiElement('label', self.element);
                self.vars.iconEl = shmi.getUiElement('icon', self.element);
                self.vars.optionEl = shmi.getUiElement('option', self.element); // To be duplicated!
                self.vars.optionContainerEl = shmi.getUiElement('option-container', self.element);

                // Verify options set and available in HTML template
                if (!self.config['options']) {
                    shmi.log("[IQ:iq-select-radio] No options defined in config", 1);
                    return;
                } else {
                    shmi.log("[IQ:iq-select-radio] Options newly defined", 1);
                }
                if (!self.vars.optionEl || !self.vars.optionContainerEl) {
                    shmi.log("[IQ:iq-select-radio] Missing option and/or option container element in template!", 1);
                    return;
                }

                // Remove initial option from DOM after cloning it for reuse
                var clone = self.vars.optionEl.cloneNode(true);
                self.vars.optionContainerEl.removeChild(self.vars.optionEl);
                self.vars.optionEl = clone;

                /************/
                /*** ICON ***/
                /************/
                if (!self.vars.iconEl) {
                    shmi.log('[IQ:iq-select-radio] no button-icon element provided', 1);
                } else if (!self.config['show-icon']) {
                    // Icon disabled
                    self.vars.iconEl.style.display = 'none';
                } else if (self.config['icon-src']) {
                    if (self.vars.iconEl) {
                        self.vars.iconEl.style.backgroundImage = `url(${self.config['icon-src']})`;
                    }
                } else if (self.config['icon-class']) {
                    if (self.vars.iconEl) {
                        shmi.removeClass(self.vars.iconEl, "iq-icon");
                        shmi.addClass(self.vars.iconEl, "iq-icon-class");
                        var iconClasses = self.config['icon-class'].trim().split(" ");
                        iconClasses.forEach(function(cls) {
                            shmi.addClass(self.vars.iconEl, cls);
                        });
                    }
                }

                // Label
                setLabelImpl(self, self.config.label);

                self.rebuildOptions();
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                shmi.log("[IQ:iq-select-radio] Enabled", 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                if (self.config.item) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                shmi.log("[IQ:iq-select-radio] disabled", 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;
                //self.vars.currentValueEl.blur();
                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                shmi.addClass(this.element, 'locked');

                shmi.log("[IQ:iq-select-radio] Locked", 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;
                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.removeClass(self.element, 'locked');

                shmi.log("[IQ:iq-select-radio] unlocked", 1);
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                var self = this,
                    changed = (self.vars.value !== value);

                self.vars.value = value;

                const optionIdx = self.config.options.findIndex((option) => option.value === value);
                if (optionIdx !== -1) {
                    self.setSelected(self.vars.optionEls[optionIdx]);
                } else {
                    self.setSelected(null);
                }

                shmi.log("[IQ:iq-select-radio] value set: " + value, 1);
                if (changed) {
                    this.fire('change', { value: self.vars.value });
                }
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                return this.vars.value;
            },

            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(min, max, step) {
                var self = this;
                self.imports.nv.setProperties(self, arguments);
            },

            setLabel: function(labelText) {
                var self = this;

                if (self.vars.labelEl && self.config['auto-label']) {
                    setLabelImpl(self, labelText);
                }
            },

            /**
             * (Re-)Creates the widgets options elements.
             *
             * @param {boolean} enableListeners Whether or not to enable listeners attached to the option elements.
             */
            rebuildOptions: function(enableListeners) {
                var self = this;

                // Remove container listeners
                self.vars.listeners.forEach(function(listener) {
                    listener.disable();
                });
                self.vars.listeners = [];

                // We're going to build the dynamic option list by cloning the option element in the template which we have removed from the DOM before
                const containerEl = self.vars.optionContainerEl,
                    optionTemplate = self.vars.optionEl;

                // Clear content
                self.vars.optionEls = [];
                containerEl.innerHTML = '';

                // Add options one-by-one to the container
                self.config.options.forEach(function(option, index) {
                    const newOption = optionTemplate.cloneNode(true);
                    newOption.dataset.value = option.value;
                    newOption.dataset.no = index;

                    const labelEl = shmi.getUiElement('option-label', newOption);
                    if (option.label && labelEl) {
                        labelEl.textContent = shmi.localize(option.label);
                    } else {
                        shmi.addClass(newOption, "no-label");
                    }

                    // Prefer icon over icon class
                    const iconEl = shmi.getUiElement('option-icon', newOption);
                    if (option['icon-src'] && iconEl) {
                        iconEl.style.backgroundImage = `url(${option['icon-src']})`;
                    } else if (option['icon-class'] && iconEl) {
                        shmi.addClass(iconEl, option['icon-class']);
                    } else {
                        shmi.addClass(newOption, "no-icon");
                    }

                    if (option['tooltip'] && iconEl) {
                        iconEl.setAttribute("title", shmi.localize(option['tooltip']));
                    }

                    containerEl.appendChild(newOption);

                    self.vars.optionEls.push(newOption);
                });

                // Event listeners
                var eventFunctions = {
                    onPress: function onPress(x, y, e) {
                        shmi.addClass(e.currentTarget, 'pressed');
                    },

                    onRelease: function onRelease(x, y, e) {
                        self.vars.optionEls.forEach((element) => {
                            shmi.removeClass(element, 'pressed');
                        });
                    },

                    onClick: function onClick(x, y, e) {
                        var targetEl = e.target,
                            useTargetEl = null,
                            changed = false,
                            value = null;

                        if (typeof targetEl.dataset.ui !== 'undefined' && targetEl.dataset.ui === 'option') {
                            // We already have the correct option element!
                            useTargetEl = targetEl;
                        } else {
                            // We have to traverse the tree up until we find it
                            useTargetEl = targetEl.closest('[data-ui="option"]');
                        }
                        if (!useTargetEl) {
                            console.error(self.uiType, "Could not match clicked option");
                        } else if (typeof useTargetEl.dataset.no === 'undefined') {
                            // Should not happen as we add it ourselves!
                            console.error(self.uiType, "Could not match clicked option (data-no missing!?)");
                            console.log(useTargetEl);
                        } else {
                            value = self.config.options[parseInt(useTargetEl.dataset.no)];
                            changed = (value.value !== self.vars.value);
                            self.vars.value = value.value;
                            self.setSelected(useTargetEl);
                            self.updateValue();
                            if (changed) {
                                self.fire('change', { value: self.vars.value });
                            }
                        }
                    }
                };
                self.vars.listeners.push(new shmi.visuals.io.MouseListener(self.vars.optionContainerEl, eventFunctions));
                self.vars.listeners.push(new shmi.visuals.io.TouchListener(self.vars.optionContainerEl, eventFunctions));
                if (enableListeners) {
                    self.listeners.forEach(function(listener) {
                        listener.enable();
                    });
                }
            },

            /**
             * Sets the selected element, not called by VISUALS
             *
             * @param element - element to select
             */
            setSelected: function(element) {
                const self = this;

                self.vars.optionEls.forEach((el) => {
                    if (element !== el) {
                        shmi.removeClass(el, 'selected');
                    } else {
                        shmi.addClass(el, 'selected');
                    }
                });

                self.vars.selectedEl = element;
            },

            /**
             * @returns {number}
             */
            getCurrentIndex: function() {
                var self = this,
                    selectedEl = self.vars.selectedEl;

                return self.vars.optionEls.findIndex((element) => selectedEl === element);
            },

            /**
             * Can be called externally, not called by VISUALS
             * @param options
             */
            setOptions: function(options) {
                var self = this;

                self.config.options = options;

                // reset properties
                self.vars.value = -1;
                self.vars.selectedEl = null;

                self.rebuildOptions(self.active);
            },

            /**
             * Writes current value to connected data source
             */
            updateValue: function() {
                var self = this;
                if (self.config.item) {
                    shmi.visuals.session.ItemManager.writeValue(this.config.item, self.vars.value);
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * iq-select-time
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-select-time",
 *     "name": null,
 *     "template": "custom/controls/iq-select-time"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * "label": Label of the widget
 * "item": The item
 * "type": Type is INT
 * "auto-label": Whether to use the auto-label (from item)
 * "tooltip": Tooltip
 * "isUTC": Whether the timestamp given is UTC time (and not local time)
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    var className = "iq-select-time", // control name in camel-case
        uiType = "iq-select-time", // control keyword (data-ui)
        isContainer = false;

    // default configuration
    var defConfig = {
        "class-name": "iq-select-time",
        "name": null,
        "template": "default/iq-select-time.iq-variant-01",
        "label": '[Label]',
        "item": null,
        "auto-label": true,
        "numpad-enabled": false,
        "tooltip": null,
        "isUTC": false
    };

    // setup module-logger
    var ENABLE_LOGGING = true,
        RECORD_LOG = false;
    var logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG);
    var fLog = logger.fLog,
        log = logger.log;

    /**
     * Sets the label text and handles toggling the `no-label` style.
     *
     * @param {*} self Reference to the widget
     * @param {?string} labelText Label text to set
     */
    function setLabelImpl(self, labelText) {
        if (!self.vars.labelEl) {
            // Nothing to do.
        } else if (labelText === "" || labelText === null || typeof labelText === "undefined") {
            self.vars.label = "";
            self.vars.labelEl.textContent = "";
            shmi.addClass(self.element, "no-label");
        } else {
            self.vars.label = labelText;
            self.vars.labelEl.textContent = shmi.localize(labelText);
            shmi.removeClass(self.element, "no-label");
        }
    }

    function setInputDisabledState(self, disabled) {
        if (self.vars.hourEl) {
            self.vars.hourEl.disabled = disabled;
        }

        if (self.vars.minuteEl) {
            self.vars.minuteEl.disabled = disabled;
        }

        if (self.vars.secondEl) {
            self.vars.secondEl.disabled = disabled;
        }
    }

    // Definition of new control extending BaseControl - START
    var definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,

        /* instance variables */
        vars: {
            listeners: [],
            value: 0,
            active: false,
            label: null,
            subscriptionTargetId: null,

            // DOM Elements
            labelEl: null,
            hourEl: null,
            minuteEl: null,
            secondEl: null,

            hours: 0,
            minutes: 0,
            seconds: 0,
            evListeners: [],

            itemNeedsString: false
        },

        /* imports added at runtime */
        imports: {
            im: "visuals.session.ItemManager"
        },

        /* array of custom event types fired by this control */
        events: ["change"],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function() {
                var self = this;

                /***********/
                /*** DOM ***/
                /***********/
                self.vars.labelEl = shmi.getUiElement('label', self.element);

                self.setupRockerComponent({
                    varName: "hourEl",
                    uiElementName: 'hour-select',
                    inputValidator: self.validateGeneric.bind(this, /^[0-1]?[0-9]$|^2[0-3]$/, 'hours'),
                    addHandler: self.addHours.bind(self),
                    max: 23,
                    label: '${V_HOURS}'
                });

                self.setupRockerComponent({
                    varName: "minuteEl",
                    uiElementName: 'minute-select',
                    inputValidator: self.validateGeneric.bind(this, /^[0-5]?[0-9]$/, 'minutes'),
                    addHandler: self.addMinutes.bind(self),
                    max: 59,
                    label: '${V_MINUTES}'
                });

                self.setupRockerComponent({
                    varName: "secondEl",
                    uiElementName: 'second-select',
                    inputValidator: self.validateGeneric.bind(this, /^[0-5]?[0-9]$/, 'seconds'),
                    addHandler: self.addSeconds.bind(self),
                    max: 59,
                    label: '${V_SECONDS}'
                });

                self.updateInputFields(self.vars);

                // Label
                setLabelImpl(self, self.config.label);
            },

            /* called when control is enabled */
            onEnable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                if (self.config.item) {
                    self.vars.subscriptionTargetId = self.imports.im.subscribeItem(self.config.item, self);
                }

                setInputDisabledState(self, false);

                self.log('Enabled', 1);
            },

            /* called when control is disabled */
            onDisable: function() {
                var self = this;

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                if (self.config.item) {
                    self.imports.im.unsubscribeItem(self.config.item, self.vars.subscriptionTargetId);
                }

                setInputDisabledState(self, true);

                self.log('Disabled', 1);
            },

            /* called when control is locked - disable mouse- and touch-listeners */
            onLock: function() {
                var self = this;

                setInputDisabledState(self, true);

                self.vars.listeners.forEach(function(l) {
                    l.disable();
                });

                shmi.addClass(self.element, 'locked');

                self.log('Locked', 1);
            },

            /* called when control is unlocked - enable mouse- and touch-listeners */
            onUnlock: function() {
                var self = this;

                setInputDisabledState(self, false);

                self.vars.listeners.forEach(function(l) {
                    l.enable();
                });

                shmi.removeClass(self.element, 'locked');

                self.log('Unlocked', 1);
            },

            onDelete: function() {
                var self = this;

                self.vars.evListeners.forEach(function(info) {
                    info.element.removeEventListener(info.eventType, info.listener);
                });
                self.vars.evListeners = [];
                self.vars.listeners = [];
            },

            /* called by ItemManager when subscribed item changes and once on initial subscription */
            onSetValue: function(value, type, name) {
                var self = this;

                value = parseInt(value);
                var date = new Date(value * 1000);

                self.vars.itemNeedsString = (typeof (value) === 'string');

                if (self.config.isUTC) {
                    self.vars.hours = date.getUTCHours();
                    self.vars.minutes = date.getUTCMinutes();
                    self.vars.seconds = date.getUTCSeconds();
                } else {
                    self.vars.hours = date.getHours();
                    self.vars.minutes = date.getMinutes();
                    self.vars.seconds = date.getSeconds();
                }

                self.updateInputFields(self.vars);

                if (self.vars.value !== value) {
                    self.vars.value = value;
                    self.fire('change', { value: value });
                }
            },

            /**
             * Retrieves current value of the Rocker Button
             *
             * @return value - current value
             */
            getValue: function() {
                var self = this;

                return self.vars.value;
            },

            /** Returns the number of hours. */
            getHours: function() {
                var self = this;

                return self.vars.hours;
            },

            /** Returns the number of minutes. */
            getMinutes: function() {
                var self = this;

                return self.vars.minutes;
            },

            /** Returns the number of seconds. */
            getSeconds: function() {
                var self = this;

                return self.vars.seconds;
            },

            /* Sets min & max values and stepping of subscribed variable */
            onSetProperties: function(min, max, step, name, type, warnmin, warnmax, prewarnmin, prewarnmax, precision) {

            },

            setLabel: function(labelText) {
                var self = this;

                if (self.vars.labelEl && self.config['auto-label']) {
                    setLabelImpl(self, labelText);
                }
            },

            log: function(msg, level) {
                shmi.log('[IQ:iq-select-time] '+msg, level);
            },

            getType: function(v) {
                if (Array.isArray(v)) {
                    return 'array';
                } else if (v === null) {
                    return 'null';
                } else {
                    return typeof (v);
                }
            },

            /**
             * Checks whether or not the given value is of one of the expected types.
             *
             * @param {*} value Value to check
             * @param {string|string[]} expected Expected type or types of the provided value.
             */
            checkType: function(value, expected) {
                var self = this;

                if (Array.isArray(expected)) {
                    if (!(value in expected)) {
                        return false;
                    }
                } else if (expected !== self.getType(value)) {
                    return false;
                }

                return true;
            },

            /**
             * Throws an exception if the given value is not of the provided type.
             *
             * @param {*} value Value to check
             * @param {string|string[]} expected Expected type or types of the provided value.
             * @param {string} [message] Error message
             *
             * @throws {TypeError}
             */
            assertType: function(value, expected, message) {
                var self = this;

                if (!self.checkType(value, expected)) {
                    throw TypeError(message);
                }
            },

            /**
             * Compute new control value from the controls current state.
             */
            computeNewValue: function() {
                var self = this,
                    vars = self.vars,
                    date = new Date(vars.value * 1000);

                if (self.config.isUTC) {
                    date.setUTCHours(vars.hours);
                    date.setUTCMinutes(vars.minutes);
                    date.setUTCSeconds(vars.seconds);
                } else {
                    date.setHours(vars.hours);
                    date.setMinutes(vars.minutes);
                    date.setSeconds(vars.seconds);
                }

                return date.getTime() / 1000;
            },

            /**
             * Compute the controls new value, update its items' value and fire a
             * `change` event if the value actually changed.
             */
            updateValue: function() {
                var self = this;

                var newValue = self.computeNewValue();

                if (self.config.item) {
                    // Make sure we write a string to the item if it was a string before
                    if (self.vars.itemNeedsString) {
                        self.imports.im.writeValue(self.config.item, String(newValue));
                    } else {
                        self.imports.im.writeValue(self.config.item, newValue);
                    }
                }

                // Only fire a 'changed' event if the value actually changed.
                if (self.vars.value !== newValue) {
                    self.vars.value = newValue;
                    self.fire('change', { value: newValue });
                }

                self.updateInputFields(self.vars);
            },

            /**
             * @param {int} val Integer to convert to a number within the group
             * @param {int} N Modulo of the group
             */
            integerToGroup: function(val, N) {
                return (N + (val % N)) % N;
            },

            /**
             * Adds seconds to the current time. Does overflow around into minutes.
             *
             * @param {int} x Number of seconds to add
             */
            addSeconds: function(x) {
                var self = this;

                self.assertType(x, 'number');

                if (self.vars.secondEl) {
                    self.vars.secondEl.blur();
                }
                self.vars.seconds = self.integerToGroup(self.vars.seconds + x, 60);

                self.updateValue();
            },

            /**
             * Adds minutes to the current time. Does overflow around into hours.
             *
             * @param {int} x Number of minutes to add
             */
            addMinutes: function(x) {
                var self = this;

                self.assertType(x, 'number');

                if (self.vars.minuteEl) {
                    self.vars.minuteEl.blur();
                }

                self.vars.minutes = self.integerToGroup(self.vars.minutes + x, 60);
                self.updateValue();
            },

            /**
             * Adds hours to the current time. Does overflow around into days.
             *
             * @param {int} x Number of hours to add
             */
            addHours: function(x) {
                var self = this;

                self.assertType(x, 'number');

                if (self.vars.hourEl) {
                    self.vars.hourEl.blur();
                }

                self.vars.hours = self.integerToGroup(self.vars.hours + x, 24);

                self.updateValue();
            },

            /**
             * Validates and stores the nodes value. If the nodes value is invalid, it
             * is overwritten with the last known-to-be-good value.
             *
             * @param {RegExp} regex A regex that has to match the elements textContent
             * @param {string} timecomponent Component to verify. Can be 'seconds', 'minutes' or 'hours'
             * @param {Node} element Node of the input field to validate
             */
            validateGeneric: function(regex, timecomponent, element) {
                var self = this;

                if (RegExp(regex).test(element.value)) {
                    self.vars[timecomponent] = parseInt(element.value);
                    self.updateValue(self);
                    if (self.vars[timecomponent] < 10 && element.value.length === 1) {
                        element.value = '0' + element.value;
                    }
                } else if (self.vars[timecomponent] < 10) {
                    element.value = '0' + self.vars[timecomponent];
                } else {
                    element.value = self.vars[timecomponent];
                }
            },

            /**
             * Keypress listener for h/m/s input fields.
             *
             * @param {Node} element Node the listener is attached to
             * @param {Event} event The event
             */
            inputKeypressListener: function(element, event) {
                if (Number(event.keyCode) === 13) {
                    event.preventDefault();
                    window.getSelection().removeAllRanges();
                    element.blur();
                }
            },

            /**
             * Keyup listener for h/m/s input fields.
             *
             * @param {Node} element Node the listener is attached to
             * @param {Event} event The event
             */
            inputChangeListener: function(element, event) {
                if (Number(event.keyCode) === 13) {
                    // Don't handle enter/return key
                } else if (element.value.length > 1) {
                    window.getSelection().removeAllRanges();
                    element.blur();
                }
            },

            /**
             * Blur listener for h/m/s input fields.
             *
             * @param {Node} element Node the listener is attached to
             * @param {function} validateFunc
             * @param {Event} event The event
             */
            inputBlurListener: function(element, validateFunc, event) {
                window.getSelection().removeAllRanges();
                shmi.removeClass(element, 'selectableText');
                validateFunc(element);
            },

            /**
             * Attaches mouse and touch listeners to a given node. The listeners
             * references are stored in the controls variable store and are enabled
             * or disabled by the control.
             *
             * @param {object} vars Reference to the controls variable storage
             * @param {Node} element Element to attach the listeners to
             * @param {Function} functions Listener callback
             */
            addClickTouchListener: function(vars, element, functions) {
                vars.listeners.push(new shmi.visuals.io.MouseListener(element, functions));
                vars.listeners.push(new shmi.visuals.io.TouchListener(element, functions));
            },

            /**
             * Attach an event listener to a given node. The listeners are detached
             * when the control is destroyed.
             *
             * @param {object} vars Reference to the controls variable storage
             * @param {Node} element Element to attach the event listener to
             * @param {string} eventType Type of the event to attach the listener to.
             * @param {Function} listener Listener callback
             */
            addEventListener: function(vars, element, eventType, listener) {
                element.addEventListener(eventType, listener);

                vars.evListeners.push({
                    element: element,
                    eventType: eventType,
                    listener: listener
                });
            },

            /**
             * Creates input field on specified elemnt
             *
             * @param {object} vars Reference to the controls variable storage
             * @param {Node} element Element for input field
             * @param {Function} validateFunc Validation function
             * @param {object} numpadConfig Numpad configuration
             */
            createInputField: function(vars, element, validateFunc, numpadConfig) {
                var self = this;

                if (!element) {
                    self.log('no base element provided', 1);
                    return;
                }

                var appConfig = shmi.requires("visuals.session.config"),
                    keyboardEnabled = (appConfig.keyboard && appConfig.keyboard.enabled); // get the keyboard config from `project.json`

                numpadConfig = shmi.cloneObject(numpadConfig);
                numpadConfig.callback = function(res) {
                    element.value = res;
                    validateFunc(element);
                };

                self.addClickTouchListener(vars, element, {
                    onPress: (x, y, event) => {
                        if (numpadConfig.numpadEnabled || keyboardEnabled) {
                            event.preventDefault();
                            event.stopPropagation();
                        }
                    },
                    onClick: function onClick(x, y, event) {
                        if (numpadConfig.numpadEnabled || keyboardEnabled) {
                            numpadConfig.value = element.value;
                            shmi.numpad(numpadConfig);
                        }
                    }
                });

                self.addEventListener(vars, element, 'keypress', self.inputKeypressListener.bind(null, element));
                self.addEventListener(vars, element, 'input', self.inputChangeListener.bind(null, element));
                self.addEventListener(vars, element, 'blur', self.inputBlurListener.bind(null, element, validateFunc));
                self.addEventListener(vars, element, 'focus', (event) => {
                    if (numpadConfig.numpadEnabled || keyboardEnabled) {
                        numpadConfig.value = element.value;
                        shmi.numpad(numpadConfig);
                    } else {
                        element.select();
                    }
                });
            },

            /**
             * Adds listeners to a "rocker component" (+/- buttons and input field)
             *
             * @param {object} configuration Rocker component configuration
             */
            setupRockerComponent: function(configuration) {
                var self = this;

                var elementComponent = shmi.getUiElement(configuration.uiElementName, self.element);
                if (!elementComponent) {
                    self.log('Element not found: ' + configuration.uiElementName, 1);
                    return false;
                }

                var elementPrev = shmi.getUiElement('previous', elementComponent);
                if (!elementPrev) {
                    self.log('No UI element found for ' + configuration.uiElementName + '.previous', 1);
                    return false;
                }

                var elementCurrent = shmi.getUiElement('current', elementComponent);
                if (!elementCurrent) {
                    self.log('No UI element found for ' + configuration.uiElementName + '.current', 1);
                    return false;
                }

                var elementNext = shmi.getUiElement('next', elementComponent);
                if (!elementNext) {
                    self.log('No UI element found for ' + configuration.uiElementName + '.next', 1);
                    return false;
                }

                self.createInputField(
                    self.vars,
                    elementCurrent,
                    configuration.inputValidator.bind(self),
                    {
                        min: 0,
                        max: configuration.max,
                        label: configuration.label,
                        numpadEnabled: self.config["numpad-enabled"] || false
                    }
                );

                self.addClickTouchListener(self.vars, elementPrev, {
                    onPress: shmi.addClass.bind(null, elementPrev, 'pressed'),
                    onRelease: shmi.removeClass.bind(null, elementPrev, 'pressed'),
                    onClick: configuration.addHandler.bind(null, -1)
                });

                self.addClickTouchListener(self.vars, elementNext, {
                    onPress: shmi.addClass.bind(null, elementNext, 'pressed'),
                    onRelease: shmi.removeClass.bind(null, elementNext, 'pressed'),
                    onClick: configuration.addHandler.bind(null, 1)
                });

                self.vars[configuration.varName] = elementCurrent;

                return true;
            },

            updateInputFields: function(vars) {
                var self = this;

                if (self.vars.hourEl) {
                    self.vars.hourEl.value = (vars.hours > 9) ? vars.hours : ('0' + vars.hours);
                }
                if (self.vars.minuteEl) {
                    self.vars.minuteEl.value = (vars.minutes > 9) ? vars.minutes : ('0' + vars.minutes);
                }
                if (self.vars.secondEl) {
                    self.vars.secondEl.value = (vars.seconds > 9) ? vars.seconds : ('0' + vars.seconds);
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    var cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * WebIQ visuals shape widget.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-shape",
 *     "name": null,
 *     "template": "default/iq-shape",
 *     "polygons": [],
 *     "rotation": 0,
 *     "scale": 1
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 * polygons {number[][]}: Array of 2d vectors used as polygons. Must contain at least 3 vectors in polygon mode.
 * rotation {number}: Rotation in degrees to apply to the shape.
 * scale {number}: Scaling factor to apply to the shape.
 *
 * @version 1.1
 */
(function() {
    'use strict';

    // variables for reference in control definition
    const className = "IqShape", // control name in camel-case
        uiType = "iq-shape", // control keyword (data-ui)
        isContainer = false;

    // example - default configuration
    const defConfig = {
        "class-name": "iq-shape",
        "name": null,
        "template": "default/iq-shape",
        "polygons": [],
        "rotation": 0,
        "scale": 1
    };

    // declare private functions - START

    /**
     * Calculates a rotation matrix for a given angle.
     *
     * @param {number} rotation Rotation angle in radiants
     * @returns {number[][]} 2D rotation matrix for the given angle as array of arrays
     */
    function calculateRotationMatrix2D(rotation) {
        return [
            [Math.cos(rotation), -Math.sin(rotation)],
            [Math.sin(rotation), Math.cos(rotation)]
        ];
    }

    /**
     * Returns the sum of two 2D vectors.
     *
     * @param {number[]} vector
     * @param {number[]} translate
     * @returns {number[][]}
     */
    function translateVector2D(vector, translate) {
        return [vector[0] + translate[0], vector[1] + translate[1]];
    }

    /**
     * Returns the result of the matrix multiplication of M x V.
     *
     * @param {number[]} vector
     * @param {number[][]} matrix
     * @returns {number[]}
     */
    function rotateVector2D(vector, matrix) {
        return [
            matrix[0][0] * vector[0] + matrix[0][1] * vector[1],
            matrix[1][0] * vector[0] + matrix[1][1] * vector[1]
        ];
    }

    /**
     * Rotates a vector using [0.5, 0.5] as origin.
     *
     * @param {number[]} vector
     * @param {number[][]} matrix
     * @returns {number[]}
     */
    function rotateVector2DCenter(vector, matrix) {
        return translateVector2D(
            rotateVector2D(
                translateVector2D(vector, [-0.5, -0.5]),
                matrix
            ),
            [0.5, 0.5]
        );
    }

    function scaleVector2D(vector, scale) {
        return [vector[0] * scale, vector[1] * scale];
    }

    /**
     * Scales a vector using [0.5, 0.5] as origin.
     *
     * @param {number[]} vector
     * @param {number[][]} matrix
     * @returns {number[]}
     */
    function scaleVector2DCentered(vector, scale) {
        return translateVector2D(
            scaleVector2D(
                translateVector2D(vector, [-0.5, -0.5]),
                scale
            ),
            [0.5, 0.5]
        );
    }

    /**
     * Creates a point-string as understood by the <polygon>-tag from the given
     * polygons.
     *
     * @param {number} width Width of the SVG
     * @param {number} height Height of the SVG
     * @param {number[][]} polygons Polygons
     * @param {number} strokeWidth
     * @returns {string}
     */
    function makePolygonPoints(width, height, polygons) {
        return polygons.map(([x, y]) => `${x * width},${y * height}`).join(" ");
    }

    /**
     * Returns an object with elements.
     *
     * @param {string[]} elements UI Elements to get.
     * @param {Element} baseElement Root node for the search.
     * @returns {Object}
     */
    function getUiElements(elements, baseElement) {
        const obj = {};
        const allElementsPresent = elements.every((elementName) => {
            const element = shmi.getUiElement(elementName, baseElement);
            if (!element) {
                console.error(`[${className}] "${elementName}" element is missing in template.`);
                return false;
            }

            obj[elementName] = element;
            return true;
        });

        return allElementsPresent ? obj : null;
    }
    // declare private functions - END

    // definition of new control extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {},
        /* imports added at runtime */
        imports: {},

        /* array of custom event types fired by this control */
        events: [],

        /* functions to extend or override the BaseControl prototype */
        prototypeExtensions: {
            /* called when config-file (optional) is loaded and template (optional) is inserted into base element */
            onInit: function onInit() {
                const svgElement = shmi.getUiElement("shape", this.element),
                    rotMatrix = calculateRotationMatrix2D((this.config.rotation || 0) / 360 * 2 * Math.PI);

                if (!svgElement) {
                    console.error(`[${className}] "shape" element is missing in template.`);
                    return;
                }

                const elements = getUiElements(["shape-polygon", "clip-path", "clip-path-polygon"], svgElement);
                if (!elements) {
                    return;
                }

                // Setup "unique" IDs for the svg.
                const randomId = Math.floor(Math.random() * Number.MAX_SAFE_INTEGER);
                Object.entries(elements).forEach(([key, el]) => el.setAttribute("id", `${key}-${randomId}`));
                elements["shape-polygon"].setAttribute("clip-path", `url(#clip-path-${randomId})`);

                // Compute polygons.
                const polygons = (this.config.polygons || []).map((poly) => scaleVector2DCentered(rotateVector2DCenter(poly, rotMatrix), this.config.scale / 100)),
                    points = makePolygonPoints(svgElement.viewBox.baseVal.width, svgElement.viewBox.baseVal.height, polygons);

                // Set attributes.
                switch (elements["shape-polygon"].tagName) {
                case "polygon":
                    elements["shape-polygon"].setAttribute("points", points);
                    elements["clip-path-polygon"].setAttribute("points", points);
                    break;
                default:
                }
            }
        }
    };

    // definition of new control extending BaseControl - END

    // generate control constructor & prototype using the control-generator tool
    const cg = shmi.requires("visuals.tools.control-generator");
    cg.generate(definition);
})();

/**
 * WebIQ visuals iq-slider control.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "iq-slider",
 *     "name": null,
 *     "template": "default/iq-slider.iq-variant-01"
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default css class applied on control root element
 * name {string}: Name of control set to data-name attribute
 * template {string}: Path to template file
 *
 * @version 1.0
 */
(function() {
    'use strict';

    // variables for reference in control definition
    const className = "IqSlider", // control name in camel-case
        uiType = "iq-slider", // control keyword (data-ui)
        isContainer = false;

    // example - default configuration
    const defConfig = {
            "class-name": uiType,
            "name": null,
            "template": "default/iq-slider.iq-variant-01",
            "send-interval": 100,
            "animation-duration": 100,
            "continuous": true,
            "min": 0,
            "max": 100,
            "step": 0,
            "precision": -1,
            "type": shmi.c("TYPE_FLOAT"),
            "label": "[Label]",
            "unit-text": "[Unit]",
            "decimal-delimiter": ".",
            "auto-min": true,
            "auto-max": true,
            "auto-step": true,
            "auto-label": true,
            "auto-unit-text": true,
            "auto-type": true,
            "auto-precision": true,
            "initial-text": "-",
            "show-text": true,
            "show-icon": false,
            "icon-src": null,
            "icon-class": null,
            "inverted": false
        },
        cssClasses = {
            NO_LABEL: "no-label",
            NO_ICON: "no-icon",
            NO_UNIT: "no-unit",
            LOCKED: "locked",
            PRESSED: "pressed",
            INVERTED: "invert-slider"
        },
        orientations = {
            VERTICAL: "vertical",
            HORIZONTAL: "horizontal"
        };

    // declare private functions - START

    /**
     * setLabel - set label text for widget. applies "no-label" CSS class when no text is specified
     *
     * @param {object} self control instance
     * @param {string} [value] text value
     */
    function setLabel(self, value) {
        const { elements } = self.vars;

        if (typeof value === "string" && value.length) {
            elements.label.textContent = shmi.localize(value);
            shmi.removeClass(self.element, cssClasses.NO_LABEL);
        } else {
            elements.label.textContent = "";
            shmi.addClass(self.element, cssClasses.NO_LABEL);
        }
    }

    /**
     * setUnit - set unit text for widget. applies "no-unit" CSS class when no text is specified
     *
     * @param {object} self control instance
     * @param {string} [value] text value
     */
    function setUnit(self, value) {
        const { elements } = self.vars;

        if (typeof value === "string" && value.length) {
            elements.unit.textContent = shmi.localize(value);
            shmi.removeClass(self.element, cssClasses.NO_UNIT);
        } else {
            elements.unit.textContent = "";
            shmi.addClass(self.element, cssClasses.NO_UNIT);
        }
    }

    /**
     * initIcon - initialize icon element according to widget configuration
     *
     * @param {object} self control instance
     */
    function initIcon(self) {
        const { elements } = self.vars,
            config = self.getConfig();

        if (!elements.icon) {
            shmi.addClass(self.element, cssClasses.NO_ICON);
        } else if (config["show-icon"] && config["icon-src"]) {
            elements.icon.style.backgroundImage = `url(${self.config['icon-src']})`;
        } else if (config["show-icon"] && config["icon-class"]) {
            shmi.addClass(elements.icon, config["icon-class"]);
        } else {
            shmi.addClass(self.element, cssClasses.NO_ICON);
        }
    }

    /**
     * updateDimensions - recalculate widget dimensions
     *
     * @param {object} self control instance
     */
    function updateDimensions(self) {
        const { dims, elements, orientation } = self.vars;

        switch (orientation) {
        case orientations.HORIZONTAL:
            dims.track = elements.track.clientWidth;
            dims.handle = elements.handle.offsetWidth;
            dims.touchzone = elements.touchzone.offsetWidth;
            break;
        case orientations.VERTICAL:
            dims.track = elements.track.clientHeight;
            dims.handle = elements.handle.offsetHeight;
            dims.touchzone = elements.touchzone.offsetHeight;
            break;
        default:
        }

        updateHandle(self, getTranslation(self, self.getValue()));
    }

    /**
     * getInputHandler - create handler for Mouse-/Touch-Listeners
     *
     * @param {object} self control instance
     * @returns {object} input handler
     */
    function getInputHandler(self) {
        return {
            onPress: function() {
                self.vars.dragging = false;
                shmi.addClass(self.vars.elements.handle, cssClasses.PRESSED);
            },
            onDrag: function(dx, dy, event) {
                event.preventDefault();

                const { movable, dims } = self.vars;

                if (!self.vars.dragging) {
                    self.vars.dragging = true;
                    if (self.config.continuous !== false) {
                        startSend(self);
                    }
                }
                if (self.vars.orientation === orientations.VERTICAL) {
                    if ((movable.ty + dy) > 0) {
                        movable.ty = 0;
                        movable.update();
                    } else if ((movable.ty + dy) < Math.min(0, dims.handle - dims.track)) {
                        movable.ty = Math.min(0, dims.handle - dims.track);
                        movable.update();
                    } else {
                        movable.translate(0, dy);
                    }
                    updateScale(self, movable.ty);
                    updateTouchZone(self, movable.ty);
                } else {
                    if ((movable.tx + dx) < 0) {
                        movable.tx = 0;
                        movable.update();
                    } else if ((movable.tx + dx + dims.handle) > dims.track) {
                        movable.tx = Math.max(0, dims.track - dims.handle);
                        movable.update();
                    } else {
                        movable.translate(dx, 0);
                    }
                    updateScale(self, movable.tx);
                    updateTouchZone(self, movable.tx);
                }
            },
            onRelease: function() {
                self.vars.dragging = false;
                stopSend(self);
                updateValue(self);
                shmi.removeClass(self.vars.elements.handle, cssClasses.PRESSED);
            }
        };
    }

    /**
     * getInputHandler - create handler for Mouse-/Touch-Listeners for the slider track.
     *
     * @param {object} self control instance
     * @returns {object} input handler
     */
    function getSliderTrackInputHandler(self) {
        const handler = getInputHandler(self),
            originalOnPress = handler.onPress,
            originalOnRelease = handler.onRelease;

        return Object.assign(handler, {
            onPress: function onPress(startX, startY, event) {
                const { movable, dims } = self.vars,
                    { min, max } = self.vars.valueSettings;

                // Prevent animations
                self.vars.dragging = true;

                // Prevents pointer events for the actual handle and applies
                // its hover style.
                shmi.addClass(self.vars.elements.track, cssClasses.PRESSED);

                // Set handle position
                const boundingBox = self.vars.elements.track.getBoundingClientRect();
                if (self.vars.orientation === orientations.VERTICAL) {
                    const posY = startY - boundingBox.y;
                    if (self.config.inverted) {
                        movable.ty = getTranslation(self, posY / dims.track * (max - min) + min);
                    } else {
                        movable.ty = getTranslation(self, (1 - posY / dims.track) * (max - min) + min);
                    }
                    updateScale(self, movable.ty);
                    updateTouchZone(self, movable.ty);
                } else {
                    const posX = startX - boundingBox.x;
                    if (self.config.inverted) {
                        movable.tx = getTranslation(self, (1 - posX / dims.track) * (max - min) + min);
                    } else {
                        movable.tx = getTranslation(self, posX / dims.track * (max - min) + min);
                    }
                    updateScale(self, movable.tx);
                    updateTouchZone(self, movable.tx);
                }
                movable.update();

                // Update widget
                updateValue(self);

                // Call original onPress
                originalOnPress.call(handler, startX, startY, event);
            },

            onRelease: function onRelease(...args) {
                // Call original onRelease
                originalOnRelease.call(handler, ...args);

                // Remove fake hover style and enable pointer events on the
                // handle again.
                shmi.removeClass(self.vars.elements.track, cssClasses.PRESSED);

                // Focus handle
                self.vars.elements.handle.focus();
            }
        });
    }

    /**
     * startSend - start sending value updates while slider handle is dragging
     *
     * @param {object} self control instance
     */
    function startSend(self) {
        if (self.vars.sendInterval === 0) {
            self.vars.sendInterval = setInterval(updateValue.bind(null, self), self.getConfig()["send-interval"]);
        }
    }

    /**
     * stopSend - stop sending value updates when slider handle stops dragging
     *
     * @param {object} self control instance
     */
    function stopSend(self) {
        if (self.vars.sendInterval !== 0) {
            clearInterval(self.vars.sendInterval);
            self.vars.sendInterval = 0;
        }
    }

    /**
     * updateValue - update widget value after handle is moved
     *
     * @param {object} self control instance
     */
    function updateValue(self) {
        const { nv, im } = self.imports,
            { valueSettings, orientation, movable, dims, elements } = self.vars,
            { min, max } = valueSettings,
            config = self.getConfig(),
            lastValue = self.getValue(),
            ds = (orientation === orientations.VERTICAL) ? -movable.ty : movable.tx,
            maxTranslation = dims.track - dims.handle;

        let value = null;

        if (maxTranslation <= 0) {
            return; /* value cannot be calculated when handle is not movable */
        } else if (max === min) {
            value = max;
        } else if (self.config.inverted) {
            value = (1 - ds / maxTranslation) * (max - min) + min;
        } else {
            value = (ds * (max - min) / maxTranslation) + min;
        }

        value = nv.applyInputSettings(value, self);

        if (value !== lastValue) {
            if (elements.value) {
                elements.value.textContent = nv.formatOutput(value, self);
            }
            self.vars.value = value;
            if (config.item) {
                im.writeValue(config.item, value);
            } else {
                self.fire("change", {
                    value: value
                });
            }
        }
    }

    /**
     * updateHandle - update handle position to match current value
     *
     * @param {object} self control instance
     * @param {number} translation handle translation
     * @returns
     */
    function updateHandle(self, translation) {
        if (isNaN(translation)) {
            return;
        }

        if (self.vars.orientation === orientations.VERTICAL) {
            self.vars.movable.ty = translation;
        } else {
            self.vars.movable.tx = translation;
        }
        self.vars.movable.update();
        updateScale(self, translation);
        updateTouchZone(self, translation);
    }

    /**
     * updateScale - update value scale to match current value
     *
     * @param {object} self control instance
     * @param {number} translation handle translation
     */
    function updateScale(self, translation) {
        let trackLength = self.vars.dims.track - self.vars.dims.handle,
            ratio = (trackLength > 0) ? Math.abs(translation) / trackLength : 1;

        const { elements: { scale } } = self.vars;

        scale.style.setProperty("--internal-fill-level", `${ratio * 100}%`);
    }

    /**
     * updateTouchZone - update touch zone element to remain within widget container boundaries
     *
     * @param {object} self
     * @param {number} translation handle translation
     */
    function updateTouchZone(self, translation) {
        const { MIN_MOVED_PX } = shmi.Constants,
            { dims, orientation, elements } = self.vars,
            maxTranslation = Math.max(0, dims.track - dims.handle);

        shmi.caf(self.vars.touchzoneRafId);
        self.vars.touchzoneRafId = shmi.raf(() => {
            translation = Math.abs(translation);

            if (orientation === orientations.VERTICAL) {
                if (translation < MIN_MOVED_PX) {
                    elements.touchzone.style.height = `calc(100% + ${Math.floor(translation) + MIN_MOVED_PX}px)`;
                    if (self.vars.touchzoneState.offsetModified) {
                        elements.touchzone.style.top = "";
                        self.vars.touchzoneState.offsetModified = false;
                    }
                    self.vars.touchzoneState.lengthModified = true;
                } else if (translation > maxTranslation - MIN_MOVED_PX) {
                    elements.touchzone.style.height = `calc(100% + ${Math.floor(maxTranslation - translation) + MIN_MOVED_PX}px)`;
                    elements.touchzone.style.top = `-${Math.floor(maxTranslation - translation)}px`;
                    self.vars.touchzoneState.lengthModified = true;
                    self.vars.touchzoneState.offsetModified = true;
                } else {
                    if (self.vars.touchzoneState.lengthModified) {
                        elements.touchzone.style.height = "";
                        self.vars.touchzoneState.lengthModified = false;
                    }
                    if (self.vars.touchzoneState.offsetModified) {
                        elements.touchzone.style.top = "";
                        self.vars.touchzoneState.offsetModified = false;
                    }
                }
            } else if (translation < MIN_MOVED_PX) {
                elements.touchzone.style.width = `calc(100% + ${Math.floor(translation) + MIN_MOVED_PX}px)`;
                elements.touchzone.style.left = `-${Math.floor(translation)}px`;
                self.vars.touchzoneState.lengthModified = true;
                self.vars.touchzoneState.offsetModified = true;
            } else if (translation > maxTranslation - MIN_MOVED_PX) {
                elements.touchzone.style.width = `calc(100% + ${Math.floor(maxTranslation - translation) + MIN_MOVED_PX}px)`;
                if (self.vars.touchzoneState.offsetModified) {
                    elements.touchzone.style.left = "";
                    self.vars.touchzoneState.offsetModified = false;
                }
                self.vars.touchzoneState.lengthModified = true;
            } else {
                if (self.vars.touchzoneState.lengthModified) {
                    elements.touchzone.style.width = "";
                    self.vars.touchzoneState.lengthModified = false;
                }
                if (self.vars.touchzoneState.offsetModified) {
                    elements.touchzone.style.left = "";
                    self.vars.touchzoneState.offsetModified = false;
                }
            }
        });
    }

    /**
     * getTranslation - calculate handle translation for given value
     *
     * @param {object} self control instance
     * @param {number} value widget value
     * @returns {number} handle translation
     */
    function getTranslation(self, value) {
        const { min, max } = self.vars.valueSettings,
            { dims, orientation } = self.vars;

        let tx = null;

        if (max === min) {
            tx = Math.max(0, dims.track - dims.handle);
        } else if (self.config.inverted) {
            tx = (1 - ((value - min) / (max - min))) *
                Math.max(0, dims.track - dims.handle);
        } else {
            tx = ((value - min) / (max - min)) *
                Math.max(0, dims.track - dims.handle);
        }

        if (orientation === orientations.VERTICAL) {
            tx *= -1;
        }

        return tx;
    }

    // declare private functions - END

    // definition of new control extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        /* default configuration settings - all available options have to be initialized! */
        config: defConfig,
        /* schema of configuration object for validation according to json-schema v4
         * (http://json-schema.org/draft-04/schema#)
         */
        configSchema: null,
        /* instance variables */
        vars: {
            elements: {
                lab