/**
 * WebIQ Visuals widget template.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "cx-table", // {%= ui_type %}
 *     "name": null,
 *     "template": "custom/controls/cx-table" // {%= template_path %}
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default CSS class applied on widget root element
 * name {string}: Name of widget set to data-name attribute
 * template {string}: Path to template file
 *
 * @version 1.2 changed by boschrexroth
 * 2024-01-31 bugfix display debug vars
 *            editCell(): add client ID to PLC edit request
 *            add multi client support (edit table from multiple client at the same time)
 */
// let myData
(function () {
    'use strict';
    // variables for reference in widget definition
    const templatePath = "custom/controls/cx-table" // "{%= template_path %}"
    const uiType = "cx-table"    // widget keyword (data-ui) {%= ui_type %}
    const className = uiType     // widget name in camel case {%= constructor_name %}
    const isContainer = false

    // example - default configuration
    const defaultConfig = {
        "class-name": uiType,
        "name": null,
        "template": templatePath,
        "label": uiType,
        "item": null
    };

    // setup module-logger
    const ENABLE_LOGGING = false //enable logging with 'log' function
    const RECORD_LOG = false //store logged messages in logger memory
    let logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG)
    let fLog = logger.fLog //force log - logs message even if logger is disabled
    let log = logger.log; //log message if logger is enabled

    let mySelect = {}        // active key-value list
    const CHAR_SORT_UP = "▼"
    const CHAR_SORT_DN = "▲"
    const CHAR_SORT = "⇵" //" ⇵ ⇕ ⇳ ↕" 
    const CHAR_SORT_UNKNOWN = "❔" // ⚠ ❔❕⚫
    let cfgBtn = {
        "ui": "cx-button",
        "class-name": "cx-button",
        "name": "cx-button",
        "template": "custom/controls/cx-button",
        "heigth": "40px",
        "label": null,
        "btnType": null,
        "labelOn": null,
        "imgPosition": "hide",
    }
    // declare private functions - START
    // (CUSTOM ADDITION++)
    /**
     * creates command string and sends it
     * @param {dom} self   widget
     * @param {string}   sCmd   command to send
     * @param {int}      iId2   CopyId or column
     * @param {string}   sValue value to send
     */
    function setCmd(self, sCmd, iId1, iId2 = 0, sValue = "") {
        let sTmp = `cmd ${sCmd} ${cxt.getClientId()} ${iId1} ${iId2} ${sValue}`
        cxt.setItem(self.config.itemCmd, sTmp, true)
    }
    /**
     * updates content of existing table cells
     * @param {dom} self widget root element
     */
    async function updateCells(self) {
        const vr = self.vars
        const cf = self.config
        const el = self.vars.elements

        cxt.setItem(cf.itemCmd, '')
        let sTmp = (cf.itemData) ? await cxt.getItem(cf.itemData, true) : ""
        if (vr.bUtf8) sTmp = shmi.from_utf8(sTmp)
        const arTmp = sTmp.split(vr.FS_DROW)[0].split(vr.FS_DCOL)
        for (let i = 1; i < arTmp.length; i++) {
            mySelect = getTextList(vr.arLists[i], self)
            // because of timing problem the row migth not exist
            if (el.tabTable.rows[i + 1]) {
                if (mySelect)
                    el.tabTable.rows[i + 1].cells[2].innerText = parseTextList(arTmp[i])
                else
                    el.tabTable.rows[i + 1].cells[2].innerText = arTmp[i]
            }
        }
    }
    /**
     * create a button with a table command
     * @param {String}  sName    name of button
     * @param {String}  sLabel   label of button
     * @param {String}  sCmd     command to execute
     * @param {Boolean} bConfirm true:confirm command before execute
     * @returns 
     */
    function createButton(self, sName, sLabel, sCmd, sConfirm) {
        const el = self.vars.elements
        const vr = self.vars
        let cfg = shmi.cloneObject(cfgBtn)
        cfg.name = cfg.label = sName
        cfg.btnType = "on-off"
        cfg.labelOn = sLabel
        let btn = shmi.createControl("cx-button", el.cxInputXBtns, cfg, "DIV")
        btn.element.style.height = "40px"
        if (sCmd)
            cfg.actionUp = async () => {
                let bRes = true
                if (sConfirm !== '') bRes = await cxt.msgBox("confirm", sConfirm)
                if (bRes) setCmd(self, sCmd, vr.iIdSelected)
            }
        return cfg
    }
    /**
     * enable/disable listeners
     * @param {object}  self root element of widgets
     * @param {boolean} bEnable true:enable false:disable
     */
    function enableListeners(self, bEnable = true) {
        log("enableListeners: " + bEnable)
        if (!self.vars)
            return
        const vr = self.vars
        const el = self.vars.elements
        vr.bEnabled = bEnable
        el.txtFilter.forEach(txt => {
            if (txt !== null) {
                txt.readOnly = !bEnable
                log("txt.readOnly=" + txt.readOnly)
            }
        });
    }
    /**
     * Hides rows, which does not match filter criteria
     * @param {object}  self  root element of widget
     * @param {object}  me    clicked table cell
     * @param {integer} iFCol filter column number
     */
    function startFilter(self, me, iFCol) {
        const vr = self.vars
        const el = vr.elements
        // " " is used in search string to seperate multiple words
        if (me !== null)
            vr.arFilterStr[iFCol] = me.value.toLowerCase().split(" ")
        let iRowMax = el.tabTable.rows.length
        for (let iRow = vr.iRow1Data; iRow < iRowMax; iRow++) {
            let bOk = true
            for (let iCol = 0; iCol < el.tabTable.rows[iRow].cells.length; iCol++) {
                // for (let iCol = 0; iCol < vr.arFilterStr.length; iCol++) {
                let sTdText = el.tabTable.rows[iRow].cells[iCol].innerText
                if (!cxt.filterMatch(sTdText.toLowerCase(), vr.arFilterStr[iCol])) {
                    bOk = false
                    break
                }
            }
            el.tabTable.rows[iRow].hidden = !bOk
        }
    }
    /**
     * Switch highligthning of a row (on/off)
     * @param {number}  iRow    row number to switch mark/highlight
     * @param {boolean} bOn     true=on false=off
     * @param {boolean} bScroll true=scroll false=do not scroll for bOn=true
     * @returns {null}
     */
    function markRow(self, iRow, bOn = true, bScroll = false) {
        let vr = self.vars
        let el = vr.elements
        let jScrollOpt = { block: "center" }
        let cl = el.tabTable.rows[iRow]
        // if mark row is disabled return
        if (!vr.bMarkOn) return
        if (vr.arMarkCols.length > 0) {
            if (bOn === true)
                for (let i = 0; i < vr.arMarkCols.length; i++)
                    cl.cells[vr.arMarkCols[i]].classList.add("selected")
            else
                for (let i = 0; i < vr.arMarkCols.length; i++)
                    cl.cells[vr.arMarkCols[i]].classList.remove("selected")
        } else {
            if (bOn === true) {
                cl.classList.add("selected")
                if (bScroll)
                    cl.scrollIntoView(jScrollOpt)
            } else {
                cl.classList.remove("selected")
            }
        }
    }
    /**
     * Sends edit command to PLC if row changed, 
     * otherwise row is selected
     * @param {*} self 
     * @param {integer} iRow row number of clicked cell
     * @param {integer} iCol col number of clicked cell
     */
    async function editCell(self, iRow, iCol) {
        let vr = self.vars
        let el = vr.elements
        let cf = self.config
        let iRowNew = parseInt(el.tabTable.rows[vr.iRow1Data + iRow].cells[0].innerText)
        let bEdit = (iRowNew === vr.iIdSelected) ? true : false
        vr.iIdSelected = iRowNew

        let iID = 0
        // unmark previously marked row, mark selected row
        for (let i = vr.iRow1Data; i < el.tabTable.rows.length; i++) {
            iID = parseInt(el.tabTable.rows[i].cells[0].innerText)
            markRow(self, i, (iID === vr.iIdSelected))
        }
        let bSingleEditClick = self.element.classList.contains("singleClickEdit")

        if (!(bEdit || bSingleEditClick) || (iCol == 0)) return
        if (cf.selectType === "string") {
            if (vr.bIsArray) {
                setCmd(self, 'editIdCol', vr.iIdSelected, iCol)
            } else {
                setCmd(self, 'editIdCol', iCol, vr.iIdSelected)
            }
        } else {
            const im = shmi.requires("visuals.session.ItemManager")
            let sItem = `${cf.itemStruct}${vr.sAryStart}${iRowNew}${vr.sAryEnd}${vr.arMember[iCol]}`
            let myProps = im.getItem(sItem).getProperties()
            try {
                myProps = tableEditCell(self, sItem, iRow, iCol, myProps)
                if (!myProps) {
                    // cxt.msgTop("warning", "Not editable")
                    return
                }
            } catch (e) {
                // use properties
                myProps = myProps
            }

            let rMin = (myProps.min !== null) ? myProps.min : DEFAULT_MIN
            let rMax = (myProps.max != null) ? myProps.max : DEFAULT_MAX
            if (myProps.type === TYP_BOOL) {
                let iVal = await cxt.getItem(sItem)
                iVal = (iVal === 0) ? 1 : 0
                cxt.setItem(sItem, iVal)
            } else if (vr.arLists[iCol] !== "") {
                let sOpt = ""
                let jTL = getTextList(vr.arLists[iCol], self)
                for (const key in jTL) {
                    sOpt += `${jTL[key]}\t${key}\n`
                }
                let iVal = await cxt.getItem(sItem)
                let ret = await cxt.msgBox("select", "select option", "ok", sOpt, "\n", "\t", iVal)
                if (ret)
                    cxt.setItem(sItem, ret)
            } else {
                // keypad(sItem, sType, iDigits, iType, rMin, rMax, bSelectLanguage, bIsUtf8)
                cxt.keypad(sItem, "webiq", 0, myProps.type, rMin, rMax, true, false)
            }
        }
    }
    /**
     * sorts table by clicked column
     * @param {*} self  root element of widget
     * @param {integer} iCol column number for sort
     * @returns 
     */
    function sortClick(self, iCol, bKeepColumn = false) {
        log("sortClick()")
        const vr = self.vars
        const el = self.vars.elements

        if (!vr.bEnabled)
            return
        if (vr.arLists[iCol])
            mySelect = getTextList(vr.arLists[iCol], self)
        // return when sort is hidden/disabled
        if (self.element.classList.contains("disableSort"))
            return
        if (bKeepColumn === false) {
            if (vr.iSortCol === iCol) {
                vr.bSortUp = !vr.bSortUp
            } else {
                if (vr.iSortCol >= 0)
                    el.divSort[vr.iSortCol].innerHTML = CHAR_SORT
                vr.iSortCol = iCol
                vr.bSortUp = false
            }
        }
        if (vr.arSort[vr.iSortCol] === "I") { // integer
            vr.arData = vr.arData.sort(cxt.sortByKey(vr.bSortUp, vr.iSortCol, null, parseInt, null));
        } else if (vr.arSort[vr.iSortCol] === "F") { // float
            vr.arData = vr.arData.sort(cxt.sortByKey(vr.bSortUp, vr.iSortCol, null, parseFloat, null));
        } else if (vr.arSort[vr.iSortCol] === "L") { // list
            mySelect = getTextList(vr.arLists[iCol], self)
            vr.arData = vr.arData.sort(cxt.sortByKey(vr.bSortUp, vr.iSortCol, null, parseTextList, null));
        } else // text
            vr.arData = vr.arData.sort(cxt.sortByKey(vr.bSortUp, vr.iSortCol, null, cxt.parseToLower, null));
        createTable(self)
    }
    /**
     * Retrieves json text list, which is empty when not defined
     * @param {string} sName name of text list
     * @returns {object} text list object
     */
    function getTextList(sName, self) {
        let myList = undefined
        if (typeof cxt.textList !== 'undefined') {
            myList = cxt.textList[sName]
        }
        if (cxt.isNullOrUndef(myList)) {
            try {
                let myList = self.vars.textLists[sName]
                return myList
            } catch {
                return {}
            }
        } else {
            return myList
        }
    }

    /**
     * Returns value for key from JSON list (e.g. 0=off 1=on)
     * @param {string} key name of list
     * @returns json list
     */
    function parseTextList(key) {
        try {
            let res = shmi.localize(mySelect["" + key])
            return res ? res : `?TxtLst?=${key}`
        } catch (e) {
            return `?TxtLst?=${key}`
        }
    }
    /**
     * Creates all cmd buttons defined in JSON
     * @param {dom} self 
     */
    function createAllButtons(self) {
        const el = self.vars.elements
        const vr = self.vars
        const iMax = el.cxInputXBtns.children.length
        el.createdElements = []
        for (let i = 0; i < iMax; i++) {
            el.cxInputXBtns.children[0].remove()
        }
        if (!vr.cmds) return
        vr.cmds.forEach(cmd => {
            let btn = createButton(self, "cx-btn", cmd.lbl, cmd.cmd, cmd.cfm)
            el.createdElements.push(btn)
            if (cmd.cmd === "copy") {
                btn.actionUp = () => {
                    // store id of selected row for paste
                    vr.iCopyRow = vr.iIdSelected
                    cxt.msgTop("info", "Copied #" + vr.iCopyRow)
                }
            } else if (cmd.cmd === "paste") {
                btn.actionUp = async () => {
                    let bRes = await cxt.msgBox("confirm", cmd.cfm)
                    if (bRes) setCmd(self, "pasteRow", vr.iIdSelected, vr.iCopyRow)
                }
            }
        });
    }
    /**
    * create a table
    * @param {domElement} self  root element of widget
    */
    function createTable(self) {
        log("createTable")
        const vr = self.vars
        const el = self.vars.elements
        const bDebug = true

        el.tabTable = cxt.addChild(null, "table", "myTable")
        if (bDebug) {
            for (let i = 0; i < el.divWidget.children.length; i++)
                el.divWidget.children[0].remove()
            el.divWidget.appendChild(el.tabTable)
        }

        let myRow = null
        // create table header
        let myHead = cxt.addChild(el.tabTable, "thead")
        let myBody = cxt.addChild(el.tabTable, "tbody")

        let imgLock = cxt.addChild(myHead, "img", "cxLock")
        imgLock.src = cxt.cfgMsgBox.imgLock

        // create table filter row
        myRow = cxt.addChild(myHead, "tr")
        let iFiltersActive = 0
        if (!vr.bIsArray) {
            vr.arFilterOn = ['+', '+', '+']
            vr.arWidth = ['40px', '200px', '200px']
            vr.arSort = ['', '', '']
        }
        // show buttons only when displaying an array not single vars
        el.cxInputXBtns.style.display = vr.bIsArray ? "" : "none"
        for (let iCol = 0; iCol < vr.arFilterOn.length; iCol++) {
            let th = cxt.addChild(myRow, "th")
            if (vr.arWidth[iCol])
                if (vr.arWidth[iCol].length > 0)
                    th.setAttribute("style", `width:${vr.arWidth[iCol]}`)
            let txt = null
            if (vr.arFilterOn[iCol]) {
                iFiltersActive++
                txt = cxt.addChild(th, "input")
                txt.value = vr.arFilterStr[iCol]
                // if (vr.arWidth[iCol])
                //     if (vr.arWidth[iCol].length > 0)
                //         txt.style.width = vr.arWidth[iCol] // "90%"
                txt.style.width = "calc( 100% - 2px )"
                txt.style.marginTop = "4px"
                txt.style.height = "70%"
                txt.readOnly = !vr.bEnabled
                if (cxt.keypadEnable())
                    txt.readOnly = cxt.keypadEnable() || (vr.bEnabled === false)
                txt.placeholder = shmi.localize("${cxVisuals.filter}")
                txt.addEventListener("change", (e) => startFilter(self, e.target, iCol));
                txt.addEventListener("keyup", (e) => startFilter(self, e.target, iCol));
                txt.addEventListener("dblclick", (e) => {
                    if (!vr.bEnabled || shmi.isDesignerEditor() || self.element.classList.contains("locked"))
                        return
                    txt.value = "";
                    startFilter(self, e.target, iCol)
                });
                txt.addEventListener("click", async (e) => {
                    if (!vr.bEnabled || shmi.isDesignerEditor() || self.element.classList.contains("locked"))
                        return
                    await cxt.keypad(txt, "value", 10, TYP_STRING)
                    startFilter(self, e.target, iCol)
                });
                if (iCol === 0) el.txtIdFilter = txt
            }
            el.txtFilter.push(txt)
        }
        if ((el.txtIdFilter) && self.element.classList.contains("locked")) el.txtIdFilter.style.visibility = "hidden";
        if (iFiltersActive === 0) myRow.hidden = true
        // create table title with sort chars
        myRow = cxt.addChild(myHead, "tr")

        // create table header row
        let arTitle = (vr.bIsArray) ? vr.arTitle : vr.arHeader
        for (let iCol = 0; iCol < arTitle.length; iCol++) {
            let th = cxt.addChild(myRow, "th")
            let div = cxt.addChild(th, "div", "flexOn")
            let sTmp = ""
            if (vr.arSort[iCol]) {
                th.onclick = () => {
                    if (self.element.classList.contains("locked")) return
                    sortClick(self, iCol)
                }
                sTmp = (vr.iSortCol !== iCol) ? CHAR_SORT : (vr.bSortUp) ? CHAR_SORT_UP : CHAR_SORT_DN
            }
            let oTmp = cxt.addChild(div, "div", "flexFix", sTmp)
            if (sTmp !== CHAR_SORT) el.divSortChar = oTmp
            let sUnit = (vr.arUnit[iCol].length) ? ` [${vr.arUnit[iCol]}]` : ""
            cxt.addChild(div, "div", "flexResize", shmi.localize(arTitle[iCol] + `${sUnit}`))
            el.divSort.push(oTmp)
        }

        // add data rows
        if (vr.iIdSelected < 0)
            vr.iIdSelected = vr.arData[vr.iRowFirst][0]
        for (let iRow = vr.iRowFirst; iRow < vr.iRowLast; iRow++) {
            myRow = cxt.addChild(myBody, "tr")
            for (let iCol = 0; iCol < vr.arData[iRow].length; iCol++) {
                let val = vr.arData[iRow][iCol]
                if (vr.bIsArray) {
                    // ToDo Delete
                    if (vr.arLists[iCol]) {
                        mySelect = getTextList(vr.arLists[iCol], self)
                        val = parseTextList(val)
                    } else if (vr.arFormat[iCol]) {
                        // mySelect = getTextList(vr.arLists[iCol], self)
                        eval("val=" + vr.arFormat[iCol])
                    }

                } else {
                    if (iCol === 2) {
                        if (vr.arLists[iRow + 1]) {
                            mySelect = getTextList(vr.arLists[iRow + 1], self)
                            val = parseTextList(val)
                        }
                    }
                }
                let td = cxt.addChild(myRow, "td", "", shmi.localize(val))
                td.onclick = () => {
                    if (self.element.classList.contains("locked")) return
                    editCell(self, iRow, iCol)
                }
                if (!cxt.isNullOrUndef(vr.arTalign[iCol])) {
                    if (vr.arTalign[iCol] == "L")
                        td.style.textAlign = "left"
                    else if (vr.arTalign[iCol] == "C")
                        td.style.textAlign = "center"
                }
            }
            if (vr.arData[iRow][0] == vr.iIdSelected)
                markRow(self, iRow + vr.iRow1Data)
            el.cxInputXBtns.style.display = self.element.classList.contains("locked") ? "none" : ""
        }
        cxt.overlayWait(true)
        // remove all children
        for (let i = 0; i < el.divWidget.children.length; i++)
            el.divWidget.children[0].remove()
        // append generated table to widget
        startFilter(self, null, 0)
        // sortClick(self, 1)
        el.divWidget.appendChild(el.tabTable)
        createAllButtons(self)
        cxt.overlayWait(false)
    }
    /**
     * Parse different input data & converts it into requested format
     * @param {Object} self  root element of widget
     * @param {*} value 
     * @returns 
     */
    async function parseData(self, value, bUpdate = false, sParent = "") {
        log("parseData Parent=" + sParent)
        const vr = self.vars
        const cf = self.config
        let sTmp = ""
        let sErr = "" // stores invalid item names
        let iErrCnt = 0
        vr.arData = [] // clear array
        // myData = vr.arData
        if (cf.selectType === "string") {
            // do nothing when params not provided
            if ((!cf.itemCfg) || (!cf.itemData))
                return
            // reset table vars
            vr.arTitle = []    // table title strings
            vr.arWidth = []    // table column width
            vr.arUnit = []     // table unit strings
            vr.arSort = []     // parse fcts for sorting
            vr.arFilterOn = [] // filter strings
            vr.arFormat = []   // format functions
            try {
                sTmp = await cxt.getItem(cf.itemCfg, true)
                vr.jCfg = JSON.parse(sTmp)
                log("parseData Parent=" + sParent + " CFG:\n" + JSON.stringify(vr.jCfg, null, 2))
                vr.arTitle = (vr.jCfg.title !== undefined) ? vr.jCfg.title.split(vr.FS_COL) : []
                vr.arFilterOn = (vr.jCfg.title !== undefined) ? vr.jCfg.filter.split(vr.FS_COL) : []
                vr.arSort = (vr.jCfg.title !== undefined) ? vr.jCfg.sort.split(vr.FS_COL) : []
                vr.arUnit = (vr.jCfg.title !== undefined) ? vr.jCfg.unit.split(vr.FS_COL) : []
                vr.arFormat = (vr.jCfg.title !== undefined) ? vr.jCfg.format.split(vr.FS_COL) : []
                vr.arTalign = (vr.jCfg.talign !== undefined) ? vr.jCfg.talign.split(vr.FS_COL) : []
                vr.arLists = (vr.jCfg.title !== undefined) ? vr.jCfg.list.split(vr.FS_COL) : []
                vr.bUtf8 = (vr.jCfg.utf8 !== undefined) ? vr.jCfg.utf8 : false
                vr.arWidth = (vr.jCfg.width !== undefined) ? vr.jCfg.width.split(vr.FS_COL) : []
                vr.iSortCol = (vr.jCfg.sortCol !== undefined) ? vr.jCfg.sortCol : vr.iSortCol
                vr.bSortUp = (vr.jCfg.sortUp !== undefined) ? vr.jCfg.sortUp : vr.bSortUp
                vr.bIsArray = (vr.jCfg.isArray !== undefined) ? vr.jCfg.isArray : true
                vr.arHeader = ["ID", "Variable name", "Value"]
                vr.textLists = (vr.jCfg.textLists !== undefined) ? vr.jCfg.textLists : {}
                vr.cmds = (vr.jCfg.cmds !== undefined) ? vr.jCfg.cmds : []
            } catch (e) {
                console.error("Parsing itemCfg failed")
                return
            }
            // create data array
            sTmp = (self.config.itemData) ? await cxt.getItem(self.config.itemData, true) : ""
            if (vr.bUtf8) sTmp = shmi.from_utf8(sTmp)
            log("parseData Parent=" + sParent + " DATA:\n" + sTmp)
            const arTmp = sTmp.split(vr.FS_DROW)
            if (vr.bIsArray) {
                for (let iRow = 0; iRow < arTmp.length; iRow++) {
                    if (arTmp[iRow] !== "")
                        vr.arData.push(arTmp[iRow].split(vr.FS_DCOL))
                }
            } else {
                // vr.arSort = ["L","L","L",]
                let arCol = arTmp[0].split(vr.FS_DCOL)
                for (let iRow = 1; iRow < arCol.length; iRow++) {
                    let arRow = [iRow, vr.arTitle[iRow], arCol[iRow]]
                    vr.arData.push(arRow)
                }
            }
            vr.iRowFirst = 0
            vr.iRowLast = vr.arData.length
        } else { // STRUCT
            vr.bIsArray = true
            // create data array
            for (let iR = vr.iRowFirst; iR < vr.iRowLast; iR++) {
                let arCols = []
                arCols.push(iR) // add array ID
                for (let iC = 1; iC < vr.arMember.length; iC++) {
                    let sVal = ""
                    let sItem = `${cf.itemStruct}${vr.sAryStart}${iR}${vr.sAryEnd}${vr.arMember[iC]}`
                    try {
                        sVal = await cxt.getItem(sItem, true, false)
                        if (sVal === true) {
                            sVal = 1
                        } else if (sVal === false)
                            sVal = 0
                    } catch (e) {
                        sVal = "invalid item: " + sItem
                        iErrCnt++
                        if (sErr.length < 500)
                            sErr += sItem + " "
                    }
                    arCols.push(sVal)
                }
                vr.arData.push(arCols)
            }
        }
        // create table body with data
        if (vr.arData.length > 0) {
            vr.iRowFirst = (vr.iRowFirst < 0) ? 0 : vr.iRowFirst
            // vr.iRowLast = (vr.iRowLast < 0) ? vr.arData.length : vr.iRowLast
            vr.iRowLast = (vr.iRowLast > vr.arData.length) ? vr.arData.length : vr.iRowLast
        }
        if (iErrCnt > 0) {
            cxt.msgBox("bug", `${iErrCnt} invalid item name configured:\n` + sErr)
        } else if (value !== "") {
            for (let i = 0; i < vr.arTitle.length; i++) {
                if (vr.arFilterOn.length < vr.arTitle.length) vr.arFilterOn.push(null)
                if (vr.arFilterStr.length < vr.arTitle.length) vr.arFilterStr.push([""])
                if (vr.arUnit.length < vr.arTitle.length) vr.arUnit.push("")
                if (vr.arFormat.length < vr.arTitle.length) vr.arFormat.push(null)
                // ToDo list length is independent 
                // if (vr.arLists.length < vr.arTitle.length) vr.arLists.push(null)

                vr.arFormat[i] = (["", "-"].includes(vr.arFormat[i])) ? null : vr.arFormat[i]

                vr.arFilterOn[i] = (["", "-"].includes(vr.arFilterOn[i])) ? null : vr.arFilterOn[i]
                if (vr.arSort.length < vr.arTitle.length)
                    vr.arSort.push(null)
                vr.arSort[i] = (["", "-"].includes(vr.arSort[i])) ? null : vr.arSort[i]
            }
            if (bUpdate)
                sortClick(self, vr.iSortCol, true) // calls createTable after sort
            else
                createTable(self)

        }
    }
    // (/CUSTOM ADDITION--)
    // declare private functions - END

    // definition of new widget extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        fctEdit: null,
        fctMark: null,

        // default configuration settings - all available options have to be initialized!
        config: defaultConfig,
        // instance variables
        vars: {
            // (CUSTOM ADDITION++)
            elements: {
                //references for DOM elements accessed with JS code
                divSort: [],
                tabTable: null,
                txtFilter: [],
                divWidget: null, //
                txtIdFilter: null,
                divSortChar: null
            },
            //reference for PLC item subscription token
            itemCmd: null,
            listeners: [],    //event listeners
            sAryStart: '',
            sAryEnd: '',
            bakNow: 0,        // value of previous click
            bakRow: -1,       // value of previous click
            bakCol: -1,       // value of previous click
            iRowFirst: -1,    // 1st  array row to display
            iRowLast: -1,     // last array row to display
            iIdSelected: -1,  // -1 no row selected
            iRow1Data: 2,     // first data row, header rows start at 0
            bEnabled: false,  // true:widget enabled false:disabled
            iSortCol: -1,     // selected sort column (-1 nothing selected)
            bSortUp: false,   // true=sort up false=sort dn
            bClearDemo: true, // true=clear demo table
            arMarkCols: [],   // empty array[]=> mark row, else[0,1] columns to mark
            bMarkOn: true,    // true mark/highlight selected row
            bUtf8: false,     // true, convert data strings between utf16<->utf8
            arMember: [],     // table members of struct
            arTitle: [],      // table header strings
            arWidth: [],      // table column width
            arData: [],       // table body strings
            arUnit: [],       // table unit strings
            arSort: [],       // parse fcts for sorting
            arFilterOn: [],   // filter enable
            arFilterStr: [],  // filter strings
            arFormat: [],     // format functions
            arLists: [],      // list name to use
            arTalign: [],     // text align of columns
            jCfg: {},         // table config
            arSubscribe: [],  // subscribe struct items

            FS_DROW: "\n",
            FS_DCOL: "\t",
            FS_COL: "/"
            // (/CUSTOM ADDITION--)
        },
        // imports added at runtime
        imports: {
            // example - add import via shmi.requires(...)
            im: "visuals.session.ItemManager",
            // (CUSTOM ADDITION++)
            io: "visuals.io"
            // (/CUSTOM ADDITION--)
        },

        // array of custom event types fired by this widget
        events: [],

        // functions to extend or override the BaseControl prototype
        prototypeExtensions: {
            // called when config-file (optional) is loaded and template (optional) is inserted into base element
            onInit: function () {
                // (CUSTOM ADDITION++)
                const self = this
                const el = self.vars.elements
                const cf = self.config
                const vr = self.vars
                // disable context menu
                self.element.addEventListener('contextmenu', (e) => e.preventDefault(), false)

                logger = shmi.requires("visuals.tools.logging").createLogger(cf.name, ENABLE_LOGGING, RECORD_LOG)
                log = logger.log
                log("*** onInit ***")
                if (cf.selectType === "string") {
                    if (cf.itemStrStruct) {
                        cf.itemStrStruct = cf.itemStrStruct.endsWith(".") ? cf.itemStrStruct : cf.itemStrStruct + "."
                        cf.itemCmd = cf.itemStrStruct + "strCmd"
                        cf.itemCfg = cf.itemStrStruct + "strCfg"
                        cf.itemData = cf.itemStrStruct + "strData"
                        cf.itemState = cf.itemStrStruct + "strState"
                    }
                } else {
                    if (cf.selectType === "struct[x]") {
                        vr.sAryStart = "["
                        vr.sAryEnd = "]."
                    } else {
                        vr.sAryStart = "."
                        vr.sAryEnd = "."
                    }
                    if (cf.stringMember !== null) cf.stringMember = "#" + vr.FS_COL + cf.stringMember
                    if (!cf.stringHeader) cf.stringHeader = ""
                    if (!cf.stringUnit) cf.stringUnit = ""
                    if (!cf.stringFilter) cf.stringFilter = ""
                    if (!cf.stringSort) cf.stringSort = ""
                    if (!cf.stringFormat) cf.stringFormat = ""
                    if (!cf.stringList) cf.stringList = "" // text list name
                    if (!cf.stringListDef) cf.stringListDef = "{}" // text list definition
                    if (!cf.stringWidth) cf.stringWidth = ""

                    cf.stringHeader = "ID" + vr.FS_COL + cf.stringHeader
                    cf.stringUnit = vr.FS_COL + cf.stringUnit
                    cf.stringFilter = "+" + vr.FS_COL + cf.stringFilter
                    cf.stringFormat = "i2s(val,10,3)" + vr.FS_COL + cf.stringFormat
                    cf.stringList = vr.FS_COL + cf.stringList
                    cf.stringWidth = "30px" + vr.FS_COL + cf.stringWidth
                    cf.stringSort = "i" + vr.FS_COL + cf.stringSort
                    // cf.stringSort = "i"

                }
                el.divWidget = shmi.getUiElement("divTable", self.element);
                if (shmi.isDesignerEditor()) {
                    let sTmp = `<table style="border:1px solid black;border-collapse:collapse;">
                        <tr>
                        <th style="border:1px solid;">&nbsp;table&nbsp;</th><th style="border:1px solid;">&nbsp;placeholder&nbsp;</th>
                        </tr>
                        <tr>
                        <td style="border:1px solid;">&nbsp;11</td><td style="border:1px solid;">&nbsp;12</td>
                        </tr>
                        <tr>
                        <td style="border:1px solid;">&nbsp;21</td><td style="border:1px solid;">&nbsp;22</td>
                        </tr>
                        </table>`
                    cxt.addChild(el.divWidget, "div", "", sTmp)
                }

                el.cxInputXBtns = shmi.getUiElement("cmdBtns", self.element)
                // (CUSTOM ADDITION--)
            },
            // called when widget is enabled
            onEnable: async function () {
                // (CUSTOM ADDITION++)
                log("onEnable")
                const self = this
                const { im } = self.imports;
                const cf = self.config
                const vr = self.vars
                const el = vr.elements

                enableListeners(self, true);

                // remove all children
                if (vr.bClearDemo) {
                    vr.bClearDemo = false
                    for (let i = 0; i < el.divWidget.children.length; i++) {
                        el.divWidget.children[0].remove()
                    }
                }
                if (cf.selectType === "string") {
                    // Subscribe to item updates
                    if (cf.itemState) vr.itemState = im.subscribeItem(cf.itemState, self)
                    if (cf.itemCmd) setCmd(self, "updateTable", 4, 1)
                } else { // WebIQ structure
                    // structure
                    if (!cf.name.endsWith("-sba")) return
                    if ((!cf.itemStruct) || (!cf.stringMember)) return
                    // vr.jCfg.lists = JSON.parse(cf.stringListDef)

                    vr.arMember = cf.stringMember.split(vr.FS_COL)
                    vr.arTitle = cf.stringHeader.split(vr.FS_COL)
                    for (let i = vr.arTitle.length; i < vr.arMember.length; i++) {
                        vr.arTitle.push("...")
                    }
                    // show item titles or names, when title=null
                    vr.iRowFirst = cf.intIndexMin
                    vr.iRowLast = cf.intIndexMax + 1

                    vr.arUnit = cf.stringUnit.split(vr.FS_COL)
                    vr.arFilterOn = cf.stringFilter.split(vr.FS_COL)
                    vr.arSort = cf.stringSort.toUpperCase().split(vr.FS_COL)
                    vr.arFormat = cf.stringFormat.split(vr.FS_COL)
                    vr.arLists = cf.stringList.split(vr.FS_COL)
                    vr.arWidth = cf.stringWidth.split(vr.FS_COL)

                    await parseData(self, "x", false, "onEnable2")
                    // subscribe all table items
                    for (let iR = vr.iRowFirst; iR < vr.iRowLast; iR++) {
                        for (let iC = 0; iC < vr.arMember.length; iC++) {
                            let sItem = `${cf.itemStruct}${vr.sAryStart}${iR}${vr.sAryEnd}${vr.arMember[iC]}`
                            im.subscribeItem(sItem, self);
                        }
                    }
                }

                // (/CUSTOM ADDITION--)
            },
            // called when widget is disabled
            onDisable: function () {
                // (CUSTOM ADDITION++)
                log("onDisable")
                const vr = this.vars
                enableListeners(self, false);
                // Stop listening to item, i.e. unsubscribe from item updates
                if (vr.itemState) {
                    vr.itemState.unlisten();
                    vr.itemState = null;
                }
                // (/CUSTOM ADDITION--)
            },
            // called when widget is locked - disable mouse- and touch-listeners
            onLock: function () {
                // (CUSTOM ADDITION++)
                log("onLock")
                const self = this
                const el = self.vars.elements
                self.element.classList.add("locked")
                el.cxInputXBtns.style.display = "none"
                if (el.txtIdFilter) el.txtIdFilter.style.visibility = "hidden"
                // (/CUSTOM ADDITION--)
            },
            // called when widget is unlocked - enable mouse- and touch-listeners
            onUnlock: function () {
                // (CUSTOM ADDITION++)
                const self = this
                const el = self.vars.elements
                log("onUnlock")
                self.element.classList.remove("locked")
                el.cxInputXBtns.style.display = ""
                if (el.txtIdFilter) el.txtIdFilter.style.visibility = "visible"
                // (/CUSTOM ADDITION--)
            },
            // called by ItemManager when value of subscribed item changes and once on initial subscription
            onSetValue: async function (value, type, name) {
                // (CUSTOM ADDITION++)
                const self = this;
                const vr = self.vars
                const cf = self.config
                const el = self.vars.elements
                log("onSetValue" + ` ${name}='${value}'`)

                if (cf.selectType === "string") {
                    if (name === cf.itemState) {
                        // ignore commands which are not for me
                        if ((value === '') || (value === null)) return
                        let arW = value.split(" ")
                        // console.log("onSetValue:" + value)
                        if (el.tabTable === null) arW[1] = "updateTable"
                        if (["makeTable", "updateTable"].includes(arW[1])) {
                            cxt.setItem(cf.itemState, '')

                            // select id, provided by plc
                            // @SBA do not use id from PLC
                            if ((arW[2] === cxt.getClientId()) && (arW[3] !== "")) vr.iIdSelected = parseInt(arW[3])
                            await parseData(self, value, ["updateTable"].includes(arW[1]), "onSetvalue() ")
                        } else if (["markRow"].includes(arW[1])) {
                            if (arW[2] !== "") vr.iIdSelected = parseInt(arW[2])
                            for (let iRow = vr.iRowFirst; iRow < vr.iRowLast; iRow++) {
                                if (vr.arData[iRow][0] == vr.iIdSelected)
                                    markRow(self, iRow + vr.iRow1Data, true, true)
                                else
                                    markRow(self, iRow + vr.iRow1Data, false)
                            }
                        } else if (["updateCells"].includes(arW[1])) {
                            updateCells(self)
                        } else if (value.startsWith("{")) {
                            let jMsg = {}
                            jMsg = JSON.parse(value)
                            if (jMsg.stat === "edit") {
                                // return, when edit request is from other clients
                                if (jMsg.clientID !== cxt.getClientId())
                                    return
                                cxt.setItem(cf.itemState, '') // clear content
                                let iType = TYP_FLOAT
                                // keypad(sItem, sType = "webiq", iLen = -1, iType = TYP_STRING, rMin = -100, rMax = 100, bSelectLanguage = true)
                                jMsg.editor = jMsg.editor.toUpperCase()
                                let myId = vr.iIdSelected
                                let myCol = jMsg.col
                                if (jMsg.editor === "B") {
                                    // let bVal = "TRUE"
                                    // if ((jMsg.value === "TRUE") || (jMsg.value === 1)) {
                                    //     bVal = "0"
                                    // }
                                    let iVal = (jMsg.value === "1") ? "0" : "1"
                                    setCmd(self, "setIdCol", myId, myCol, iVal)
                                } else {
                                    // dependent on item type, select matching editor
                                    if (jMsg.editor === "F")
                                        iType = TYP_FLOAT
                                    else if (jMsg.editor === "I")
                                        iType = TYP_INT
                                    else if (jMsg.editor === "S")
                                        iType = TYP_STRING
                                    let ret = null
                                    if ((jMsg.editor === "I") && (vr.arLists[jMsg.col] !== "")) {
                                        let sOpt = ""
                                        let jTL = getTextList(vr.arLists[jMsg.col], self)
                                        for (const key in jTL) {
                                            sOpt += `${jTL[key]}\t${key}\n`
                                        }
                                        ret = await cxt.msgBox("select", "select option", "ok", sOpt, "\n", "\t", jMsg.value)

                                    } else {
                                        let bBak = cxt.keypadEnable()
                                        cxt.keypadEnable(true)
                                        let sTmp = (vr.bUtf8) ? shmi.from_utf8(jMsg.value) : jMsg.value

                                        ret = await cxt.keypad(sTmp, "input", 0, iType, jMsg.min, jMsg.max)
                                        cxt.keypadEnable(bBak)
                                    }
                                    if (ret !== null) {
                                        let sUtf8 = shmi.to_utf8(ret)
                                        setCmd(self, "setIdCol", myId, myCol, sUtf8)
                                    }
                                }
                            }
                        }

                    }
                } else {
                    if (name.startsWith(cf.itemStruct)) {
                        let sTmp = cf.itemStruct + vr.sAryStart
                        if (name.startsWith(sTmp)) {
                            // extract row & col from item name
                            let sTmp2 = name.substr(sTmp.length)
                            let iPos = sTmp2.indexOf(vr.sAryEnd)
                            let iID = parseInt(sTmp2.substr(0, iPos))
                            let iRow = iID
                            let iCol = vr.arMember.indexOf(sTmp2.substr(iPos + vr.sAryEnd.length))
                            let iRowMax = vr.arData.length
                            for (let i = 0; i < iRowMax; i++) {
                                if (vr.arData[i][0] === iID) {
                                    vr.arData[i][iCol] = value
                                    iRow = i + vr.iRow1Data
                                    break;
                                }
                            }
                            // display value in table
                            let val = value
                            if (vr.arLists[iCol]) {
                                mySelect = getTextList(vr.arLists[iCol], self)
                                val = parseTextList(val)
                            } else if (vr.arFormat[iCol]) {
                                eval("val=" + vr.arFormat[iCol])
                            }

                            // show ? when value in sort column changed
                            if (iCol === Math.abs(vr.iSortCol))
                                if (val !== el.tabTable.rows[iRow].cells[iCol].innerText) {
                                    if (el.divSortChar !== null) el.divSortChar.innerHTML = CHAR_SORT_UNKNOWN
                                    log("iCol = " + iCol + " ?")
                                }

                            el.tabTable.rows[iRow].cells[iCol].innerText = val

                            log(`id=${iID} row=${iRow} col=${iCol} val=${val}`)
                        }

                    }
                }
                // (/CUSTOM ADDITION--)
            },
            // called by ItemManager to provide properties (min & max values etc.) of subscribed item
            onSetProperties: function (min, max, step, name, type, warnMin, warnMax, prewarnMin, prewarnMax, digits) {
                // (CUSTOM ADDITION++)
                // const self = this;
                // const vr = self.vars
                // const cf = self.config
                // // func is called once for every item without valid paramters => ignore it
                // if (type === null)
                //     return
                // log("onSetProperties " + `name:${name} type:${type} min:${min} max:${max}`)

                // (/CUSTOM ADDITION--)
            },
            // called when widget is deleted - used for instance clean-up
            onDelete: function () {
                log("onDelete")
                // const self = this;
                // const el = self.vars.elements
                // for (let i = 0; i < el.createdElements.length; i++) {
                //     if (el.createdElements[i])
                //         shmi.deleteControl(element)
                // }
            }
        }
    };

    // definition of new widget extending BaseControl - END
    // generate widget constructor & prototype using the control-generator tool
    shmi.requires("visuals.tools.control-generator").generate(definition);
})();
