/**
 * WebIQ Visuals widget template.
 *
 * Configuration options (default):
 *
 * {
 *     "class-name": "cx-dbspy", // {%= ui_type %}
 *     "name": null,
 *     "template": "custom/controls/cx-dbspy" // {%= template_path %}
 * }
 *
 * Explanation of configuration options:
 *
 * class-name {string}: Sets default CSS class applied on widget root element
 * name {string}: Name of widget set to data-name attribute
 * template {string}: Path to template file
 *
 * @version 1.2 changed by boschrexroth
 */
(function () {
    'use strict';

    // variables for reference in widget definition
    const templatePath = "custom/controls/cx-dbspy" // "{%= template_path %}"
    const uiType = "cx-dbspy"    // widget keyword (data-ui) {%= ui_type %}
    const className = uiType           // widget name in camel case {%= constructor_name %}
    const isContainer = false
    const ROW_OFFSET_DATA = 1

    // example - default configuration
    const defaultConfig = {
        "class-name": uiType,
        "name": null,
        "template": templatePath,
        "label": uiType,
        "item": null
    };

    // setup module-logger
    const ENABLE_LOGGING = true //enable logging with 'log' function
    const RECORD_LOG = false //store logged messages in logger memory
    let logger = shmi.requires("visuals.tools.logging").createLogger(uiType, ENABLE_LOGGING, RECORD_LOG)
    let log = logger.log; //log message if logger is enabled
    const fLog = logger.fLog //force log - logs message even if logger is disabled

    // declare private functions - START
    // (CUSTOM ADDITION++)
    /**
     * Retrieves json text list, which is empty when not defined
     * @param {string} sName name of text list
     * @returns {object} text list object
     */
    function getTextList(self, sName) {
        let myList = cxt.textList[sName]
        if (typeof cxt.textList !== 'undefined') {
            myList = cxt.textList[sName]
        }
        if (cxt.isNullOrUndef(myList)) {
            try {
                let myList = self.vars.textLists[sName]
                return myList
            } catch {
                return {}
            }
        } else {
            return myList
        }
    }
    /**
     * Returns value for key from JSON list (e.g. 0=off 1=on)
     * @param {string} key name of list
     * @returns json list
     */
    function parseTextList(self, key) {
        const vr = self.vars
        try {
            let res = shmi.localize(vr.txtList["" + key])
            return res ? res : `?TxtLst?=${key}`
        } catch (e) {
            return `?TxtLst?=${key}`
        }
    }
    /**
     * enable/disable all listeners stored in listenrs[]
     * @param {Dom} self reference to widget element
     */
    function enableListeners(self, bEnable = true) {
        if (bEnable)
            self.vars.listeners.forEach((l) => { l.enable(); });
        else
            self.vars.listeners.forEach((l) => { l.disable(); });
    }

    /**
     * subscribe item if defined & stores it in an array
     * @param {object} self  reference to WebIQ widget
     * @param {string} sItem name of item
     */
    function subscribeIfExists(self, sItem) {
        const { im } = self.imports;
        if (sItem)
            self.vars.subscribes.push(im.subscribeItem(sItem, self))
    }
    /**
     * Creates table with spy items. Double entries are removed
     * @param {DOM}    self      dom widget element
     * @param {string} sItemList string with all spy items (FS=\n)
     */
    function createTable(self, sItemList) {
        const el = self.vars.elements
        const vr = self.vars
        const im = shmi.requires("visuals.session.ItemManager")
        // remove existing children
        for (let i = 0; i < el.divTable.children.length; i++)
            el.divTable.children[0].remove()

        // Stop listening to all items in subscribes[]
        vr.subscribes.forEach((s) => {
            s.unlisten();
        })

        // create new table
        el.tabTable = cxt.addChild(el.divTable, "table", "myTable")

        let myRow = null
        // create table header
        let myHead = cxt.addChild(el.tabTable, "thead")
        myRow = cxt.addChild(myHead, "tr")
        let th = cxt.addChild(myRow, "th", "", shmi.localize("${itemName}"))
        th.setAttribute("style", "max-width:80px")
        th = cxt.addChild(myRow, "th", "", shmi.localize("${itemValue}"))
        th.setAttribute("style", "max-width:100px")

        // create table data rows
        vr.textLists = {}
        let arItems = sItemList.split(vr.sRowFs)
        for (let i = 0; i < arItems.length; i++) {
            let arW = arItems[i].split(vr.sColFs)
            if (arW.length > 1) {
                arItems[i] = arW[0]
                vr.textLists[arItems[i]] = arW[1]
            }
        }
        vr.itemList = {}

        let myBody = cxt.addChild(el.tabTable, "tbody")
        let iID = 0
        let sItemNew = ""
        let sFS = ""
        let bUpdate = false
        for (let i = 0; i < arItems.length; i++) {
            // item can only be added once
            // if ((vr.itemList[arItems[i]] === undefined) && (arItems[i] !== "")) {
            if ((vr.itemList[arItems[i]] === undefined)) {
                if (arItems[i] === "") arItems[i] = "&nbsp;"
                vr.itemList[arItems[i]] = iID + ROW_OFFSET_DATA
                iID++
                sItemNew += sFS + arItems[i]
                sFS = "\n"
                myRow = cxt.addChild(myBody, "tr")
                myRow.onclick = async () => {
                    if (self.element.classList.contains("locked")) return
                    const sItem = el.tabTable.rows[i + ROW_OFFSET_DATA].cells[0].innerHTML
                    if (sItem === "&nbsp;") return
                    const myItem = im.getItem(sItem) // item must be subscribed or virtual otherwise null is retured
                    let myProps = myItem.getProperties()
                    if (myProps.type === TYP_BOOL) {
                        let bVal = await cxt.getItem(sItem) == 1 ? 0 : 1
                        cxt.setItem(sItem, bVal)
                    } else {
                        if (vr.textLists[sItem] !== undefined) {
                            let sOpt = ""
                            let oVal = await cxt.getItem(sItem)
                            vr.txtList = getTextList(self, vr.textLists[sItem])
                            for (const key in vr.txtList)
                                sOpt += `${vr.txtList[key]}\t${key}\n`
                            oVal = await cxt.msgBox("select", "select option", "ok", sOpt, "\n", "\t", oVal)
                            if (oVal !== null) cxt.setItem(sItem, oVal)
                        } else {
                            // min & max properties must be filled in WebIQ, otherwise min & max = null
                            let oMin = (myProps.min === null) ? DEFAULT_MIN : myProps.min
                            let oMax = (myProps.max === null) ? DEFAULT_MAX : myProps.max
                            let bBak = cxt.keypadEnable()
                            cxt.keypadEnable(true)
                            await cxt.keypad(sItem, "webiq", -1, myProps.type, oMin, oMax, true)
                            cxt.keypadEnable(bBak)
                        }
                    }
                }
                cxt.addChild(myRow, "td", "", arItems[i])
                cxt.addChild(myRow, "td")
                subscribeIfExists(self, arItems[i])

            } else {
                bUpdate = true
            }
        }
        arItems = null
        if (bUpdate) {
            cxt.setItem(self.config.item, sItemNew)
        }
    }
    // (/CUSTOM ADDITION--)
    // declare private functions - END

    // definition of new widget extending BaseControl - START
    const definition = {
        className: className,
        uiType: uiType,
        isContainer: isContainer,
        // default configuration settings - all available options have to be initialized!
        config: defaultConfig,
        // instance variables
        vars: {
            // (CUSTOM ADDITION++)
            elements: {
                //references for DOM elements accessed with JS code
            },
            sRowFs: "\n",
            // sColFs: "\t",
            sColFs: "=",
            item: null,      //reference for PLC item subscription token - will be set when widget is enabled
            listeners: [],   //event listeners
            subscribes: []   //subscribed items
            // (CUSTOM ADDITION--)
        },
        // imports added at runtime
        imports: {
            // example - add import via shmi.requires(...)
            im: "visuals.session.ItemManager",
            // (CUSTOM ADDITION++)
            io: "visuals.io"
            // (CUSTOM ADDITION--)
        },

        // array of custom event types fired by this widget
        events: [],

        // functions to extend or override the BaseControl prototype
        prototypeExtensions: {
            // called when config-file (optional) is loaded and template (optional) is inserted into base element
            onInit: function () {
                // (CUSTOM ADDITION++)
                log("onInit")

                const self = this
                const vr = self.vars
                const el = self.vars.elements
                const cf = self.config
                logger = shmi.requires("visuals.tools.logging").createLogger(cf.name, ENABLE_LOGGING, RECORD_LOG)
                log = logger.log
                // disable context menu
                self.element.addEventListener('contextmenu', (e) => e.preventDefault(), false)

                el.divTable = shmi.getUiElement("divTable", self.element)
                el.cxInputXBtns = shmi.getUiElement("divCmdBtns", self.element)
                if (cf.separator === "=;") {
                    vr.sColFs = "="
                    vr.sRowFs = ";"
                }
                // (CUSTOM ADDITION--)
            },
            // called when widget is enabled
            onEnable: function () {
                // (CUSTOM ADDITION++)
                log("onEnable")
                const self = this
                const { im } = self.imports;
                enableListeners(self, true);

                // Subscribe to item updates
                // subscribeIfExists(self, self.config.item)
                if (self.config.item)
                    self.vars.itemSubscribe = im.subscribeItem(self.config.item, self)
                // self.vars.toolTip = cxt.isTooltipEnabled() ? im.subscribeItem(TOOLTIP_ITEM, self) : null
                // (CUSTOM ADDITION--)
            },
            // called when widget is disabled
            onDisable: function () {
                // (CUSTOM ADDITION++)
                log("onDisable")
                const self = this;
                enableListeners(self, false);
                // Stop listening to all items in subscribes[]
                self.vars.subscribes.forEach((s) => {
                    s.unlisten();
                })
                if (self.vars.itemSubscribe) self.vars.itemSubscribe.unlisten()
                self.vars.subscribes = []
                // if (self.vars.toolTip) self.vars.toolTip.unlisten(); self.vars.toolTip = null
                // (CUSTOM ADDITION--)
            },
            // called when widget is locked - disable mouse- and touch-listeners
            onLock: function () {
                // (CUSTOM ADDITION++)
                log("onLock")
                const self = this
                enableListeners(self, false)
                self.element.classList.add("locked")
                // (/CUSTOM ADDITION--)
            },
            // called when widget is unlocked - enable mouse- and touch-listeners
            onUnlock: function () {
                // (CUSTOM ADDITION++)
                log("onUnlock")
                const self = this
                enableListeners(self, true)
                self.element.classList.remove("locked")
                // (CUSTOM ADDITION--)
            },
            // called by ItemManager when value of subscribed item changes and once on initial subscription
            onSetValue: function (value, type, name) {
                // (CUSTOM ADDITION++)
                log("onSetValue: " + name + "=" + value)
                if (type === null)  // when type===null do nothing, because params are invalid
                    return
                const self = this
                const el = self.vars.elements
                const vr = self.vars
                if (name === self.config.item)
                    createTable(self, value)
                // else if (name === SHOW_TOOLTIP)
                //     cxt.tooltip(self, value)
                else {
                    if (type === TYP_BOOL) {
                        el.tabTable.rows[vr.itemList[name]].cells[1].innerHTML = (value === 1) ? "TRUE" : "FALSE"
                    } else {
                        if (type === TYP_FLOAT)
                            value = value.toFixed(3)
                        else
                            if (vr.textLists[name]) {
                                vr.txtList = getTextList(self, vr.textLists[name])
                                value = parseTextList(self, value)
                            }
                        el.tabTable.rows[vr.itemList[name]].cells[1].innerHTML = value
                    }
                }
                // (CUSTOM ADDITION--)
            },
            // called by ItemManager to provide properties (min & max values etc.) of subscribed item
            onSetProperties: function (min, max, step, name, type, warnMin, warnMax, prewarnMin, prewarnMax, digits) {
            },
            // called when widget is deleted - used for instance clean-up
            onDelete: function () {
                log("onDelete")
            }
        }
    };

    // definition of new widget extending BaseControl - END

    // generate widget constructor & prototype using the control-generator tool
    shmi.requires("visuals.tools.control-generator").generate(definition);
})();
