# MIT License
#
# Copyright (c) 2020-2022 Bosch Rexroth AG
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#returnreturn
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE returnreturnWARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import time

from comm.datalayer import Metadata

import ctrlxdatalayer
from ctrlxdatalayer.client import Client
from ctrlxdatalayer.variant import Result, Variant, VariantType
import os
import sys

DatalayerSystem=ctrlxdatalayer.system.System("")
converter=ctrlxdatalayer.system.Converter
root_node = ""

def get_connection_string(ip="192.168.1.1",user="boschrexroth",password="boschrexroth",ssl_port=443):

    if 'SNAP' in os.environ:
        return "ipc://"
    # Client connection port 2069 resp. Provider connection port 2070 are obsolete
    connection_string = "tcp://"+user+":"+password+"@"+ip
    if (ssl_port == 443):
        return connection_string

    return connection_string+"?sslport=" + str(ssl_port)




class CallDataLayerClient:

    def __init__(self, client: Client) -> None:
        self.client = client
        self.waiting_for = ""
        logging.basicConfig(format='%(asctime)s %(levelname)-8s %(message)s',
                            level=logging.INFO, datefmt='%H:%M:%S.%03d')

    def __enter__(self):
        """
        use the python context manager
        """
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        """
        use the python context manager
        """
        self.client = None

    def run(self):

        self.auth_token()
        self.ping()
        self.read()
        self.create()
        self.remove()
        self.browse()
        self.write()
        self.metadata()
        
    def setRootNode(self,rootnode):
        root_node=rootnode
 
     
    def wait_for_async_callback(self, result: Result):
        if result != Result.OK:
            return

        count = 0
        while count < 5:
            if self.waiting_for is None:
                return

            if count > 0:
                logging.debug('Waiting for %s %ss', self.waiting_for)

            count = count + 1
            time.sleep(1.0)

        logging.error("%s TIME OUT", self.waiting_for)

    def log_result(self, msg: str, result: Result):
        if result == Result.OK:
            logging.debug('%s --> %s', msg, result)
            return
        logging.error('%s failed with: %s', msg, result)

    def auth_token(self):

        logging.info("get_auth_token()")
        the_auth_token = self.client.get_auth_token()
        if the_auth_token is None:
            self.log_result('get_auth_token()', Result.FAILED)
            return

        logging.debug(the_auth_token)

        logging.info("set_auth_token()")
        self.client.set_auth_token(the_auth_token)

    def ping_async_callback(self, result: Result, data: Variant, userdata: ctrlxdatalayer.clib.userData_c_void_p):
        self.waiting_for = None
        logging.info(">>>ping_async_callback: %s", result)

    def ping(self):

        self.waiting_for = "ping_async_callback"
        logging.info("ping_async()")
        result = self.client.ping_async(self.ping_async_callback, 105)
        self.log_result('ping_async()', result)
        self.wait_for_async_callback(result)

        logging.info("ping_sync()")
        result = self.client.ping_sync()
        self.log_result('ping_sync()', result)

    def read_async_callback(self, result: Result, data: Variant, userdata: ctrlxdatalayer.clib.userData_c_void_p):
        self.waiting_for = None
        self.log_result(">>>read_async_callback(): ", result)
        self.print_data(">>>read_async_callback(): ", result, "", data)

    def read_sync(self, node: str):
        addressBase = root_node 
        address = addressBase + node

        logging.info("read_async() %s", address)
        result, data = self.client.read_sync(address)
        with data:
            self.log_result("read_sync() " + address, result)

            return self.returnData("read_async()", result, address, data),data.get_type()

        
    def get_typeaddress(self, address: str):

        result, metadata = self.client.metadata_sync(address)
        if result != Result.OK:
            print("ERROR Reading metadata of ", address, " failed with: ", result)
            return

        metadata_root = Metadata.Metadata.GetRootAsMetadata(
            metadata.get_flatbuffers())

        if metadata_root.ReferencesLength() == 0:
            print("ERROR Metadata references are empty")
            return

        for i in range(0, metadata_root.ReferencesLength()):
            reference = metadata_root.References(i)

            if reference is None:
                continue

            if reference.Type().decode('utf-8').lower() == "readtype":
                read_typeaddress = reference.TargetAddress().decode('utf-8')
                break

        return read_typeaddress
    
    def returnData(self, msg: str, result: Result, address: str, data: Variant):

        vt = data.get_type()

        if vt == VariantType.ARRAY_BOOL8:
            return data.get_array_bool8()

        if vt == VariantType.ARRAY_FLOAT32:
            return data.get_array_float32()

        if vt == VariantType.ARRAY_FLOAT64:
            return data.get_array_float64()

        if vt == VariantType.ARRAY_INT16:
            return data.get_array_int16()

        if vt == VariantType.ARRAY_INT32:
            return data.get_array_int32()

        if vt == VariantType.ARRAY_INT64:
            return data.get_array_int64()

        if vt == VariantType.ARRAY_INT8:
            return data.get_array_int8()

        if vt == VariantType.ARRAY_STRING:
            return data.get_array_string()

        if vt == VariantType.ARRAY_UINT16:
            return data.get_array_uint16()

        if vt == VariantType.ARRAY_UINT32:
            return data.get_array_uint32()

        if vt == VariantType.ARRAY_UINT64:
            return data.get_array_uint64()

        if vt == VariantType.ARRAY_UINT8:
            return data.get_array_uint8()

        if vt == VariantType.BOOL8:
            return data.get_bool8()

        if vt == VariantType.FLATBUFFERS:

            # Get type address for flatbuffers information
            typeAddress = self.get_typeaddress(address)
            if typeAddress is None:
                print("ERROR Type Address is none")
                return

            # Read type address as variant
            result, typeVar = self.client.read_sync(typeAddress)
            if result != Result.OK:
                print("ERROR Reading Type Value failed with: ", result)
                return

            # Convert variant flatbuffers data to json type
            result, json = converter.converter_generate_json_complex(data, typeVar, -1)
            if result != Result.OK:
                print("ERROR Converting json failed with: ", result)
                return

            return json.get_string()

        if vt == VariantType.FLOAT32:
            return data.get_float32()

        if vt == VariantType.FLOAT64:
            return data.get_float64()

        if vt == VariantType.INT16:
            return data.get_int16()

        if vt == VariantType.INT32:
            return data.get_int32()

        if vt == VariantType.INT64:
            return data.get_int64()

        if vt == VariantType.INT8:
            return data.get_int8()

        if vt == VariantType.STRING:
            return data.get_string()

        if vt == VariantType.UINT16:
            return data.get_uint16()

        if vt == VariantType.UINT32:
            return data.get_uint32()

        if vt == VariantType.UINT64:
            return data.get_uint64()

        if vt == VariantType.UINT8:
            return data.get_uint8()

        print("WARNING Unknown Variant Type:", vt)
        return None

 
    def create_async_callback(self, result: Result, data: Variant, userdata: ctrlxdatalayer.clib.userData_c_void_p):
        self.waiting_for = None
        logging.info(">>>create_async_callback(): %s %s %s",
                     result, data, userdata)

    def create_async(self, path, node, data: Variant):
        address = path + node
        # Remove node so that create will succeed
        result = self.client.remove_sync(address)  # Ignore error

        self.waiting_for = "create_async_callback"
        logging.info("create_async() %s", address)
        result = self.client.create_async(
            address, data, self.create_async_callback, 122)
        self.log_result("create_async()", result)
        self.wait_for_async_callback(result)

    def create_sync(self, path, node, data: Variant):
        address = path + node
        # Remove node so that create will succeed
        result = self.client.remove_sync(address)  # Ignore error

        logging.info("create_sync() %s", address)
        result, dataReturned = self.client.create_sync(address, data)
        # !!! dataReturned is a reference on data (dataReturned==data)
        self.log_result("create_sync() " + address, result)

    def create(self):
        with Variant() as data:

            addressBase = root_node + "/dynamic/_py/"

            data.set_bool8(True)
            self.create_sync(addressBase, "bool8", data)
            self.create_async(addressBase, "bool8", data)

            data.set_int8(-127)
            self.create_sync(addressBase, "int8", data)

            data.set_uint8(255)
            self.create_sync(addressBase, "uint8", data)

            data.set_int16(32767)
            self.create_sync(addressBase, "int16", data)

            data.set_uint16(65535)
            self.create_sync(addressBase, "uint16", data)

            data.set_int32(2147483647)
            self.create_sync(addressBase, "int32", data)

            data.set_uint32(4294967294)
            self.create_sync(addressBase, "uint32", data)

            data.set_int64(9223372036854775807)
            self.create_sync(addressBase, "int64", data)

            data.set_uint64(9223372036854775807 * 2)
            self.create_sync(addressBase, "uint64", data)

            data.set_float32(0.123456789)
            self.create_sync(addressBase, "float32", data)

            data.set_float64(0.987654321)
            self.create_sync(addressBase, "float64", data)

            data.set_string("This is string")
            self.create_sync(addressBase, "string", data)

            # Flatbuffers
            """
        def set_flatbuffers(self, data: bytearray) -> Result:
            buf = (ctypes.c_byte * len(data)).from_buffer(data)
            c_data = ctypes.cast(buf, ctypes.POINTER(ctypes.c_byte))
            return Result(libcomm_datalayer.DLR_variantSetFlatbuffers(self.c_variant, c_data, len(data)))
            """

            data.set_array_bool8([False, True, False])
            self.create_sync(addressBase, "array-of-bool8", data)

            data.set_array_int8([-127, -1, 0, 127])
            self.create_sync(addressBase, "array-of-int8", data)

            data.set_array_uint8([0, 127, 128, 255])
            self.create_sync(addressBase, "array-of-uint8", data)

            data.set_array_int16([-32767, -1, 0, 32767])
            self.create_sync(addressBase, "array-of-int16", data)

            data.set_array_uint16([0, 32767, 32768, 65535])
            self.create_sync(addressBase, "array-of-uint16", data)

            data.set_array_int32([-2147483647, -1, 0, 2147483647])
            self.create_sync(addressBase, "array-of-int32", data)

            data.set_array_uint32([0, 2147483647, 2147483648, 4294967295])
            self.create_sync(addressBase, "array-of-uint32", data)

            data.set_array_int64(
                [-9223372036854775807, -1, 0, 9223372036854775807])
            self.create_sync(addressBase, "array-of-int64", data)

            data.set_array_uint64(
                [0, 9223372036854775807, 9223372036854775808, 18446744073709551615])
            self.create_sync(addressBase, "array-of-uint64", data)

            data.set_array_float32([32.1, 32.2, 32.3, 32.4])
            self.create_sync(addressBase, "array-of-float32", data)

            data.set_array_float64([64.1, 64.2, 64.3, 64.4])
            self.create_sync(addressBase, "array-of-float64", data)

            data.set_array_string(["Red", "Green", "Yellow", "Blue"])
            self.create_sync(addressBase, "array-of-string", data)

    def remove_async_callback(self, result: Result, data: Variant, userdata: ctrlxdatalayer.clib.userData_c_void_p):
        self.waiting_for = None
        logging.info(">>>create_async_callback(): %s %s", result, userdata)

    def remove(self):

        with Variant() as data:

            addressBase = root_node + "/dynamic/_py/"
            addressNode = "xxx"
            address = addressBase + addressNode
            data.set_string("Will be removed synch")
            self.client.create_sync(address, data)

            logging.info("remove_sync() %s", address)
            result = self.client.remove_sync(address)
            self.log_result("remove_sync()" + address, result)

            self.client.create_sync(address, data)

            logging.info("remove_async() %s", address)
            self.waiting_for = "remove_async_callback"
            result = self.client.remove_async(
                address, self.remove_async_callback, 243)
            self.log_result("remove_async()", result)
            self.wait_for_async_callback(result)

    def browse_async_callback(self, result: Result, data: Variant, userdata: ctrlxdatalayer.clib.userData_c_void_p):
        self.waiting_for = None
        logging.info(">>>browse_async_callback: %s %s %s",
                     result, data.get_array_string(), userdata)

    def browse(self, node: str):

        logging.info("browse_sync() /")
        result, data = self.client.browse_sync(node)
        return data.get_array_string(), result

    def write_async_callback(self, result: Result, data: Variant, userdata: ctrlxdatalayer.clib.userData_c_void_p):
        self.waiting_for = None
        self.log_result(">>>write_async_callback:", result)

    def write_sync(self, node: str, data: Variant):
        logging.info("write_sync() %s", node)
        result, _ = self.client.write_sync(node, data)
        self.log_result("write_sync()", result)
        return result

    def write(self):
        with Variant() as data:

    

            data.set_bool8(False)
            self.write_sync(addressBase, "bool8", data)

            data.set_float32(-0.123456789)
            self.write_sync(addressBase, "float32", data)

            data.set_float64(-0.987654321)
            self.write_sync(addressBase, "float64", data)

            data.set_int8(-127)
            self.write_sync(addressBase, "int8", data)

            data.set_int16(-32767)
            self.write_sync(addressBase, "int16", data)

            data.set_int32(0x80000001)
            self.write_sync(addressBase, "int32", data)

            data.set_int64(0x8000000000000001)
            self.write_sync(addressBase, "int64", data)

            data.set_string("Changed by python Data Layer Client")
            self.write_sync(addressBase, "string", data)

    def print_metadata(self, text: str, result: Result, data: Variant):
        if result != Result.OK:
            logging.error("%s failed with %s", text, result)
            return

        if data is None:
            logging.error("%s failed: data is None", text)
            return

        logging.info("%s %s", text, result)

        # Print Metadata (Flatbuffers)
        metadata = Metadata.Metadata.GetRootAsMetadata(
            data.get_flatbuffers(), 0)
        allowedoperations = metadata.Operations()
        print("metadata.NodeClass()", metadata.NodeClass(),
              " allowedOperations",
              "read=", allowedoperations.Read(),
              "write=", allowedoperations.Write(),
              "create=", allowedoperations.Create(),
              "delete=", allowedoperations.Delete(),
              "metadata.DisplayName()", metadata.DisplayName(),
              "metadata.DisplayFormat()", metadata.DisplayFormat())

    def metadata_async_callback(self, result: Result, data: Variant, userdata: ctrlxdatalayer.clib.userData_c_void_p):
        self.waiting_for = None
        self.print_metadata(">>>metadata_async_callback", result, data)


    def metadata(self):

        address = root_node + "dynamic/bool8"
        self.waiting_for = "metadata_async_callback"
        logging.info("metadata_async() %s", address)
        result = self.client.metadata_async(
            address, self.metadata_async_callback, 490)
        self.log_result("metadata_async()", result)
        self.wait_for_async_callback(result)

        logging.info("metadata_sync() %s", address)
        result, data = self.client.metadata_sync(address)
        with data:
            self.log_result("metadata_async()", result)
            self.print_metadata("metadata_sync() " + address, result, data)

        address = root_node + "/static/bool8"
        logging.info("metadata_sync() %s", address)
        result, data = self.client.metadata_sync(address)
        with data:
            self.log_result("metadata_async()", result)
            self.print_metadata("metadata_sync() " + address, result, data)
    
    
    #AuxFunctions



    def get_client(ip="192.168.1.1", user="boschrexroth", password="boschrexroth",ssl_port=443):
        system=DatalayerSystem
        DatalayerSystem.start(False)
        connection_string = get_connection_string(ip, user, password, ssl_port)
        client = system.factory().create_client(connection_string)
        if client.is_connected():
            print("INFO Connecting", connection_string, "succeeded.")
            return client, connection_string
        client.close()
        print("WARNING Connecting", connection_string, "failed.")
        datalayer_system.stop(False)
        sys.exit(1)
        return None, connection_string
    
    def get_provider(self,system: ctrlxdatalayer.system.System,ip="192.168.1.1",user="boschrexroth",password="boschrexroth",ssl_port=443):
        connection_string = get_connection_string(Ip, user, password, ssl_port)
        provider = system.factory().create_provider(connection_string)
        if (provider.start() == ctrlxdatalayer.variant.Result.OK) & provider.is_connected():
            return provider, connection_string
        provider.close()

        return None, connection_string
